/*
** 1998-11-26 -	A neat little module to manage directory histories in the panes. Initially, this
**		will just remember the actual paths, but future versions might include storing
**		info about selected files, sizes, and so on...
** 1998-12-06 -	Rewritten after a suggestion from J. Hanson (<johan@tiq.com>), to simply hash on
**		the inode numbers rather than the file names. Should reduce time and memory complex-
**		ities by a whole lot.
** 1998-12-15 -	Now stores the vertical position in a relative way, rather than using absolute pixel
**		coordinates. Not good, but better.
** 1998-12-23 -	Eh. Inode numbers are _not_ unique across devices (which might be why there's
**		a st_dev field in the stat structure). This of course makes them a bad choice
**		for unique file identifiers - when buffering a directory containing inodes from
**		various devices (such as / typically does), things went really haywire. To fix
**		this, we now store the device identifier too. The pair (device,inode) really should
**		be unique.
** 1999-09-05 -	Cut away the fancy browser-esque history system implemented for 0.11.8, since it wasn't
**		complete, and I didn't like it much. Going back to a simple combo as before.
** 1999-11-13 -	Prettied up handling of vertical position remembering somewhat, exported it separately.
*/

#include "gentoo.h"

#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>

#include "odemilbutton.h"

#include "cmdseq.h"
#include "dirpane.h"
#include "errors.h"
#include "fileutil.h"
#include "strutil.h"

#include "dirhistory.h"

/* ----------------------------------------------------------------------------------------- */

#define	HISTORY_SIZE	(16)	/* Maximum number of items in history. Should be dynamic. */

/* ----------------------------------------------------------------------------------------- */

typedef struct {				/* A history key. Just a (device,inode) pair. */
	dev_t	dev;
	ino_t	inode;
} HKey;

/* This is used to store the selected files in a pane, and nothing else. */
struct DHSel {
	GHashTable	*hash;			/* It's a hash of HKeys, as above. For fast 'apply' operations. */
};

/* Info about a single remembered directory. The 'hist' field of dirpanes stores a linked
** list of these.
*/
typedef struct {
	gchar		path[PATH_MAX];		/* The path to this directory. Must be first! */
	HKey		key;			/* The directory's unique identifier. */
	DHSel		*sel;			/* The selection last time we were here. */
	gfloat		vpos;			/* Vertical position. */
	gint		focus_row;		/* Row that had the keyboard-controlled focus. */
	GTimeVal	a_time;
} DHEntry;

struct DirHistory {
	DHEntry	*current;		/* Description of current directory. */
	GList	*history;		/* Visited directories. Recent are close to head. */
};

/* Here's a glib GMemChunk we use to allocate HKeys, in the hope of getting better efficiency
** than g_malloc() would give us. Note that there is just one such GMemChunk, from which we
** allocate HKeys for _all_ directories (DHEntries).
*/
static GMemChunk	*the_chunk = NULL;

/* ----------------------------------------------------------------------------------------- */

static DHSel *	dirsel_set(DHSel *sel, DirPane *dp);

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-08 -	Create a new dirpane history structure. */
DirHistory * dph_dirhistory_new(void)
{
	DirHistory	*dh;

	dh = g_malloc(sizeof *dh);
	dh->current = NULL;
	dh->history = NULL;

	return dh;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-12-23 -	Set a history key according to information in <st>. */
static void hkey_set(HKey *hk, struct stat *st)
{
	if(hk != NULL && st != NULL)
	{
		hk->dev   = st->st_dev;
		hk->inode = st->st_ino;
	}
}

/* 1999-06-05 -	Set a key from a name. Returns FALSE on failure, TRUE on success. */
static gboolean hkey_set_from_name(HKey *hk, const gchar *path)
{
	struct stat	stbuf;

	if(stat(path, &stbuf) == 0)
	{
		hkey_set(hk, &stbuf);
		return TRUE;
	}
	return FALSE;
}

/* 1998-12-23 -	Create a new history key, initialized with information from <st>. */
static HKey * hkey_new(struct stat *st)
{
	HKey	*hk = NULL;

	if(the_chunk == NULL)
		the_chunk = g_mem_chunk_new("HKey", sizeof *hk, 1024, G_ALLOC_AND_FREE);
	if(the_chunk != NULL)
	{
		if((hk = g_mem_chunk_alloc(the_chunk)) != NULL)
			hkey_set(hk, st);
	}
	return hk;
}

/* 1998-12-23 -	Compare two history keys for equality. */
static gint hkey_equal(gconstpointer a, gconstpointer b)
{
	HKey	*ha = (HKey *) a, *hb = (HKey *) b;

	return (ha->dev == hb->dev) && (ha->inode == hb->inode);
}

/* 1998-12-23 -	Compute hash value from a history key. Nothing fancy. */
static guint hkey_hash(gconstpointer a)
{
	HKey	*ha = (HKey *) a;

	return (guint) ha->dev ^ (guint) ha->inode;
}

/* 1998-12-23 -	A g_hash_table_foreach() callback that frees a hash key. Note that both <key>
**		and <value> point at the same HKey structure here.
*/
static void hkey_free(gpointer key, gpointer value, gpointer user)
{
	g_mem_chunk_free(the_chunk, key);
}

/* 1999-06-06 -	A routine just like the one above, but for g_hash_table_foreach_remove(). */
static gboolean hkey_free2(gpointer key, gpointer value, gpointer user)
{
	g_mem_chunk_free(the_chunk, key);

	return TRUE;		/* Remove me, please. */
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-05 -	Create a new empty DHEntry structure. */
static DHEntry * dhentry_new(void)
{
	DHEntry	*data;

	data = g_malloc(sizeof *data);
	data->path[0] = '\0';
	data->key.dev = data->key.inode = 0;
	data->sel = NULL;
	data->vpos = 0.0f;
	data->focus_row = -1;

	return data;
}

/* 1999-06-09 -	Update <dhe> to mimic state of <dp>. */
static void dhentry_update(DHEntry *dhe, DirPane *dp)
{
	dhe->sel  = dirsel_set(dhe->sel, dp);
	dhe->vpos = dph_vpos_get(dp);
	if(dp->focus_row != -1)
		dhe->focus_row = dp->focus_row;
	else
		dhe->focus_row = -1;
	g_get_current_time(&dhe->a_time);
}

/* 1999-06-09 -	Apply state conserved in <dhe> to the rows of <dp>. */
static void dhentry_apply(DHEntry *dhe, DirPane *dp)
{
	dph_dirsel_apply(dp, dhe->sel);
	dph_vpos_set(dp, dhe->vpos);
	if(dp->old_focus_row != -1)	/* Was focus active? */
	{
		gint	row = dhe->focus_row;

		if(row == -1)
			row = 0;
		dp_focus(dp, row);
	}
	g_get_current_time(&dhe->a_time);
}

/* 1999-06-06 -	Destroy a directory history entry. */
static void dhentry_destroy(DHEntry *dh)
{
	if(dh->sel != NULL)
		g_hash_table_foreach_remove(dh->sel->hash, hkey_free2, NULL);
	g_free(dh->sel);
	g_free(dh);
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-05 -	Create a new empty selection. */
static DHSel * dirsel_new(void)
{
	DHSel	*sel;

	sel = g_malloc(sizeof *sel);
	sel->hash = g_hash_table_new(hkey_hash, hkey_equal);

	return sel;
}

static gint dummy_func(gpointer a, gpointer b, gpointer c)
{
	return TRUE;
}

static void dirsel_clear(DHSel *sel)
{
	if((sel == NULL) || (sel->hash == NULL))
		return;

	g_hash_table_foreach_remove(sel->hash, dummy_func, NULL);
}

/* 1999-03-05 -	Given an <sel> structure, replace selection with that of <dp>. Returns the
**		new selection (or NULL if <dp> had no selected items). If <sel> is NULL on
**		entry, a new selection will be created and returned.
*/
static DHSel * dirsel_set(DHSel *sel, DirPane *dp)
{
	GSList	*slist;

	if((slist = dp_get_selection_full(dp)) != NULL)
	{
		HKey	*key;
		GSList	*iter;

		if(sel == NULL)
			sel = dirsel_new();
		else
			dirsel_clear(sel);
		
		for(iter = slist; iter != NULL; iter = g_slist_next(iter))
		{
			key = hkey_new(&DP_SEL_LSTAT(iter));
			g_hash_table_insert(sel->hash, key, key);	/* *Value* is returned on lookup!! */
		}
		dp_free_selection(slist);
		return sel;
	}
	return NULL;
}

/* 1999-03-05 -	This returns an opaque representation of all selected rows of <dp>. The
**		selection is not related to the order in which these rows are displayed
**		in the pane, so it's handy to use before e.g. resorting the pane.
**		Note that NULL is a valid representation if there is no selection.
*/
DHSel * dph_dirsel_new(DirPane *dp)
{
	return dirsel_set(NULL, dp);
}

/* 1999-03-05 -	Apply given given <sel> selection to <dp>, making those rows selected
**		again. <dp> need not be the same as when the selection was created,
**		and it need not have the same contents. This is not terribly efficient,
**		but I think it'll be OK.
*/
void dph_dirsel_apply(DirPane *dp, DHSel *sel)
{
	HKey	key;
	guint	i;

	if((sel == NULL) || (sel->hash == NULL))
		return;

	for(i = 0; i < dp->dir.num_rows; i++)
	{
		hkey_set(&key, &DP_ROW_LSTAT(&dp->dir.row[i]));
		if(g_hash_table_lookup(sel->hash, &key))
			dp_select(dp, i);
	}
}

/* 1999-03-05 -	Destroy a selection, this is handy when you're done with it (like after
**		having applied it).
*/
void dph_dirsel_destroy(DHSel *sel)
{
	if(sel == NULL)
		return;

	if(sel->hash != NULL)
	{
		g_hash_table_foreach(sel->hash, hkey_free, NULL);
		g_hash_table_destroy(sel->hash);
	}
	g_free(sel);
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-11-13 -	Return a floating-point number that somehow encodes the vertical position of
**		the given pane. The only use for that number is to pass it to dph_vpos_set().
*/
gfloat dph_vpos_get(DirPane *dp)
{
	GtkAdjustment	*adj;

	if((adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(dp->scwin))) != NULL)
		return adj->value;
	return -1.0f;
}

/* 1999-11-13 -	Set the given pane's vertical position to resemble what is was when <vpos>
**		was returned by dph_vpos_get().
*/
void dph_vpos_set(DirPane *dp, gfloat vpos)
{
	GtkAdjustment	*adj;

	if((vpos >= 0.0f) && (adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(dp->scwin))) != NULL)
		gtk_adjustment_set_value(adj, vpos);
}

/* ----------------------------------------------------------------------------------------- */

#if 0
/* Here's a handy history display routine, for debugging. */
static void hist_dump(DirHistory *dh)
{
	GList	*iter;

	printf("(");
	for(iter = dh->backward; iter != NULL; iter = g_list_next(iter))
	{
		printf("%s", ((DHEntry *) iter->data)->path);
		if(g_list_next(iter))
			putchar(' ');
	}
	printf(") '%s' (", ((DHEntry *) dh->current)->path);
	for(iter = dh->forward; iter != NULL; iter = g_list_next(iter))
	{
		printf("%s", ((DHEntry *) iter->data)->path);
		if(g_list_next(iter))
			putchar(' ');
	}
	printf(") [");
	for(iter = dh->cache; iter != NULL; iter = g_list_next(iter))
	{
		printf("%s", ((DHEntry *) iter->data)->path);
		if(g_list_next(iter))
			putchar(' ');
	}
	printf("]\n");
}
#endif

/* 1999-06-06 -	Enter the directory specified by <entry>, and apply any historic information
**		it may contain.
** 2000-11-30 -	Did tricky resize-queuing + event-gobbling before setting vpos. Better.
*/
static void enter_and_apply(DirPane *dp, DHEntry *entry)
{
	dp_freeze(dp);
	dp_enter_dir(dp, entry->path);
	dp_redisplay(dp);
	dp_thaw(dp);
	gtk_widget_queue_resize(GTK_WIDGET(dp->list));
	while(gtk_events_pending())
		gtk_main_iteration();
	dhentry_apply(entry, dp);
	dp->hist->current = entry;
}

/* 1999-06-05 -	Enter <new_path> in <dp>, and update the history data accordingly. If the <new_path>
**		isn't stat()-able, return FALSE while setting errno to something sensible.
** 1999-09-05 -	Rewritten to deal with a simple, single, history list instead.
*/
gboolean dph_direnter(DirPane *dp, const gchar *new_path)
{
	DHEntry	*entry = NULL;
	HKey	nkey;
	GList	*iter;

	/* First make (reasonably) sure we can really enter the directory. */
	if(access(new_path, R_OK) != 0 || !hkey_set_from_name(&nkey, new_path))
		return FALSE;

	/* Save current pane state in the current history entry. */
	if(dp->hist->current != NULL)
		dhentry_update(dp->hist->current, dp);

	/* Find the history entry that describes the target directory, if there is one. */
	for(iter = dp->hist->history; iter != NULL; iter = g_list_next(iter))
	{
		entry = iter->data;
		if(hkey_equal(&nkey, &entry->key))
		{
			dp->hist->history = g_list_remove_link(dp->hist->history, iter);
			g_list_free_1(iter);
			break;
		}
	}

	/* No old entry found? Then construct a new one. */
	if(iter == NULL)
		entry = dhentry_new();

	stu_strncpy(entry->path, new_path, sizeof entry->path);
	entry->key = nkey;

	dp->hist->history = g_list_prepend(dp->hist->history, entry);

	/* Has the list grown too long? Then we prune the end off. */
	if(g_list_length(dp->hist->history) > HISTORY_SIZE)
	{
		GList	*tail, *next;

		tail = g_list_nth(dp->hist->history, HISTORY_SIZE);
		for(; tail != NULL; tail = next)
		{
			next = g_list_next(tail);
			dp->hist->history = g_list_remove_link(dp->hist->history, tail);
			dhentry_destroy(tail->data);
			g_list_free_1(tail);
		}
	}
	enter_and_apply(dp, entry);

	return TRUE;
}

/* 1999-09-05 -	Rescan the directory currently in <dp>. Implemented lazily. */
gboolean dph_dirrescan(DirPane *dp)
{
	if(dph_direnter(dp, dp->dir.path))
	{
		dph_set_path_widgets(dp);
		return TRUE;
	}
	return FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-25 -	Update path-related widgets for <dp>. This is a little bit besides the core "point" of this module
**		perhaps, but since the data is so incredibly opaque, this has to go here. Big deal.
*/
void dph_set_path_widgets(DirPane *dp)
{
	DirHistory	*hist = dp->hist;

	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dp->path)->entry), hist->current->path);
	gtk_editable_changed(GTK_EDITABLE(GTK_COMBO(dp->path)->entry));
	gtk_combo_set_popdown_strings(GTK_COMBO(dp->path), hist->history);	/* Since path is first in DHEntry, this works. */
}
