static char rcsid[] = "$Id: tabchannel.c,v 1.8 1997/05/29 08:35:19 dhb Exp $";

/*
** $Log: tabchannel.c,v $
** Revision 1.8  1997/05/29 08:35:19  dhb
** Update from Antwerp GENESIS version 21e
**
 * EDS21e revison: EDS BBF-UIA 97/05/03
 * Added code to check if EREST_ACT exists
 *
 * EDS21c revison: EDS BBF-UIA 95/08/02-95/10/05
 * Added ADD_GBAR msg and updated tabcurrent in TabChannel_SAVE and TabChannel_READ
 *
 * EDS20j revison: EDS BBF-UIA 95/07/24-95/08/11
 * Added DOMAINCONC msg and surface field
 * Updated ShowInterpol function call
 * Minor corrections to TabChannel_READ and TabChannel_SAVE
 *
 * EDS20i revison: EDS BBF-UIA 95/06/02
 * Added TAB_SAVE and TAB_READ actions.
 * Added olf_defs.h
 *
 * Revision 1.7  1996/08/06  20:52:43  dhb
 * Use GetScriptDouble() rather than get_script_float().
 *
 * Revision 1.6  1995/03/23  01:37:06  dhb
 * Added number of values to SAVE2 file format
 *
 * Revision 1.5  1995/03/21  01:00:46  venkat
 * Upi changes: Typecasted second argument to TabInterp() to double
 *
 * Revision 1.4  1994/10/04  23:01:21  dhb
 * Added TABDELETE action.
 *
 * Revision 1.3  1994/08/08  22:31:03  dhb
 * Changes from Upi.
 *
 * Revision 1.4  1994/06/13  22:52:24  bhalla
 * Added the SHOW and DUMP actions
 *
 * Revision 1.3  1994/05/26  13:52:10  bhalla
 * Changed syntax for SetTable
 *
 * Revision 1.2  1994/03/22  18:13:18  dhb
 * Change by Upi Bhalla for setting tables.
 *
 * Revision 1.1  1992/11/14  00:37:22  dhb
 * Initial revision
 *
*/

#include "olf_ext.h"
#include "olf_defs.h"
#include "olf_g@.h"
#include "result.h"
#include "symtab.h"

static double       savedata[3];

/* E. De Schutter Caltech 1/91, modified Z 4/91 */
/* Tabulated hh-like channel.  The standard voltage dependent 
**	activation (X) and inactivation (Y) factors are present. For
**	concentration dependent processes a third factor (Z) has
**	added.  Z can do a lot of things, most important is that 
**	it gets another message (type1: c) than X and Y (type 0: v). 
**	Standard use for Z is inactivation, just use a Zpower>0.  
**	Alternative is to use it for codependent activation, use
**	Zpower<0, z will equal c times v-dependent table-value.
** For computation efficiency the forward rate factor alpha (A) and 
**	1/tau (B) are tabulated. 
*/
/* debugged, TABFILL added, CALC_MINF, CALC_ALPHA, CALC_BETA added
** by U.S. Bhalla Caltech 2/91 */
/*
** Generalized integration scheme implemented July 1991 by Upi Bhalla 
*/

int TabChannel(channel,action)
register struct tab_channel_type *channel;
Action		*action;
{
double 	c,v;
double	A,B;
double	dy;
double	g;
double	dt;
int     int_method;
char    *field;
int     xdivs;
float   xmin,xmax;
Interpol *create_interpol();
short	fill_mode;
MsgIn	*msg;
extern double GetScriptDouble();
int	n;
Result  *rp,*SymtabLook();
extern Symtab GlobalSymbols;

    if(debug > 1){
	ActionHeader("tabchannel",channel,action);
    }

    SELECT_ACTION(action){
    case INIT:
	channel->activation = 0;
	break;

    case PROCESS:
	dt = Clockrate(channel);
	g = channel->Gbar;
	MSGLOOP(channel,msg) {
	case VOLTAGE:		/* membrane VOLTAGE */
	    /* 0 = membrane potential */
	    v = MSGVALUE(msg,0);
	    break;
	case CONCEN1:		/* ionic CONCENTRATION */
	    /* 0 = concentration */
	    c = MSGVALUE(msg,0);
	    break;
	case DOMAINCONC:	/* domain CONCENTRATION: is a scaled Ik */
	    /* 0 = Ik */
	    /* 1 = scaling factor */
	    c = MSGVALUE(msg,0)*MSGVALUE(msg,1)/channel->surface;
	    break;
	case EK:		/* reversal potential EK */
	    /* 0 = EK */
		channel->Ek = MSGVALUE(msg,0);
		break;
        case ADD_GBAR:
            /* 0 = scale_factor */
      	    g = MSGVALUE(msg,0) + g;
       	    break;
	}

	/* calculate the active state variables x, y and z and
	**  and calculate the conductance */
	int_method = (int)(channel->object->method);
	if(channel->Xpower != 0){
	    A = TabInterp(channel->X_A,v);
	    B = TabInterp(channel->X_B,v);
	    channel->X = IntegrateMethod(int_method,channel,
			channel->X,A,B,dt,"X");
	    g *= pow(channel->X,channel->Xpower);
	}
	if(channel->Ypower != 0){
	    A = TabInterp(channel->Y_A,v);
	    B = TabInterp(channel->Y_B,v);
	    channel->Y = IntegrateMethod(int_method,channel,
			channel->Y,A,B,dt,"Y");
	    g *= pow(channel->Y,channel->Ypower);
	}
	if(channel->Zpower > 0){
	    A = TabInterp(channel->Z_A,c);
	    B = TabInterp(channel->Z_B,c);
	    channel->Z = IntegrateMethod(int_method,channel,
			channel->Z,A,B,dt,"Z");
	    g *= pow(channel->Z,channel->Zpower);
	} else if(channel->Zpower < 0){
	    A = c * TabInterp(channel->Z_A,v);
	    B = TabInterp(channel->Z_B,v);
	    channel->Z = IntegrateMethod(int_method,channel,
			channel->Z,A,B,dt,"Z");
	    g *= pow(channel->Z,-channel->Zpower);
	}

	/* calculate the transmembrane current */
	channel->Gk = g;
	channel->Ik = (channel->Ek - v) * g;
	break;

    case RESET:
	channel->activation = 0;
	rp=SymtabLook(&GlobalSymbols,"EREST_ACT"); /* check if script global exists */
	if (rp) v = GetScriptDouble("EREST_ACT");
	c = 0;
	/* calculate the conductance at Erest */
	g = channel->Gbar;
	MSGLOOP(channel,msg) {
	    case VOLTAGE:		/* Vm */
			v = MSGVALUE(msg,0);
			break;
	    case CONCEN1:		/* Co */
			c = MSGVALUE(msg,0);
			break;
	    case DOMAINCONC:		/* domain CONCENTRATION: is a scaled Ik */
			c = MSGVALUE(msg,0)*MSGVALUE(msg,1)/channel->surface;
			break;
            case ADD_GBAR:
      			g = MSGVALUE(msg,0) + g;
       			break;
	}
	/* calculate the active state variables x, y and z at steady 
	**  state and and calculate the conductance */
	if(channel->Xpower != 0){
	    A = TabInterp(channel->X_A,v);
	    B = TabInterp(channel->X_B,v);
	    channel->X = A / B;
	    g = g * pow(channel->X,channel->Xpower);
	}
	if(channel->Ypower != 0){
	    A = TabInterp(channel->Y_A,v);
	    B = TabInterp(channel->Y_B,v);
	    channel->Y = A / B;
	    g = g * pow(channel->Y,channel->Ypower);
	}
	if(channel->Zpower > 0){
	    A = TabInterp(channel->Z_A,c);
	    B = TabInterp(channel->Z_B,c);
	    channel->Z = A / B;
	    g *= pow(channel->Z,channel->Zpower);
	} else if(channel->Zpower < 0){
	    A = c * TabInterp(channel->Z_A,v);
	    B = TabInterp(channel->Z_B,v);
	    channel->Z = A / B;
	    g *= pow(channel->Z,-channel->Zpower);
	}
	channel->Gk = g;
	channel->Ik = (channel->Ek - v) * g;
	break;

    case CHECK:
	v = 0;
	c = 0;
	MSGLOOP(channel,msg) {
	    case VOLTAGE:		/* Vm */
			v = 1;
			break;
	    case CONCEN1:		/* Co */
			c = c+1;
			break;
	    case DOMAINCONC:		/* Co */
			c = c+1;
			if (channel->surface<=0) {
				ErrorMessage("tabchannel","surface field not set.",channel);
			}
			break;
	}
	if(v == 0){
	    ErrorMessage("tabchannel","Missing VOLTAGE msg.",channel);
	}
	if(c > 1){
	    ErrorMessage("tabchannel","CONCEN msg cannot be combined with DOMAINCONC msg.",channel);
	}
	if((channel->Zpower != 0) && (c == 0)){
	    ErrorMessage("tabchannel","Missing CONCEN msg.",channel);
	}
	if ((channel->Xpower != 0 && !channel->X_alloced) || 
	    (channel->Ypower != 0 && !channel->Y_alloced) ||
	    (channel->Zpower != 0 && !channel->Z_alloced)) { 
            ErrorMessage("tabchannel", "Factor tables not allocated.",channel);
	}
	break;
    case SET :
        if (action->argc != 2)
            return(0); /* do the normal set */
		if (SetTable(channel,2,action->argv,action->data,
			"X_A X_B Y_A Y_B Z_A Z_B"))
			return(1);
        if (strncmp(action->argv[0],"X_A",3) == 0)
            scale_table(channel->X_A,action->argv[0] + 5,action->argv[1]);
        else if (strncmp(action->argv[0],"X_B",3) == 0)
            scale_table(channel->X_B,action->argv[0] + 5,action->argv[1]);
        else if (strncmp(action->argv[0],"Y_A",3) == 0)
            scale_table(channel->Y_A,action->argv[0] + 5,action->argv[1]);
        else if (strncmp(action->argv[0],"Y_B",3) == 0)
            scale_table(channel->Y_B,action->argv[0] + 5,action->argv[1]);
            return(0); /* do the normal set */
		break;
    case SHOW:
		if (ShowInterpol(action,"X_A",channel->X_A,channel->X_alloced,100)) 
			return(1);
		if (ShowInterpol(action,"X_B",channel->X_B,channel->X_alloced,100)) 
			return(1);
		if (ShowInterpol(action,"Y_A",channel->Y_A,channel->Y_alloced,100)) 
			return(1);
		if (ShowInterpol(action,"Y_B",channel->Y_B,channel->Y_alloced,100)) 
			return(1);
		if (ShowInterpol(action,"Z_A",channel->Z_A,channel->Z_alloced,100)) 
			return(1);
		return(ShowInterpol(action,"Z_B",channel->Z_B,channel->Z_alloced,100));
		break;
    case SAVE2:
	savedata[0] = channel->X;
	savedata[1] = channel->Y;
	savedata[2] = channel->Z;
	/* action->data contains the file pointer */
        n=3;
        fwrite(&n,sizeof(int),1,(FILE*)action->data);
	fwrite(savedata,sizeof(double),3,(FILE*)action->data);
	break;

    case RESTORE2:
	/* action->data contains the file pointer */
        fread(&n,sizeof(int),1,(FILE*)action->data);
        if (n != 3) {
            ErrorMessage("TabChannel", "Invalid savedata length", channel);
            return n;
        }
	fread(savedata,sizeof(double),3,(FILE*)action->data);
	channel->X = savedata[0];
	channel->Y = savedata[1];
	channel->Z = savedata[2];
	break;
	case DUMP:
	/* action->data contains the file pointer */
    	if (action->argc == 1) {
        	if (strcmp(action->argv[0],"pre") == 0) {
            	return(0);
        	}
        	if (strcmp(action->argv[0],"post") == 0) {
            	FILE *fp = (FILE *)action->data ;
            	DumpInterpol(fp,channel,channel->X_A,"X_A");
            	DumpInterpol(fp,channel,channel->X_B,"X_B");
            	DumpInterpol(fp,channel,channel->Y_A,"Y_A");
            	DumpInterpol(fp,channel,channel->Y_B,"Y_B");
            	DumpInterpol(fp,channel,channel->Z_A,"Z_A");
            	DumpInterpol(fp,channel,channel->Z_B,"Z_B");
            	return(0);
        	}
    	}
	break;
    case TABCREATE:
	if (action->argc < 4) {
		printf("usage : %s field xdivs xmin xmax\n","tabcreate");
		return(0);
	}
	field = action->argv[0];
	xdivs = atoi(action->argv[1]);
/* mds3 changes */
	xmin = Atof(action->argv[2]);
	xmax = Atof(action->argv[3]);
	if (strcmp(field,"X") == 0) {
		channel->X_A = create_interpol(xdivs,xmin,xmax);
		channel->X_B = create_interpol(xdivs,xmin,xmax);
		channel->X_alloced = 1;
	} else if (strcmp(field,"Y") == 0) {
		channel->Y_A = create_interpol(xdivs,xmin,xmax);
		channel->Y_B = create_interpol(xdivs,xmin,xmax);
		channel->Y_alloced = 1;
	} else if (strcmp(field,"Z") == 0) {
		channel->Z_A = create_interpol(xdivs,xmin,xmax);
		channel->Z_B = create_interpol(xdivs,xmin,xmax);
		channel->Z_alloced = 1;
	} else {
		printf("field '%s' not known\n",field);
		return(0);
	}
	break;
    case TABDELETE: /* MCV's fix to delete tabchannel interpol memory
		    ** NOTE: Can't do this in a DELETE action since we
		    ** don't know if some other table might be a reference
		    ** to this one.  Need reference counting.
		    */
	if (channel->X_alloced) {
	    free(channel->X_A->table);
	    free(channel->X_A);
	    channel->X_A = NULL;
	    free(channel->X_B->table);
	    free(channel->X_B);
	    channel->X_B = NULL;
	    channel->X_alloced = 0;
	}
	if (channel->Y_alloced) {
	    free(channel->Y_A->table);
	    free(channel->Y_A);
	    channel->Y_A = NULL;
	    free(channel->Y_B->table);
	    free(channel->Y_B);
	    channel->Y_B = NULL;
	    channel->Y_alloced = 0;
	}
	if (channel->Z_alloced) {
	    free(channel->Z_A->table);
	    free(channel->Z_A);
	    channel->Z_A = NULL;
	    free(channel->Z_B->table);
	    free(channel->Z_B);
	    channel->Z_B = NULL;
	    channel->Z_alloced = 0;
	}
	break;
    case TABFILL:
        if (action->argc < 3) {
            printf("usage : %s field xdivs fill_mode\n","tabfill");
            return(0);
        }
        field = action->argv[0];
        xdivs = atoi(action->argv[1]);
        fill_mode = atoi(action->argv[2]);
        if (strcmp(field,"X") == 0) {
            fill_table(channel->X_A,xdivs,fill_mode);
            fill_table(channel->X_B,xdivs,fill_mode);
        } else if (strcmp(field,"Y") == 0) {
            fill_table(channel->Y_A,xdivs,fill_mode);
            fill_table(channel->Y_B,xdivs,fill_mode);
        } else if (strcmp(field,"Z") == 0) {
            fill_table(channel->Z_A,xdivs,fill_mode);
            fill_table(channel->Z_B,xdivs,fill_mode);
        } else {
            printf("field '%s' not known\n",field);
            return(0);
        }
        break;
    }
    return(0);
}

TabChannel_CALC_MINF(channel,action)
register struct tab_channel_type *channel;
Action      *action;
{
double	alpha,beta;
char	*gate;
Interpol	*ipa,*ipb;
double	m;

    if(action->argc == 2){
	gate = action->argv[0];
		if (strcmp(gate,"X") == 0) {
			ipa = channel->X_A;
			ipb = channel->X_B; 
		} else if (strcmp(gate,"Y") == 0) {
			ipa = channel->Y_A;
			ipb = channel->Y_B;
		} else if (strcmp(gate,"Z") == 0) {
			ipa = channel->Z_A;
			ipb = channel->Z_B;
		}
    	channel->activation = Atof(action->argv[1]);
    } else {
    Error();
    printf("usage : CALC_MINF gate voltage\n");
    }
    /*
    ** calculate the steady state value of the state variable
    */
	alpha = TabInterp(ipa,(double)(channel->activation));
	/* remember that in tabchannels beta is 1/tau */
	beta = TabInterp(ipb,(double)(channel->activation));
    m = alpha/beta;
    action->passback = ftoa(m);
}


TabChannel_CALC_ALPHA(channel,action)
register struct tab_channel_type *channel;
Action		*action;
{
double	alpha;
char	*gate;
Interpol	*ip;

    if(action->argc == 2){
	gate = action->argv[0];
		if (strcmp(gate,"X") == 0) {
			ip = channel->X_A;
		} else if (strcmp(gate,"Y") == 0) {
			ip = channel->Y_A;
		} else if (strcmp(gate,"Z") == 0) {
			ip = channel->Z_A;
		}
    	channel->activation = Atof(action->argv[1]);
    } else {
    Error();
    printf("usage : CALC_ALPHA gate voltage\n");
    }
    /*
    ** calculate the steady state value of the state variable
    */
	alpha = TabInterp(ip,(double)(channel->activation));
    action->passback = ftoa(alpha);
}

TabChannel_CALC_BETA(channel,action)
register struct tab_channel_type *channel;
Action		*action;
{
double	alpha,beta;
char	*gate;
Interpol	*ipa,*ipb;

    if(action->argc == 2){
	gate = action->argv[0];
		if (strcmp(gate,"X") == 0) {
			ipa = channel->X_A;
			ipb = channel->X_B; 
		} else if (strcmp(gate,"Y") == 0) {
			ipa = channel->Y_A;
			ipb = channel->Y_B;
		} else if (strcmp(gate,"Z") == 0) {
			ipa = channel->Z_A;
			ipb = channel->Z_B;
		}
    	channel->activation = Atof(action->argv[1]);
    } else {
    Error();
    printf("usage : CALC_MINF gate voltage\n");
    }
    /*
    ** calculate the steady state value of the state variable
    */
	alpha = TabInterp(ipa,(double)(channel->activation));
	/* remember that in tabchannels beta is 1/tau */
	beta = TabInterp(ipb,(double)(channel->activation));
    beta -= alpha;
    action->passback = ftoa(beta);
}
/*
** Routines to save/read all interpol tables to/from a file.
** Routines can be used by tabchannel, tab2Dchannel and tabcurrent.
** Includes extensive error checking.  Maybe part of these
**  routines should be relocated to Interpol(2D) code.
** E. De Schutter, BBF-UIA 12/94
*/
TabChannel_SAVE(elm,action)
Element *elm;
Action	*action;
{
int	i,j,k,n=0,ttype,ntype,itype;
int	dsize,isize;
FILE	*fp,*fopen();
Tchan	*chan1;
T2chan	*chan2;
Tcurr	*curr;
Interpol *ipolA,*ipolB;
Interpol2D *ipol2A,*ipol2B;
double	*A,*B;

    if (action->argc != 1){
		Error();
		printf(" usage: TABSAVE filename\n");
		return;
	}
	dsize=sizeof(double);
	isize=sizeof(int);
	/* identify table type */
	if (strcmp(elm->object->name,"tabchannel")==0) {
		ttype=TABCHAN_T;
	} else if (strcmp(elm->object->name,"tab2Dchannel")==0) {
		ttype=TAB2CHAN_T;
	} else if (strcmp(elm->object->name,"tabcurrent")==0) {
        ttype=TABCURR_T;
	} else if (strcmp(elm->object->name,"tabflux")==0) {
		ttype=TABFLUX_T;
	} else {
		Error();
		printf(" TABSAVE bug #1\n");
		return;
	}
	/* count number of tables we need to save */
	switch (ttype) {
		case TABCHAN_T:
		case TAB2CHAN_T:
			chan1=(Tchan *)elm;
			if (chan1->X_alloced) n++;
			if (chan1->Y_alloced) n++;
			if (chan1->Z_alloced) n++;
			break;
		
        	case TABCURR_T:
 			curr=(Tcurr *)elm;
           		if (curr->alloced) n++;
            		break;

		default:
			break;
	}
	if (n==0) return;	/* nothing to write */
	/* open the file */
	if (!(fp = fopen(action->argv[0],"w"))) {
		Error();
		printf(" can't open file '%s'\n",action->argv[0]);
		return;
	}
	/* write header */
	i=3;	/* version number */
	fwrite(&i,isize,1,fp);
	fwrite(&ttype,isize,1,fp);
	fwrite(&n,isize,1,fp);
	/* loop through all the tables */
	for (i=0; i<n; i++) {
		TabChannel_ptrs(elm,i,ttype,&itype,&ntype,&ipolA,&ipolB,&ipol2A,&ipol2B);
		fwrite(&itype,isize,1,fp);
		fwrite(&ntype,isize,1,fp);
		if (itype==1) {
			/* write table sizes */
			fwrite(&(ipolA->xdivs),isize,1,fp);
			fwrite(&(ipolA->xmin),dsize,1,fp);
			fwrite(&(ipolA->xmax),dsize,1,fp);
			fwrite(&(ipolA->dx),dsize,1,fp);
			/* write the tables */
			A=ipolA->table;
			if (ntype>1) B=ipolB->table;
			for (j=0; j<=ipolA->xdivs; j++) {
				fwrite(A++,dsize,1,fp);
				if (ntype>1) fwrite(B++,dsize,1,fp);
			}
		} else {	/* itype==2 */
			/* write table sizes */
			fwrite(&(ipol2A->xdivs),isize,1,fp);
			fwrite(&(ipol2A->xmin),dsize,1,fp);
			fwrite(&(ipol2A->xmax),dsize,1,fp);
			fwrite(&(ipol2A->dx),dsize,1,fp);
			fwrite(&(ipol2A->ydivs),isize,1,fp);
			fwrite(&(ipol2A->ymin),dsize,1,fp);
			fwrite(&(ipol2A->ymax),dsize,1,fp);
			fwrite(&(ipol2A->dy),dsize,1,fp);
			/* write the tables */
			for (j=0; j<=ipol2A->xdivs; j++) {
				A=ipol2A->table[j];
				if (ntype>1) B=ipol2B->table[j];
				for (k=0; k<=ipol2A->ydivs; k++) {
					fwrite(A++,dsize,1,fp);
					if (ntype>1) fwrite(B++,dsize,1,fp);
				}
			}
		}
	}
	fclose(fp);
}

TabChannel_READ(elm,action)
Element *elm;
Action	*action;
{
int	i,j,k,n=0,ttype,itype,ntype,version;
int	dsize,isize;
FILE	*fp,*fopen();
Tchan	*chan1;
T2chan	*chan2;
Tcurr	*curr;
Interpol *ipolA,*ipolB;
Interpol2D *ipol2A,*ipol2B;
double	*A,*B;

    if (action->argc != 1){
		Error();
		printf(" usage: TABREAD filename\n");
		return;
	}
	dsize=sizeof(double);
	isize=sizeof(int);
	/* identify table type */
	if (strcmp(elm->object->name,"tabchannel")==0) {
		ttype=TABCHAN_T;
	} else if (strcmp(elm->object->name,"tab2Dchannel")==0) {
		ttype=TAB2CHAN_T;
	} else if (strcmp(elm->object->name,"tabcurrent")==0) {
		ttype=TABCURR_T;
	} else if (strcmp(elm->object->name,"tabflux")==0) {
		ttype=TABFLUX_T;
	} else {
		Error();
		printf(" TABREAD bug #1\n");
	}
	/* count number of tables we need to save */
	switch (ttype) {
		case TABCHAN_T:
		case TAB2CHAN_T:
			chan1=(Tchan *)elm;
			if (chan1->X_alloced) n++;
			if (chan1->Y_alloced) n++;
			if (chan1->Z_alloced) n++;
			break;
		
        	case TABCURR_T:
 			curr=(Tcurr *)elm;
           		if (curr->alloced) n++;
             		break;

		default:
			break;
	}
	if (n==0) {	/* nothing to read */
		Error();
		printf(" no tables allocated in %s\n",Pathname(elm));
		return;
	}
	/* open the file */
	if (!(fp = fopen(action->argv[0],"r"))) {
		Error();
		printf(" can't open file '%s'\n",action->argv[0]);
		return;
	}
	/* read header */
	fread(&version,isize,1,fp);
	if ((version<1)||((version==2)&&(ttype==TABCURR_T))||(version>3)) {
		Error();
		printf(" can't read file '%s': wrong version #%d\n",action->argv[0],version);
		fclose(fp);
		return;
	}
	fread(&i,isize,1,fp);
	if (i!=ttype) {
		Error();
		printf(" can't read file '%s': wrong object type (%d)\n",action->argv[0],i);
		fclose(fp);
		return;
	}
	fread(&i,isize,1,fp);
	if (i!=n) {
		Error();
		printf(" can't read file '%s': wrong number of tables (%d)\n",action->argv[0],i);
		fclose(fp);
		return;
	}
	/* loop through all the tables */
	for (i=0; i<n; i++) {
		TabChannel_ptrs(elm,i,ttype,&itype,&ntype,&ipolA,&ipolB,&ipol2A,&ipol2B);
		fread(&k,isize,1,fp);
		if (k!=itype) {
			Error();
			printf("can't read file '%s': wrong dimension of table (%d: %d)\n",action->argv[0],itype,k);
			return;
		}
		if (version>1) {
			fread(&k,isize,1,fp);
			if (k!=ntype) {
				Error();
				printf("can't read file '%s': wrong type of table (%d: %d)\n",action->argv[0],ntype,k);
				return;
			}
		} else {
			ntype==2;	/* two tables */
		}
		if (itype==1) {
			/* read table sizes */
			fread(&k,isize,1,fp);
			if ((k!=ipolA->xdivs) || 
				((ttype<TABCURR_T)&&(k!=ipolB->xdivs))) {
				Error();
				printf("can't read file '%s': wrong xdivs for table (%d: %d)\n",action->argv[0],k,ipolA->xdivs);
				return;
			}
			fread(&(ipolA->xmin),dsize,1,fp);
			fread(&(ipolA->xmax),dsize,1,fp);
			fread(&(ipolA->dx),dsize,1,fp);
			ipolA->invdx=1.0/ipolA->dx;
			if (ntype>1) {
				ipolB->xmin=ipolA->xmin;
				ipolB->xmax=ipolA->xmax;
				ipolB->dx=ipolA->dx;
				ipolB->invdx=1.0/ipolA->dx;
			}
			/* read the tables */
			A=ipolA->table;
			if (ntype>1) B=ipolB->table;
			if (version==1) {
				k=ipolA->xdivs-1;
			} else {	/* small bug corrected in version 2 */
				k=ipolA->xdivs;
			}
			for (j=0; j<=k; j++) {
				fread(A++,dsize,1,fp);
				if (ntype>1) fread(B++,dsize,1,fp);
			}
		} else {	/* itype==2 */
			/* read table sizes */
			fread(&k,isize,1,fp);
			if ((k!=ipol2A->xdivs) || ((n>1)&&(k!=ipol2B->xdivs))) {
				Error();
				printf(" can't read file '%s': wrong xdivs for table (%d: %d)\n",action->argv[0],k,ipol2A->xdivs);
				return;
			}
			fread(&(ipol2A->xmin),dsize,1,fp);
			fread(&(ipol2A->xmax),dsize,1,fp);
			fread(&(ipol2A->dx),dsize,1,fp);
			ipol2A->invdx=1.0/ipol2A->dx;
			fread(&k,isize,1,fp);
			if ((k!=ipol2A->ydivs) || ((ntype>1)&&(k!=ipol2B->ydivs))) {
				Error();
				printf(" can't read file '%s': wrong ydivs for table (%d: %d)\n",action->argv[0],k,ipol2A->ydivs);
				return;
			}
			fread(&(ipol2A->ymin),dsize,1,fp);
			fread(&(ipol2A->ymax),dsize,1,fp);
			fread(&(ipol2A->dy),dsize,1,fp);
			ipol2A->invdy=1.0/ipol2A->dy;
			if (ntype>1) {
				ipol2B->xmin=ipol2A->xmin;
				ipol2B->xmax=ipol2A->xmax;
				ipol2B->dx=ipol2A->dx;
				ipol2B->invdx=1.0/ipol2A->dx;
				ipol2B->ymin=ipol2A->ymin;
				ipol2B->ymax=ipol2A->ymax;
				ipol2B->dy=ipol2A->dy;
				ipol2B->invdy=1.0/ipol2A->dy;
			}
			/* read the tables */
			for (j=0; j<=ipol2A->xdivs; j++) {
				A=ipol2A->table[j];
				if (ntype>1) B=ipol2B->table[j];
				for (k=0; k<=ipol2A->ydivs; k++) {
					fread(A++,dsize,1,fp);
					if (ntype>1) fread(B++,dsize,1,fp);
				}
			}
		}
	}
	fclose(fp);
}

TabChannel_ptrs(elm,i,ttype,itype,ntype,ipolA,ipolB,ipol2A,ipol2B)
	Element *elm;
	int		i,ttype,*ntype,*itype;
	Interpol	**ipolA,**ipolB;
	Interpol2D	**ipol2A,**ipol2B;
{
Tchan	*chan1;
T2chan	*chan2;
Tcurr	*curr;

	/* get table pointers */
	switch (ttype) {
		case TABCHAN_T:
			chan1=(Tchan *)elm;
			*itype=1;
			*ntype=2;
			if (chan1->X_alloced && (i==0)) {
				*ipolA=chan1->X_A;
				*ipolB=chan1->X_B;
			} else if (chan1->Y_alloced && ((i==0)||(i==1))) {
				*ipolA=chan1->Y_A;
				*ipolB=chan1->Y_B;
			} else if (chan1->Z_alloced) {
				*ipolA=chan1->Z_A;
				*ipolB=chan1->Z_B;
			}
			break;

		case TAB2CHAN_T:
			chan2=(T2chan *)elm;
			*itype=2;
			*ntype=2;
			if (chan2->X_alloced && (i==0)) {
				*ipol2A=chan2->X_A;
				*ipol2B=chan2->X_B;
			} else if (chan2->Y_alloced && (i<2)) {
				*ipol2A=chan2->Y_A;
				*ipol2B=chan2->Y_B;
			} else if (chan2->Z_alloced) {
				*ipol2A=chan2->Z_A;
				*ipol2B=chan2->Z_B;
			}
			break;

		case TABCURR_T:
			curr=(Tcurr *)elm;
			*itype=2;
			*ntype=2;
			if (i==0) {
				*ipol2A=curr->G_tab;
				*ipol2B=curr->I_tab;
			}
			break;
	}
}
