/* 
   transaction.c
   
   Part of GNU Enterprise Application Server (GEAS)

   Copyright (C) 2001 Free Software Foundation

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: transaction.c,v 1.12 2001/06/07 23:37:05 ntiffin Exp $

*/

#include "config.h"

#include "geas.h"
#include "geas-server.h"
#include "globals.h"
#include "geas-skeleton.h"
#include "transaction.h"
#include "exceptions.h"
#include "datamonitor/datamonitor.h"

/** \file transaction.c
 *  \brief GEAS::Transaction and GEAS::TransactionFactory implementation
 * 
 * This code is mostly incomplete, but exists to provide an initial framework.
 */


/* factory methods */
GEAS_Transaction
TransactionFactory_new (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_Transaction retval = CORBA_OBJECT_NIL;
  GEAS_DataObject obj = CORBA_OBJECT_NIL;
  char *newid;

  /* Create a new transaction */
  obj = GEAS_Connection_newObject (id->server, "geas::transaction", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      return (CORBA_OBJECT_NIL);
    }
  GEAS_DataObject_setField (obj, "_user", id->username, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      GEAS_DataObject_delete (obj, ev);
      CORBA_Object_release (obj, ev);
      return (CORBA_OBJECT_NIL);
    }

  GEAS_DataObject_setField (obj, "open", "F", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      GEAS_DataObject_delete (obj, ev);
      CORBA_Object_release (obj, ev);
      return (CORBA_OBJECT_NIL);
    }

  newid = GEAS_DataObject__get_objectID (obj, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      GEAS_DataObject_delete (obj, ev);
      CORBA_Object_release (obj, ev);
      return (CORBA_OBJECT_NIL);
    }

  retval =
    make_transaction_reference (id->username, newid, id->sessionid, ev);
  CORBA_free (newid);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      GEAS_DataObject_delete (obj, ev);
      CORBA_Object_release (obj, ev);
      return (CORBA_OBJECT_NIL);
    }

  CORBA_Object_release (obj, ev);
  return (retval);
}

GEAS_Transaction
TransactionFactory_current (GEAS_object_reference * id,
                            CORBA_Environment * ev)
{
  GEAS_Transaction retval = CORBA_OBJECT_NIL;
  GEAS_DataObject user;
  char *tid;

  /* Find the transaction that the current user is associated with */

  /* find the user */
  user = GEAS_Connection_loadSingleObject (id->server,
                                           "geas::user", "username",
                                           id->username, ev);
  if (ev->_major != CORBA_NO_EXCEPTION || user == CORBA_OBJECT_NIL)
    {
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      return (CORBA_OBJECT_NIL);
    }

  /* get transaction object */
  tid = GEAS_DataObject_getField (user, "_currenttransaction", ev);
  if (ev->_major != CORBA_NO_EXCEPTION || tid == NULL)
    {
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_Object_release (user, ev);
      if (tid)
        CORBA_free (tid);
      return (CORBA_OBJECT_NIL);
    }

  retval = make_transaction_reference (id->username, tid, id->sessionid, ev);
  CORBA_free (tid);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_Object_release (user, ev);
      return (CORBA_OBJECT_NIL);
    }

  CORBA_Object_release (user, ev);
  return (retval);
}

/* transaction methods */

/* raises: TransactionInProgress, DatabaseClosed */
void
Transaction_begin (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_DataObject transaction;
  GEAS_DataObject user;
  char *open;
  char *transactionid = NULL;

  /* find record */
  transaction = GEAS_Connection_loadSingleObject (id->server,
                                                  "geas::transaction",
                                                  "objectid", id->objectid,
                                                  ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
    }

  /* check it's closed */
  open = GEAS_DataObject_getField (transaction, "open", ev);
  if (ev->_major != CORBA_NO_EXCEPTION || strcmp (open, "T") == 0)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      /* send a generic 'no can do' error */
      make_TransactionInProgress_exception (ev, "");
      return;
    }

  /* get its identifier */
  transactionid = GEAS_DataObject__get_objectID (transaction, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      /* send a generic 'no can do' error */
      make_TransactionInProgress_exception (ev, "");
      return;
    }

  /* mark it as open */
  GEAS_DataObject_setField (transaction, "open", "T", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions, really */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);
      if (transactionid)
        CORBA_free (transactionid);

      /* send a generic 'no can do' error */
      make_TransactionInProgress_exception (ev, "");
      return;
    }
  CORBA_free (open);

  /* make it the 'current' transaction for user 'id->username' */
  user = GEAS_Connection_loadSingleObject (get_server_connection_object (),
                                           "geas::user", "username",
                                           id->username, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      printf ("eep? another error handler TODO %s/%d\n", __FILE__, __LINE__);
      exit (0);
    }
  GEAS_DataObject_setField (user, "_currenttransaction", transactionid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      printf ("eep? another error handler TODO %s/%d\n", __FILE__, __LINE__);
      exit (0);
    }
  CORBA_Object_release (user, ev);

  /* done */
  dm_event (id->username, transactionid, DM_TRANSACTION_BEGIN);
  if (transactionid)
    {
      CORBA_free (transactionid);
    }
  CORBA_Object_release (transaction, ev);
}

/* raises: TransactionNotInProgress */
void
Transaction_commit (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_DataObject transaction, user;
  char *open;
  char *transactionid;

  transaction = GEAS_Connection_loadSingleObject (id->server,
                                                  "geas::transaction",
                                                  "objectid", id->objectid,
                                                  ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
    }

  /* check it's open */
  open = GEAS_DataObject_getField (transaction, "open", ev);
  if (ev->_major != CORBA_NO_EXCEPTION || strcmp (open, "T") != 0)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      /* send a generic 'no can do' error */
      make_TransactionNotInProgress_exception (ev, "");
      return;
    }
  CORBA_free (open);

  /* get its identifier */
  transactionid = GEAS_DataObject__get_objectID (transaction, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      /* send a generic 'no can do' error */
      make_TransactionNotInProgress_exception (ev, "");
      return;
    }

  /* commit transaction now */
  dm_event (id->username, transactionid, DM_TRANSACTION_COMMIT);
  CORBA_free (transactionid);

  /* mark it as closed */
  GEAS_DataObject_setField (transaction, "open", "F", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions, really */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      return;
    }

  /* clear the 'current' transaction for user 'id->username' */
  user = GEAS_Connection_loadSingleObject (get_server_connection_object (),
                                           "geas::user", "username",
                                           id->username, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      printf ("eep? another error handler TODO %s/%d\n", __FILE__, __LINE__);
      exit (0);
    }
  GEAS_DataObject_setField (user, "_currenttransaction", "", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      printf ("eep? another error handler TODO %s/%d\n", __FILE__, __LINE__);
      exit (0);
    }
  CORBA_Object_release (user, ev);

  /* done */
  CORBA_Object_release (transaction, ev);
}

/* raises: TransactionNotInProgress */
void
Transaction_abort (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_DataObject transaction, user;
  char *open;
  char *transactionid;

  transaction = GEAS_Connection_loadSingleObject (id->server,
                                                  "geas::transaction",
                                                  "objectid", id->objectid,
                                                  ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
    }

  /* check it's open */
  open = GEAS_DataObject_getField (transaction, "open", ev);
  if (ev->_major != CORBA_NO_EXCEPTION || strcmp (open, "T") != 0)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      /* send a generic 'no can do' error */
      make_TransactionNotInProgress_exception (ev, "");
      return;
    }
  CORBA_free (open);

  /* get its identifier */
  transactionid = GEAS_DataObject__get_objectID (transaction, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      /* send a generic 'no can do' error */
      make_TransactionNotInProgress_exception (ev, "");
      return;
    }

  /* abort/rollback transaction now */
  dm_event (id->username, transactionid, DM_TRANSACTION_ROLLBACK);
  CORBA_free (transactionid);

  /* mark it as closed */
  GEAS_DataObject_setField (transaction, "open", "F", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions, really */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      return;
    }

  /* clear the 'current' transaction for user 'id->username' */
  user = GEAS_Connection_loadSingleObject (get_server_connection_object (),
                                           "geas::user", "username",
                                           id->username, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      printf ("eep? another error handler TODO %s/%d\n", __FILE__, __LINE__);
      exit (0);
    }
  GEAS_DataObject_setField (user, "_currenttransaction", "", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      printf ("eep? another error handler TODO %s/%d\n", __FILE__, __LINE__);
      exit (0);
    }
  CORBA_Object_release (user, ev);

  /* done */
  CORBA_Object_release (transaction, ev);
}

/* raises: TransactionNotInProgress */
void
Transaction_checkpoint (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_DataObject transaction;
  char *open;
  char *transactionid;

  transaction = GEAS_Connection_loadSingleObject (id->server,
                                                  "geas::transaction",
                                                  "objectid", id->objectid,
                                                  ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
    }

  /* check it's open */
  open = GEAS_DataObject_getField (transaction, "open", ev);
  if (ev->_major != CORBA_NO_EXCEPTION || strcmp (open, "T") != 0)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      /* send a generic 'no can do' error */
      make_TransactionNotInProgress_exception (ev, "");
      return;
    }
  CORBA_free (open);

  /* get its identifier */
  transactionid = GEAS_DataObject__get_objectID (transaction, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_free (open);
      CORBA_Object_release (transaction, ev);

      /* send a generic 'no can do' error */
      make_TransactionNotInProgress_exception (ev, "");
      return;
    }

  /* checkpoint transaction now */
  dm_event (id->username, transactionid, DM_TRANSACTION_CHECKPOINT);
  CORBA_free (transactionid);

  /* leave it open */

  /* done */
  CORBA_Object_release (transaction, ev);
}

/* raises: TransactionNotInProgress */
void
Transaction_join (GEAS_object_reference * id, CORBA_Environment * ev)
{
}

/* raises: TransactionNotInProgress */
void
Transaction_leave (GEAS_object_reference * id, CORBA_Environment * ev)
{
}

CORBA_boolean
Transaction_isOpen (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_DataObject transaction;
  char *open;
  CORBA_boolean retval;

  transaction = GEAS_Connection_loadSingleObject (id->server,
                                                  "geas::transaction",
                                                  "objectid", id->objectid,
                                                  ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
    }
  open = GEAS_DataObject_getField (transaction, "open", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore exceptions */
      message ("error : %s  (%s,%d)", CORBA_exception_id (ev), __FILE__,
               __LINE__);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_Object_release (transaction, ev);

      /* an error, so we can't be active */
      return (CORBA_FALSE);
    }

  CORBA_Object_release (transaction, ev);
  if (open[0] == 'T')
    {
      retval = CORBA_TRUE;
    }
  else
    {
      retval = CORBA_FALSE;
    }
  CORBA_free (open);
  return (retval);
}
