/* 
   security.c -
   
   Part of GNU Enterprise Application Server (GEAS)

   Copyright (C) 2001 Free Software Foundation

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: security.c,v 1.16 2001/07/25 17:48:24 reinhard Exp $

*/

/** \file security.c
 *  \brief Support functions for the security system
 */

#include <ctype.h>

#include "config.h"

#include "geas.h"
#include "geas-server.h"
#include "globals.h"
#include "config/configuration.h"
#include "security.h"

/** \brief Set up the security system
 * Called by the main function, when initialising the server. Currently
 * it ensures that there is a guest user and an admin user.
 */
void init_security_system(void)
{
   GEAS_ObjectList users = CORBA_OBJECT_NIL;
   CORBA_Environment ev;
   char *admin,*adminpw,*p;

   CORBA_exception_init( &ev );

   /* get name of admin user, make sure it's all ASCII */
   adminpw =
     (char *)get_global_option_str(configdata, "adminpassword", NULL);
   admin =
     (char *)get_global_option_str(configdata, "adminuser", "admin");
   if( !admin || strlen(admin) < 1 )
     admin = "admin";
   p = admin;
   while( p && *p != '\0' )
     {
	if( isascii(*p) == 0 )
	  {
	     admin = "admin";
	     p = NULL;
	  }
	else
	  p++;
     }

   /* load all users */
   users = GEAS_Connection_loadAll( get_server_connection_object() , "geas::user" , &ev );
   if( ev._major != CORBA_NO_EXCEPTION )
     {
	errormsg( "Error loading list of available users: %s" , CORBA_exception_id(&ev) );
	return;
     }
   else
     {
	int count;
        count = GEAS_ObjectList__get_length( users , &ev );
	if( ev._major != CORBA_NO_EXCEPTION )
	  {
	     errormsg( "Error counting users: %s" , CORBA_exception_id(&ev) );
	  }
	else if( count == 0 )
	  {
	     GEAS_DataObject user;

	     message( "No users found." );
	     message( "  Adding 'guest' user.");
	     user = GEAS_Connection_newObject(get_server_connection_object(),"geas::user",&ev);
	     if( ev._major != CORBA_NO_EXCEPTION )
	       {
		  errormsg( "Could not add new user: %s" , CORBA_exception_id(&ev) );
	       }
	     else
	       {
		  GEAS_DataObject_setField(user,"username","guest",&ev);
		  if( ev._major != CORBA_NO_EXCEPTION )
		    {
		       errormsg( "Could not set username." );
		    }
		  else
		    {
		       GEAS_DataObject_setField(user,"password","guest",&ev);
		       if( ev._major != CORBA_NO_EXCEPTION )
			 {
			    errormsg( "Could not set password." );
			 }
		       else
			 {
			    GEAS_DataObject_setField(user,"_currenttransaction","",&ev);
			    if( ev._major != CORBA_NO_EXCEPTION )
			      {
				 errormsg( "Could not set _currenttransaction." );
			      }
			 }
		    }
		  CORBA_Object_release(user,&ev);
	       }

	     message( "  Adding '%s' user." , admin );
	     user = GEAS_Connection_newObject(get_server_connection_object(),"geas::user",&ev);
	     if( ev._major != CORBA_NO_EXCEPTION )
	       {
		  errormsg( "Could not add new user: %s" , CORBA_exception_id(&ev) );
	       }
	     else
	       {
		  GEAS_DataObject_setField(user,"username",admin,&ev);
		  if( ev._major != CORBA_NO_EXCEPTION )
		    {
		       errormsg( "Could not set username." );
		    }
		  else
		    {
		       char password[33];

		       if( adminpw != NULL )
			 {
			    strncpy( password , adminpw , 30 );
			    password[31] = '\0';
			 }
		       else
			 {
		       do {
			  printf( "\nWarning: this system provides no real security.\n" );
			  printf( "Please enter a new administrator password (max 32 characters): " );
			  fgets( password , 32 , stdin );
			  password[33] = '\0';
			  while( password[strlen(password)-1] == 10 ||
				 password[strlen(password)-1] == 13 ||
				 isspace(password[strlen(password)-1]) )
			    password[strlen(password)-1] = '\0';
		       } while( strlen(password) == 0 );
			 }
		       GEAS_DataObject_setField(user,"password",password,&ev);
		       if( ev._major != CORBA_NO_EXCEPTION )
			 {
			    errormsg( "Could not set password." );
			 }
		       else
			 {
			    GEAS_DataObject_setField(user,"_currenttransaction","",&ev);
			    if( ev._major != CORBA_NO_EXCEPTION )
			      {
				 errormsg( "Could not set _currenttransaction." );
			      }
			 }
		    }
		  CORBA_Object_release(user,&ev);
	       }
	  }
	else
	  {
	     message( "%d user records identified." , count );
	  }
     }
   GEAS_ObjectList_release( users , &ev );

   CORBA_exception_free( &ev );
}
