/*
   query.c -
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2000-2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: query.c,v 1.12 2001/06/29 04:08:25 treshna Exp $
 
*/

#include "config.h"

#include "geas.h"
#include "geas-server.h"
#include "dataobject.h"
#include "connection.h"
#include "exceptions.h"
#include "classdata.h"
#include "objectcache/objectcache.h"
#include "oql/oql.h"
#include "objectstore/objectstore.h"
#include "collectiondata.h"
#include "methods/methods.h"
#include "datamonitor/datamonitor.h"
#include "connection.h"
#include "query.h"

GEAS_Query_logictype
Constraint__get_logic (GEAS_object_reference * id, CORBA_Environment * ev)
{
  char *tmp;
  GEAS_Query_logictype retval = GEAS_Query_UNKNOWN;
  GEAS_DataObject q;

  q = GEAS_Connection_loadSingleObject (id->server, "geas::searchcriteria",
                                        "objectid", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return (retval);
    }
  tmp = GEAS_DataObject_getField (q, "logic", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      CORBA_Object_release (q, ev);
      return (retval);
    }
  retval = GEAS_Query_UNKNOWN;
  if (strcmp (tmp, "1") == 0)
    {
      retval = GEAS_Query_AND;
    }
  if (strcmp (tmp, "2") == 0)
    {
      retval = GEAS_Query_OR;
    }
  CORBA_free (tmp);
  CORBA_Object_release (q, ev);
  return (retval);
}

void
Constraint__set_logic (GEAS_object_reference * id, GEAS_Query_logictype value,
                       CORBA_Environment * ev)
{
  GEAS_DataObject q;

  q = GEAS_Connection_loadSingleObject (id->server, "geas::searchcriteria",
                                        "objectid", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return;
    }
  if (value == GEAS_Query_AND)
    {
      GEAS_DataObject_setField (q, "logic", "1", ev);
    }
  if (value == GEAS_Query_OR)
    {
      GEAS_DataObject_setField (q, "logic", "2", ev);
    }
  CORBA_Object_release (q, ev);
}

void
Constraint_addField (GEAS_object_reference * id, GEAS_Query_Field * field,
                     CORBA_Environment * ev)
{
  GEAS_DataObject f;
  char *buf = NULL;

  /* create storage object */
  f = GEAS_Connection_newObject (id->server, "geas::searchfield", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return;
    }
  /* fill it in */
  GEAS_DataObject_setField (f, "constraint", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }

  GEAS_DataObject_setField (f, "field", field->field, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }

  GEAS_DataObject_setField (f, "value", field->value, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  switch (field->test)
    {
    case GEAS_Query_equals:
      buf = "1";
      break;
    case GEAS_Query_lessthan:
      buf = "2";
      break;
    case GEAS_Query_greaterthan:
      buf = "3";
      break;
    case GEAS_Query_contains:
      buf = "4";
      break;
    case GEAS_Query_startswith:
      buf = "5";
      break;
    case GEAS_Query_endswith:
      buf = "6";
      break;
    }
  GEAS_DataObject_setField (f, "test", buf, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  if (field->casesensitive == CORBA_TRUE)
    {
      GEAS_DataObject_setField (f, "casesensitive", "TRUE", ev);
    }
  else
    {
      GEAS_DataObject_setField (f, "casesensitive", "FALSE", ev);
    }
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  if (field->invert)
    {
      GEAS_DataObject_setField (f, "invert", "TRUE", ev);
    }
  else
    {
      GEAS_DataObject_setField (f, "invert", "FALSE", ev);
    }
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  /* done */
  CORBA_Object_release (f, ev);
  return;

error:
  errormsg ("%s", CORBA_exception_id (ev));
  CORBA_exception_free (ev);
  CORBA_exception_init (ev);
  GEAS_DataObject_delete (f, ev);
  CORBA_Object_release (f, ev);
  return;
}

GEAS_Query_Constraint
Constraint_newConstraint (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_Query_Constraint retval;
  GEAS_DataObject c;
  char *uniqueid;

  c = Connection_newObject (id->serverid, "geas::searchcriteria", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      return (CORBA_OBJECT_NIL);
    }
  GEAS_DataObject_setField (c, "parent", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  GEAS_DataObject_setField (c, "classname", "geas::unknownclass", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  GEAS_DataObject_setField (c, "orderby", "", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  GEAS_DataObject_setField (c, "reverse", "F", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  GEAS_DataObject_setField (c, "logic", "1", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      goto error;
    }
  /* done */
  uniqueid = GEAS_DataObject__get_objectID (c, ev);
  retval =
    make_constraint_reference (uniqueid, id->uniqueid, id->username,
                               id->sessionid, ev);
  CORBA_free (uniqueid);
  CORBA_Object_release (c, ev);
  return (retval);

error:
  /* ignore details of the error - is a good plan, honest */
  CORBA_exception_free (ev);
  CORBA_exception_init (ev);
  GEAS_DataObject_delete (c, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      /* ignore this exception */
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
    }
  CORBA_Object_release (c, ev);
  make_ServerError_exception (ev, "Error allocating new query.");
  return (CORBA_OBJECT_NIL);
}

void
Query_release (GEAS_object_reference * id, CORBA_Environment * ev)
{
  /* QueryData *q; */
  struct query_result *result;

  /* delete the geas::searchcriteria object */
  result =
    delete_from_objectstore ("geas::searchcriteria", id->uniqueid, NULL,
                             NULL);
  if (result)
    {
      free_query_result (result);
    }

  /* delete the associated geas::searchfield instances */
  result =
    delete_all_from_objectstore ("geas::searchfield", "constraint",
                                 id->uniqueid, NULL, NULL);
  if (result)
    {
      free_query_result (result);
    }
}

void
Query_clear (GEAS_object_reference * id, CORBA_Environment * ev)
{
  fatal_error ("This isn't done yet...");
}

CORBA_char *
Query__get_orderby (GEAS_object_reference * id, CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  GEAS_DataObject q;

  q = GEAS_Connection_loadSingleObject (id->server, "geas::searchcriteria",
                                        "objectid", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return (retval);
    }
  retval = GEAS_DataObject_getField (q, "orderby", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      CORBA_Object_release (q, ev);
      return (retval);
    }

  CORBA_Object_release (q, ev);
  return (retval);
}

void
Query__set_orderby (GEAS_object_reference * id, CORBA_char * value,
                    CORBA_Environment * ev)
{
  GEAS_DataObject q;

  q = GEAS_Connection_loadSingleObject (id->server, "geas::searchcriteria",
                                        "objectid", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return;
    }
  GEAS_DataObject_setField (q, "orderby", value, ev);
  CORBA_Object_release (q, ev);
}

CORBA_boolean
Query__get_reverse (GEAS_object_reference * id, CORBA_Environment * ev)
{
  char *tmp;
  CORBA_boolean retval = CORBA_FALSE;
  GEAS_DataObject q;

  q = GEAS_Connection_loadSingleObject (id->server, "geas::searchcriteria",
                                        "objectid", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return (retval);
    }
  tmp = GEAS_DataObject_getField (q, "reverse", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      CORBA_Object_release (q, ev);
      return (retval);
    }
  if (strcmp (tmp, "1") == 0)
    {
      retval = CORBA_TRUE;
    }
  if (strcmp (tmp, "0") == 0)
    {
      retval = CORBA_FALSE;
    }
  CORBA_free (tmp);

  CORBA_Object_release (q, ev);
  return (retval);
}

void
Query__set_reverse (GEAS_object_reference * id, CORBA_boolean value,
                    CORBA_Environment * ev)
{
  GEAS_DataObject q;

  q = GEAS_Connection_loadSingleObject (id->server, "geas::searchcriteria",
                                        "objectid", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return;
    }
  if (value)
    {
      GEAS_DataObject_setField (q, "reverse", "T", ev);
    }
  else
    {
      GEAS_DataObject_setField (q, "reverse", "F", ev);
    }
  CORBA_Object_release (q, ev);
}

void
Query__set_classname (GEAS_object_reference * id,
                      CORBA_char * value, CORBA_Environment * ev)
{
  GEAS_DataObject q;

  q = GEAS_Connection_loadSingleObject (id->server, "geas::searchcriteria",
                                        "objectid", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return;
    }
  GEAS_DataObject_setField (q, "classname", value, ev);

  CORBA_Object_release (q, ev);
}

CORBA_char *
Query__get_classname (GEAS_object_reference * id, CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  GEAS_DataObject q;

  q = GEAS_Connection_loadSingleObject (id->server, "geas::searchcriteria",
                                        "objectid", id->uniqueid, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      return (retval);
    }
  retval = GEAS_DataObject_getField (q, "classname", ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      errormsg ("%s", CORBA_exception_id (ev));
      CORBA_Object_release (q, ev);
      return (retval);
    }

  CORBA_Object_release (q, ev);
  return (retval);
}

GEAS_ObjectReference
Query__get_id (GEAS_object_reference * id, CORBA_Environment * ev)
{
  return ((GEAS_ObjectReference) CORBA_string_dup (id->uniqueid));
}
