/*
   objectstore_private.h - Common header file for database interfaces.
                           Private to objectstore library.

   Copyright (C) 2001 Free Software Foundation

   This file is part of the GNU Enterprise Application Server (GEAS)

   GEAS is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GEAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GEAS; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   $Id: objectstore_private.h,v 1.3 2001/07/24 21:19:38 reinhard Exp $
*/

#ifndef _OBJECTSTORE_PRIVATE_H
#define _OBJECTSTORE_PRIVATE_H

#include "compare.h"

/* forward references */
struct active_connection;
struct database_handle;


typedef GList *DatabaseResultRow_t;

/* database operations supported */
typedef gboolean (*connect_function) (struct database_handle * h,
                                      unsigned long int index);
typedef gboolean (*disconnect_function) (struct database_handle * h,
                                         unsigned long int index);

typedef struct query_result *(*delete_object_function) (struct database_handle
                                                        * h,
                                                        const char *classname,
                                                        const char *key,
                                                        int *errorcode,
                                                        char **errormsg);
typedef struct query_result *(*delete_all_objects_function) (struct
                                                             database_handle *
                                                             h,
                                                             const char
                                                             *classname,
                                                             const char
                                                             *fieldname,
                                                             const char *key,
                                                             int *errorcode,
                                                             char **errormsg);
typedef struct query_result *(*write_object_function) (struct database_handle
                                                       * h,
                                                       const gchar *classname,
                                                       const gchar *key,
                                                       GHashTable *values,
                                                       gboolean update,
                                                       int *errorcode,
                                                       char **errormsg);

typedef struct query_result *(*update_tables_function) (struct database_handle
                                                        * h,
                                                        gboolean remove_items,
                                                        int *errorcode,
                                                        char **errormsg);

typedef void (*delete_database_function) (struct database_handle * h);
typedef struct active_connection *(*get_connection_function) (struct
                                                              database_handle
                                                              * h);

typedef struct query_result *(*execute_query_function) (struct database_handle
                                                        * h,
                                                        QueryData * query,
                                                        int *errorcode,
                                                        char **errormsg);
gboolean generic_database_config (configuration config,
                                         const char *database,
                                         struct database_handle *h);

typedef void (*hack_function) (struct database_handle * ph);


struct active_connection
{
  /* general data */
  int index;
  gboolean available;             /* TRUE if no other threads are using it */
  gboolean connected;             /* TRUE if currently connected to database */
  unsigned long int queue;        /* count of threads waiting for this
                                   connection */
};

/* base structures */
struct database_handle
{
  /* generic data */
  char *name;                        /* name this entry */
  char *type;                        /* what DB server is on the other end? */

  /* semi-generic database config */
  /* used by most servers */
  char *dbname, *username, *password, *hostname, *unixsocket;
  signed long int port;
  unsigned long int retries;        /* times to retry before giving up */

  /* list of connections in use */
  unsigned long int num_connections;
  void *connections;

  /* functions to execute database operations */
  execute_query_function execute;
  connect_function connect;
  disconnect_function disconnect;
  delete_database_function delete_database;
  get_connection_function get_connection;
  delete_object_function delete_object;
  delete_all_objects_function delete_all_objects;
  write_object_function write_object;
  update_tables_function update_tables;

  hack_function begintransaction;
  hack_function committransaction;
};

void free_generic_database (struct database_handle *h);
struct query_result *new_query_result (void);
void remove_message (FILE * fp, DBchange * c); 
void add_message (FILE * fp, DBchange * c);


/* NOTE regarding adding additional databases
 *
 * Each different database known to GEAS has a single function that is
 * known at compile time.  That function prototype is defined here.
 * It is the responsibliity of this function to register in the
 * database_handle object other functions that provide the interface
 * between GEAS and the backend.
 *
 * There are only two places outside of the database specific .c file
 * (i.e. mysql.c and postgresql.c)
 * that have any references to a specific database.  This is one of them.
 *
 * The other is in objectstore.c where the *_create_database_handle()
 * prototypes defined here are actually called.
 *
 */
 
/* link to postgresql database */
struct database_handle *
postgresql_create_database_handle (configuration config, const char *dbname);

/* link to mysql database */
struct database_handle *
MySQL_create_database_handle (configuration config, const char *dbname);


#endif /* _OBJECTSTORE_PRIVATE_H */
