/*
  geas-server.c -
  
  Part of GNU Enterprise Application Server (GEAS)
 
  Copyright (C) 2000-2001 Free Software Foundation
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
  
  $Id: geas-server.c,v 1.107 2001/07/25 17:48:24 reinhard Exp $
 
*/

/** \file geas-server.c
 *  \brief Main function, and several utility functions
 */

#include "config.h"

#include <glib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#ifdef USE_GNU_GETOPT
#include <getopt.h>
#endif

#include <ctype.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>

#include <errno.h>

#include "geas.h"
#include "geas-server.h"
#include "globals.h"
#include "geas-skeleton.h"

#include "config/configuration.h"
#include "objectcache/objectcache.h"
#include "objectstore/objectstore.h"
#include "oql/oql.h"
#include "classdata.h"
#include "datamonitor/datamonitor.h"
#include "methods/methods.h"
#include "security.h"

#include "geas-internal-classes.h"

#ifdef GEAS_SYSRC
#define DEFAULT_CONFIG_FILE GEAS_SYSRC
#else
#define DEFAULT_CONFIG_FILE "./geas.conf"
#endif

#ifdef ENABLE_NAMESERVICE
#include <ORBitservices/CosNaming.h>

CosNaming_NamingContext name_srv;
CosNaming_Name *main_name;

CosNaming_Name *create_name (const char *name);
static void register_with_name_service (GEAS_ConnectionFactory factory);
#endif

#ifdef ENABLE_INTERCEPT_SIGSEGV
#include <signal.h>
#endif

#define DEFAULT_IOR_FILE    GEAS_IOR_DIR"/geas-server.ior"
#define DEFAULT_PID_FILE    GEAS_PID_DIR"/geas-server.pid"

#define SERVER_OBJECT_CONNECTION_ID "2d16e1c6-0958-4ce3-95e3-a514e9195be9"

/* global variables */
int debuglevel;
CORBA_ORB the_orb;
struct ServerSettings settings;
odl_tree *all_classes = NULL;

static GEAS_Connection server_connection_object = CORBA_OBJECT_NIL;
static GEAS_object_reference *server_connection_id = NULL;
static char *debug_file_name = NULL;
static int filedebuglevel = DEBUGLEVEL_HIGH;

/* private function prototypes */
static void readOptions (int argc, char **argv);
static void usage (void);
static void usage_debuglevel (void);
static void loadConfig (const char *filename, const char *shadowname);
static void load_class_files (void);
static void reserve_sql_words (const char *file, unsigned long int dbtype);
static void loadTableAliases (const char *filename);
GEAS_object_reference * get_server_connection_id (void);
struct DatabaseAliases *make_database_alias (const char *name);
struct TableAlias *make_table_alias (const char *classname, const char *tablename,
                  gboolean modifyable, const char *pkey);
struct ColumnAlias *make_column_alias (struct TableAlias *table,
                   const char *fieldname, const char *columnname);

gboolean allow_show_version (void);
gboolean allow_show_options (void);
gboolean allow_show_message (void);
gboolean allow_show_errors (void);
gboolean allow_show_config (void);
gboolean allow_show_classes (void);
void show_version (FILE * fp);

#ifdef ENABLE_INTERCEPT_SIGSEGV
void sigsegv_handler (int arg);
typedef void (*sighandler_t) (int arg);
sighandler_t old_sigsegv = NULL;

void
sigsegv_handler (int arg)
{
  fprintf (stderr,
           "kernel segmentation fault: root file system corrupted. system terminating.\n");
  sleep (3);
  fprintf (stderr, "Just kidding  :)\n\n");
  fprintf (stderr,
           "Remember: Detailed bug reports are a valuable tool, if you want the problem fixed.\n\n  :)\n\n");
  fprintf (stderr,
           "A full stack trace from 'gdb' in particular is useful for identifying\n"
           "exactly where and why an error occured.\n\n");
  if (old_sigsegv)
    old_sigsegv (arg);
  else
    {
      /* restore original handler, and call it */
      signal (SIGSEGV, SIG_DFL);
      raise (SIGSEGV);
    }
}

#endif

#ifdef TIMER_TESTING
static int active_timer_sections[ TIMER_SECTIONS ];

unsigned long int timer_base_seconds;

unsigned long int timer_get_current( void );
unsigned long int
timer_get_current( void )
{
    struct timeval tv;

    gettimeofday( &tv , NULL );
    return ((tv.tv_sec-timer_base_seconds)*1000000)+(tv.tv_usec);
}
#define us_to_seconds(x)    ((float) (x)/1000000.0 )

struct timer_profile
{
    unsigned long int max,min,total,count,timercount;
    float avg;
};
struct timer_profile timer_profiles[ TIMER_FUNC_COUNT ];

void timer_display_profiles( void )
{
  int i;
  unsigned int unaccounted;

  printf( "\n----------------------------------\n"
	    "Profiles: (time in seconds to 6dp)\n" );

  unaccounted = timer_profiles[ TIMER_FUNC_OPERATION ].total;

  for( i=0 ; i<TIMER_FUNC_COUNT ; i++ ) {
    if( timer_profiles[i].count > 0 ) {
      printf( "PROFILE: " );
      switch( i )
      {
      case TIMER_FUNC_MAKEREF    : printf( "make ref()          : " ); break;
      case TIMER_FUNC_MAKEOID    : printf( "make oid()          : " ); break;
      case TIMER_FUNC_HANDLERESULTS: printf( "handle results      : " ); break;
      case TIMER_FUNC_GETFIELD   : printf( "getField()          : " ); break;
      case TIMER_FUNC_SETFIELD   : printf( "setField()          : " ); break;
      case TIMER_FUNC_NEWOBJECT  : printf( "newObject()         : " ); break;
      case TIMER_FUNC_LOADALL    : printf( "loadAll()           : " ); break;
      case TIMER_FUNC_MAKEID     : printf( "make id             : " ); break;
      case TIMER_FUNC_GET_OBJECTS: printf( "_get_objects()      : " ); break;
      case TIMER_FUNC_OPERATION  : printf( "operation           : " ); break;
      case TIMER_FUNC_FIND_BY_KEY: printf( "find by key         : " ); break;
      case TIMER_FUNC_FIND_CACHED_BY_KEY: printf( "find cached by key  : " ); break;
      case TIMER_FUNC_GET_FIELD  : printf( "get field           : " ); break;
      case TIMER_FUNC_SET_FIELD  : printf( "set field           : " ); break;
      case TIMER_FUNC_FLUSH      : printf( "flush               : " ); break;
      case TIMER_FUNC_TEMP1      : printf( "filling object array: " ); break;
      case TIMER_FUNC_GETENTRY   : printf( "getEntry()          : " ); break;
      case TIMER_FUNC_LOOKUP     : printf( "hash table lookup   : " ); break;
      case TIMER_FUNC_EXECQUERY  : printf( "exec query          : " ); break;
      case TIMER_FUNC_RELEASELIST: printf( "release list        : " ); break;

      case TIMER_FUNC_A          : printf( "temp A              : " ); break;
      case TIMER_FUNC_B          : printf( "temp B              : " ); break;
      case TIMER_FUNC_C          : printf( "temp C              : " ); break;
      case TIMER_FUNC_D          : printf( "temp D              : " ); break;
      case TIMER_FUNC_E          : printf( "temp E              : " ); break;
      case TIMER_FUNC_F          : printf( "temp F              : " ); break;
      case TIMER_FUNC_G          : printf( "temp G              : " ); break;
      case TIMER_FUNC_H          : printf( "temp H              : " ); break;
      case TIMER_FUNC_I          : printf( "temp I              : " ); break;
      case TIMER_FUNC_J          : printf( "temp J              : " ); break;
      case TIMER_FUNC_K          : printf( "temp K              : " ); break;
      case TIMER_FUNC_L          : printf( "temp L              : " ); break;
      }
      printf( "count:%5lu min:%9.6f avg:%9.6f max:%9.6f tot:%9.6f\n" ,
	      timer_profiles[i].count,
	      us_to_seconds( timer_profiles[i].min ),
	      us_to_seconds( timer_profiles[i].avg ),
	      us_to_seconds( timer_profiles[i].max ),
	      us_to_seconds( timer_profiles[i].total ) );
      if( i != TIMER_FUNC_OPERATION ) unaccounted -= timer_profiles[i].total;
    }
    timer_profiles[i].count = 0;
  }
  refill_oid_buffer (1024 * 1024);
  oc_print_stats();
}

void timer_start_profile( int function )
{
    timer_profiles[function].timercount = timer_get_current();
}

void timer_update_profile( int function )
{
  unsigned long int diff;
  diff = timer_get_current() - timer_profiles[function].timercount;

    if( function == TIMER_FUNC_OPERATION )
    printf( "diff: %lu  max: %lu  min:%lu\n" , diff ,
	     timer_profiles[function].max ,
	     timer_profiles[function].min );

  if( timer_profiles[function].count == 0 )
  {
    timer_profiles[function].max = diff;
    timer_profiles[function].min = diff;
    timer_profiles[function].count = 1;
    timer_profiles[function].total = diff;
    timer_profiles[function].avg = diff;
  }
  else
  {
    if( diff > timer_profiles[function].max ) {
      if( function == TIMER_FUNC_OPERATION ) printf( "inc max\n" );
      timer_profiles[function].max = diff;
    }

    if( diff < timer_profiles[function].min ) {
      if( function == TIMER_FUNC_OPERATION ) printf( "inc max\n" );
      timer_profiles[function].min = diff;
    }

    timer_profiles[function].count++;
    timer_profiles[function].total += diff;
    timer_profiles[function].avg =
      timer_profiles[function].total/timer_profiles[function].count;
  }
/*
  printf( "PROFILE: " );
  switch( function )
   {
   case TIMER_FUNC_GETFIELD   : printf( "getField()     : " ); break;
   case TIMER_FUNC_SETFIELD   : printf( "setField()     : " ); break;
   case TIMER_FUNC_NEWOBJECT  : printf( "newObject()    : " ); break;
   case TIMER_FUNC_LOADALL    : printf( "loadAll()      : " ); break;
   case TIMER_FUNC_GET_OBJECTS: printf( "_get_objects() : " ); break;
   case TIMER_FUNC_OPERATION  : printf( "operation      : " ); break;
   }
  printf( "min: %.6f avg: %.6f max: %.6f\n" ,
	      us_to_seconds( timer_profiles[i].max ).
	      us_to_seconds( timer_profiles[i].avg ).
	      us_to_seconds( timer_profiles[i].min ) );
*/
}

struct timeable
{
    char name[128];
    unsigned long int timeval;
};

/* statically allocated structure, to avoid memory allocation
   during timing operations */
#define MAX_TIMEABLE_COUNT 256
struct timeable timeablestack[ MAX_TIMEABLE_COUNT ];
static signed int timer_operation_count = (-1);

#define TIMER_INDENT "   "

void timer_start_operation( int section , char *fmt , ... )
{
  int i;
  va_list ap;
return;
  /* increment stack poitner */
  timer_operation_count++;

  /* die if too many nested operations */
  g_assert( timer_operation_count < MAX_TIMEABLE_COUNT );

  /* record data on stack */
  va_start (ap, fmt);
  vsnprintf (timeablestack[timer_operation_count].name,128, fmt, ap);
  va_end (ap);

  timeablestack[timer_operation_count].timeval = timer_get_current();

  /* display start event */
  if( active_timer_sections[section] == 1 ) {
    for( i=0 ; i<timer_operation_count ; i++ )
      printf( TIMER_INDENT );
    printf( "TIMER: START: %s\n" , timeablestack[timer_operation_count].name );
  }
}

void timer_done_operation( int section )
{
  unsigned long int diff;
  int i;
return;
  g_assert( timer_operation_count >= 0 );

  if( active_timer_sections[section] == 1 ) {
    diff = timer_get_current() - timeablestack[timer_operation_count].timeval;
    for( i=0 ; i<timer_operation_count ; i++ )
      printf( TIMER_INDENT );
    printf( "TIMER: END  : %.6f %s\n" ,
	    us_to_seconds(diff) ,
	    timeablestack[timer_operation_count].name );
  }

  timer_operation_count -= 1;
}

void timer_fail_operation( int section )
{
  unsigned long int diff;
  int i;
return;
  g_assert( timer_operation_count >= 0 );

  if( active_timer_sections[section] == 1 ) {
    diff = timer_get_current() - timeablestack[timer_operation_count].timeval;
    for( i=0 ; i<timer_operation_count ; i++ )
      printf( TIMER_INDENT );
    printf( "TIMER: FAIL : %.6f %s\n" ,
	    us_to_seconds(diff) ,
	    timeablestack[timer_operation_count].name );
  }

  timer_operation_count -= 1;
}

#endif

/* Public functions */

/** \brief Main function
 *  Reads options, initialises the server, loads class definitions.
 */
int
main (int argc, char **argv)
{
#ifdef SELF_TEST
  int doneselftest = 0;
#endif
  char *filename = NULL;
  FILE *fp = NULL;
#ifdef TIMER_TESTING
  struct timeval tv;
  int i;
#endif

  PortableServer_POA root_poa;
  PortableServer_POAManager pm;
  CORBA_Environment ev;
  GEAS_ConnectionFactory factory;
  CORBA_char *objref;

#ifdef ENABLE_INTERCEPT_SIGSEGV
  /* initialise SIGSEGV handler */
  old_sigsegv = signal (SIGSEGV, sigsegv_handler);
  if (old_sigsegv == SIG_ERR)
    old_sigsegv = NULL;
#endif

#ifdef DEBUG_SECURITY
  /* NO CODE BEFORE THIS LINE:                                          */
  /* prevents a system compiled with debug security from being used     */
  /* accidentally. */

  /* this section requires manual confirmation if the security code is  */
  /* being debugged prevents accidental writing of passwords and other  */
  /* data to plain text logs                                            */
  {
    char buf[8];

    fprintf (stdout,
             "The security system is being debugged. This may result in weaker security\nbecause of debugging output.\n\n");
    fprintf (stdout,
             "Do you wish to continue? (Type 'yes' to continue.)\n\n> ");

    fgets (buf, 6, stdin);
    buf[7] = '\0';
    while (buf[strlen (buf) - 1] == '\r' || buf[strlen (buf) - 1] == '\n')
      buf[strlen (buf) - 1] = '\0';
    if (g_strcasecmp (buf, "yes") != 0)
      {
        fprintf (stdout,
                 "\nServer exiting. Recompile the server to turn off debugging features.\n"
                 "(Binary distributions should be compiled in release mode. If you did not compile the\n"
                 " server yourself, please contact the place you received it from, or contact the\n"
                 " GNUe team at http://www.gnue.org/ (email info@gnue.org)\n\n");
        exit (0);
      }
    fprintf (stdout,
             "\nThank you for your cooperation. Have a nice day.\n\n");
  }
#endif

  /* set up a timer for speed testing */
#ifdef TIMER_TESTING
  /* select active sections for timing info display */

  gettimeofday( &tv , NULL );
  timer_base_seconds = tv.tv_sec;

  /* clear all */
  for( i=0 ; i<TIMER_SECTIONS ; i++ )
    active_timer_sections[i] = 0;

  for( i=0 ; i<TIMER_FUNC_COUNT ;i++ ) {
    timer_profiles[i].max   = 0;
    timer_profiles[i].min   = 0;
    timer_profiles[i].total = 0;
    timer_profiles[i].count = 0;
    timer_profiles[i].avg   = 0.0;
  }

  active_timer_sections[ TIMER_MISC     ] = 1;
  active_timer_sections[ TIMER_SKELETON ] = 0;
  active_timer_sections[ TIMER_SKELMISC ] = 0;
  active_timer_sections[ TIMER_CACHE    ] = 0;
  active_timer_sections[ TIMER_LISTS    ] = 0;

#endif

  timer_start_operation(TIMER_MISC,"Server initialisation");

  /* read command line arguments, including a config file */
  readOptions (argc, argv);

  /* make lots and lots of OIDs for use */
  refill_oid_buffer (1024 * 1024);

  /* initialise various modules */
  /* Order of steps is significant */

  /* this must be done before any classes are created or loaded */
  message ("Preparing server side data handling classes");
  all_classes = odl_create_empty_tree ();
  make_server_classes ();

#ifdef USE_OAF
  message ("Initialising OAF code");
  oaf_init (argc, argv);
#endif

  message ("Initialising data monitor and log system.");
  init_datamonitor ();          /* set up system */

  /* initialise all appropriate log files here */
  if (get_global_option (configdata, "mainlog"))
    {
      const char *logfile = get_global_option (configdata, "mainlog");
      message ("%s tracks all data accesses.", logfile);

      add_datamonitor_logfile (logfile, DM_ALL, DM_LOG_ALLOW_TRANSACTIONS);
    }
  else
    message ("No log file active.");

#ifdef DEBUG
  if (debug_file_name)
    message ("Debugging output logged to '%s'", debug_file_name);
#endif

  /* needs to be done before any SQL queries */
  message ("Loading initial list of reserved words");
  filename =
    (char *) get_global_option_str (configdata, "mysqlreserved",
                                    "mysql.reserved.lst");
  if (filename)
    reserve_sql_words (filename, OQL_DBTYPE_MYSQL);
  filename =
    (char *) get_global_option_str (configdata, "postgresqlreserved",
                                    "postgresql.reserved.lst");
  if (filename)
    reserve_sql_words (filename, OQL_DBTYPE_POSTGRESQL);

  /* note: server side classes must be first, so can be inherited from */
  message ("Load business class definitions");
  load_class_files ();

  /* note: classes must be loaded */
  message ("Initialising business class method handling.");
  if (initialise_method_handling (configdata))
    {
      if (!load_method_handlers (configdata))
        {
          errormsg ("Method handler could not be initialised.");
        }
    }
  else
    {
      errormsg ("Method handling not supported initialised.");
    }
  message ("Initialising object store.");
  initialise_objectstore (configdata);

  message ("Initialising object cache.");
  create_object_cache (settings.cache_maximum_size);

  /* note: requires objectstore and classes to be loaded */
  message ("Validating/updating database table definitions.");
  update_objectstore_databases (FALSE);

  if (allow_show_classes ())
    odl_display_tree (stdout, all_classes, FALSE);


#ifdef SELF_TEST
  /* optionally run a self test section */
  fprintf (stdout,
           "\n\n----------------------------------------------------------------------------\n"
           "Self test mode has been enabled. Various modules will have a self test\n");
  fprintf (stdout,
           "function for testing purposes. Please make a selection from the\n");
  fprintf (stdout, "available tests:\n");

  while (doneselftest == 0)
    {
      char buf[256];

      buf[255] = '\0';

      fprintf (stdout, "\nOptions:\n\n");
      fprintf (stdout, " Q - Exit program\n");
      fprintf (stdout, " S - Start server\n\n");
      fprintf (stdout,
               " 1 - Object store module (SQL database access system)\n");
      fprintf (stdout, " 2 - Object container system\n");

      fprintf (stdout, "\n> ");
      fflush (NULL);

      fgets (buf, 254, stdin);
      buf[0] = tolower (buf[0]);
      switch (buf[0])
        {
        case 'q':
          fprintf (stdout,
                   "\n\nThank you for playing. Please come again.\n\n");
          exit (0);
        case 's':
          doneselftest = 1;
          break;
        case '1':
          self_test_objectstore ();
          break;
        case '2':
          self_test_objectcontainer ();
          break;
        default:
          break;
        }
    }

#endif

  /* initialise ORB */
  message ("Initialising ORB");
  CORBA_exception_init (&ev);
  the_orb = CORBA_ORB_init (&argc, argv, "orbit-local-orb", &ev);
  root_poa =
    (PortableServer_POA) CORBA_ORB_resolve_initial_references (the_orb,
                                                               "RootPOA",
                                                               &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    fatal_error ("Could not initialise ORB.");
  initialise_object_servants (root_poa);
#if 0
  /* may not be needed: */
  create_fake_servants (&ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    {
      fatal_error ("error: %s", CORBA_exception_id (&ev));
    }
#endif
  pm = PortableServer_POA__get_the_POAManager (root_poa, &ev);
  PortableServer_POAManager_activate (pm, &ev);

  /* create ConnectionFactory object */

  message ("Creating ConnectionFactory instance for new connections");
  factory =
    (GEAS_ConnectionFactory) impl_GEAS_ConnectionFactory__create (root_poa,
                                                                  &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    fatal_error ("Could not create GEAS.ConnectionFactory object.");

  /* create a connection for GEAS to use when talking to itself  :)  */

  server_connection_id = g_new0 (GEAS_object_reference, 1);
  g_assert (server_connection_id != NULL);
  if (server_connection_id)
    {
      server_connection_id->listid = NULL;
      server_connection_id->username = g_strdup (SERVER_PRIVATE_USERNAME);
      server_connection_id->classname = NULL;
      server_connection_id->objectid = NULL;
    }

  server_connection_object =
    make_connection_reference (server_connection_id->username,
                               SERVER_OBJECT_CONNECTION_ID, &ev);

  /* can't do this til the server is, essentially, active */
  message ("Initialising security system");
  init_security_system ();

  /* publish server information */
  message ("Publishing server details.");

#ifdef ENABLE_NAMESERVICE
  register_with_name_service (factory);
#endif

  filename =
    (char *) get_global_option_str (configdata, "pidfile", DEFAULT_PID_FILE);
  fp = fopen (filename, "w");
  if (fp)
    {
      fprintf (fp, "%lu", (long)getpid ());
      fclose (fp);
      fp = NULL;
    }
  else
    {
      warnmsg ("Could not write to file %s", filename);
    }
  message ("PID = %d", getpid ());

  objref = CORBA_ORB_object_to_string (the_orb, factory, &ev);
  if (ev._major != CORBA_NO_EXCEPTION)
    fatal_error ("Could not create GEAS.ConnectionFactory object reference.");

  filename =
    (char *) get_global_option_str (configdata, "iorfile", DEFAULT_IOR_FILE);
  fp = fopen (filename, "w");
  if (!fp)
    {
      fprintf (stderr, "Could not create '%s'\n", filename);
      fprintf (stderr, "IOR String: %s\n", objref);
    }
  else
    {
      fprintf (fp, "%s", objref);
      fclose (fp);
    }

  /* start server */
  timer_done_operation(TIMER_MISC);
  fprintf (stderr, "The GNU Enterprise Application Server is ready.\n");

  CORBA_ORB_run (the_orb, &ev);

  /* Code reaches here once ORB has been shut down */
  message ("Server shutting down.");

  /* TODO: really need to put cleanup code here */

  message ("Server has been shut down.");
  return 0;
}

/** \brief Used for internal operations only
 * Returns a statically allocated CORBA object references, suitable for
 * non-transactional internal operations only.
 */
GEAS_Connection
get_server_connection_object ()
{
  return (server_connection_object);
}

/** \brief Used for internal oeprations only
 * Returns a statically allocated GEAS object identifier, suitable for
 * non-transactional internal operations only.
 */
GEAS_object_reference *
get_server_connection_id (void)
{
  return (server_connection_id);
}

/** \brief Termiante the application
 *  Displays an error message, and terminates the application.
 *  Usually called via a macro (fatal_error()) in geas-server.h
 */
void
do_fatal_error (char *file, int line, char *func, char *fmt, ...)
{
  va_list a;

  /* write to stderr */
  fprintf (stderr, "\n\n*** SERVER EXITING ***\n\n");
  fprintf (stderr, "Error detected in %s() (file %s, at line %d)\n", func,
           file, line);
  va_start (a, fmt);
  vfprintf (stderr, fmt, a);
  va_end (a);
  fprintf (stderr, "\n\n");
  fprintf (stderr,
           "Remember the value of a complete bug report: everyone involved will\n"
           "thank you for a useful description of the problem.\n\n");

  /* write to log file */
  va_start (a, fmt);
  dm_vlogentry (DM_EVENT_ERROR, NULL, fmt, a);
  va_end (a);

  fflush (NULL);

  abort ();
}

/* logging function : for debug output   */

#ifdef DEBUG

/* link error if this is recompiled in release mode, without recompiling
   all other modules that use this */

/** \brief Debugging output function
 * Outputs data to stderr, dependant on the debuggling level. Usually
 * called via a macro defined in geas-server.h
 */
void
debug_output_f (char *type, int minlevel, char *file, char *func,
                unsigned int line, const char *msg, ...)
{
  FILE *debug_out_fp = NULL;
  va_list a;

  /* write debugging info to a file */
  if (debug_file_name
      && (minlevel == DEBUGLEVEL_ALWAYS || minlevel <= filedebuglevel))
    {
      debug_out_fp = fopen (debug_file_name, "a");
      if (debug_out_fp)
        {
          fprintf (debug_out_fp, "%02d: ", minlevel);
          if (type)
            fprintf (debug_out_fp, "[%s] ", type);
          if (file != NULL && func != NULL)
            fprintf (debug_out_fp, "[%s/%d] [%s] ", file, line, func);
          va_start (a, msg);
          vfprintf (debug_out_fp, msg, a);
          va_end (a);
          fprintf (debug_out_fp, "\n");
          fclose (debug_out_fp);
        }
    }

  /* if minlevl == ALWAYS, always display it */
  if (minlevel != DEBUGLEVEL_ALWAYS)
    {
      /* if debugging is off, don't display it */
      /* else if debugging level is lower than the message level */
      /* don't display it */
      /* eg a 'high' detail message isn't displayed if the */
      /* user selected 'medium' detail */
      if (debuglevel == DEBUGLEVEL_OFF || minlevel > debuglevel)
        {
          return;
        }
    }

  if (type)
    fprintf (stderr, "[%s] ", type);
  if (file != NULL && func != NULL)
    fprintf (stderr, "[%s/%d] [%s] ", file, line, func);

  va_start (a, msg);
  vfprintf (stderr, msg, a);
  va_end (a);
  fprintf (stderr, "\n");
}
#endif

/* function for displaying errors to users. extra info in debug mode */

/* should be extended to include logging of errors and warnings */

/** \brief Message output function
 * Outputs data to stderr. Usually called via a macro defined in geas-server.h
 */
void
error_message_out (char *type, char *file, char *func,
                   unsigned int line, char *msg, ...)
{
  va_list a;
  unsigned long int dmtype = 0;

  /* write to log file */
  if (g_strcasecmp (type, "error"))
    dmtype = DM_EVENT_ERROR;
  if (g_strcasecmp (type, "warning"))
    dmtype = DM_EVENT_WARNING;
  va_start (a, msg);
  dm_vlogentry (dmtype, NULL, msg, a);
  va_end (a);

#ifdef DEBUG
  fprintf (stderr, "%s: ", type);
  /* where was the error detected - debug mode only, because most users
     won't be able to do much with it. */
  if (file != NULL)
    fprintf (stderr, "[%s/%d] [%s] ", file, line, func);
#else
  /* if quiet, no errors are printed to stderr, but are still logged */
  if (settings.quiet)
    return;
  fprintf (stderr, "%s: ", type);
#endif

  va_start (a, msg);
  vfprintf (stderr, msg, a);
  va_end (a);
  fprintf (stderr, "\n");
}

/** \brief Display the program version
 */
void
show_version (FILE * fp)
{
  fprintf (fp, "\nGNU Enterprise Application Server\nVersion " VERSION);

  fprintf (fp, ""               /* blank bit so that in the worst
                                   case, the function still
                                   compiles */
#ifdef DEBUG
           " DEBUG"
#endif
#ifdef DEBUG_SECURITY
           " (Security debugging)"
#endif
#ifdef SELF_TEST
           " SELF TEST"
#endif
           "\nDatabase support compiled for:"
#ifdef USE_MYSQL
           " MySQL"
#endif
#ifdef USE_POSTGRESQL
           " PostgreSQL"
#endif
    );

  fprintf (fp,
           "\nCopyright 2001 Free Software Foundation\n"
           "\nSee the file 'COPYING' for details of"
           " the Gnu Public License\nthat covers this program\n\n"
           "http://www.gnue.org/\n\n");
}

/** \brief Long command line options options
 */
#ifdef USE_GNU_GETOPT
static struct option long_options[] = {
  {"config", 1, 0, 'c'},
  {"debug", 1, 0, 'd'},
  {"usage", 0, 0, 'h'},
  {"help", 0, 0, 'h'},
  {"verbose", 0, 0, 'v'},
  {"version", 0, 0, 'V'},
  {"quiet", 0, 0, 'q'},
  {"showconfig", 0, 0, 'C'},
  {"classes", 0, 0, 'l'},
  {0, 0, 0, 0}                  /* end of table marker */
};
#endif

#define OPTION_STRING "c:d:hvVqClO:"

/* private functions                     */
/** \brief Process options
 * Reads options from the configuration file, and command line, using defaults
 * if neither the configuration system or command line specify an option.
 */
static void
readOptions (int argc, char **argv)
{
  int c;
  /* int digit_optind = 0; */
  unsigned int i;
  int fatal = 0;                /* set fatal > 0 to quit. if fatal > 1 print

                                   usage first */
  char *shadowfile = "shadowpw";
  gboolean firstoption = TRUE;

  /* some deaults. order for determining an oiption is: * 1: this default *
     2: config file (overrides default) * 3: cmd line options (overrides
     config file) */
  settings.cache_maximum_size = 50;
  settings.loggeasclasses = CORBA_FALSE;
#ifdef DEBUG
  /* in debug mode, default to verbose, so will display config */
  settings.verbose = CORBA_TRUE;
#else
  settings.verbose = CORBA_FALSE;
#endif
  settings.showconfig = FALSE;

#ifdef DEBUG
  /* default to low debug */
  debuglevel = DEBUGLEVEL_LOW;
#else
  /* debugging not needed */
  debuglevel = DEBUGLEVEL_OFF;
#endif

  settings.showclasses = FALSE;

  /* command line args override config file */
  while (1)
    {
      /* int this_option_optind = optind ? optind : 1; */
      /* int option_index = 0; */

#ifdef USE_GNU_GETOPT
      c =
        getopt_long (argc, argv, OPTION_STRING, long_options, &option_index);
#else
      c = getopt (argc, argv, OPTION_STRING);
#endif
      /* if the very first option is a different configuration file, use
         that. */
      if (firstoption == TRUE)
        {
          char *tmp;
          firstoption = FALSE;

          if (c == 'c')
            {
              /* printf("Using config file '%s'\n" , optarg ); */
              settings.configfile = g_strdup (optarg);
              loadConfig (settings.configfile, shadowfile);
            }
          else
            {
              /* old:  settings.configfile = g_strdup("geas.conf"); */
              settings.configfile = g_strdup (DEFAULT_CONFIG_FILE);
              /* printf("Using default config file '%s'\n" , settings.configfile ); */
              loadConfig (settings.configfile, shadowfile);
            }
          debug_file_name =
            (char *) get_global_option (configdata, "debugfile");
          if (debug_file_name)
            {
              FILE *fp;
              debug_file_name = g_strdup (debug_file_name);
              fp = fopen (debug_file_name, "w");
              if (fp)
                {
                  fprintf (fp, "GEAS debugging log\n------------------\n\n");
                  fclose (fp);
                }
            }
          tmp = (char *) get_global_option (configdata, "debugfilelevel");
          if (tmp)
            {
              filedebuglevel = atoi (tmp);
            }

          /* use global settings from config file, set defaults if not
             in the config * file - uses defaults as set at the start of 
             this function: * get_global_option() returns the last
             argument with no changes * so this effectively means "if
             not defined, then no change" */
          tmp = (char *) get_global_option (configdata, "debuglevel");
          if (tmp)
            {
              if (g_strcasecmp (tmp, "high") == 0)
                debuglevel = DEBUGLEVEL_HIGH;
              else if (g_strcasecmp (tmp, "med") == 0)
                debuglevel = DEBUGLEVEL_MEDIUM;
              else if (g_strcasecmp (tmp, "medium") == 0)
                debuglevel = DEBUGLEVEL_MEDIUM;
              else if (g_strcasecmp (tmp, "low") == 0)
                debuglevel = DEBUGLEVEL_LOW;
              else if (g_strcasecmp (tmp, "off") == 0)
                debuglevel = DEBUGLEVEL_OFF;
              else
                debuglevel =
                  get_global_option_int (configdata, "debuglevel",
                                         debuglevel);
            }
          settings.verbose =
            get_global_option_bool (configdata, "verbose", settings.verbose);
          settings.loggeasclasses =
            get_global_option_bool (configdata, "logserverclasses",
                                    settings.loggeasclasses);

          settings.cache_maximum_size =
            get_global_option_int (configdata, "maximumcache",
                                   settings.cache_maximum_size);
          firstoption = FALSE;

          /* if used a different config file, get next option */
          if (c == 'c')
            continue;
        }

      if (c == -1)
        break;

      switch (c)
        {
        case 'O':
          /* ignore - is an ORBit option */
          break;
        case 'C':
          settings.showconfig = TRUE;
          break;
        case 'l':
          settings.showclasses = TRUE;
          break;
        case 'q':
          settings.quiet = TRUE;
          break;
        case 'V':
          /* display version then quit */
          show_version (stdout);
          if (fatal < 1)
            fatal = 1;
          break;
        case 'c':
          fprintf (stderr,
                   "error: option -c or -config must be the first option given.\n");
          if (fatal < 2)
            fatal = 2;
          break;
        case 'd':
          if (g_strcasecmp (optarg, "off") == 0
              || g_strcasecmp (optarg, "0") == 0)
            debuglevel = DEBUGLEVEL_OFF;
          else if (g_strcasecmp (optarg, "low") == 0)
            debuglevel = DEBUGLEVEL_LOW;
          else if (g_strcasecmp (optarg, "medium") == 0)
            debuglevel = DEBUGLEVEL_MEDIUM;
          else if (g_strcasecmp (optarg, "high") == 0)
            debuglevel = DEBUGLEVEL_HIGH;
          else
            {
              /* check it's just numbers */
              for (i = 0; i < strlen (optarg); i++)
                if (!isdigit (optarg[i]))
                  {
                    i = strlen (optarg);
                    usage_debuglevel ();
                    if (fatal < 1)
                      fatal = 1;
                  }
              debuglevel = atoi (optarg);
              /* check it's in range */
              if (debuglevel < DEBUGLEVEL_LOW || debuglevel > DEBUGLEVEL_HIGH)
                {
                  usage_debuglevel ();
                  if (fatal < 1)
                    fatal = 1;
                }
            }
          break;
        case 'v':
          settings.verbose = CORBA_TRUE;
          break;
        case 'h':
          usage ();
          exit (0);
        case '?':
          if (fatal < 2)
            fatal = 2;          /* show usage */
          break;
        default:
          fprintf (stderr, "Unknown option (ASCII 0x%02x)\n", c);
          if (fatal < 2)
            fatal = 2;          /* show usage */
          break;
        }
    }

  /* stop now if there was an error */
  if (fatal > 1)
    usage ();                   /* print usage? */
  if (fatal > 0)
    exit (0);                   /* exit now if any errors */

#ifndef DEBUG
  /* warning for debug level option in non debug version */
  if (debuglevel > 0)
    {
      warnmsg
        ("Debugging support has not been compiled in. (Debug level %d ignored.)",
         debuglevel);
      debuglevel = DEBUGLEVEL_OFF;
    }
#endif

#ifdef SELF_TEST
  if (settings.verbose == CORBA_TRUE)
    {
      if (settings.quiet)
        message ("Self test mode active (quiet mode ignored)");
      else
        message ("Self test mode active");

      settings.quiet = FALSE;
    }
  else
    {
      if (settings.quiet)
        message
          ("Self test mode active: verbose mode turned on automatically (quiet mode ignored)");
      else
        message
          ("Self test mode active: verbose mode turned on automatically");

      settings.quiet = FALSE;
      settings.verbose = CORBA_TRUE;
    }
#endif

  if (allow_show_version ())
    show_version (stdout);

  /* display options in debug mode */
  if (allow_show_config ())
    {
      char *msg;

      fprintf (stdout, "Configuration file loaded:\n\n");
#ifdef DEBUG
      if (debuglevel >= DEBUGLEVEL_1)
        {
#ifdef DEBUG_SECURITY
          /* show passwords for security debugging */
          show_configuration (configdata, TRUE);
#else
          /* hide passwords when not debugging security */
          show_configuration (configdata, FALSE);
#endif
          fprintf (stdout, "\n");
        }
#endif

      if (debuglevel < 4)
        msg = "low detail";     /* 1  2  3    */
      else if (debuglevel < 8)
        msg = "medium detail";  /* 4  5  6  7 */
      else
        msg = "high detail";    /* 8  9 10    */

      if (debuglevel != DEBUGLEVEL_OFF)
        fprintf (stdout, "Debug level         : %d (%s)\n", debuglevel, msg);
      else
        fprintf (stdout, "Debug level         : off\n");

      fprintf (stdout, "Verbose             : %s\n",
               (settings.verbose == CORBA_TRUE) ? "On" : "Off");
      fprintf (stdout, "Max cache size      : %d\n",
               (int)settings.cache_maximum_size);
      fprintf (stdout, "Log geas classes    : %s\n",
               (settings.loggeasclasses ? "Yes" : "No"));
      fprintf (stdout, "\n");
      fprintf (stdout, "PID file            : %s\n",
               (char *) get_global_option_str (configdata, "pidfile",
                                               DEFAULT_PID_FILE));
      fprintf (stdout, "IOR file            : %s\n",
               (char *) get_global_option_str (configdata, "iorfile",
                                               DEFAULT_IOR_FILE));

      fprintf (stdout, "GCD directory       : %s\n",
               (char *) get_global_option_str (configdata, "classdir", "."));
      fprintf (stdout, "Database change log : %s\n",
               (char *) get_global_option_str (configdata,
                                               "databasechangefile",
                                               "database.changes.txt"));
      fprintf (stdout, "\n");

      if (optind < argc)
        {
          fprintf (stdout, "Non-option arguments: (ignored)\n");
          while (optind < argc)
            fprintf (stdout, "  %s\n", argv[optind++]);
          fprintf (stdout, "\n");
        }
    }
  else
    {
#ifdef DEBUG
      /* DEBUG mode, but debuglevel == DEBUGLEVEL_OFF */
      if (debuglevel == DEBUGLEVEL_OFF)
        warnmsg ("Debugging support has been turned off.");
#endif
    }

  if (fatal > 1)
    usage ();                   /* print usage? */
  if (fatal > 0)
    exit (0);                   /* exit now if any errors */
}

/** \brief Test if the version should be displayed
 */
gboolean
allow_show_version ()
{
  if (settings.verbose == CORBA_TRUE && !settings.quiet)
    return (TRUE);
  else
    return (FALSE);
}

/** \brief Test if informational messages should be displayed
 */
gboolean
allow_show_message (void)
{
  if (settings.verbose == CORBA_TRUE && !settings.quiet)
    return (TRUE);
  else
    return (FALSE);
}

/** \brief Test if the current configuration should be displayed
 */
gboolean
allow_show_config (void)
{
  if (settings.showconfig == CORBA_TRUE)
    return (TRUE);
  else
    return (FALSE);
}

/** \brief Test if error messages should be displayed
 */
gboolean
allow_show_errors (void)
{
  if (!settings.quiet)
    return (TRUE);
  else
    return (FALSE);
}

/** \brief Test if the options in use should be displayed
 */
gboolean
allow_show_options (void)
{
  if (debuglevel >= DEBUGLEVEL_LOW || settings.verbose == TRUE)
    return (TRUE);
  else
    return (FALSE);
}

/** \brief Test if the class tree should be shown
 */
gboolean
allow_show_classes (void)
{
  if (settings.quiet == FALSE && settings.showclasses == TRUE)
    return (TRUE);
  else
    return (FALSE);
}

/** \brief Test if the logfile should include server private classes
 */
gboolean
allow_log_class_event (const char *classname)
{
#ifdef DEBUG
  /* debug mode */
  if (strncmp (classname, "geas::", 6) == 0)
    {
      /* in the GEAS module, only log if configured to log GEAS classes */
      if (settings.loggeasclasses == CORBA_TRUE)
        return (TRUE);
      return (FALSE);
    }
  /* always log non geas:: classes */
  return (TRUE);
#else
  /* normal mode */
  if (strncmp (classname, "geas::", 6) == 0)
    {
      /* in the GEAS module, do not log geas:: classes */
      return (FALSE);
    }
  /* always log non geas:: classes */
  return (TRUE);
#endif
}

/** \brief Display valid command line options
 */
static void
usage ()
{
  fprintf (stdout,
           "\nUsage: geas-server [-c configfile] [other options]\n\n");

#ifdef USE_GNU_GETOPT
  fprintf (stdout,
           "--config FILE,   -c FILE   : Set configuration file to use.\n");

#ifdef DEBUG
  fprintf (stdout,
           "--debug LEVEL,   -d LEVEL  : Set debug level to 'LEVEL'.\n");
#endif

  fprintf (stdout,
           "--verbose,       -v        : Verbose server output on. (Defaults to 'off'.)\n");
  fprintf (stdout,
           "--quiet,         -q        : Hide error messages (Defaults to 'off')\n");
  fprintf (stdout,
           "--version,       -V        : Show version, then quit immediately.\n");
  fprintf (stdout, "--showconfig,    -C        : Show configuration file.\n");

  fprintf (stdout, "--usage,         -u        :\n");
  fprintf (stdout, "--help,          -h        : Show this help text.\n");

  fprintf (stdout,
           "\n(Note: -c must be the first command line option, if it is used.)\n\n");

  fprintf (stdout, "ORBit related options:\n\n");
  fprintf (stdout,
           "-ORBIIOPIPv4=1             : Enable IIOP over TCP/IPIP\n");
  fprintf (stdout,
           "-ORBIIOPIPv4=0             : Disable IIOP over TCP/IPIP\n");
  fprintf (stdout,
           "-ORBIIOPUSock=1            : Enable IIOP over UNIX sockets\n");
  fprintf (stdout,
           "-ORBIIOPUSock=0            : Disable IIOP over UNIX sockets\n");
  fprintf (stdout, "\n");

#else
  fprintf (stdout, " -c FILE        : Set configuration file to use.\n");

#ifdef DEBUG
  fprintf (stdout, " -d LEVEL       : Set debug level to 'LEVEL'.\n");
#endif

  fprintf (stdout,
           " -v             : Verbose server output on. (Defaults to 'off'.)\n");
  fprintf (stdout,
           " -q             : Hide error messages (Defaults to 'off')\n");
  fprintf (stdout,
           " -V             : Show version, then quit immediately.\n");
  fprintf (stdout, " -C             : Show configuration file.\n");

  fprintf (stdout, " -u OR -h       : Show this help text.\n");
  fprintf (stdout, "\n");
  fprintf (stdout, "ORBit related options:\n\n");
  fprintf (stdout, "-ORBIIOPIPv4=1  : Enable IIOP over TCP/IPIP\n");
  fprintf (stdout, "-ORBIIOPIPv4=0  : Disable IIOP over TCP/IPIP\n");
  fprintf (stdout, "-ORBIIOPUSock=1 : Enable IIOP over UNIX sockets\n");
  fprintf (stdout, "-ORBIIOPUSock=0 : Disable IIOP over UNIX sockets\n");
  fprintf (stdout, "\n");
#endif

  fprintf (stdout,
           "Debug level: This sets the level of debugging output. It can be one of 'off',\n");
  fprintf (stdout,
           "'low', 'medium' or 'high'. Alternatively, an integer from 0 to 10 may be used,\n");
  fprintf (stdout, "where '0' is 'off' and '10' is maximum detail.\n\n");
  fprintf (stdout, "Note: 'quiet' mode cancels 'verbose' mode.\n");

#ifdef SELF_TEST
  fprintf (stdout,
           "In self test mode, verbose is turned on automatically, but\n");
  fprintf (stdout, "quiet mode can cancel this.\n");
#endif

  fprintf (stdout, "\n");
}

/** \brief Describe the debuglevel system
 */
static void
usage_debuglevel ()
{
  fprintf (stdout,
           "\nDebug level must be either off, low, medium, or high. Numeric values must be\n"
           "between %d and %d (inclusive).\n\n",
           DEBUGLEVEL_LOW, DEBUGLEVEL_HIGH);
}

/** \brief Load the configuration file
 */
static void
loadConfig (const char *filename, const char *shadowpasswordfile)
{
  configuration c;
  FILE *fp;

  printf ("Trying to open config file '%s'\n", filename);
  fp = fopen (filename, "r");
  if (fp != NULL)
    {
      /* file exists!  :) */
      fclose (fp);
    }
  else
    {
      /* nope, so try a final fallback */
      filename = "geas.conf";
      printf ("nope, so trying config file '%s'\n", filename);
    }

  debug_output (DEBUGLEVEL_2, "Loading configuration from: %s", filename);

  c = load_configuration_file (filename, shadowpasswordfile);
  if (!c)
    {
      {
        /* load_configuration() prints error messages to stderr */
        errormsg ("Failed to read configuration data.");
        exit (0);
      }
    }
  if (get_global_option (c, "configfile") != NULL &&
      strcmp ("unknown",
              get_global_option_str (c, "configfile", "unknown")) == 0)
    {
      /* load_configuration() prints error messages to stderr */
      errormsg ("Failed to find or load the configuration file.");
      exit (0);
    }

  /* store the config data */
  configdata = c;

  loadTableAliases (get_global_option_str
                    (configdata, "tablealiases", "./geas-table-aliases"));
}

/* always printed */
/** \brief Display serious errors (can not be disabled at run time)
 */
void
criticalerror (char *fmt, ...)
{
  va_list a;

  fprintf (stderr,
           "\n\n**********************************************************************\n");
  va_start (a, fmt);
  vfprintf (stderr, fmt, a);
  va_end (a);
  fprintf (stderr, "\n");
}

/** \brief Display tect messages (if not in 'quiet' mode)
 */
void
message (char *fmt, ...)
{
  va_list a;

  /* write to log file */
  va_start (a, fmt);
  dm_vlogentry (DM_EVENT_MESSAGE, NULL, fmt, a);
  va_end (a);

  if (!allow_show_message ())
    return;

  va_start (a, fmt);
  vfprintf (stdout, fmt, a);
  va_end (a);
  fprintf (stdout, "\n");
}

#ifdef SELF_TEST
void
do_self_test_message (char *fmt, ...)
{
  va_list a;

  if (settings.quiet == TRUE)
    return;

  if (settings.quiet)
    return;

  va_start (a, fmt);
  vfprintf (stderr, fmt, a);
  va_end (a);
  fprintf (stderr, "\n");
}
#endif

/** \brief Load the GCD files
 */
void
load_class_files (void)
{
  char *name;
  odl_filenamelist *fl = NULL;

  name = (char *) get_global_option_str (configdata, "classdir", ".");
  if (name == NULL)
    {
      perror ("Cannot parse classdir option");
      exit (-1);
    }

  fl = odl_get_files_from_dir (fl, name);

  if (!fl)
    {
      errormsg
        ("No class definitions were found. Please ensure the configuration file\ndefines a directory with valid '.gcd' files.");
      exit (-1);
    }

  /* load GCD files */
  all_classes = odl_load_files (fl, all_classes);
  odl_filenamelist_free (fl);

  /* load methods */
  /* TODO: */
}

#ifdef ENABLE_NAMESERVICE

static void
register_with_name_service (GEAS_ConnectionFactory factory)
{
  CORBA_Environment ev;
  FILE *fp;
  char ior[1024];

  fp =
    fopen (get_global_option_str
           (configdata, "nameserviceior", "/tmp/orbit-name"), "r");
  if (!fp)
    {
      errormsg ("Could not read name service IOR");
      return;
    }
  fgets (ior, 2044, fp);
  fclose (fp);
  ior[1023] = '\0';             /* make sure it's terminated, just in case */
  while (ior[strlen (ior) - 1] == 10 || ior[strlen (ior) - 1] == 13)
    ior[strlen (ior) - 1] = '\0';

  CORBA_exception_init (&ev);
  message ("Attempting to register with naming service");
  name_srv = CORBA_ORB_string_to_object (the_orb, ior, &ev);
  if (ev._major == CORBA_NO_EXCEPTION)
    {
      main_name = create_name ("GEAS");
      CosNaming_NamingContext_bind (name_srv, main_name, factory, &ev);
      if (ev._major != CORBA_NO_EXCEPTION)
        {
          errormsg ("Could not bind name to name service : %s",
                    CORBA_exception_id (&ev));
        }
      main_name = create_name ("ConnectionFactory");
      CosNaming_NamingContext_bind (name_srv, main_name, factory, &ev);
      if (ev._major != CORBA_NO_EXCEPTION)
        {
          errormsg ("Could not bind name to name service : %s",
                    CORBA_exception_id (&ev));
        }
    }
  else
    {
      warnmsg ("Could not locate name service.");
    }
  CORBA_exception_free (&ev);
}

CosNaming_Name *
create_name (const char *name)
{

  CosNaming_Name *cos_name;
  CosNaming_NameComponent *cos_name_cmp;
  char *id;
  char *kind;
  char *dummy_kind = "filler";

  /* Allocate a CosNaming::Name (sequence of CosNaming::NameComponent) */
  cos_name = (CosNaming_Name *) g_malloc (sizeof (CosNaming_Name));
  cos_name->_maximum = 1L;
  cos_name->_length = 1L;

  /* Relinquish ownership of the NameComponent to the sequence. When
     CORBA_free is called on it later, the NameComponent will be freed */
  CORBA_sequence_set_release (cos_name, FALSE);

  /* Create the naming component.  We don't care about the kind, so
     we give it a dummy value */
  cos_name_cmp = (CosNaming_NameComponent *)
    g_malloc (sizeof (CosNaming_NameComponent));

  /* Create id and kind value components. id is the name of our object */
  id = CORBA_string_alloc (strlen (name));
  strcpy (id, name);
  cos_name_cmp->id = id;
  kind = CORBA_string_alloc (strlen (dummy_kind));
  strcpy (kind, dummy_kind);
  cos_name_cmp->kind = kind;

  cos_name->_buffer = cos_name_cmp;

  return cos_name;
}

#endif

/** \brief Marks standard words as reserved
*/

static void
reserve_sql_words (const char *file, unsigned long int dbtype)
{
  FILE *fp;
  char line[256];
  char *p, *q;

  trace_functioncall ();

  fp = fopen (file, "r");
  g_assert (fp != NULL);
  if (!fp)
    {
      printf ("Error: Opening reserved SQL words file %s, error number %d.\n",
              file, errno);
      return;
    }

  while (!feof (fp))
    {
      if (fgets (line, 254, fp) != NULL)
        {
          /* remove comments */
          p = strchr (line, '#');
          if (p)
            *p = '\0';

          /* trim line */
          line[255] = '\0';
          p = line;
          while (p[strlen (p) - 1] == '\n' || isspace (p[strlen (p) - 1]))
            p[strlen (p) - 1] = '\0';
          p = line;
          while (isspace (*p))
            p++;

          while ((q = strtok (p, " \t")) != NULL)
            {
              /* printf( "reserving '%s'\n" , q ); */
              add_reserved_word (q, dbtype, TRUE);
              p = NULL;
            }
        }
    }
  fclose (fp);

#if 0
  printf ("%s %s reserved in %s\n",
          "position", is_word_reserved ("position",
                                        OQL_DBTYPE_MYSQL) ? "is" : "isn't",
          "MySQL");
  printf ("%s %s reserved in %s\n", "all",
          is_word_reserved ("all", OQL_DBTYPE_MYSQL) ? "is" : "isn't",
          "MySQL");

  printf ("%s %s reserved in %s\n",
          "position", is_word_reserved ("position",
                                        OQL_DBTYPE_POSTGRESQL) ? "is" :
          "isn't", "PostgreSQL");
  printf ("%s %s reserved in %s\n", "all",
          is_word_reserved ("all", OQL_DBTYPE_POSTGRESQL) ? "is" : "isn't",
          "PostgreSQL");
#endif
}

struct DatabaseAliases
{
  char *databasename;
  GList *tables;
  GList *columns;
};

struct TableAlias
{
  char *classname;
  char *tablename;
  gboolean modifyable;
  char *pkey;
};

struct ColumnAlias
{
  struct TableAlias *table;
  char *fieldname;
  char *columnname;
};

GList *database_alias_store = NULL;

struct DatabaseAliases *
make_database_alias (const char *name)
{
  struct DatabaseAliases *d = g_new0 (struct DatabaseAliases, 1);
  g_assert (d != NULL);
  if (d)
    {
      d->databasename = g_strdup (name);
      d->tables = NULL;
      d->columns = NULL;
    }
  return (d);
}

struct TableAlias *
make_table_alias (const char *classname, const char *tablename,
                  gboolean modifyable, const char *pkey)
{
  struct TableAlias *a = g_new0 (struct TableAlias, 1);
  g_assert (a != NULL);
  if (a)
    {
      a->classname = g_strdup (classname);
      a->tablename = g_strdup (tablename);
      a->modifyable = modifyable;
      a->pkey = g_strdup (pkey);
    }
  return (a);
}

struct ColumnAlias *
make_column_alias (struct TableAlias *table,
                   const char *fieldname, const char *columnname)
{
  struct ColumnAlias *c = g_new0 (struct ColumnAlias, 1);
  g_assert (fieldname != NULL);
  g_assert (columnname != NULL);
  g_assert (table != NULL);
  if (c)
    {
      c->table = table;
      c->fieldname = g_strdup (fieldname);
      c->columnname = g_strdup (columnname);
    }
  return (c);
}

static struct DatabaseAliases *
find_alias_database (char *name)
{
  GList *l = database_alias_store;
  while (l)
    {
      struct DatabaseAliases *d = (struct DatabaseAliases *) l->data;
      if (g_strcasecmp (name, d->databasename) == 0)
        return (d);
      l = g_list_next (l);
    }
  return (NULL);
}
static struct TableAlias *
find_alias_table (GList * tables, const char *name)
{
  while (tables)
    {
      struct TableAlias *t = (struct TableAlias *) tables->data;
      if (g_strcasecmp (t->classname, name) == 0)
        return (t);
      tables = g_list_next (tables);
    }
  return (NULL);
}

/*
static struct ColumnAlias *
find_alias_column (GList * cols, const char *name)
{
  while (cols)
    {
      struct ColumnAlias *c = (struct ColumnAlias *) cols->data;
      if (g_strcasecmp (c->fieldname, name) == 0)
        return (c);
      cols = g_list_next (cols);
    }
  return (NULL);
}
*/

static void
loadTableAliases (const char *filename)
{
  struct DatabaseAliases *db = NULL;
  FILE *fp;
  char buf[1024];
  char *p, *q;
  int lineno;
  char classname[1024];
  char database[1024];
  char tablename[1024];
  char pkey[1024];
  char column[1024];
  gboolean modify = FALSE;

  printf ("Loading aliases from file '%s'\n", filename);

  fp = fopen (filename, "r");
  if (!fp)
    return;

  buf[1023] = '\0';

  lineno = 0;
  while (!feof (fp))
    {
      fgets (buf, 1022, fp);
      lineno++;
      q = strchr (buf, '#');
      if (q)
        *q = '\0';

      p = buf;
      while (isspace (p[strlen (p) - 1]) ||
             p[strlen (p) - 1] == 10 ||
             p[strlen (p) - 1] == 12 || p[strlen (p) - 1] == 13)
        p[strlen (p) - 1] = '\0';
      if (strlen (p) == 0)
        continue;

      /* get classname */
      while (p && *p != '\0' && isspace (*p))
        p++;
      if (!p || *p == '\0')
        goto linefailed;

      q = p;
      while (!isspace (*q))
        q++;
      if (*q == '\0')
        goto linefailed;
      *q = '\0';
      strcpy (classname, p);
      p = strchr (classname, '.');
      if (p)
        {
          *p = '\0';
          strcpy (column, p + 1);
        }
      else
        strcpy (column, "");
      p = q + 1;


      /* get tablename */
      while (p && *p != '\0' && isspace (*p))
        p++;
      if (!p || *p == '\0')
        goto linefailed;

      if (*p == '"')
        {
          p++;
          q = p;
          while (*q != '\0' && *q != '"')
            q++;
          if (*q == '\0')
            goto linefailed;
          *q = '\0';
          strcpy (tablename, p);
          p = q + 1;
        }
      else
        {
          q = p;
          while (!isspace (*q))
            q++;
          if (*q == '\0')
            goto linefailed;
          *q = '\0';
          strcpy (tablename, p);
          p = q + 1;
        }

      /* get database */
      while (p && *p != '\0' && isspace (*p))
        p++;
      if (!p || *p == '\0')
        goto linefailed;

      q = p;
      while (!isspace (*q))
        q++;
      if (*q == '\0')
        goto linefailed;
      *q = '\0';
      strcpy (database, p);
      p = q + 1;

      /* get can modify table? */
      if (strlen (column) > 0)
        goto linedone;
      while (p && *p != '\0' && isspace (*p))
        p++;
      if (!p || *p == '\0')
        goto linefailed;

      q = p;
      while (!isspace (*q) && *q != '\0')
        q++;
      if (*q == '\0')
        {
          if (g_strcasecmp (p, "YES") == 0 || g_strcasecmp (p, "TRUE") == 0)
            modify = TRUE;
          if (g_strcasecmp (p, "NO") == 0 || g_strcasecmp (p, "FALSE") == 0)
            modify = FALSE;
          p = q;
        }
      else
        {
          *q = '\0';
          if (g_strcasecmp (p, "YES") == 0 || g_strcasecmp (p, "TRUE") == 0)
            modify = TRUE;
          if (g_strcasecmp (p, "NO") == 0 || g_strcasecmp (p, "FALSE") == 0)
            modify = FALSE;
          p = q + 1;
        }

      /* get primary key (in existing table) */
      strcpy (pkey, "");
      while (p && *p != '\0' && isspace (*p))
        p++;
      if (!p || *p == '\0')
        {
          goto linedone;        /* optional part */
        }

      q = p;
      while (!isspace (*q))
        q++;
      if (*q == '\0')
        goto linefailed;
      *q = '\0';
      strcpy (pkey, p);
      p = q + 1;

    linedone:

      db = find_alias_database (database);
      if (!db)
        {
          db = make_database_alias (database);
          database_alias_store = g_list_append (database_alias_store, db);
        }

      printf ("Classname   : %s\n", classname);
      if (strlen (column) > 0)
        {
          struct TableAlias *t;
          struct ColumnAlias *c;
          t = find_alias_table (db->tables, classname);
          c = make_column_alias (t, column, tablename);
          db->columns = g_list_append (db->columns, c);

          printf ("Field name  : %s\n", column);
          printf ("Column name : %s\n", tablename);
          printf ("Database    : %s\n", database);
        }
      else
        {
          struct TableAlias *t;

          t = find_alias_table (db->tables, classname);
          if (!t)
            {
              t = make_table_alias (classname, tablename, modify, pkey);
              db->tables = g_list_append (db->tables, t);
            }

          printf ("Table name  : %s\n", tablename);
          printf ("Database    : %s\n", database);
          printf ("Can modify? : %s\n", (modify ? "YES" : "NO"));

        }
      if (strlen (pkey) > 0)
        printf ("Primary key : %s\n", pkey);
      printf ("\n");
      continue;                 /* go to top of for loop */

    linefailed:
      /* reaches here if a line has an error */
      errormsg ("Parse error in line %d of file '%s'", lineno, filename);
    }


  {
    GList *l, *m;
    struct DatabaseAliases *d;
    struct TableAlias *t;
    struct ColumnAlias *c;

    l = database_alias_store;
    while (l)
      {
        d = l->data;
        printf ("Database: %s\n", d->databasename);
        m = d->tables;
        while (m)
          {
            t = m->data;
            printf ("    Table : %s -> %s\n", t->classname, t->tablename);
            m = g_list_next (m);
          }
        m = d->columns;
        while (m)
          {
            c = m->data;
            printf ("    Column : %s.%s -> %s\n",
                    (c->table != NULL ? (c->table->tablename) : "(unknown)"),
                    c->fieldname, c->columnname);
            m = g_list_next (m);
          }
        l = g_list_next (l);
      }
  }
}
