/*
   dataobject.c - 

   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2000 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: dataobject.c,v 1.60 2001/07/25 19:57:05 reinhard Exp $
   
*/

/** \file dataobject.c
 *  \brief GEAS::DataObject implementation
 */

#include <ctype.h>
#include <stdlib.h>

#include "config.h"

#include "geas.h"
#include "geas-server.h"
#include "dataobject.h"
#include "connection.h"
#include "exceptions.h"
#include "classdata.h"
#include "objectcache/objectcache.h"
#include "oql/oql.h"
#include "collectiondata.h"
#include "methods/methods.h"
#include "datamonitor/datamonitor.h"
#include "config/configuration.h"
#include "objectstore/objectstore.h"
#include "transaction.h"
#include "schema.h"

/* raises: */
GEAS_ObjectReference
DataObject__get_objectID (GEAS_object_reference * id, CORBA_Environment * ev)
{
  return ((GEAS_ObjectReference) DataObject_getField (id, "objectid", ev));
}

/* raises: */
CORBA_boolean
DataObject_ofclass (GEAS_object_reference * id, CORBA_char * classname,
                    CORBA_Environment * ev)
{
  CORBA_boolean retval = CORBA_FALSE;
  odl_class *cl;

  cl = odl_find_class (all_classes, id->classname, NULL);
  if (!cl)
    return (CORBA_FALSE);

  if (odl_class_is_instanceof (cl, classname))
    retval = CORBA_TRUE;
  else
    retval = CORBA_FALSE;

  return (retval);
}

/* raises: */
CORBA_char *
DataObject__get_classname (GEAS_object_reference * id, CORBA_Environment * ev)
{
  CORBA_char *retval = CORBA_string_dup (id->classname);
  return (retval);
}

/* raises: */
GEAS_classnames *
DataObject__get_classes (GEAS_object_reference * id, CORBA_Environment * ev)
{
  GEAS_classnames *retval = NULL;
  odl_class *c;
  GList *parents, *l;
  int count, i;
  g_assert (id != NULL);
  g_assert (ev != NULL);

  c = odl_find_class (all_classes, id->classname, NULL);
  if (!c)
    {
      return (NULL);
    }
  retval = GEAS_classnames__alloc ();
  CORBA_sequence_set_release (retval, CORBA_TRUE);
  if (!retval)
    return (NULL);

  /* get list of parent classes */
  parents = odl_class_get_parentnames (c);
  count = g_list_length (parents);
  printf ("count = %d\n", count);
  /* store data to return */
  retval->_maximum = (count + 1);
  retval->_length = (count + 1);
  retval->_buffer = CORBA_sequence_CORBA_string_allocbuf (count + 1);
  retval->_buffer[0] = CORBA_string_dup (id->classname);
  printf ("%s\n", retval->_buffer[0]);
  /* if there were any parent classes, record their name too */
  l = parents;
  i = 1;
  while (l)
    {
      retval->_buffer[i] = CORBA_string_dup (l->data);
      printf ("%s\n", retval->_buffer[i]);
      i++;
      l = g_list_next (l);
    }
  if (parents)
    g_list_free (parents);

  return (retval);
}

/* raises: */
GEAS_fieldlist *
DataObject__get_allfields (GEAS_object_reference * id, CORBA_Environment * ev)
{
  return (schema_get_fields (id, id->classname, TRUE, ev));
}

/* raises: UnknownField, NotLookupField, Interrupted, ServerError, Locked, TransactionNotInProgress */
GEAS_LookupOptions *
DataObject_getLookupFieldOptions (GEAS_object_reference * id, char *field,
                                  CORBA_Environment * ev)
{
  /* Note: in the proposed security system, the Connection class's mehtod
   * will believe it's been called directly by the current user, so can
   * do appropriate security checks there as well */
  return (GEAS_Connection_getLookupFieldOptions
          (id->server, id->classname, field, ev));
}

/* raises: UnknownField, NotWriteable, Interrupted, ServerError, Locked, TransactionNotInProgress */
void
DataObject_setLookupField (GEAS_object_reference * id,
                           GEAS_LookupOption * setting,
                           CORBA_Environment * ev)
{
  unsigned int i;
  GEAS_setlookuplist *list;

  /* TODO */
  list = &setting->setdata;

  for (i = 0; i < list->_length; i++)
    {
      DataObject_setField (id, list->_buffer[i].field, list->_buffer[i].value,
                           ev);
      if (ev->_major != CORBA_NO_EXCEPTION)
        {
          return;

        }
    }
}

/* raises: UnknownField, NotReadable, Interrupted, NullField, ServerError, NotDataField, Locked, TransactionNotInProgress */
extern int sillytest;
CORBA_char *
DataObject_getField (GEAS_object_reference * id, CORBA_char * fieldname,
                     CORBA_Environment * ev)
{
  CORBA_char *retval = NULL;
  ObjectData *obj;
  odl_class *cl;
  odl_field *f;
  /* int err; */

  trace_functioncall ();
  /*  printf( "get %s.%s" , id->classname , fieldname ); */

  /* which class are we interested in? */
  if (strcasecmp (fieldname, "objectid") == 0)
    return (CORBA_string_dup (id->objectid));

  cl = odl_find_class (all_classes, id->classname, NULL);
  if (!cl)
    {
      /* should never happen */
      make_ServerError_exception (ev, "Unknown class %s", id->classname);
      return (NULL);
    }

  cl = odl_field_defined_in (cl, fieldname);
  if (!cl)
    {
      make_UnknownField_exception (ev,
                                   "Field '%s' is not defined in class '%s'",
                                   fieldname, id->classname);
      return (NULL);
    }

  f = odl_class_get_field (cl, fieldname);
  switch (odl_field_get_type (f))
    {
    case FT_basic:
    case FT_lookup:
      /* normal fields, handled by cache */
      break;

    case FT_reference:
    case FT_list:
      make_NotDataField_exception (ev,
                                   "Field '%s' is not a plain data field.",
                                   fieldname);
      return (NULL);

    case FT_method:
    case FT_calculated:
    case FT_readonly:
      make_ServerError_exception (ev,
                                  "Field '%s' has an unsupported type identifier (%s).",
                                  fieldname,
                                  odl_fieldtype_name (odl_field_get_type
                                                      (f)));
      return (NULL);

    case FT_unknown:
      make_ServerError_exception (ev,
                                  "Field '%s' has an unknown type identifier.",
                                  fieldname);
      return (NULL);
    }

  /* find object in cache, and get field */
  /* printf( "looking for %s/%s\n" , odl_class_get_full_name(cl), id->objectid ); */
  /* don't log GEAS classes */


  if (allow_log_class_event (odl_class_get_full_name (cl)))
    dm_event (id->username, id->currenttransaction, DM_EVENT_FIELD_READ,
              odl_class_get_full_name (cl), id->objectid, fieldname);

//if( sillytest == 1 )
//   return CORBA_string_dup( "hey" );

  obj = oc_find_object_by_key (odl_class_get_full_name (cl), id->objectid);
  if (obj)
    {
//if( sillytest == 0 )
      retval = (char *) oc_get_object_field (obj, fieldname);
//else retval = CORBA_string_dup( "hello" );
      if (retval)
        {
          char *t = CORBA_string_dup (retval);
          g_free (retval);
          retval = t;
        }
      else
        {
          retval = CORBA_string_dup ("");
          /* make_NullField_exception(ev, "No data in field %s.%s" , id->classname , fieldname ); */
        }
    }
  else
    {
    return CORBA_string_dup( "hey" );
      /* this should, in theory, be impossible */
      make_ServerError_exception (ev,
                                  "1 Impossible error: object went missing. (%s/%s)",
                                  id->classname, id->objectid);
      errormsg ("Impossible error: object %s/%s went missing.", id->classname,
                id->objectid);
    }

  /* message( "returning '%s'" , retval ); */
  /* if( ev->_major != CORBA_NO_EXCEPTION ) message( "exception: %s\n" ,
     CORBA_exception_id(ev) ); */
  return (retval);
}

/* raises: UnknownField, NotWriteable, Interrupted, ServerError, NotDataField, Locked, TransactionNotInProgress, Format */
void
DataObject_setField (GEAS_object_reference * id, CORBA_char * fieldname,
                     CORBA_char * value, CORBA_Environment * ev)
{
  ObjectData *obj;
  odl_class *cl;
  odl_field *f;

  /* message( "\n%s/%s.%s = %s" , id->classname , id->objectid , fieldname , value );  */
  /* which class are we interested in? */
  cl = odl_find_class (all_classes, id->classname, NULL);
  if (!cl)
    {
      make_ServerError_exception (ev, "Unknown class %s", id->classname);
      return;
    }
  cl = odl_field_defined_in (cl, fieldname);
  if (!cl)
    {
      make_UnknownField_exception (ev,
                                   "Field '%s' is not defined in class '%s'",
                                   fieldname, id->classname);
      return;
    }

  f = odl_class_get_field (cl, fieldname);
  switch (odl_field_get_type (f))
    {
    case FT_basic:
      /* normal fields, handled by cache */
      if (odl_field_has_property (f, ODL_PROP_READONLY))
        {
          make_NotWriteable_exception (ev, "Field '%s' is not writeable.",
                                       fieldname);
          return;
        }
      break;

    case FT_reference:
    case FT_list:
      make_NotDataField_exception (ev,
                                   "Field '%s' is not a plain data field.",
                                   fieldname);
      return;

    case FT_lookup:
    case FT_method:
    case FT_calculated:
    case FT_readonly:
      make_NotWriteable_exception (ev, "Field '%s' is not writeable.",
                                   fieldname);
      return;

    case FT_unknown:
      make_ServerError_exception (ev,
                                  "Field '%s' has an unknown type identifier.",
                                  fieldname);
      return;
    }

  /* find object in cache, and get field */
  obj = oc_find_object_by_key (odl_class_get_full_name (cl), id->objectid);
  if (obj)
    {
      if (allow_log_class_event (odl_class_get_full_name (cl)))
        dm_event (id->username, id->currenttransaction, DM_EVENT_FIELD_WRITE,
                  odl_class_get_full_name (cl), id->objectid, fieldname,
                  value);

      /* check the datatype being set */
      switch (odl_field_get_datatype (f))
        {
        case DT_char:
        case DT_text:
          /* accept without validation */
          break;
        case DT_int16:
        case DT_int32:
        case DT_int64:
        case DT_int:
        case DT_unsignedint:
          /* validate an integer */
          {
            unsigned int i;
            for (i = 0; i < strlen (value); i++)
              if (isdigit (value[i]) == 0
                  || (i == 0 && (value[0] == '-' || value[0] == '-')))
                {
                  make_Format_exception (ev, "Expected an integer.");
                  return;
                }
            if (odl_field_get_datatype (f) == DT_unsignedint
                && value[0] == '-')
              {
                make_Format_exception (ev, "Expected an unsigned integer.");
                return;
              }
          }
          break;
        case DT_boolean:
        case DT_bool:
          /* I'ld like to take this opportunity to say that
             defining a boolean like this just really isn't bright
             and it wasn't my decision.

             encouraging code like this is bad:
             if( strcmp(value,"T") == 0 || strcmp(value,"0")==0 ) ... ;
           */
          if (g_strcasecmp (value, "-1") == 0)
            value[0] = 'F';
          value[0] = toupper (value[0]);
          value[1] = '\0';
          if (g_strcasecmp (value, "T") == 0);
          else if (g_strcasecmp (value, "F") == 0);
          else if (g_strcasecmp (value, "1") == 0)
            value[0] = 'T';
          else if (g_strcasecmp (value, "0") == 0)
            value[0] = 'F';
          else if (g_strcasecmp (value, "Y") == 0)
            value[0] = 'T';
          else if (g_strcasecmp (value, "N") == 0)
            value[0] = 'F';
          else
            {
              make_Format_exception (ev,
                                     "Expected 'TRUE' or 'FALSE' or 'T' or 't' or 'F' or 'f' or '0' or '1' or '-1' or 'YES' or 'Y' or 'NO' or 'N'");
              return;
            }
          if (value[0] == 'T')
            value[0] = '1';
          if (value[0] == 'F')
            value[0] = '0';
          break;
        case DT_float:
          {
            char *end = NULL;
            strtod (value, &end);
            if (end == value)
              make_Format_exception (ev,
                                     "Floating point number expected (eg -123.45 or 123.45)");
          }
          break;
        case DT_date:
          /* make sure the date is in the correct format (YYYY-MM-DD) */
          /* 'value' is a NULL terminated string containing the date */
          break;
        case DT_time:
          /* make sure the time is in the correct format (HH:MM:SS) */
          break;
        case DT_datetime:
          /* make sure the time is in the correct format (YYYY-MM-DD HH:MM:SS) */
          break;
        default:
          /* accept anything else */
          break;
        }

      if (oc_set_object_field (obj, fieldname, value, FALSE) == FALSE)
        {
          make_ServerError_exception (ev, "Write failure.");
        }
      else
        {
          /* immediately write it to the database */
          /* message( "attempting write" ); */
          Connection_logEdit (id->classname, id->objectid, fieldname, id);
          oc_flush_object_to_store (obj);
        }
    }
  else
    {
      /* this should, in theory, be impossible */
      make_ServerError_exception (ev,
                                  "2 Impossible error: object went missing. (%s/%s)",
                                  id->classname, id->objectid);
      errormsg ("Impossible error: object went missing.");
    }
}

/* raises: UnknownField, NotReferenceField, Interrupted, ServerError, Locked, TransactionNotInProgress */
GEAS_DataObject
DataObject_getReference (GEAS_object_reference * id, CORBA_char * field,
                         CORBA_Environment * ev)
{
  GList *s, *t;
  char *value;
  /* char *targetfield; */
  odl_class *cl, *c;
  odl_field *f;
  char *loadclass;
  QueryData *q;
  ObjectData *o;
  char *keystr;
  int idfield;
  GEAS_DataObject retval = CORBA_OBJECT_NIL;
  struct query_result *result;
  int err;
  char *errmsg;

  /* find and validate the field */
  cl = odl_find_class (all_classes, id->classname, NULL);
  if (!cl)
    {
      make_ServerError_exception (ev, "Class '%s' has gone away",
                                  id->classname);
      return (CORBA_OBJECT_NIL);
    }
  f = odl_class_get_field (cl, field);
  if (!f)
    {
      make_UnknownField_exception (ev,
                                   "Field '%s' is not defined in class '%s'",
                                   field, id->classname);
      return (CORBA_OBJECT_NIL);
    }
  if (odl_field_get_type (f) != FT_reference)
    {
      make_WrongType_exception (ev,
                                "Field '%s' in class '%s' is not a reference field.",
                                field, id->classname);
      return (CORBA_OBJECT_NIL);
    }
  loadclass = (char *) odl_field_get_sourceclass (f);

  /* create query */
  q = create_base_query (loadclass);
  if (!q)
    {
      make_ServerError_exception (ev, "Could not create query");
      return (CORBA_OBJECT_NIL);
    }

  /* add constraints to find just members of this list */
  /* ... WHERE loadclass.t1 = this.s1 AND loadclass.t2 = this.s2 ... */

  s = odl_field_get_this_fields (f);
  t = odl_field_get_source_fields (f);
  while (s && t)
    {
      /* value = this.(s->data) */
      /* note: inheritance has to be considered as well */
      c = odl_field_defined_in (cl, (const char *) s->data);
      if (!c)
        {
          make_ServerError_exception (ev,
                                      "Reference requires field %s, but this is not defined in class %s",
                                      s->data, id->classname);
          oql_free_query (q);
          return (CORBA_OBJECT_NIL);
        }
      o = oc_find_object_by_key (odl_class_get_full_name (c), id->objectid);
      if (!o)
        {
          make_ServerError_exception (ev,
                                      "Reference requires field %s.%s, but this could not be found in the database",
                                      id->classname, s->data);
          oql_free_query (q);
          return (CORBA_OBJECT_NIL);
        }
      value = (char *) oc_get_object_field (o, s->data);
      if (!value)
        value = g_strdup ("");

      /* add constraint to query */
      if (!oql_add_query_constraint
          (q, id->classname, (const char *) value, "=", loadclass,
           (const char *) t->data))
        {
          oql_free_query (q);
          make_ServerError_exception (ev, "Error making query.");
          g_free (value);
          return (CORBA_OBJECT_NIL);
        }
      g_free (value);
      s = g_list_next (s);
      t = g_list_next (t);
    }
  if (s != NULL || t != NULL)
    {
      oql_free_query (q);
      make_ServerError_exception (ev,
                                  "Mismatch in field definition for %s.%s",
                                  id->classname, field);
      return (CORBA_OBJECT_NIL);
    }
  /* perform query */
  result = query_objectstore (q, &err, &errmsg);
  if (errmsg)
    {
      if (result)
        free_query_result (result);
      make_ServerError_exception (ev, errmsg);
      errormsg (errmsg);
      g_free (errmsg);
      return (CORBA_OBJECT_NIL);
    }
  if (!result)
    {
      oql_free_query (q);
      make_ServerError_exception (ev, "Error locating object");
      return (CORBA_OBJECT_NIL);
    }

  /* create DataObject reference */
  idfield = oql_query_get_field_position (q, "objectid");
  keystr = (char *) get_result_field (result, 0, idfield);
  retval =
    (GEAS_DataObject) make_dataobject_reference (loadclass, keystr,
                                                 id->username, id->sessionid,
                                                 ev);
  oql_free_query (q);
  if (result)
    free_query_result (result);
  return retval;
}

/* raises: UnknownField, NotReferenceField, Interrupted, ServerError, Locked, TransactionNotInProgress */
GEAS_ObjectList
DataObject_getList (GEAS_object_reference * id, CORBA_char * field,
                    CORBA_Environment * ev)
{
  GList *s, *t;
  char *value;
  /* char *targetfield; */
  odl_class *cl, *c;
  odl_field *f;
  char *loadclass;
  QueryData *q;
  ObjectData *o;
  char *keystr;
  GEAS_ObjectList retval;
  /* int idfield; */
  /* int err; */

  /* find and validate the field */
  cl = odl_find_class (all_classes, id->classname, NULL);
  if (cl)
    f = odl_class_get_field (cl, field);
  if (!cl)
    {
      make_ServerError_exception (ev, "Class '%s' has gone away",
                                  id->classname);
      return (CORBA_OBJECT_NIL);
    }
  if (!f)
    {
      make_UnknownField_exception (ev,
                                   "Field '%s' is not defined in class '%s'",
                                   field, id->classname);
      return (CORBA_OBJECT_NIL);
    }
  if (odl_field_get_type (f) != FT_list)
    {
      make_WrongType_exception (ev,
                                "Field '%s' in class '%s' is not a list field.",
                                field, id->classname);
      return (CORBA_OBJECT_NIL);
    }

  loadclass = (char *) odl_field_get_sourceclass (f);

  /* create query */
  q = create_base_query (loadclass);
  if (!q)
    {
      make_ServerError_exception (ev, "Could not create query");
      return (CORBA_OBJECT_NIL);
    }

  /* add constraints to find just members of this list */
  /* ... WHERE loadclass.t1 = this.s1 AND loadclass.t2 = this.s2 ... */

  s = odl_field_get_this_fields (f);
  t = odl_field_get_source_fields (f);
  while (s && t)
    {
      /* value = this.(s->data) */
      /* note: inheritance has to be considered as well */
      /* message( "looking for class containing field %s" , s->data ); */
      c = odl_field_defined_in (cl, (const char *) s->data);
      if (!c)
        {
          make_ServerError_exception (ev,
                                      "Reference requires field %s, but this is not defined in class %s",
                                      s->data, id->classname);
          oql_free_query (q);
          return (CORBA_OBJECT_NIL);
        }
      o = oc_find_object_by_key (odl_class_get_full_name (cl), id->objectid);
      if (!o)
        {
          make_ServerError_exception (ev,
                                      "Reference requires field %s.%s, but this could not be found in the database",
                                      id->classname, s->data);
          oql_free_query (q);
          return (CORBA_OBJECT_NIL);
        }
      value = (char *) oc_get_object_field (o, s->data);
      if (!value)
        g_free (value);

      /* add constraint to query */
      if (!oql_add_query_constraint
          (q, id->classname, (const char *) value, "=", loadclass,
           (const char *) t->data))
        {
          g_free (value);
          oql_free_query (q);
          make_ServerError_exception (ev, "Error making query.");
          return (CORBA_OBJECT_NIL);
        }
      g_free (value);
      s = g_list_next (s);
      t = g_list_next (t);
    }
  if (s != NULL || t != NULL)
    {
      oql_free_query (q);
      make_ServerError_exception (ev,
                                  "Mismatch in field definition for %s.%s",
                                  id->classname, field);
      return (CORBA_OBJECT_NIL);
    }

  /* get data */
  keystr = new_object_collection_from_query (q, id, ev);
  oql_free_query (q);
  if (!keystr)
    {
      make_ServerError_exception (ev,
                                  "3 Could not create list storage structure");
      return (CORBA_OBJECT_NIL);
    }

  /* create ObjectList reference */
  retval =
    (GEAS_ObjectList) make_list_reference (keystr, id->username,
                                           id->sessionid, ev);
  CORBA_free (keystr);

  return retval;
}

/* raises: UnknownField, NotListField, WrongClass, Interrupted, NotWriteable, ServerError, Locked, TransactionNotInProgress */
void
DataObject_insert (GEAS_object_reference * id, CORBA_char * fieldname,
                   GEAS_DataObject obj, CORBA_Environment * ev)
{
  /* TODO */
  /* add 'obj' to the list in classname.fieldname */

  /*
   * eg cust_obj.insert( "invoices" , invoice_obj );
   * invoice_obj.customerid = cust_obj.id
   * 
   * find fields in 'object' that need to be changed to make it appear in
   * this.fieldname
   * 
   * get lookup_field struct for this.fieldname
   * set object.(lookup_field.targetfield<N>) = this.(lookup_field.searchfield<N>)
   * set obj1.fieldlist1<N> = obj2.fieldlist<N>
   * 
   * this.fieldname must be type LF_LIST
   */

  ObjectData *container, *object;
  odl_field *f;
  char *classname, *key;
  odl_class *cl_container, *cl_object;
  GList *fields = NULL, *values = NULL, *l = NULL;

/* debug_output( DEBUGLEVEL_ALWAYS , "Inserting object" ); */

  /* find container data */
  container = oc_find_object_by_key (id->classname, id->objectid);
  cl_container = odl_find_class (all_classes, id->classname, NULL);
  if (!container)
    {
      make_ServerError_exception (ev, "Container object %s/%s has gone away",
                                  id->classname, id->objectid);
      return;
    }

  /* find object data */
  classname = GEAS_DataObject__get_classname (obj, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      make_ServerError_exception (ev,
                                  "Could not find classname of object to insert");
      return;
    }
  key = GEAS_DataObject__get_objectID (obj, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      CORBA_free (classname);
      make_ServerError_exception (ev,
                                  "Could not find object identifier of object to insert");
      return;
    }
  object = oc_find_object_by_key (classname, key);
  cl_object = odl_find_class (all_classes, classname, NULL);
  if (!object)
    {
      make_ServerError_exception (ev, "Object %s/%s has gone away", classname,
                                  key);
      CORBA_free (classname);
      CORBA_free (key);
      return;
    }

  if (!cl_container)
    {
      make_UnknownClass_exception (ev, "%s", id->classname);
      CORBA_free (classname);
      CORBA_free (key);
      return;
    }
  if (!cl_object)
    {
      make_UnknownClass_exception (ev, "%s", classname);
      CORBA_free (classname);
      CORBA_free (key);
      return;
    }

  /* is 'field' a container field in the container class */
  f = odl_class_get_field (cl_container, fieldname);
  if (!f)
    {
      CORBA_free (classname);
      CORBA_free (key);
      make_UnknownField_exception (ev,
                                   "Field '%s' is not defined in class '%s'",
                                   fieldname, id->classname);
      return;
    }
  if (odl_field_get_type (f) != FT_list)
    {
      CORBA_free (classname);
      CORBA_free (key);
      make_NotListField_exception (ev,
                                   "Field '%s' in class '%s' is not a list field",
                                   fieldname, id->classname);
      return;
    }

  /* does this class go in that field? */
  if (!odl_class_is_instanceof (cl_object, odl_field_get_sourceclass (f)))
    {
      CORBA_free (classname);
      CORBA_free (key);
      make_WrongClass_exception (ev,
                                 "%s.%s requires objects of class '%s' but received class '%s'",
                                 fieldname, id->classname,
                                 odl_field_get_sourceclass (f),
                                 odl_class_get_full_name (cl_object));
      return;
    }

  /* validation complete. now attempt to place 'object' into 'container.fieldname' */
/* debug_output( DEBUGLEVEL_ALWAYS , "Validation complete" ); */

  /* read data from this.<fields> */
  fields = odl_field_get_this_fields (f);
  values = NULL;
  while (fields)
    {
      char *fn, *data;
      odl_class *cl;
      ObjectData *o;
      /* int err; */

      /* find data in this.field<N> */
      fn = fields->data;        /* name of field in container */
      cl = odl_field_defined_in (cl_container, fn);     /* what class is it defined in */
      o = oc_find_object_by_key (odl_class_get_full_name (cl), id->objectid);
      data = (char *) oc_get_object_field (o, fn);

      /* record data */
/* debug_output( DEBUGLEVEL_ALWAYS , "read: container : %s.%s = %s" , odl_class_get_full_name(cl_container) , fn , data ); */
      values = g_list_append (values, data);
      fields = g_list_next (fields);
    }

  /* validate that the writes are all possible */

  /* perform writes to source.<fields> */
  fields = odl_field_get_source_fields (f);
  l = values;
  while (fields)
    {
      char *fn;
      /* char *data; */
      odl_class *cl;
      ObjectData *o;
      /* int err; */

      /* find obj.field<N> */
      fn = fields->data;        /* name of field in container */
      cl = odl_field_defined_in (cl_object, fn);        /* what class is it defined in */
      o = oc_find_object_by_key (odl_class_get_full_name (cl), key);

      /* write data */
      if (allow_log_class_event (oc_get_object_class (o)))
        dm_event (id->username, id->currenttransaction, DM_EVENT_FIELD_WRITE,
                  oc_get_object_class (o), oc_get_object_key (o), fn,
                  l->data);
      oc_set_object_field (o, fn, l->data, FALSE);

      oc_flush_object_to_store (o);
      fields = g_list_next (fields);
      l = g_list_next (l);
    }

  /* done */
  g_list_free (values);
  CORBA_free (classname);
  CORBA_free (key);
}

/* raises: UnknownField, NotReferenceField, WrongClass, Interrupted, NotWriteable, ServerError, Locked, TransactionNotInProgress */
void
DataObject_setReference (GEAS_object_reference * id, CORBA_char * fieldname,
                         GEAS_DataObject obj, CORBA_Environment * ev)
{
  ObjectData *container, *object;
  odl_field *f;
  char *classname, *key;
  odl_class *cl_container, *cl_object;
  GList *fields = NULL, *values = NULL, *l = NULL;

/* debug_output( DEBUGLEVEL_ALWAYS , "Setting reference" ); */

  /* find container data */
  container = oc_find_object_by_key (id->classname, id->objectid);
  cl_container = odl_find_class (all_classes, id->classname, NULL);
  if (!container)
    {
      make_ServerError_exception (ev, "Container object %s/%s has gone away",
                                  id->classname, id->objectid);
      return;
    }

  /* find object data */
  classname = GEAS_DataObject__get_classname (obj, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      make_ServerError_exception (ev,
                                  "Could not find classname of object to reference");
      return;
    }
  key = GEAS_DataObject__get_objectID (obj, ev);
  if (ev->_major != CORBA_NO_EXCEPTION)
    {
      CORBA_free (classname);
      make_ServerError_exception (ev,
                                  "Could not find object identifier of object to reference");
      return;
    }
  object = oc_find_object_by_key (classname, key);
  cl_object = odl_find_class (all_classes, classname, NULL);
  if (!object)
    {
      make_ServerError_exception (ev, "Object %s/%s has gone away", classname,
                                  key);
      CORBA_free (classname);
      CORBA_free (key);
      return;
    }

  if (!cl_container)
    {
      make_UnknownClass_exception (ev, "%s", id->classname);
      CORBA_free (classname);
      CORBA_free (key);
      return;
    }
  if (!cl_object)
    {
      make_UnknownClass_exception (ev, "%s", classname);
      CORBA_free (classname);
      CORBA_free (key);
      return;
    }

  /* is 'field' a container field in the container class */
  f = odl_class_get_field (cl_container, fieldname);
  if (!f)
    {
      CORBA_free (classname);
      CORBA_free (key);
      make_UnknownField_exception (ev,
                                   "Field '%s' is not defined in class '%s'",
                                   fieldname, id->classname);
      return;
    }
  if (odl_field_get_type (f) != FT_reference)
    {
      CORBA_free (classname);
      CORBA_free (key);
      make_NotReferenceField_exception (ev,
                                        "Field '%s' in class '%s' is not a list field",
                                        fieldname, id->classname);
      return;
    }

  /* does this class go in that field? */
  if (!odl_class_is_instanceof (cl_object, odl_field_get_sourceclass (f)))
    {
      CORBA_free (classname);
      CORBA_free (key);
      make_WrongClass_exception (ev,
                                 "%s.%s requires objects of class '%s' but received class '%s'",
                                 fieldname, id->classname,
                                 odl_field_get_sourceclass (f),
                                 odl_class_get_full_name (cl_object));
      return;
    }

  /* validation complete. now attempt to place 'object' into 'container.fieldname' */
/* debug_output( DEBUGLEVEL_ALWAYS , "Validation complete" ); */

  /* read data from source.<fields> */
  fields = odl_field_get_source_fields (f);
  values = NULL;
  while (fields)
    {
      char *fn, *data;
      odl_class *cl;
      ObjectData *o;
      /* int err; */

      /* find data in this.field<N> */
      fn = fields->data;        /* name of field in container */
      cl = odl_field_defined_in (cl_object, fn);        /* what class is it defined in */
      o = oc_find_object_by_key (odl_class_get_full_name (cl), key);
      data = (char *) oc_get_object_field (o, fn);

      /* record data */
      values = g_list_append (values, data);
      fields = g_list_next (fields);
    }

  /* validate that the writes are all possible */
/* debug_output( DEBUGLEVEL_ALWAYS , "Read data" ); */

  /* perform writes */
  fields = odl_field_get_this_fields (f);
  l = values;
  while (fields)
    {
      char *fn;
      /* char *data; */
      odl_class *cl;
      ObjectData *o;
      /* int err; */

      /* find obj.field<N> */
      fn = (char *) fields->data;       /* name of field in container */
      cl = odl_field_defined_in (cl_container, fn);     /* what class is it defined in */
      o = oc_find_object_by_key (odl_class_get_full_name (cl), id->objectid);

      /* write data to fields in this/container object */
      if (allow_log_class_event (oc_get_object_class (o)))
        dm_event (id->username, id->currenttransaction, DM_EVENT_FIELD_WRITE,
                  oc_get_object_class (o), oc_get_object_key (o), fn,
                  l->data);
      oc_set_object_field (o, fields->data, l->data, FALSE);
      oc_flush_object_to_store (o);
      fields = g_list_next (fields);
      l = g_list_next (l);
    }

  /* done */
  g_list_free (values);
  CORBA_free (classname);
  CORBA_free (key);
/* debug_output( DEBUGLEVEL_ALWAYS , "done setReference" ); */
}

/* raises: */
void
DataObject_flush (GEAS_object_reference * id, CORBA_Environment * ev)
{
  /* not really needed: leave empty until further notice */
}

/* raises: Interrupted, ServerError, Locked, TransactionNotInProgress */
void
DataObject_delete (GEAS_object_reference * id, CORBA_Environment * ev)
{
  odl_class *cl = odl_find_class (all_classes, id->classname, NULL);
  if (cl)
    {
      if (strncmp (id->classname, "geas::", 6) != 0)
        dm_event (id->username, id->currenttransaction,
                  DM_EVENT_OBJECT_DELETE, odl_class_get_full_name (cl),
                  id->objectid);
      Connection_logDelete (odl_class_get_full_name (cl), id->objectid, id);
      oc_delete_object (odl_class_get_full_name (cl), id->objectid);
    }
}

/* raises: UnknownMethod, InvalidArgument, MethodError, ServerError, Locked, TransactionNotInProgress */
CORBA_char *
DataObject_callMethod (GEAS_object_reference * id, CORBA_char * methodname,
                       GEAS_Arguments * args, CORBA_Environment * ev)
{
  odl_class *cl = NULL;
  odl_field *f = NULL;

  /* validate class/object/method */
  cl = odl_find_class (all_classes, id->classname, NULL);
  if (!cl)
    {
      make_ServerError_exception (ev, "Class %s has gone missing",
                                  id->classname);
      return (NULL);
    }
  cl = odl_field_defined_in (cl, methodname);
  if (cl)
    {
      f = odl_class_get_field (cl, methodname);
    }
  if (!f)
    {
      make_UnknownMethod_exception (ev, "Method %s was not found in class %s",
                                    methodname, id->classname);
      return (NULL);
    }
  /* done */

  /* validate arguments */
  if (args->_length != odl_method_argument_count (f))
    {
      make_ArgumentCount_exception (ev, odl_method_argument_count (f),
                                    args->_length,
                                    "%s.%s requires %d arguments, but received %d",
                                    id->classname, methodname,
                                    odl_method_argument_count (f),
                                    args->_length);
      return (NULL);
    }

#ifdef DEBUG
  /* display arguments */
  if (debuglevel >= DEBUGLEVEL_HIGH)
    {
      unsigned int i;

      for (i = 0; i < args->_length; i++)
        debug_output (DEBUGLEVEL_ALWAYS, "Arg %2d: %s", i, args->_buffer[i]);
    }
#endif

  return (execute_method (id, methodname, args, ev));
}
