/* 
   $RCSfile: gcdparser.h,v $

   This file is part of:
   
       GNU Enterprise Application Server (GEAS)

   Copyright (C) 2001 Free Software Foundation

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   $Id: gcdparser.h,v 1.30 2001/08/24 21:30:58 ntiffin Exp $
   
*/


/** \file gcdparser.h
 * \warning This file should not be used outside of the ODL class library.
 * \brief Contains typedefs and function prototypes used inside the
 * GCD parser.  Should not contain any code called from outside of the
 * module.
 */

#ifndef GCD_PARSER_H
#define GCD_PARSER_H

#include "classdata.h"

extern int            yyerrorcount;
extern int            yywarncount;

/** \struct _odl_base
 *  \brief Generic data for an odl object.
 *  Only used in conjuction with _odl_container and
 *  _odl_item.
 */
struct _odl_base
{
  char                 *name;
  char                 *fullname;
  char                 *mangledname;
  enum odl_itemtype     type;
  unsigned long int     access;
  unsigned long int     triggers;
  struct _odl_base     *parent;
  odl_tree             *tree;
};

/** \struct _odl_container
 *  \brief Stores modules, classes and any other containers.
 */
struct _odl_container
{
  struct _odl_base      base;       /* WARNING: nothing can be put before base */
                                    /*   in structure as the code make heavy   */
                                    /*   use of casts that rely on this being  */
                                    /*   first item in structure.              */
                                    /* generic data                            */
  GList                *contents;   /* list of (struct _odl_base *)            */
                                    /* business class data                     */
  gboolean              istype;     /* true if 'type' keyword was used         */
  char                 *orderby;
  GList                *parents;    /* list of (char *)                        */
  GList                *indexes;    /* list of (struct _odl_index *)           */
  char                 *filename;   /* filename that defined this container    */
  gboolean              desc;       /* ORDER BY uses descending order          */
};

/** \struct _odl_item
 *  \brief Stores items that can not have containers.
 */
struct _odl_item
{
  struct _odl_base      base;       /* WARNING: nothing can be put before base */
                                    /*   in structure as the code make heavy   */
                                    /*   use of casts that rely on this being  */
                                    /*   first item in structure.              */
                                        /* class fields - generic data */
  enum odl_fieldtype    fieldtype;
  enum odl_datatype     datatype;
  char                 *datatypeclass;  /* when datatype == DT_class */
  unsigned long int     properties;
                                        /* basic fields */
  char                 *format;
  char                 *defaultval;
  char                 *tmpdefault;
                                        /* lookup/reference/list */
                                        /* readonly fields */
  char                 *sourcefield;
  char                 *sourceclass;
  odl_namelist         *this_fields;
  odl_namelist         *source_fields;
  signed long int       bounds;         /*     -1 = reference,    */
                                        /*      0 = unlimited list, */
                                        /* +other = bounded list, not implemented */
  odl_argumentlist     *arguments;      /* method */
  char                 *calculation;    /* calculated */
  GList                *elements;       /* enum */
};

/** \struct _odl_argument
 *  \brief Stores method arguments.
 */
struct _odl_argument
{
  struct _odl_item     *method;
  char                 *name;
  enum odl_datatype     datatype;
  char                 *classdatatype;
};

/** \struct _odl_tree
 *  \brief Root of the ODL definition tree.
 */
struct _odl_tree
{
  struct _odl_container *root;
};

/** \struct _odl_datatype
 *  \brief ODL data type structure for passing around in functions.
 *  Used for implicit references and lists only.
 */
typedef struct
{
  gboolean              unsignedval;
  char                 *name;
  unsigned long int     bound;
  enum odl_datatype     dt;
}
_odl_datatype;

/** \struct _odl_datasource
 *  \brief 
 */
typedef struct
{
  GList                *fields;
  char                 *classname;
  char                 *field;
}
_odl_datasource;

/** \struct _odl_index
 *  \brief For indexing the class.
 */
typedef struct
{
  gboolean              unique;
  gboolean              primary;
  GList                *fields;
}
_odl_index;

/** \brief Create new ODL data type.
 *  \param unsignedval ?.
 *  \param name Name of datatype.
 *  \param bound Bounds for list (not implemented).
 *  \param dt Data type.
 *  \return Pointer to created ODL data type.
 */
_odl_datatype        *alloc_odl_datatype (gboolean unsignedval, char *name,
                                          unsigned long int bound,
                                          enum odl_datatype dt);
void                  free_odl_datatype (_odl_datatype * _datatype);
_odl_datasource      *alloc_odl_datasource (void);
void                  free_odl_datasource (_odl_datasource * ds);

 /* memory functions */
struct _odl_tree     *alloc_odl_tree (void);
void                  free_odl_tree (struct _odl_tree *tree);
struct _odl_container *alloc_odl_container (void);
void                  free_odl_container (struct _odl_container *c);
struct _odl_item     *alloc_odl_item (void);
void                  free_odl_item (struct _odl_item *item);
struct _odl_argument *alloc_odl_argument (void);
void                  free_odl_argument (struct _odl_argument *argument);
void                  free_odl_unknown_item (struct _odl_base *base);
void                  init_odl_base (struct _odl_base *base);
void                  free_odl_base (struct _odl_base *base);

 /* functions for creating a class tree */
struct _odl_container *odl_tree_get_root (struct _odl_tree *tree);
void                  odl_container_insert_container (struct _odl_container
                                                      *parent,
                                                      struct _odl_container
                                                      *child);
odl_module           *odl_new_module (unsigned long int access,
                                      const char *name);
odl_class            *odl_new_class (unsigned long int access,
                                     const char *name);
odl_item             *odl_new_enum (unsigned long int access,
                                    const char *name);
void                  odl_enum_add_element (odl_item * e, char *element);

/**
 * i = pointer to current item to convert
 * 
 */
void                  odl_make_reference(odl_item * i, struct _odl_container * container);

/**
 * Anytime a field is processed it comes here.  The purpose of
 * this routine is to expand multiple fields on one line,
 * and create reference and list structures.  Types are
 * set up for further processing but not expanded.
 *
 * for a defintion like:
 *   address  home, business, girlfriend;
 *
 * \param datatype Is the address _odl_datatype struct
 * \param fields Is a list of (struct _odl_item *)
 *     fields -> GList[ home, business, girlfriend ]
 */
GList                *odl_reprocess_fields (_odl_datatype * datatype,
                                            GList * fields);
odl_item             *set_method_datatype (_odl_datatype * datatype,
                                           odl_item * method);
const char           *odl_get_base_full_name (odl_base * b);
void                  odl_standardise_parents (odl_class * classd);
_odl_index           *odl_make_index (odl_class * c, gboolean unique,
                                      GList * fieldnames);
int                   odl_sort_string_name (gpointer a, gpointer b);

#endif
