
#include <glib.h>
#include <stdio.h>

#ifndef GCD_PARSER_H
#include "gcdparser.h"
#endif

/** \file classdata.h
 * \brief GCD parser interface
 */

#ifndef ODL_CLASSDATA_H
#define ODL_CLASSDATA_H

/*********************************************************/

/* opaque data types - do not rely on the indicated type */
typedef struct _odl_item odl_item;
typedef struct _odl_base odl_base;
typedef struct _odl_container odl_container;
typedef struct _odl_container odl_class;
typedef struct _odl_container odl_module;
typedef struct _odl_item odl_field;
typedef struct _odl_argument odl_argument;
typedef GList odl_fieldlist;
typedef GList odl_namelist;
typedef GList odl_filenamelist;
typedef GList odl_argumentlist;
typedef struct _odl_indexes odl_indexes;
typedef struct _odl_index odl_index;
typedef struct _odl_tree odl_tree;
typedef struct _odl_item odl_enum;

/* note triggers can overlap with triggers in another section:  */

/* a field specific trigger and a class specific trigger always */

/* apply to different items */

/* generic triggers */
#define ODL_TRIG_NONE       0x00000000

/* class triggers */
#define ODL_TRIG_PREDELETE  0x80000000
#define ODL_TRIG_PREWRITRE  0x80000001
#define ODL_TRIG_POSTLOAD   0x80000002

/* field triggers */
#define ODL_TRIG_PRESET     0x80000001
#define ODL_TRIG_PREGET     0x80000002

/* class/field access */
#define ODL_ACCESS_UNKNOWN   0x0000			 /* internal use */
#define ODL_ACCESS_UNCHANGED 0x0001			 /* internal use */
#define ODL_ACCESS_PUBLIC    0x0002			 /* anything can access */
#define ODL_ACCESS_PRIVATE   0x0004			 /* not yet supported */
#define ODL_ACCESS_PROTECTED 0x0008			 /* not yet supported */
#define ODL_ACCESS_SYSTEM    0x0010			 /* limited to object se */
#define ODL_ACCESS_DEFAULT   ODL_ACCESS_PUBLIC

/* class types */
#define ODL_CLASSTYPE_NORMAL    0x00000001
#define ODL_CLASSTYPE_ABSTRACT  0x00000002 /* not yet supported */
#define ODL_CLASSTYPE_FINAL     0x00000004 /* not yet supported */
#define ODL_CLASSTYPE_SINGLETON 0x00000008 /* not yet supported */

/* field properties */
#define ODL_PROP_NONE           0x00000000
#define ODL_PROP_NOTNULL        0x00000001
#define ODL_PROP_READONLY       0x00000002
#define ODL_PROP_SERVER         0x00000004
#define ODL_PROP_HIDDEN         0x00000008

/** \brief Convert an integer representation of an access type to a string
 * \param access The integer representing the access level
 * \return A pointer to a statically allocated string containing the access
 * level name.
 */
const char *odl_access_name(unsigned long int access);

/************************/

/* Various enumerations */

/** \brief type of item in a tree
 * A tree is made up of a number of different items, some of which
 * may contain further items. enum odl_itemtype defines the type
 * of each item.
 */
enum odl_itemtype
{
   IT_unknown, /** \brief An unknown item. Should not be used */
   IT_module,  /** \brief A module. (Modules are used to manage name spaces) */
   IT_class,   /** \brief A class. (Objects are instances of a class) */
   IT_field,   /** \brief A field in a class */
   IT_enum,    /** \brief A new datatype that can only contain specific values */
   IT_type,    /** \brief abstract class that can only be used as a field in another class */
   IT_ignore   /** \brief ignore this item for all processing */
};

/* conversion to/from ASCII */
/** \brief Convert an enum odl_itemtype to a string
 * \param type The type of an item
 * \returns A pointer to a statically allocated string containing the name of the item type
 */
const char *odl_itemtype_name(enum odl_itemtype type);
enum odl_itemtype odl_itemtype_id(const char *name);

/* datatype for fields/arguments/method return values */

/** \brief The available types of data
 * To add a new datatype:
 *   1) Add an entry to this list
 *   2) Add code to odl_datatype_name() and odl_datatype_id()
 *   3) (TODO) Add code to (undecided) to validate defaults and formats
 *   4) Add code to the geas/src/objectstore module in each database
 *      driver, in order to manage database tables correctly
 */
enum odl_datatype
{
   DT_unknown = 0,
   DT_char, /* Character */
   DT_int, /* Integer same as DT_int32 */
   DT_int16, /* 16 bit integer */
   DT_int32, /* 32 bit integer */
   DT_int64, /* 64 bit integer */
   DT_boolean, /* boolean */
   DT_text, /* Arbitrary length  */
   DT_class, /* placeholder - must also store the name of the class */
   DT_date, /* YYYY-MM-DD */
   DT_time, /* HH:MM:SS */
   DT_datetime, /* YY-MM-DD HH:MM:SS */
   DT_bool, /* TRUE/FALSE */
   DT_float, /*  */
   DT_void, /* indicates 'no valid data' */
   DT_object, /* badly named - means an object's UUID */
   DT_enum, /* an enum - must store the name of the enum as well */
   DT_unsignedint, /* an unsigned integer */

   /* internal datatypes only - will not appear in a class tree */
   DT_type /* indicates a field that will be replaced by several fields derived from a 'type' item */
};

/* conversion to/from ASCII */
const char *odl_datatype_name(enum odl_datatype type);
enum odl_datatype odl_datatype_id(const char *name);

/* type of a given field in a class */
enum odl_fieldtype
{
   FT_basic,
   FT_lookup,
   FT_reference,
   FT_list,
   FT_method,
   FT_calculated,
   FT_readonly,
   FT_unknown
};

/* conversion to/from ASCII */
const char *odl_fieldtype_name(enum odl_fieldtype type);
enum odl_fieldtype odl_fieldtype_id(const char *name);

/**********************/

/* List handling code */

/* list filenames */
void odl_filenamelist_free(odl_filenamelist * list);
odl_filenamelist *odl_filenamelist_add(odl_filenamelist * list,
                                       const char *filename);

/* name list handling code */
void odl_namelist_free(odl_namelist * list);
const char *odl_namelist_entry(odl_namelist * list, int index);
gboolean  odl_namelist_contains( odl_namelist * list, const char *name,gboolean casesensitive);

/* field list handling code */
void odl_fieldlist_free(odl_fieldlist * list);
odl_field *odl_fieldlist_entry(odl_fieldlist * list, int index);

/* argument list handling code */
void odl_argumentlist_free(odl_argumentlist * list);
odl_argument *odl_argumentlist_entry(odl_argumentlist * list, int index);

/*****************/

/* Item location */

/* locate items based on name */
odl_class *odl_find_class(odl_tree * tree, const char *name, const char *id);
odl_module *odl_find_module(odl_tree * tree, const char *name, const char *id);
odl_field *odl_find_field(odl_tree * tree, const char *name, const char *id);
odl_base *odl_find_item(odl_tree * tree, const char *name, const char *id);
odl_base *odl_container_get_parent(odl_container * container); 	/* not parent
																					   class */
odl_base *odl_find_relative(odl_base * start, const char *name);

/* identify a specific item */
enum odl_itemtype odl_find_named_item_type(odl_tree * tree, const char *name);
const char *odl_item_get_full_name(odl_base * item);

/* locate item in a container */
odl_class *odl_field_defined_in(odl_class * startclass, const char *name);

/**************************/

/* read data from: tree */
odl_namelist *odl_tree_list_classes(odl_tree * tree);

/**************************/

/* read data from: module */
const char *odl_module_get_name(odl_module * module);
const char *odl_module_get_full_name(odl_module * module);
unsigned long int odl_module_get_access(odl_module * module);
odl_base *odl_module_get_parent(odl_module * module);

/*************************/

/* read data from: class */
const char *odl_class_get_name(odl_class * classd);
const char *odl_class_get_full_name(odl_class * classd);
const char *odl_class_get_mangled_name(odl_class * classd);
unsigned long int odl_class_get_access(odl_class * classd);
unsigned long int odl_class_get_triggers(odl_class * classd);
odl_base *odl_class_get_parent(odl_class * classd);
odl_namelist *odl_class_get_parentnames(odl_class * classd);
odl_field *odl_class_get_field(odl_class * classd, const char *fieldname);
odl_fieldlist *odl_class_get_fields(odl_class * classd,
                                    enum odl_fieldtype type);
odl_fieldlist *odl_class_get_all_fields(odl_class * classd,
                                        enum odl_fieldtype type);
gboolean odl_class_is_instanceof(odl_class * classd, const char *classname);

/*************************/

/* read data from: field */
const char *odl_field_get_name(odl_field * field);
const char *odl_field_get_full_name(odl_field * field);
unsigned long int odl_field_get_access(odl_field * field);
unsigned long int odl_field_get_triggers(odl_field * field);
odl_base *odl_field_get_class(odl_field * field);
enum odl_fieldtype odl_field_get_type(odl_field * field);
enum odl_datatype  odl_field_get_datatype(odl_field * field);
const char *odl_field_get_datatype_classname(odl_field * field);
gboolean odl_field_has_property(odl_field * field, unsigned long int prop);

/* basic data fields */
const char *odl_field_get_format(odl_field * field);
const char *odl_field_get_default(odl_field * field);

/* lookup/reference/list/readonly fields */
const char *odl_field_get_sourcefield(odl_field * field);
const char *odl_field_get_sourceclass(odl_field * field);
odl_fieldlist *odl_field_get_this_fields(odl_field * field);
odl_fieldlist *odl_field_get_source_fields(odl_field * field);

/* method fields */
odl_argumentlist *odl_field_get_arguments(odl_field * field);
unsigned long int odl_method_argument_count(odl_field * field);

/* calculated fields */
const char *odl_field_get_calculation(odl_field * field);

/************************************/

/* get data about a method argument */
const char *odl_argument_get_name(odl_argument * arg);
enum odl_datatype odl_argument_get_datatype(odl_argument * arg);
const char *odl_argument_get_classdatatype(odl_argument * arg);

/**************************/

/* Get data about an enum */
const char *odl_enum_get_full_name(odl_enum * e);

/* parser functions */
int odl_get_error_count(void);
int odl_get_warn_count(void);

/** \brief Resolve reference field.
 *  \param i The item from the parser that is converted to a reference field.
 *  \param append The items to be added to the container.
 *  \param datatype The datatype from the parser.
 */
void odl_resolve_implicit_reference (odl_item * i, GList * append);

/**
 *
 */
void odl_resolve_implicit_list (odl_item * i );
char * odl_make_fullname( odl_base * parent, char * name );
char * odl_make_fullname_base( odl_base * base );

odl_tree *odl_load_files(odl_filenamelist * files, odl_tree * tree);
void odl_display_tree(FILE * out, odl_tree * tree, gboolean show_full_name);
void odl_dump1_tree(FILE * out, odl_base * tree, gboolean recursive);
void odl_dump_tree(FILE * out, odl_tree * tree, gboolean recursive);

/* Utility functions */
odl_namelist *odl_split_qualified_name(char *qualifiedname);
char *odl_mangle_qualified_name( const char *qualifiedname );

void odl_free_tree(odl_tree * tree);

odl_tree *odl_create_empty_tree(void);

odl_base *odl_find_in_container(odl_base * c, const char *name);

odl_filenamelist *
odl_get_files_from_dir (odl_filenamelist * filelist, const char *name);

gboolean 
odl_is_extension(char *filename, char *ext);

#endif
