/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

/**********************************/
/* global unified rendering setup */
/**********************************/
struct render_pak
{
/* main setup */
gint type;
gdouble width;
gdouble height;
gint camera[3];
gdouble vp_dist;        /* vanishing point for perspective proj */
gchar filename[FILELEN];
/* flags & control */
gint perspective;
gint antialias;
gint shadowless;
gint fog;
gint axes;
gint atype;
gboolean wire_model;
gboolean wire_surface;
gboolean wire_show_hidden;
gint animate;
gint animate_type;       /* file output */
gint no_exec_povray;
gint no_keep_tempfiles;
gchar *animate_file;
gdouble delay;
gdouble mpeg_quality;
/* composite object quality */
gdouble sphere_quality;
gdouble cylinder_quality;
gdouble ribbon_quality;
gint halos;
gint fast_rotation;

/* light list */
gint num_lights;
GSList *light_list;

/* old lighting */
gdouble ambience;
gdouble diffuse;
gdouble specular;

gdouble fog_density;
gdouble fog_start;

/* geometry */
gdouble ball_rad;
gdouble stick_rad;
gdouble stick_thickness;
gdouble line_thickness;
gdouble frame_thickness;
gdouble geom_line_width;
gdouble cpk_scale;
gdouble ribbon_thickness;
gdouble ribbon_curvature;
gdouble phonon_scaling;

/* spatial highlighting */
gdouble shl_strength;
gdouble shl_size;
/* atom/bond highlighting */
gdouble ahl_strength;
gdouble ahl_size;

/* material properties*/
gdouble ref_index;
gdouble transmit;
gdouble ghost_opacity;
gdouble fg_colour[3];
gdouble bg_colour[3];
gdouble morph_colour[3];
gdouble rsurf_colour[3];         /* re-entrant surface */
gdouble label_colour[3];         /* geometry labels */
gdouble title_colour[3];         /* axes titles */
gdouble ribbon_colour[3];
gchar *morph_finish;
};

/***********************/
/* top level structure */
/***********************/
struct sysenv_pak
{
/* location info */
gchar *cwd;
gchar *init_file;
gchar *elem_file;

gchar *gdis_path;
gchar *gulp_path;
gchar *povray_path;
gchar *convert_path;
gchar *viewer_path;
gchar *babel_path;
gchar *grace_path;
gchar *gamess_path;
gchar *gulp_exe;
gchar *povray_exe;
gchar *convert_exe;
gchar *viewer_exe;
gchar *babel_exe;
gchar *grace_exe;
gchar *gamess_exe;

/* filetype filters */
gint file_type;
gint babel_type;
gint write_gdisrc;

/* centralized info on all supported file types */
GSList *file_list;
/* NEW - dynamic module support */
GSList *module_list;
/* NEW - simulation list */
GSList *projects;

/* database allocation size */
gint num_elements;
GSList *elements;
GHashTable *sfc_table;
GtkTreeStore *tree_store;
GtkWidget *tree;

/* approximate GTK font width */
gint gtk_fontsize;

/* OpenGL font */
gchar gl_fontname[LINELEN];
GdkFont *gl_font;

/* LEFT pane hideable boxes */
gint mtb_on;
gint mpb_on;
gint msb_on;
gint apb_on;
gint pib_on;
gint rab_on;
gint lmb_on;

/* model related */
gpointer active_model;
GSList *mal;
gint num_trees;
gint num_displayed;
gint displayed[MAX_DISPLAYED];
gint select_mode;
gint unfragment;
/* NEW - perform the periodic bond calc, slower when it's done, but */
/* gives more info & speeds up subsequent stuff (eg unfrag) */
gint calc_pbonds;
gdouble angle[3];

/* drawing */
gboolean stereo;                /* stereo on/off */
gboolean stereo_windowed;       /* is capable of windowed stereo */
gboolean stereo_fullscreen;     /* actual stereo mode (windowed/fullscreen) */
gint canvas;
gint redraw;
gint moving;
gint roving;
gint canvas_rows;
gint canvas_cols;
GSList *canvas_list;
GdkVisual *visual;
GdkColormap *colourmap;
GtkWidget *display_box, *mpane, *tpane;
/* screen dimensions */
gint x;
gint y;
gint width;
gint height;
/* model pane dimensions */
gint tree_width;
gint tray_height;
gint tree_divider;
/* for OpenGL - the smaller of width & height */
gint size;    /* pixel size */
gdouble rsize; /* real world size */
/* size of colourmap */
gint depth;
/* number of subwindows */
gint subx;
gint suby;
/* pixel dimensions */
gint subwidth;
gint subheight;
/* smallest of the above two */
gint subscale;
/* fraction dimensions */
gdouble fsubwidth;
gdouble fsubheight;
/* list of subwindow centres */
gint subcenx[MAX_DISPLAYED];
gint subceny[MAX_DISPLAYED];
/* dialog list */
GSList *dialog_list;
/* copy active select */
struct model_pak *select_source;
/* rendering */
struct render_pak render;

/* task handling */
GSList *task_list;
GtkWidget *task_label;
gint max_threads;
GThreadPool *thread_pool;
};

/********************************/
/* element properties structure */
/********************************/
struct elem_pak
{
gchar *symbol;
gchar *name;
gint number;         /* NB: keep! elem_pak is used for more than elem table */
gdouble weight;
gdouble cova;
gdouble vdw;
gdouble charge;
gdouble shell_charge;
gdouble colour[3];
};

/**********************/
/* spatial vector pak */
/**********************/
struct vec_pak
{
gdouble colour[3];
/* coords */
gdouble x[3];
gdouble rx[3];
/* normal */
gdouble n[3];
gdouble rn[3];
};

/***************************/
/* geometric notation pack */
/***************************/
struct geom_pak
{
gint type;           /* atom, bond, angle etc. */
gint image[3];       /* iff dist -> image translation of the 2nd core */
GSList *cores;
gdouble colour[3];
gchar *text;        /* text */
};

/***********************/
/* special object pack */
/***********************/
struct object_pak
{
gint id;
gint type;          /* plane, vector, ribbon */
gpointer *data;     /* object definition data */
};

/**********************/
/* general point pack */
/**********************/
struct point_pak
{
gint type;
gint num_points;
gdouble *x[3];
};

/*****************************************************************/
/* structure for bulk computation of valid cuts & their energies */
/*****************************************************************/
struct shift_pak
{
gint locked;
gdouble shift;
gint region[2];
gdouble eatt[2];
gdouble esurf[2];
gboolean dipole_computed;
gdouble dipole;
gdouble gnorm;
gchar *procfile;
};

/********************************/
/* general plane data structure */
/********************************/
struct plane_pak
{
/* is this a symmetry generated plane or not */
gint primary;

/* XRD info */
gint multiplicity;
/* real, imaginary structure factor components */
gdouble f[2];

/* raw direction (should be miller, but use float just in case) */
gdouble m[3];
/* cdd inequality - normalized/latmat'd -ve of the miller indices */
gdouble x[3];
/* hkl label */
gint index[3];
/* unrotated normal */
gdouble norm[3];
/* face ranking data */
gdouble dhkl;
gdouble eatt[2];
gdouble esurf[2];
/* attached shift list */
GSList *shifts;
/* drawing control flags */
gint present;
gint visible;
/* hull vertices */
GSList *vertices;
/* hkl label position */
gdouble rx[3];
};

/********************/
/* space group info */
/********************/
struct space_pak
{
gint lookup;
gint spacenum;
gint lattice;
gint pointgroup;
gint cellchoice;
gint centric;
gchar *spacename;
gchar *latticename;
gint order;
/* signs & ordering of x,y,z general positions */
gdouble **matrix;
/* const offset to general positions */
gdouble **offset;
};

/*****************/
/* symmetry info */
/*****************/
struct symop_pak
{
gint type;
gint order;
gdouble point[3];
gdouble norm[3];
gdouble matrix[9];
};

struct symmetry_pak
{
gint num_symops;
struct symop_pak *symops;
gint num_items;
gchar **items;
/* redundant? - pass around via attached data to callbacks? */
GtkWidget *summary;
GtkWidget *pg_entry;
gchar *pg_name;
};

/*************/
/* gulp data */
/*************/
struct gulp_pak
{
gint no_exec;
gint run;
gint dock;
gint method;
gint optimiser;
gint optimiser2;
gint switch_type;
gdouble switch_value;
gint unit_hessian;
gint qeq;
gint free;
gint zsisa;
gint phonon;
gint eigen;
gint compare;
gint nosym;
gint fix;
gint cycles;
gint ensemble;
gint coulomb;
gdouble maxcyc;
gint super[3];
gint no_esurf;     /* old gulp's can't handle sbulkenergy keyword */
gdouble energy;
gdouble esurf[2];   /* first - unre, second - re */
gint num_kpoints;  /* 0 -> don't include kpoint keyword */
gdouble kpoints[3]; /* TODO - a list of kpoints */
gchar *esurf_units;
gint no_eatt;      /* old gulp's can't handle dhkl keyword */
gdouble eatt[2];
gchar *eatt_units;
gdouble sbulkenergy;
gdouble sdipole;
gdouble sdipole_tolerance;
/* electrostatics */
gdouble qsum;
gdouble gnorm;
gdouble temp;
/* dynamics info */
gdouble frame_time;
gdouble frame_ke;
gdouble frame_pe;
gdouble frame_temp;
/* HACK - store original structure so we can recall after reading trajectory */
gint orig_fractional;
gint orig_construct_pbc;
gdouble orig_pbc[6];
gdouble orig_latmat[9];
GSList *orig_cores;
GSList *orig_shells;
/* electrostatic potential calcs */
gint epot_autoscale;
GSList *epot_vecs;
GSList *epot_vals;
gint epot_divisions;
gdouble epot_min;
gdouble epot_max;
/* widgets that need to be updated */
GtkWidget *energy_entry;
GtkWidget *sdipole_entry;
GtkWidget *esurf_entry;
GtkWidget *eatt_entry;
GtkWidget *sbe_entry;

/* element/potentials data */
GSList *ff_list;
GSList *species_list;
GSList *element_list;

/* filename control */
gchar *libfile;
gchar *srcfile;
gchar *temp_file;
gchar *out_file;
gchar *dump_file;
gchar *trj_file;
gchar *mov_file;
/* stuff gdis didn't understand */
gchar *extra;
};

/***************/
/* gamess data */
/***************/

struct GMS_keyword_pak
{
gchar *label;
gint id;
};

typedef enum {
  GMS_ANGS, GMS_BOHR
} GMSUnits;

typedef enum {
  GMS_RUN, GMS_CHECK, GMS_DEBUG
} GMSExeType;

typedef enum {
  GMS_USER, GMS_MNDO, GMS_AM1, GMS_PM3, GMS_MINI, GMS_MIDI,
  GMS_STO, GMS_N21, GMS_N31, GMS_N311, GMS_P6311G,
  GMS_DZV, GMS_DH, GMS_TZV, GMS_MC
} GMSBasisType;

typedef enum {
  /* single coordinate options */
  GMS_ENERGY, GMS_GRADIENT, GMS_HESSIAN, GMS_PROP, GMS_MOROKUMA, GMS_TRANSITN, GMS_FFIELD, GMS_TDHF, GMS_MAKEFP,
  /*  multiple geometry options */
  GMS_OPTIMIZE, GMS_TRUDGE, GMS_SADPOINT, GMS_IRC, GMS_VSCF, GMS_DRC, GMS_GLOBOP, GMS_GRADEXTR, GMS_SURFACE
} GMSRunType;

typedef enum {
  GMS_RHF, GMS_UHF, GMS_ROHF
} GMSScfType;

typedef enum {
  GMS_QA, GMS_NR, GMS_RFO, GMS_SCHLEGEL
} GMSOptType;

struct gamess_pak
{
GMSUnits units;
GMSExeType exe_type;
GMSRunType run_type;
GMSScfType scf_type;
GMSBasisType basis;
GMSOptType opt_type;
gint ngauss;
gint converged;
gint have_CI;
gint have_CC;
gint wide_output;
gint MP_level;
gdouble total_charge;
gdouble multiplicity;
gdouble time_limit;
gdouble mwords;
gdouble num_p;
gdouble num_d;
gdouble num_f;
gint have_heavy_diffuse;
gint have_hydrogen_diffuse;
gdouble nstep;
gdouble maxit;
gdouble energy;
gdouble max_grad;
gdouble rms_grad;
gint have_energy;
gint have_max_grad;
gint have_rms_grad;
gchar *title;
gchar *srcfile;
gchar *temp_file;
gchar *out_file;

/* widgets that need to be updated */
GtkWidget *energy_entry;
};

/***************/
/* siesta data */
/***************/
struct siesta_pak
{
gdouble energy;
gdouble max_grad;
gint have_energy;
gint have_max_grad;
};

/***************/
/* abinit data */
/***************/
struct abinit_pak
{
gdouble energy;
gdouble max_grad;
gdouble rms_grad;
};

/***************/
/* NWChem data */
/***************/
struct nwchem_pak
{
gdouble energy;
gdouble max_grad;
gdouble rms_grad;
gint min_ok;
gint have_energy;
gint have_max_grad;
gint have_rms_grad;
};

/***************/
/* CASTEP data */
/***************/
struct castep_pak
{
gdouble energy;
gdouble max_grad;
gdouble rms_grad;
gint min_ok;
gint have_energy;
gint have_max_grad;
gint have_rms_grad;
};

/**********************/
/* surface generation */
/**********************/
struct surface_pak
{
struct model_pak *model;
gint optimise;         /* relaxed/unrelaxed */
gint converge_eatt;    /* T - eatt, F - esurf */
gint converge_r1;
gint converge_r2;
gint include_polar;    /* calc shifts - keep polar surfaces */
gint ignore_bonding;   /* gensurf - ignore bonding */
gint create_surface;
gint true_cell;
gint keep_atom_order;
gdouble miller[3];
gdouble shift;
gdouble dspacing;
gdouble depth;
gdouble depth_vec[3];
gdouble region[2];
};

/************/
/* XRD data */
/************/
struct diffract_pak
{
gint radiation;
gdouble wavelength;
gdouble theta[3];
gdouble dhkl_min;
gint broadening;
gdouble asym;
gdouble u, v, w;
};

/************************/
/* model data structure */
/************************/
struct model_pak
{
/* main info */
gint id;
gint number;
gboolean locked;         /* prevent deletion (eg by dependent child process) */
gboolean protein;        /* cope with ugly PDB element labelling */
gint mode;
gint state;              /* the mode state (always reset when changing modes) */
gint grafted;            /* indicates if it's been put on the model tree yet */
gint redraw;             /* indicates a redraw is required */
gint redraw_count;       /* current number of redraws */
gint redraw_current;     /* current redraw time (micro-sec) */
gint redraw_cumulative;  /* cumulative redraw time (micro-sec) */
gint redraw_time;        /* average redraw time (micro-sec) */
gint fractional;
guint periodic;
guint num_frames;
guint cur_frame;
/* external animation frame info */
guint header_size;
guint frame_size;
guint file_size;
guint expected_cores;
guint expected_shells;
gboolean trj_swap;
/* bonding types */
gboolean build_molecules;
gboolean build_hydrogen;
gboolean build_polyhedra;
gboolean build_zeolite;
gint construct_pbc;
gint done_pbonds;   
gint cbu;              /* continuous update of bonds (# atoms related) */
gint colour_scheme;
gint has_sof;
gint sof_colourize;
gint atom_info;
gint shell_info;
gdouble rmax;           /* auto */
gdouble scale;          /* user */
gdouble zoom;           /* comparable between models */

/* display flags for the drawing area */
gint show_names;
gint show_title;
gint show_hbonds;
gint show_frame_number;
gint show_energy;
gint show_charge;
gint show_atom_charges;
gint show_atom_labels;
gint show_atom_types;
gint show_geom_labels;
gint show_cores;
gint show_shells;
gint show_axes;
gint show_cell;
gint show_cell_images;
gint show_cell_lengths;

/* ghosts */
gint num_ghosts;
GSList *ghosts;
/* coord data */
gint num_atoms;        /* total (incl. periodic images etc.) */
gint num_asym;         /* asym cell */
gint num_shells;
gint num_geom;

/* main lists */
GSList *cores;
GSList *shels;
GSList *bonds;    /* radii based connectivity */
GSList *ubonds;   /* user modified bond list (overlaid on the above) */
GSList *moles;
GSList *images;
GSList *selection;
GSList *unique_atom_list;
GSList *measure_list;
GSList *layer_list;
GSList *graph_list;
GSList *picture_list;
GSList *transform_list;
GSList *frame_data_list;
gpointer graph_active;
gpointer picture_active;
gpointer project;

/* property hash table and ranked list */
GHashTable *property_table;
GSList *property_list;

/* spatial partioning data */
gint zone_div[3];
gdouble zone_min[3];
gdouble zone_max[3];
gdouble zone_dx[3];
gint num_zones;
gpointer *zones;

/* special object lists */
GSList *ribbons;
GSList *spatial;
GSList *phonons;
GSList *ir_list;
GSList *raman_list;
gint num_phonons;
gint current_phonon;
gint pulse_count;
gint pulse_direction;
GtkWidget *phonon_slider;
gboolean show_eigenvectors;

/* model geometric and display data */
gdouble offset[3];
gdouble angle[3];
gdouble da;
gdouble pbc[6];
gdouble centroid[3];
gdouble cell_angles[3];
gdouble area;
gdouble volume;
/* translation matrix/vectors & its inverse */
gdouble latmat[9];
gdouble ilatmat[9];
/* reciprocal lattice vectors */
gdouble rlatmat[9];
/* cummulative rotation matrix & its inverse */
gdouble rotmat[9];
gdouble irotmat[9];
/* rotation/translation transformation matrix */
gdouble display_lattice[16];
/* special object flags */
gint axes_type;
gint mesh_on, box_on, asym_on;
gboolean ms_colour_scale;
gint ms_colour_method;

/* special object data */
struct vec_pak xlat[6];
struct vec_pak axes[6];
struct vec_pak cell[8];
gint select_box[4];

/* morphology data */
gint morph_type;                /* BFDH, etc. */
gint morph_label;               /* flag - hkl labelling */
gint num_vertices;
gint num_planes;                /* total number of faces */
gint num_facets;                /* number of present facets in plane list */
/* morphology */
GSList *vertices;
GSList *planes;

/* space group */
struct space_pak sginfo;
/* symmetry info */
struct symmetry_pak symmetry;
/* periodic image numbers - start/stop multiples */
gint num_images;
gdouble image_limit[6];

/* region display (marvin only) */
gint region_max;
gint region[4];
gint region_empty[4];

/* text info */
gchar *title;
gchar filename[FILELEN];
gchar gulplib[FILELEN];
gchar *basename;
/* energetics */
struct gulp_pak gulp;
struct gamess_pak gamess;
struct siesta_pak siesta;
struct abinit_pak abinit;
struct nwchem_pak nwchem;
struct castep_pak castep;
struct diffract_pak diffract;
/* surface config */
struct surface_pak surface;
/* element data */
gint num_elem;
struct elem_pak *elements;
/* animation sequences */
gint animation;
gint animating;
gint anim_confine;
gboolean anim_fix;
gboolean anim_loop;
gdouble anim_speed;
gpointer analysis;
GList *frame_list;
FILE *afp;
};

/**********************/
/* Supported file pak */
/**********************/
struct file_pak
{
gint id;                      /* unique identifier */
gint group;                   /* used to group inp/out types */
gint menu;                    /* (true/false) include in file menu listing? */
gchar *label;                 /* text info for the user */
GSList *ext;                  /* extension matching list */
gint (*read_file) (gchar *, struct model_pak *);     /* read routine */
gint (*write_file) (gchar *, struct model_pak *);    /* write routine */
gint (*read_frame) (FILE *, struct model_pak *);     /* animation read */
};

/******************/
/* MDI structures */
/******************/
/* TODO - combine box_pak & cand_pak? */
struct box_pak
{
gint component;
gint x;
gint y;
gint z;
};

struct mdi_pak
{
gint box_dim;
gdouble latt_sep;
gint num_comp;
gint *comp_idx;
gint *comp_req;
gint *comp_done;
gint *array;
};

struct cand_pak
{
gint pos;
gint x;
gint y;
gint z;
};

/*********************************/
/* widget convenience structures */
/*********************************/
struct callback_pak
{
gchar *label;
gint id;
};

/*******************/
/* main prototypes */
/*******************/
struct model_pak *model_ptr(gint, gint);

void sys_free(void);

void get_eledata(void);
void put_elem_data(struct elem_pak *, struct model_pak *);
gint get_elem_data(gint, struct elem_pak *, struct model_pak *);
gint construct_unique(struct model_pak *);

void init_objs(gint, struct model_pak *);
void init_coords(struct model_pak *);

void refresh_connectivity(struct model_pak *);

void make_axes(struct model_pak *);
void make_cell(struct model_pak *);
void make_mesh(struct model_pak *);
void make_latmat(struct model_pak *);
void make_rlat(struct model_pak *);

void rot(gdouble *, gdouble , gdouble , gdouble);

void view_panel(void);
void toggle_cell(struct model_pak *);

void render_dialog(void);
void mdi_setup_widget(void);
void mdi_model_create(gint);
void creation_widget(void);

void about(void);
void write_dat(struct model_pak *);

gdouble ran2(void);
gint fill(void);

void run_pov(void);
void exec_pov(gchar *);
void colour_change(gint);
void display_properties(void);
void pbc_constrain_atoms(struct model_pak *);
void pbc_constrain_mols(struct model_pak *);
void update_box(gint, gint, struct model_pak *, gint);

void new_colour(GtkWidget *, GtkColorSelection *);
void animate_setup_widget(void);
gint get_frame(struct model_pak *data, gint);

void create_new_model(void);
void delete_active(void);
void free_model(struct model_pak *);

void mem_shrink(struct model_pak *);

void template_model(struct model_pak *);
gint prep_model(struct model_pak *);

gint elem_symbol_test(const gchar *);
gint elem_number_test(const gchar *);
gint elem_test(const gchar *);

gint pdb_elem_type(const gchar *);

void save_apd(void);

gint geom_label(gint, struct model_pak *, gint *, gchar *);
void geom_info(void);
void update_creator(void);

void set_rgb_colour(GdkGC *, gint *);

void surf_dialog(void);
void surf2_dialog(void);

void gperiodic_win(void);

gint exec_cdd(struct model_pak *);
void sym_info(GtkWidget *, struct model_pak *);
void update_symlist(struct model_pak *);

gint write_gdisrc(void);
gint read_gdisrc(void);

gint generate_surface(struct model_pak *, struct model_pak *);

void unfragment(struct model_pak *);

gint event_render_modify(GtkWidget *, gpointer *); 

void check_geom(struct model_pak *);
void alter_geom(struct model_pak *, gint, gint);

void space_action(gint, gint);

void sof_colour(struct model_pak *);

void latmat_fudge(struct model_pak *);

/* NB: currently, make_ can't replace init_ due to axes transformation */
void init_rotmat(gdouble *);
void make_rotmat(gdouble *, gdouble , gint);

void gdk_draw(gint);
void force_view(gint);

void overlay_models_dialog(void);

void delay(gint);

void expose_canvas(void);

gint read_elem_data(FILE *, gint);
gint write_elem_data(FILE *);

void cmd_init(gint, gchar **);

GSList *slist_gchar_dup(GSList *);
void free_slist(GSList *);
void free_list(GList *);

