/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "gdis.h"
#include "parse.h"

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/******************************/
/* allocate & strip extension */
/******************************/
gchar *strdup_no_extension(const gchar *name)
{
gint i;
static gchar *ret;

/* FIXME - is no extension handled ok? */
for (i=0 ; i<strlen(name) ; i++)
  if (*(name+i) == '.')
    break;

ret = g_malloc((i+1)*sizeof(gchar));
strncpy(ret,name,i);
*(ret+i) = '\0';

return(ret);
}

/*******************************************************/
/* get a float from a string (even in fraction format) */
/*******************************************************/
/* TODO - fortran prints a bunch of *'s if the number is too big - */
/* find a way to cope with this */
#define DEBUG_STR_TO_FLOAT 0
gfloat str_to_float(gchar *str)
{
gint i;
gchar *ptr;
gfloat num, den, val;

/* do we have a backslash? */
ptr = rindex((char *) str, '/');
if (ptr)
  {
#if DEBUG_STR_TO_FLOAT
printf("processing: %s -> ",str);
#endif
  num = atof((char *) str);
  den = atof((char *) ++ptr);
  val = num/den;
#if DEBUG_STR_TO_FLOAT
printf("[%f]\n",val);
#endif
  }
else
  {
/* NEW - hack to remove equal signs that may crop up */
  for (i=0 ; i<strlen(str) ; i++)
    if (*(str+i) == '=')
      *(str+i) = ' ';

  val = atof(g_strstrip(str));
  }

return(val);
}

/**************************************************************/
/* return a string of keyword code (if any) found in a string */
/**************************************************************/
/* 1st item -> number actually found */
#define DEBUG_GET_KEYWORD 0
gint *get_keyword(gchar *str, gint max)
{
gint i, j, n, len;
gchar **buff;
gint *list;

#if DEBUG_GET_KEYWORD
printf("extracted: ");
#endif

list = g_malloc((max+1) * sizeof(gint));
buff = get_tokens(str, MAX_TOKENS);
n=1;
i=0;
while(i < MAX_TOKENS)
  {
/* default keyword code - nothing */
  *(list+n) = -1;
  j=0;
  while(keywords[j].code != -1)
    {
    len = strlen(keywords[j].label);
    if (g_strncasecmp(*(buff+i), keywords[j].label, len) == 0)
      {
#if DEBUG_GET_KEYWORD
printf(" %d",keywords[j].code);
#endif
      *(list+n) = keywords[j].code;
      if (++n == max+1)
        goto get_keyword_done;
      }
    j++;
    }
  i++;
  }
get_keyword_done:;
g_strfreev(buff);
*list = n-1;
#if DEBUG_GET_KEYWORD
printf("\n");
#endif

return(list);
}

gint num_keys(void)
{
gint n;

n=0;
while (keywords[n].code != -1)
  n++;
/*
printf("Found %d keywords\n",n);
*/
return(n);
}

/******************************************/
/* retrieve extension and return filetype */
/******************************************/
#define DEBUG_MODEL_TYPE 0
gint model_type(const gchar *name)
{
gint id;
gchar *tmp, *key;

/* test length */
if (strlen(name) >= FILELEN)
  {
#if DEBUG_MODEL_TYPE
printf("model_type() error: filename has exceeded maximum length."); 
#endif
  return(-1);
  }
/* get file extension (if any) */
tmp = (gchar *) rindex(name,'.');
if (!tmp)
  {
#if DEBUG_MODEL_TYPE
printf("model_type() warning: missing extension.\n");
#endif
  return(-1);
  }

key = g_strdup(tmp+1);

/* do the file extension matching */
if (g_strncasecmp(key,"gin",3) == 0 || g_strncasecmp(key,"res",3) == 0)
  id = GULP;
else if (g_strncasecmp("got",key,3) == 0 || g_strncasecmp("gout",key,4) == 0)
  id = GULPOUT;
else if (g_strncasecmp(key,"mot",3) == 0 || g_strncasecmp(key,"mvout",5) == 0
                                         || g_strncasecmp(key,"mvnout",6) == 0)
  id = MVNOUT;
/* NB: if below is before the above, .mvnout will be incorrectly */
/* identified as type MARVIN instead of type MVNOUT */
else if (g_strncasecmp("mvn",key,3) == 0 || g_strncasecmp("mvn-r",key,5) == 0)
  id = MARVIN;
else if (g_strncasecmp("gmf",key,3) == 0)
  id = MORPH;
else if (g_strncasecmp(key,"car",3) == 0 || g_strncasecmp(key,"arc",3) == 0
                                         || g_strncasecmp(key,"cor",3) == 0)
  id = BIOSYM;
else if (g_strncasecmp(key,"xyz",3) == 0)
  id = XYZ;
else if (g_strncasecmp(key,"xtl",3) == 0)
  id = XTL;
/* general types for specified babel conversion */
else if (g_strncasecmp(key,"inp",3) == 0 || g_strncasecmp(key,"out",3) == 0)
  id = BABEL;
else if (g_strncasecmp(key,"pdb",3) == 0 || g_strncasecmp(key,"dat",3) == 0)
  id = BABEL;
else if (g_strncasecmp(key,"cssr",3) == 0 || g_strncasecmp(key,"fdat",3) == 0)
  id = BABEL;
else if (g_strncasecmp(key,"cif",3) == 0)
  id = CIF;
else if (g_strncasecmp(key,"trg",3) == 0)
  id = GULP_TRJ;
/* all others are unrecognized files */
else
  {
#if DEBUG_MODEL_TYPE
printf("model_type() warning: unrecognized extension.\n");
#endif
  return(-1);
  }

g_free(key);
return(id);
}

/*********************/
/* tokenize a string */
/*********************/
/* replacement routine for copy_items */
/* aim is to have one call, rather than multiple copy_item calls */
/* trouble is, g_strsplit doesn't eliminate multiple separators */
/* return number found -> check if enough in file parsing */
/* ensure exactly num tokens returned!!! */
/* strlen = 0 if token is empty */
#define DEBUG_GET_TOKENS 0
gchar **get_tokens(gchar *src, gint num)
{
gint i, j;
gchar **buff, *tmp;
static gchar **dest;

/* duplicate & replace all whitespace with a space */
/* strange errors can be avoided if a strstrip is done */
tmp = g_strdup(src);
for (i=0 ; i<strlen(tmp) ; i++)
  if (isspace((int) *(tmp+i)))
    *(tmp+i) = ' ';
g_strstrip(tmp);

/* NB: most problems have occured by making MAX_TOKENS too small */
/* for some reason it can need many more than it would apparently seem */
buff = g_strsplit(tmp, " ", MAX_TOKENS);

/* num+1 -> last ptr is NULL, so g_strfreev works */
dest = g_malloc((num+1)*sizeof(gchar *));

i=j=0;
/* fill in the non empty tokens */
while (*(buff+i) != NULL && j<num)
  {
  if (strlen(*(buff+i)))
    *(dest+j++) = g_strdup(g_strstrip(*(buff+i)));
  i++;
  }

/* pad with empty strings */
while (j<num)
  *(dest+j++) = g_strdup("");

/* terminate */
*(dest+num) = NULL;

#if DEBUG_GET_TOKENS
for (i=0 ; i<num ; i++)
  printf("%s:",*(dest+i));
printf("%p\n",*(dest+num));
#endif

/* done */
g_strfreev(buff);
return(dest);
}

/* need another routine that gets everything in a line past */
/* a specified point - this will replace copy_items(...ALL) */
gchar *get_token_pos(gchar *src, gint num)
{
gint i,j,n,len;

/* flag the start(i) and end(j) */
len = strlen(src);
i = j = 0;
for (n=0 ; n<=num ; n++)
  {
  i = j;
/* FIXME - use the isspace function here */
  while((*(src+i) == ' ' || *(src+i) == '\t') && i<len)
    i++; 

  j = i;
  while(*(src+j) != ' ' && *(src+j) != '\t' && j<len)
    j++;
  }
 
/* return ptr to position */
return(src+i);
}

/***************************************************/
/* extract (space or tab sep.) items from a string */
/***************************************************/
gint copy_items(gchar *dest, gchar *src, gint num, gint type)
{
gint i,j,n,len;

/* flag the start(i) and end(j) */
len = strlen(src);
i = j = 0;
for (n=0 ; n<num ; n++)
  {
  i = j;
  while((*(src+i) == ' ' || *(src+i) == '\t') && i<len)
    i++; 

  j = i;
  while(*(src+j) != ' ' && *(src+j) != '\t' && j<len)
    j++;
  }
 
/* copy appropriate part of line into item */
if (type == SINGLE)
  {
  strncpy(dest,src+i,j-i);
/* I think this is necessary - not sure why though */
  *(dest+j-i) = '\0'; 
  }
else
  strcpy(dest,src+i);

/* remove any \n's */
for (i=0 ; i<strlen(dest) ; i++)
  if (*(dest+i) == '\n')
    *(dest+i) = '\0'; 

return(0);
}

/*****************************************/
/* the inverse of the copy_items routine */
/*****************************************/
#define DEBUG_put_item 0
/* FIXME - systematic way of dealing with string length mismatches */
/* TODO - this shouldn't need to return anything as *line is changed */
gint put_item(gchar *dest, gchar *item, gint num)
{
gint i,j,n,len1,len2;

#if DEBUG_put_item
printf("Replacing item number %d...\n",num);
printf("In: %s\n",dest);
printf("With: %s\n",item);
#endif

len1 = strlen(item);
len2 = strlen(dest);

i = j = 0;
for (n=0 ; n<num ; n++)
  {
  i = j;
  while(*(dest+i) == ' ' || *(dest+i) == '\t')
    i++; 

  j = i;
  while(*(dest+j) != ' ' && *(dest+j) != '\t' && j<len2)
    j++;
  }

#if DEBUG_put_item
printf("Item starts at: %d\n",i);
printf("Item end at: %d\n",j);
#endif

/* don't copy the length of the item */
/* but rather the length of what is being replaced */
/* strncpy((line+i),item,j-i); */
/* copy the length of the item  - needed for saving BIOSYM files */
strncpy((dest+i),item,len1);

#if DEBUG_put_item
printf("ret: %s\n",line);
#endif

return(0);
}

/****************************/
/* initialize a parsing pak */
/****************************/
gint parse_ptr(struct parse_pak *ptr, gint type)
{
switch (type)
  {
/* specialized routines for these filetypes */
  case GULP:
  case MVNOUT:
    break;

  case MARVIN:
    ptr->id = MARVIN;
    ptr->hdr_skip = 0;
    strcpy(ptr->coord_start1,"COORD");
    ptr->coord_start1_len = 5;
    strcpy(ptr->coord_start2,"BASIS");
    ptr->coord_start2_len = 5;
    ptr->a_pos = 1;
    ptr->l_pos = 1;
    ptr->x_pos = 3;
    ptr->y_pos = 4;
    ptr->z_pos = 5;
    ptr->tail_pos = 0;
/* make gdis read all marvin regions */
    strcpy(ptr->coord_end1,"dummy");
    ptr->coord_end1_len = 5;
    strcpy(ptr->coord_end2,"dummy");
    ptr->coord_end2_len = 5;
    break;

  case BIOSYM:
    ptr->id = BIOSYM;
    ptr->hdr_skip = 3;
    strcpy(ptr->coord_start1,"!DATE");
    ptr->coord_start1_len = 5;
/* NB: PBC+space - avoid matching PBC=ON/OFF */
    strcpy(ptr->coord_start2,"PBC ");
    ptr->coord_start2_len = 4;
    ptr->a_pos = 8;
    ptr->l_pos = 1;
    ptr->x_pos = 2;
    ptr->y_pos = 3;
    ptr->z_pos = 4;
    ptr->tail_pos = 5;
    strcpy(ptr->coord_end1,"END");
    ptr->coord_end1_len = 3;
    strcpy(ptr->coord_end2,"END");
    ptr->coord_end2_len = 3;
    break;

  case XYZ:
    ptr->id = XYZ;
    ptr->hdr_skip = 2;
    strcpy(ptr->coord_start1,"");
    ptr->coord_start1_len = 0;
    strcpy(ptr->coord_start2,"");
    ptr->coord_start2_len = 0;
    ptr->a_pos = 1;
    ptr->l_pos = 1;
    ptr->x_pos = 2;
    ptr->y_pos = 3;
    ptr->z_pos = 4;
    ptr->tail_pos = 0;
    strcpy(ptr->coord_end1,"dummy");
    ptr->coord_end1_len = 5;
    strcpy(ptr->coord_end2,"dummy");
    ptr->coord_end2_len = 5;
    break;

  case XTL:
    ptr->id = XTL;
    ptr->hdr_skip = 0;
    strcpy(ptr->coord_start1,"NAME");
    ptr->coord_start1_len = 4;
    strcpy(ptr->coord_start2,"");
    ptr->coord_start2_len = 0;
    strcpy(ptr->special_start,"CELL");
    ptr->special_start_len = 4;
    ptr->a_pos = 1;
    ptr->l_pos = 1;
    ptr->x_pos = 2;
    ptr->y_pos = 3;
    ptr->z_pos = 4;
    ptr->tail_pos = 0;
    strcpy(ptr->coord_end1,"EOF");
    ptr->coord_end1_len = 3;
    strcpy(ptr->coord_end2,"dummy");
    ptr->coord_end2_len = 5;
    break;

  default:
    return(1);
  }
return(0);
}

/******************************************************************************
  FUNCTION: SwapEndian
  PURPOSE: Swap the byte order of a structure
******************************************************************************/
