/*******************************************************************************
  Copyright(c) 2000 - 2003 Radu Corlan. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information: radu@corlan.net
*******************************************************************************/

// scope.c: telescope control functions (new and improved ;-))

#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/resource.h>
#include <termios.h>
#include <unistd.h>
#include <stdlib.h>
#include <sched.h>
#include <math.h>
#include <errno.h>
#include <ctype.h>

#include "gcx.h"
#include "misc.h"
#include "params.h"
#include "scope.h"

#define GOTO_SPEED "2" 		/* degrees per second. undef to keep telescope default */

#define SLEW_FINISH_TIME 3000 /* we wait 2 seconds from the time the scope coordinates
			       * have reached their target before we declare the slew over
			       */
#define SLEW_END_TOL (0.1) /* how close we need to be from the target point to 
			      decide the slew ended (and begin end-of-slew delay) */
#define STABILISATION_DELAY P_INT(TELE_STABILISATION_DELAY) 
				/* time we wait at the end of the slew for the 
				   mount to stabilise */


#define SLEW_TIMEOUT 120000 /* max duration of a slew (ms): 2 minutes */
#define LX_SERIAL_TIMEOUT 1000000 /* this one is in us */


#define POLL_INTERVAL 20	 /* ms */
#define SERIAL_POLL_INTERVAL 200 /* ms between issuing telescope polling commands */

static struct scope the_scope;

static int lx_response(int fd, char *buf, int size, char **patterns);
static int timeout_read(int fd, char * resp, int resp_size, int timeout);
static void sersend(int fd, char *buf);
static char *resp_ack[] = {"L", "P", "A", NULL};
static char *resp_zero_one[] = {"0", "1", NULL};
static char *resp_zero_other[] = {"0", NULL};
static int scope_poll(struct scope *scope);
static int scope_get_dec(int fd, double *dec);
static int scope_get_ra(int fd, double *ra);
static double dec_factor(double dec);
static int is_losmandy(int fd);
static void end_dec_move(struct scope *scope);
static void end_ra_move(struct scope *scope);
static void centering_move_start(struct scope *scope);
static double dec_factor(double dec);
static void gemini_checksum(char *buf);
static void flush_serial(int fd);
static int set_precision(int fd);

/* check the scope is in the right state for taking commands. 
   return 0 if so, else a negative error */

static int ping_scope (int fd)
{
	char buf[256];
	int ret;

	buf[0] = 0x06;
	buf[1] = 0;
	sersend(fd, buf);
	ret = lx_response(fd, buf, 255, resp_ack);
	
	if (ret < 0) {
		err_printf("scope timeout on ping\n");
		return -1;
	}
	if (ret == 2 || ret == 3) {
		return 0;
	}
	if (ret == 0) { 	/* check for losmandy codes */
		if (buf[0] == 'b') { /* losmady waiting for startup mode selection */
			sersend(fd, "bR#");
			return 0;
		}
		return 0;
	}
	err_printf("telescope is in land mode!\n");
	return 0;		/* perhaps we should return an error here, 
				   but this is an unlikely occurence */
}


struct scope * scope_open(char *name)
{
	struct termios tio;

	if (name == NULL)
		return NULL;
	if (the_scope.state != SCOPE_CLOSED) {
		err_printf("scope already open\n");
		return NULL;
	}
	if (the_scope.name) {
		free (the_scope.name);
		the_scope.name = strdup(name);
	}

	the_scope.fd = open(name, O_RDWR);
	if (the_scope.fd <= 0) {
		err_printf("scope open error: %s\n", strerror(errno));
		return NULL;
	}
//	d3_printf("ser opened: %d\n", the_scope.fd);
	tcgetattr(the_scope.fd, &tio);
	cfsetospeed(&tio, B9600);
	cfsetispeed(&tio, B9600);
//	cfmakeraw(&tio);
	tio.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
	tio.c_oflag &= ~OPOST;
	tio.c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
	tio.c_cflag &= ~(CSIZE|PARENB);
	tio.c_cflag |= CS8;
	tcsetattr(the_scope.fd, TCSANOW, &tio);

	if (ping_scope(the_scope.fd)) {
		close(the_scope.fd);
		return NULL;
	}
	the_scope.state = SCOPE_IDLE;
	return &the_scope;
}

void scope_close(struct scope *scope)
{
	if (scope == NULL)
		return;
	if (scope->state == SCOPE_CLOSED)
		return;
	if (scope->state != SCOPE_IDLE) {
		scope_abort(scope);
	}
	scope->state = SCOPE_CLOSED;
	close(scope->fd);
}

// send the 'obs' object coordinates to the telescope
int scope_set_object(struct scope *scope, double ra, double dec, double epoch, char *name)
{
	char lb[64];
	char out[64];
	char buf[64];
	int i, j, k, jdi, ret;

	struct timeval tv;
	struct tm *t;

	if (scope == NULL)
		return -1;
	if (scope->state != SCOPE_IDLE) {
		err_printf("telescope is busy\n");
		return -1;
	}

	gettimeofday(&tv, NULL);
	t = gmtime(&(tv.tv_sec));

	scope->last_ra = ra;
	scope->last_dec = dec;
	scope->last_epoch = epoch;

// do the julian date (per hsaa p107)
	i = 1900 + t->tm_year;
	j = t->tm_mon + 1;
	k = t->tm_mday;

	jdi = k - 32075 + 1461 * (i + 4800 + (j - 14) / 12) / 4
		+ 367 * (j - 2 - (j - 14) / 12 * 12) / 12
		- 3 * (( i + 4900 + (j - 14) / 12) / 100) / 4;

	if (P_INT(TELE_PRECESS_TO_EOD))
		precess_hiprec(epoch, (jdi - JD2000) / 365.25 + 2000.0, &ra, &dec);

	flush_serial(scope->fd);

	if (set_precision(scope->fd)) {
		err_printf("error setting coordinates mode\n");
		return -2;
	}

	if (name != NULL) {
		snprintf(out, 63, ":ON%s#", name);
		sersend(scope->fd, out);
	} else {
		sersend(scope->fd, ":ONobj#");
	}

	degrees_to_dms_pr(lb, ra / 15.0, 0);
	d3_printf("RA'%.2f %s", 2000 + (jdi - JD2000) / 365.25, lb);

	sprintf(out, ":Sr%s#", lb);
	sersend(scope->fd, out);
	ret = lx_response(scope->fd, buf, 64, resp_zero_one);
	if (ret != 2) {
		err_printf("error setting RA\n");
		return -1;
	}

	degrees_to_dms_pr(lb, dec, 0);
	d3_printf(" DEC'%.2f %s -> scope\n", 2000 + (jdi - JD2000) / 365.25, lb);

	sprintf(out, ":Sd%s#", lb);
	sersend(scope->fd, out);
	ret = lx_response(scope->fd, buf, 64, resp_zero_one);
	if (ret != 2) {
		err_printf("error setting RA\n");
		return -1;
	}

	scope->target_ra = ra;
	scope->target_dec = dec;

	return 0;
}

// start a slew to the selected object
int scope_slew(struct scope *scope)
{
	char buf[256];

	if (scope == NULL)
		return -1;
	if (scope->state != SCOPE_IDLE) {
		err_printf("telescope is busy\n");
		return -1;
	}

	buf[0] = 0;
	flush_serial(scope->fd);

#ifdef GOTO_SPEED
	sersend(scope->fd, ":Sw"GOTO_SPEED"#");
	lx_response(scope->fd, buf, 256, resp_zero_one);
#endif
	sersend(scope->fd, ":MS#");
	scope->abort = 0;
	buf[0] = 0;
	lx_response(scope->fd, buf, 256, resp_zero_other);
	if (buf[0] != '0') {
		err_printf("Error starting slew: %s\n", buf);
		sersend(scope->fd, ":Q#");
		return -1;
	}
	scope->last_dra = -1.0;
	scope->last_ddec = -1.0;
	scope->errcount = 0;
	update_timer(&scope->op_tv);
	update_timer(&scope->tv);
	scope->state = SCOPE_SLEW_START;
	g_timeout_add(10, (GSourceFunc)scope_poll, scope);
	return 0;	

}

// start a slew to the selected object
int scope_timed_center(struct scope *scope, double dra, double ddec)
{
	char buf[256];

	if (scope == NULL)
		return -1;
	if (scope->state != SCOPE_IDLE) {
		err_printf("telescope is busy\n");
		return -1;
	}

	buf[0] = 0;
	flush_serial(scope->fd);
	if (scope_get_dec(scope->fd, &scope->dec)) {
		err_printf("start_timed_center: cannot get declination\n");
		return -1;
	}
	scope->ra_time = dra * 3600.0 * 1000.0 
		/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
	scope->dec_time = ddec * 3600.0 * 1000.0 
		/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
	d1_printf("timed center: dra=%.3f(%dms), ddec=%.3f(%dms)\n", dra, scope->ra_time,
		  ddec, scope->dec_time);
	if (scope->dec_time > 20000) {
		err_printf("dec move larger than 20 seconds clipped\n");
		scope->dec_time = 20000;
	}
	if (scope->dec_time < -20000) {
		err_printf("dec move larger than 20 seconds clipped\n");
		scope->dec_time = -20000;
	}
	if (scope->ra_time > 20000) {
		err_printf("ra move larger than 20 seconds clipped\n");
		scope->ra_time = 20000;
	}
	if (scope->ra_time < -20000) {
		err_printf("ra move larger than 20 seconds clipped\n");
		scope->ra_time = -20000;
	}
	if (scope->dec_reversal)
		scope->dec_time = - scope->dec_time;
	centering_move_start(scope);
	update_timer(&scope->op_tv);
	update_timer(&scope->tv);
	scope->state = SCOPE_CENTER1;
	g_timeout_add(10, (GSourceFunc)scope_poll, scope);
	return 0;	

}


void scope_abort(struct scope *scope)
{
	if (scope == NULL)
		return;
	if (scope->state == SCOPE_CLOSED)
		return;
//	g_source_remove_by_user_data(scope);
	end_ra_move(scope);
	end_dec_move(scope);
	usleep(50000);
	sersend(scope->fd, ":Q#");
	scope->state = SCOPE_ABORTED;
	return;
}

/* this is the big state machine that handles the telescope operations. 
   called regularly by a mainloop timer. returns zero when it doesn't 
   need to be polled anymore (usually at the end of an operation) */
static int scope_poll(struct scope *scope)
{
	double dr, dd, dec;
	int ms;

	if (scope == NULL)
		return 0;
	
	d4_printf(" %d", scope->state);
	
	if (scope->errcount > 10) {
		err_printf("meridian flip or comm errors, aborting\n");
		scope_abort(scope);
		return 0;
	}
	switch(scope->state) {
	case SCOPE_CLOSED:
	case SCOPE_IDLE:
	case SCOPE_ABORTED:
	case SCOPE_ERR:
		return 0;

	case SCOPE_SLEW_START:
		scope->waitms = SERIAL_POLL_INTERVAL - get_timer_delta(&scope->tv);
		if (scope->waitms > 0) {
			return 1;
		}
		if (scope_get_ra(scope->fd, &scope->ra))
			scope->errcount++;
		update_timer(&scope->tv);
		scope->state = SCOPE_SLEW_GET_DEC;
		return 1;
	case SCOPE_SLEW_GET_DEC:
		scope->waitms = SERIAL_POLL_INTERVAL - get_timer_delta(&scope->tv);
		if (scope->waitms > 0) {
			return 1;
		}
		if (scope_get_dec(scope->fd, &scope->dec))
			scope->errcount++;
		update_timer(&scope->tv);

		d4_printf("+++ %.3f, %.3f\n", scope->ra, scope->dec);

		dr = fabs(angular_dist(scope->ra, scope->target_ra)) 
			/ dec_factor(scope->target_dec);
		dd = fabs(angular_dist(scope->dec, scope->target_dec));

		if (P_INT(TELE_ABORT_FLIP)) {
			if ((scope->last_dra > 0) && (scope->last_dra < dr) && (dr > 3.0)) {
				scope->errcount++;
			}
			if ((scope->last_ddec > 0) && (scope->last_ddec < dd) && (dd > 3.0)) {
				scope->errcount++;
			}
		}
		scope->last_dra = dr;
		scope->last_ddec = dd;

		if ((dr > SLEW_END_TOL) || (dd > SLEW_END_TOL)) {
			scope->state = SCOPE_SLEW_START;
		} else {
			scope->state = SCOPE_SLEW_WAIT_END;
		}
		return 1;
	case SCOPE_SLEW_WAIT_END:
		scope->waitms = SLEW_FINISH_TIME - get_timer_delta(&scope->tv);
		if (scope->waitms > 0) {
			return 1;
		}
		update_timer(&scope->tv);
		if (is_losmandy(scope->fd) && P_DBL(TELE_GEAR_PLAY) > 0.001) {
			scope->ra_time = P_DBL(TELE_GEAR_PLAY) * 3600.0 * 1000.0 
				/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
			scope->dec_time = P_DBL(TELE_GEAR_PLAY) * 3600.0 * 1000.0 
				/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
			centering_move_start(scope);
			scope->state = SCOPE_GEAR_PLAY1;
			/* take out gear play */
		} else {
			scope->state = SCOPE_STABILIZE;
		}
		return 1;
	case SCOPE_STABILIZE:
		scope->waitms = STABILISATION_DELAY - get_timer_delta(&scope->tv);
		if (scope->waitms > 0) {
			return 1;
		}
		scope->state = SCOPE_IDLE;
		return 0;
	case SCOPE_GEAR_PLAY1:
		ms = get_timer_delta(&scope->tv);
		scope->waitms = abs(scope->ra_time) - ms;
		if (scope->waitms > 0) 
			return 1;

		end_ra_move(scope);
		end_dec_move(scope);
		update_timer(&scope->tv);
		scope->ra_time = - P_DBL(TELE_GEAR_PLAY) * 3600.0 * 1000.0 
			/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
		scope->dec_time = - P_DBL(TELE_GEAR_PLAY) * 3600.0 * 1000.0 
			/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
		centering_move_start(scope);
		scope->state = SCOPE_GEAR_PLAY2;
		return 1;
	case SCOPE_GEAR_PLAY2:
		ms = get_timer_delta(&scope->tv);
		scope->waitms = abs(scope->ra_time) - ms;
		if (scope->waitms > 0) 
			return 1;

		end_ra_move(scope);
		end_dec_move(scope);
		update_timer(&scope->tv);
		scope->ra_time = 0;
		scope->dec_time = 0;
		scope->state = SCOPE_STABILIZE;
		return 1;
	case SCOPE_CENTER1:
		ms = get_timer_delta(&scope->tv);
		if (abs(scope->ra_time) <= ms && scope->ra_move_active) {
			end_ra_move(scope);
		}
		if (abs(scope->dec_time) <= ms && scope->dec_move_active) {
			end_dec_move(scope);
		}
		if (scope->dec_move_active || scope->ra_move_active) {
			if (abs(scope->ra_time) > abs(scope->dec_time))
				scope->waitms = abs(scope->ra_time) - ms;
			else
				scope->waitms = abs(scope->dec_time) - ms;
			return 1;
		}
		update_timer(&scope->tv);
		if (!scope_get_dec(scope->fd, &dec)) {
			int truedec;
			truedec = (scope->dec_reversal ? -1 : 1) * scope->dec_time;
			if ((truedec > 0 && (dec - scope->dec) < 0.0) ||
			    (truedec < 0 && (dec - scope->dec) > 0.0)) {
				scope->dec_time = - 2 * scope->dec_time;
				scope->ra_time = 0;
				scope->dec_reversal = !scope->dec_reversal;
				centering_move_start(scope);
				d1_printf("dec reversal\n");
				scope->state = SCOPE_CENTER2;
				return 1;
			}
		}
		update_timer(&scope->tv);
		if (is_losmandy(scope->fd) && P_DBL(TELE_GEAR_PLAY) > 0.001) {
			scope->ra_time = P_DBL(TELE_GEAR_PLAY) * 3600.0 * 1000.0 
				/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
			scope->dec_time = P_DBL(TELE_GEAR_PLAY) * 3600.0 * 1000.0 
				/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
			centering_move_start(scope);
			scope->state = SCOPE_GEAR_PLAY1;
			/* take out gear play */
		} else {
			scope->state = SCOPE_STABILIZE;
		}
		return 1;
	case SCOPE_CENTER2:
		ms = get_timer_delta(&scope->tv);
		if (abs(scope->ra_time) <= ms && scope->ra_move_active) {
			end_ra_move(scope);
		}
		if (abs(scope->dec_time) <= ms && scope->dec_move_active) {
			end_dec_move(scope);
		}
		if (scope->dec_move_active || scope->ra_move_active) {
			if (abs(scope->ra_time) > abs(scope->dec_time))
				scope->waitms = abs(scope->ra_time) - ms;
			else
				scope->waitms = abs(scope->ra_time) - ms;
			return 1;
		}
		update_timer(&scope->tv);
		if (is_losmandy(scope->fd) && P_DBL(TELE_GEAR_PLAY) > 0.001) {
			scope->ra_time = P_DBL(TELE_GEAR_PLAY) * 3600.0 * 1000.0 
				/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
			scope->dec_time = P_DBL(TELE_GEAR_PLAY) * 3600.0 * 1000.0 
				/ 15.0 / P_DBL(TELE_CENTERING_SPEED);
			centering_move_start(scope);
			scope->state = SCOPE_GEAR_PLAY1;
			/* take out gear play */
		} else {
			scope->state = SCOPE_STABILIZE;
		}
		return 1;
	}
	return 0; 		/* remove timer function */
}

static int scope_get_ra(int fd, double *ra)
{
	char buf[256];
	int ret;
	double lra;

	flush_serial(fd);
	sersend(fd, ":GR#");
	ret = lx_response(fd, buf, 256, NULL);
	if (ret < 0) {
		err_printf("Timeout reading ra\n");
		return -1;
	}
	ret = dms_to_degrees(buf, &lra);
	if (ret) {
		err_printf("Error parsing ra from scope\n");
		return -1;
	}
	if (ra)
		*ra = 15*lra;
	return 0;
}

static int scope_get_dec(int fd, double *dec)
{
	char buf[256];
	int ret;
	double ldec;

	flush_serial(fd);
	sersend(fd, ":GD#");
	ret = lx_response(fd, buf, 256, NULL);
	if (ret < 0) {
		err_printf("Timeout reading dec\n");
		return -1;
	}
	ret = dms_to_degrees(buf, &ldec);
	if (ret) {
		err_printf("Error parsing dec from scope\n");
		return -1;
	}
	if (dec)
		*dec = ldec;
	return 0;
}

static void sersend(int fd, char *buf)
{
	d4_printf(">>> %s \n", buf);
	write(fd, buf, strlen(buf));
}

static void flush_serial(int fd)
{
	char buf[256];

	buf[0]=0;
	timeout_read(fd, buf, 255, 0);
	if (buf[0])
		d4_printf("serial flush returns: '%s'\n", buf);
}

static int is_losmandy(int fd)
{
	char buf[256];
	int ret;

	flush_serial(fd);
	sersend(fd, ":GV#");
	ret = lx_response(fd, buf, 256, NULL);
	if (ret == 0 && isdigit(buf[0]) && isdigit(buf[1]) && isdigit(buf[2]))
		return 1;
	else
		return 0;
}


static void centering_move_start(struct scope *scope)
{
	char buf[256];

	d1_printf("Timed move: %d %d [%d]\n", scope->ra_time, scope->dec_time, 
		  scope->dec_reversal);

	snprintf(buf, 255, ">170:%.0f!#", P_DBL(TELE_CENTERING_SPEED));
	gemini_checksum(buf);
	sersend(scope->fd, buf);
	sersend(scope->fd, ":RC#");
	if (scope->ra_time != 0) {
		scope->ra_move_active = 1;
		if (scope->ra_time > 0) {	
			sersend(scope->fd, ":Me#");
		} else {
			sersend(scope->fd, ":Mw#");
		}
	}
	if (scope->dec_time != 0) {
		scope->dec_move_active = 1;
		if (scope->dec_time > 0) {	
			sersend(scope->fd, ":Mn#");
		} else {
			sersend(scope->fd, ":Ms#");
		}
	}
}

static void end_ra_move(struct scope *scope)
{
	if (!scope->ra_move_active)
		return;
	if (scope->ra_time >= 0) {	
		sersend(scope->fd, ":Qe#");
	} else {
		sersend(scope->fd, ":Qw#");
	}
	scope->ra_move_active = 0;
}

static void end_dec_move(struct scope *scope)
{
	if (!scope->dec_move_active)
		return;
	if (scope->dec_time >= 0) {	
		sersend(scope->fd, ":Qn#");
	} else {
		sersend(scope->fd, ":Qs#");
	}
	scope->dec_move_active = 0;
}

/* 1/cos(dec), clamped at 5 */
static double dec_factor(double dec)
{
	if (cos(dec) > 0.2)
		return (1.0 / cos(dec));
	else
		return 5.0;

}

/* replace the '!' in the buffer with a gemini-style 
   checksum of preceding charaters */
static void gemini_checksum(char *buf)
{
	int ck=0;
	while (*buf) {
		if (*buf == '!') {
			ck &= 0x7f;
			ck ^= 0x40;
			*buf = ck;
			return;
		}
		ck ^= *buf++;
	}
}

/* set the precision mode, retun 0 if successful */
static int set_precision(int fd)
{
	char buf[256];
	int ret, i;

	buf[0] = 0;
	buf[255] = 0;
	flush_serial(fd);
	sersend(fd, ":GR#");
	ret = lx_response(fd, buf, 256, NULL);
	ret = 0; 
	i = 0;
	while (buf[i]) {
		if (buf[i] == ':')
			ret ++;
		i++;
	}
	if (ret == 1) {
		d3_printf("current precision is low\n");
		sersend(fd, ":U#");
		return 0;
	} else if (ret == 2) {
		d3_printf("current precision is high\n");
		return 0;
	} else {
		err_printf("can't understand response to :GR# [%s]\n", buf);
	}
	return -1;
}

// sync to current coordinates
static int scope_sync(struct scope *scope)
{
	char buf[64];

	d3_printf("lx Sync\n");
	usleep(100000);
	sersend(scope->fd, ":CM#");
	lx_response(scope->fd, buf, 64, NULL);
	usleep(100000);
	return 0;
}

int scope_sync_coords(struct scope *scope, double ra, double dec, double epoch)
{
	int ret;

	if (scope == NULL)
		return -1;
	if (scope->state != SCOPE_IDLE) {
		err_printf("telescope is busy\n");
		return -1;
	}

	if (epoch == 0.0)
		epoch = 2000.0;

	ret = scope_set_object(scope, ra, dec, epoch, "sobj");
	if (ret)
		return ret;
	return scope_sync(scope);
}

/* do one select and one read from the given file */
static int timeout_read(int fd, char * resp, int resp_size, int timeout)
{
	struct timeval tmo;
	int ret;
	fd_set fds;

	if (fd <= 0)
		return -1;

	tmo.tv_sec = 0;
	tmo.tv_usec = timeout;
	FD_ZERO(&fds);
	FD_SET(fd, &fds);

	ret = select(FD_SETSIZE, &fds, NULL, NULL, &tmo);
	if (ret == 0) {
		if (timeout > 0) {
			d1_printf("lx_read: timeout in receive\n");
			return -1;
		} else {
			return 0;
		}
	}
	if (ret == -1 && errno) {
		err_printf("lx_read: select error (%s)\n", strerror(errno));
		return -1;
	}
	ret = read(fd, resp, resp_size);
//	d3_printf("read returns %d, val=%d\n", ret, *resp);
	if (ret == -1 && errno != 0) {
		err_printf("lx_read: error in receive [%s]\n", strerror(errno));
		return -1;
	}
	return ret;
}


/* return the index to the matched pattern (starting at 1), 0 if a # was found, or -1 if we 
 * reached timeout without matching anything */
static int lx_response(int fd, char *buf, int size, char **patterns)
{
	int ret;
	char **p;
	char *bp = buf;
	int i;

	while (size > 2) {
		ret = timeout_read(fd, bp, size - 1, LX_SERIAL_TIMEOUT);
		if (ret < 0) {
			*bp = 0;
			d4_printf("lx response timeout: '%s'\n", buf);
			return -1;
		} else {
			bp += ret;
			*bp = 0;
			i = 1;
			p = patterns;
			if (*(bp - 1) == '#') {
				d4_printf("lx response: '%s'\n", buf);
				return 0;
			}
			while ((p != NULL) && (*p != NULL)) {
				if (!strcasecmp(buf, *p)) {
					d4_printf("lx response: '%s' (%d)\n", buf, i);
					return i;
				}
				p++;
				i++;
			}
			size -= ret;
		} 
	}
	*bp = 0;
	return -1;
}

void scope_status_string(struct scope *scope, char *buf, int size)
{

	if (scope == NULL) {
		snprintf(buf, size, "Not connected to telescope");
		return;
	}
	switch(scope->state) {
	case SCOPE_CLOSED:
		snprintf(buf, size, "Telescope conection closed");
		break;
	case SCOPE_IDLE:
		snprintf(buf, size, "Telescope idle");
		break;
	case SCOPE_ABORTED:
		snprintf(buf, size, "Telescope operation aborted");
		break;
	case SCOPE_ERR:
		snprintf(buf, size, "Telescope error");
		break;
	case SCOPE_SLEW_START:
	case SCOPE_SLEW_GET_DEC:
		if (scope->last_dra < 0 || scope->last_ddec < 0) {
			snprintf(buf, size, "Starting slew");
		} else {
			snprintf(buf, size, "Slewing; %.1f, %.1f to go",
				 scope->last_dra, scope->last_ddec);
		}
		break;
	case SCOPE_SLEW_WAIT_END:
		snprintf(buf, size, "Waiting for end of slew");
		break;
	case SCOPE_STABILIZE:
		snprintf(buf, size, "Waiting to stabilize, %.1f to go",
			 scope->waitms / 1000.0);
		break;
	case SCOPE_GEAR_PLAY1:
	case SCOPE_GEAR_PLAY2:
		snprintf(buf, size, "Taking out gear play, %.1f to go",
			 scope->waitms / 1000.0);
		break;
	case SCOPE_CENTER1:
	case SCOPE_CENTER2:
		snprintf(buf, size, "Centering, %.1f to go",
			 scope->waitms / 1000.0);
		break;
	default:
		snprintf(buf, size, "Invalid telescope state");
	}
}
