package GCPanel;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################
use utf8;
use Gtk2;
use GCBorrowings;
use GCGraphicComponents;

use strict;

{
    package GCItemPanel;
    use base "Gtk2::Frame";
    
    sub new
    {
        my ($proto, $parent, $options, $layout) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new();

        bless ($self, $class);

        $self->{disableBorrowerChange} = 0;
        $self->{autoUpdate} = 1;
        $self->{previouslyHidden} = 0;

        $self->{expanders} = [];
        $self->{formatted} = [];
        #Stores for each fields the widgets which depends on it
        # (used for formatted and launcher)
        $self->{dependencies} = {};

        $self->{parent} = $parent;
        $self->{lang} = $parent->{lang};
        $self->{options} = $options;
        $self->{layout} = $layout;
        $self->set_shadow_type('none');
        
        $self->{defaultImage} = $parent->{defaultImage};
        $self->{tooltips} = $parent->{tooltips};
        
        $self->{infoBox} = new Gtk2::VBox(1,0);
        $self->{viewAllBox} = new Gtk2::EventBox;
        $self->{viewAllVBox} = new Gtk2::VBox(0,0);
        $self->{viewAll} = new Gtk2::HBox(0,0);
        $self->{viewAllButton} = new Gtk2::Button->new_from_stock('gtk-refresh');
        $self->{viewAll}->pack_start($self->{viewAllButton},1,0,50);
        $self->{viewAllVBox}->pack_start($self->{viewAll},0,0,0);
        $self->{viewAllBox}->add($self->{viewAllVBox});
        $self->{viewAllButton}->signal_connect('clicked' => sub {
            $self->{parent}->allItems;
        });

        $self->{margin} = $GCUtils::margin;
        $self->{halfMargin} = $GCUtils::halfMargin;
        $self->{quarterMargin} = $GCUtils::quarterMargin;

        return $self;
    }
    
    sub loadStyles
    {
        my ($self, $style) = @_;
    }    
    
    sub createContent
    {
        my ($self, $model) = @_;
        return if $self->{model}
              && ($self->{model}->getName eq $model->getName);

        if ($self->{model})
        {
            my @children = $self->{mainBox}->get_children;
            $self->{mainBox}->remove($_) foreach (@children);
            foreach (@{$self->{toDestroy}})
            {
                $_->destroy if $_;
            }
        }

        $self->{model} = $model;
        $self->{fieldsInfo} = $self->{model}->{fieldsInfo};
        $self->{toDestroy} = [];
        foreach (@{$model->{fieldsNames}})
        {
            $self->{$_} = $self->createItem($self->{model}->{fieldsInfo}->{$_});
            push @{$self->{toDestroy}}, $self->{$_};
            $self->{$_.'Label'} = $self->createLabel($self->{model}->{fieldsInfo}->{$_}, $self->{withSeparator});
            push @{$self->{toDestroy}}, $self->{$_.'Label'};
        }
        $self->createSpecial;

        $self->{displayedValues} = {};
        $self->createLayout($self->{layout}, $self->{layoutBox});

        $self->show_all;
    }
    
    sub initShow
    {
        my ($self, $show, $hasToShow) = @_;

        $self->{show} = $show;
       
        if ($hasToShow)
        {
            $self->show_all;
            $self->{infoBox}->hide;
            return 1;
        }
        else
        {
            $self->hide(0);
            $self->{previouslyHidden} = 1;
            return 0;
        }
    }
    
    sub show
    {
        my $self = shift;
        $self->{mainBox}->show;
        $self->{infoBox}->hide;
        $self->setShowOption($self->{show}, 1) if $self->{previouslyHidden};
        $self->{previouslyHidden} = 0;
    }
    
    sub setShowOption
    {
        my ($self, $show, $hasToShow) = @_;
        return if ! $self->initShow($show, $hasToShow);
        
        my %parentsToHide;
        foreach (keys %{$show})
        {
            $self->{$_.'Label'}->hide if (! $show->{$_}) && ($self->{$_.'Label'});
            my $widget = $self->{$_};
            if ($widget)
            {
                if ($show->{$_})
                {
                    $parentsToHide{$widget->{realParent}} = 0;
                    foreach my $dependency(@{$self->{dependencies}->{$_}})
                    {
                        $parentsToHide{$dependency->{realParent}} = 0;
                    }
                }
                else
                {
                    $widget->hide;
                    $parentsToHide{$widget->{realParent}} = $widget->{realParent} if ! exists $parentsToHide{$widget->{realParent}};
                    foreach my $dependency(@{$self->{dependencies}->{$_}})
                    {
                        $dependency->hide;
                    }
                }
            }
        }

        foreach (keys %parentsToHide)
        {
            if ($parentsToHide{$_})
            {
                $parentsToHide{$_}->hide;
                $parentsToHide{$_}->{realParent}->hide
                    if ($parentsToHide{$_}->{realParent})
                    && ($parentsToHide{$_}->{realParent}->isa('Gtk2::Expander'));
            }
        }
        
        $self->setShowOptionForSpecials($show, $hasToShow);
    }

    sub setShowOptionForSpecials
    {
        my ($self, $show, $hasToShow) = @_;
    }

    sub hide
    {
        my ($self, $filtering) = @_;
        
        $self->{infoBox}->show_all;
        if ($filtering)
        {
            $self->{warning}->set_markup('<b>'.$self->{lang}->{AllItemsFiltered}.'</b>');
        }
        else
        {
            $self->{warning}->set_markup($self->{lang}->{Warning}) if !$filtering;
            $self->{viewAllBox}->hide;
        }
        $self->{mainBox}->hide;
    }
    
    sub showMe
    {
        my $self = shift;
        my $parent = $self->parent;
        while ($parent)
        {
           last if ($parent->isa('Gtk2::Window'));
           $parent = $parent->parent;
        }
        $parent->present if $parent;
    }
    
    sub isReadOnly
    {
        return 1;
    }
    
    sub getAsHash
    {
        my $self = shift;
        my $hash = {};
        $hash->{$_} = $self->$_ foreach(@{$self->{parent}->{model}->{fieldsNames}});
        return $hash;
    }
    
    sub getValue
    {
        my ($self, $field) = @_;
        
        return $self->$field;
    }
    
    sub disableBorrowerChange
    {
        my $self = shift;
        $self->{disableBorrowerChange} = 1;
    }
    
    sub disableAutoUpdate
    {
        my $self = shift;
        $self->{autoUpdate} = 0;
    }

    sub dataChanged
    {
        my $self = shift;
        
        foreach (@{$self->{expanders}})
        {
            $self->setExpanderLabel($_, 1);
        }
        foreach (@{$self->{formatted}})
        {
            $self->setFormattedLabel($_, 1);
        }
    }

    sub createLabel
    {
        my ($self, $info, $withSeparator) = @_;

        my $label = $self->{model}->getDisplayedText($info->{label});
        $label .= $self->{parent}->{lang}->{Separator} if $withSeparator;
        my $widget = new Gtk2::Label($label);
        $widget->set_alignment(0, 0.5);
        return $widget;
    }
    
    sub formatLabel
    {
        my ($self, $format) = @_;

        (my $label = $format) =~ s/%(.*?)%/$self->{$1}->getValue(1)/eg;
        $label =~ s/@(.*?)@/$self->{model}->getDisplayedText($1)/eg;
        return $label;
    }
    
    sub setExpanderLabel
    {
        my ($self, $expander, $init) = @_;

        if ($expander->{collapsedTemplate}
        && ($expander->get_expanded xor $init))
        {
            $expander->set_label($expander->{originalLabel}.$self->{parent}->{lang}->{Separator}.
                                 $self->formatLabel($expander->{collapsedTemplate}));
        }
        else
        {
            $expander->set_label($expander->{originalLabel});
        }
    }
    
    sub setFormattedLabel
    {
        my ($self, $label, $init) = @_;

        my $name = $label->{name};
        $self->$name($self->formatLabel($label->{format}));
    }

    sub addToContainer
    {
        my ($self, $container, $widget, $info) = @_;
        if ($container->isa('Gtk2::Notebook'))
        {
            $container->append_page($widget,
                                    $self->{model}->getDisplayedText($info->{title}));
        }
        elsif ($container->isa('Gtk2::Table'))
        {
            my $left = $info->{col};
            my $right = $left + ($info->{colspan} ? $info->{colspan} : 1);
            my $top = $info->{row};
            my $bottom = $top + ($info->{rowspan} ? $info->{rowspan} : 1);
            my $fill = [];
            push @$fill, 'fill' if ! ($info->{expand} eq 'false');
            push @$fill, 'expand' if $info->{expand} eq 'true';
            if ($info->{expand} eq 'shrink')
            {
                $fill = ['expand', 'fill'];
                my $hbox = new Gtk2::HBox(0,0);
                $hbox->pack_start($widget,0,0,0);
                $widget = $hbox;
            }
            $container->attach($widget,
                               $left, $right, $top, $bottom,
                               $fill, $fill,
                               0, 0);
        }
        else
        {
            my $expand = ($info->{expand} eq 'true') ? 1 : 0;
            if ($container->isa('Gtk2::VBox'))
            {
                my $margin = $self->{halfMargin};
                $margin = 0 if $info->{nomargin} eq 'true';
                if ($info->{place} eq 'end')
                {
                    $container->pack_end($widget, $expand, $expand, $margin);
                }
                else
                {
                    $container->pack_start($widget, $expand, $expand, $margin);
                }
            }
            else
            {
                if ($container->get_children)
                {
                    my $marginBox = new Gtk2::HBox(0,0);
                    $container->pack_start($marginBox, 0, 0, $self->{halfMargin});
                }
                $container->pack_start($widget, $expand, $expand, 0);
            }
        }
    }
    
    sub setWidgetStyle
    {
        my ($self, $widget, $name, $style) = @_;
    }
    
    sub getWidget
    {
        my ($self, $field, $info) = @_;
        return $self->{$field};
    }
    
    sub createLayout
    {
        my ($self, $xml, $container, $parent, $noTab) = @_;
        #
        foreach (@{$xml->{item}})
        {
            my $widget;
            my $box;
            my $isContainer = 0;

            if ($_->{type} eq 'formatted')
            {
                my $info = {};
                $info->{type} = 'short text';
                $info->{value} = $_->{name};
                $widget = $self->createItem($info);
                $widget->{format} = $_->{value};
                $widget->{name} = $_->{name};
                $self->{$_->{name}} = $widget;
                push @{$self->{formatted}}, $widget;
                while ($_->{value} =~ /%(.*?)%/g)
                {
                    push @{$self->{dependencies}->{$1}}, $widget;
                }
                $_->{for} = $widget->{name};
            }
            elsif ($_->{type} eq 'value')
            {
                $widget = $self->getWidget($_->{for}, $_);
                $widget->setWidth($_->{width}) if $_->{width};
                $widget->setHeight($_->{height}) if $_->{height};
                $widget->expand if $_->{expand} eq 'true';
                $self->{displayedValues}->{$_->{for}} = $_->{values} if $_->{values};
            }
            elsif ($_->{type} eq 'label')
            {
                if ($_->{for})
                { 
                    $widget = $self->{$_->{for}.'Label'};
                }
                else
                {
                    $widget = $self->createLabel($_, $self->{withSeparator});
                }
                if ($_->{align} eq 'center')
                {
                    $widget->set_alignment(0.5, 0.5);
                    if ($self->{withSeparator})
                    {
                        my $label = $widget->get_label;
                        my $sep = $self->{parent}->{lang}->{Separator};
                        $label =~ s/$sep$//;
                        $widget->set_label($label);
                    }
                }
                $widget->set_alignment(0, 0) if $_->{align} eq 'top';
            }
            elsif ($_->{type} eq 'special')
            {
                $widget = $self->{$_->{for}};
                $widget->{param} = $_->{param};
            }
            elsif ($_->{type} eq 'launcher')
            {
                $widget = Gtk2::Button->new_from_stock('gtk-media-play');
                my $valueWidget = $self->{$_->{for}};
                my $format = $self->{model}->{fieldsInfo}->{$_->{for}}->{format};
                $widget->signal_connect('clicked' => sub {
                    $self->{parent}->launch($valueWidget->getValue, $format);
                });
                push @{$self->{dependencies}->{$_->{for}}}, $widget;
            }
            elsif ($_->{type} eq 'line')
            {
                $widget = new Gtk2::HBox(0,0);
                #$widget->set_border_width($self->{halfMargin});
                $widget->set_size_request(-1, $_->{height}) if $_->{height};
                $box = $widget;
                $isContainer = 1;
            }
            elsif ($_->{type} eq 'box')
            {
                $widget = new Gtk2::VBox(0,0);
                #$widget->set_border_width($self->{halfMargin});
                $box = $widget;
                if ($_->{width})
                {
#                    $widget = new Gtk2::EventBox;
#                    $widget->add($box);
                    $widget->set_size_request($_->{width}, -1);
                }
                $isContainer = 1;
            }
            elsif ($_->{type} eq 'notebook')
            {
                if ($noTab)
                {
                    $widget = new Gtk2::VBox(0,0);
                }
                else
                {
                    $widget = new Gtk2::Notebook;
                }
                #$widget->set_border_width($self->{margin});
                $box = $widget;
                $isContainer = 1;
            }
            elsif ($_->{type} eq 'tab')
            {
                $widget = new Gtk2::VBox(0,0);
                if ($noTab)
                {
                    my $label = new Gtk2::Label;
                    $label->set_markup('<b>'.$self->{model}->getDisplayedText($_->{title}).'</b>');
                    $label->set_alignment(0,0);
                    $widget->pack_start($label, 0, 0, 0);
                    $widget->set_border_width(0);
                    $box = new Gtk2::VBox(0,0);
                    $box->set_border_width($self->{margin});
                    $widget->pack_start($box, 1, 1, 0);
                }
                else
                {
                    $widget->set_border_width($self->{margin});
                    $box = $widget;
                }
                $isContainer = 1;
            }
            elsif ($_->{type} eq 'table')
            {
                $widget = new Gtk2::Table($_->{rows},$_->{cols});
                #$widget->set_border_width($self->{margin});
                $widget->set_col_spacings($self->{colSpacing});
                $widget->set_row_spacings($self->{rowSpacing});
                $box = $widget;
                $isContainer = 1;
            }
            elsif ($_->{type} eq 'frame')
            {
                $widget = new GCGroup($self->{model}->getDisplayedText($_->{title}));
                $widget->set_border_width(0);
                $widget->setPadding(0);
                $box = new Gtk2::Table($_->{rows},$_->{cols});
                $box->set_border_width($self->{halfMargin});
                $box->set_col_spacings($self->{margin});
                $box->set_row_spacings($self->{halfMargin});
                $widget->addWidget($box);
                $isContainer = 1;
            }
            elsif ($_->{type} eq 'expander')
            {
                $widget = new Gtk2::Expander($self->{model}->getDisplayedText($_->{title}));
                $widget->{originalLabel} = $self->{model}->getDisplayedText($_->{title});
                $widget->{collapsedTemplate} = $_->{collapsed};
                $self->setExpanderLabel($widget, 1);
                $widget->signal_connect('activate' => sub {
                    $self->setExpanderLabel($widget, 0);
                });
                push @{$self->{expanders}}, $widget;
                $box = new Gtk2::VBox(0,0);
                $box->set_border_width($self->{halfMargin});
                $widget->add($box);
                $isContainer = 1;
            }
            
            if (exists $_->{border})
            {
                $widget->set_border_width($_->{border});
            }
            
            if ($isContainer)
            {
                $self->createLayout($_, $box, $widget, $noTab);
            }
            push @{$self->{toDestroy}}, $widget;
            $widget->{realParent} = $parent if $parent;
            $self->setWidgetStyle($widget, $_->{for}, $_->{style}) if $_->{style};
            $self->addToContainer($container, $widget, $_) if $widget;
        }
    }
}

{
    package GCReadOnlyPanel;
    use base 'GCItemPanel';

    use File::Basename;

    sub new
    {
        my ($proto, $parent, $options, $layout) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent, $options, $layout);

        bless ($self, $class);

        $self->{withSeparator} = 0;
        $self->{colSpacing} = $self->{quarterMargin};
        $self->{rowSpacing} = $self->{quarterMargin};
        $self->loadStyles($options->panelStyle);

        $self->{inBox} = new Gtk2::VBox(0, 0);
        $self->{inBox}->set_border_width($self->{margin});
       
        $self->set_name('GCItemFrame');

        $self->{inBox} = new Gtk2::VBox(0, 0);
        my $hboxMargins = new Gtk2::HBox(0,0);
        $hboxMargins->pack_start($self->{inBox}, 1, 1, 50);
        $self->{mainBox} = new Gtk2::EventBox;
        $self->{mainBox}->modify_bg('normal', $self->{styles}->{page}->{bgColor});
        $self->{mainBox}->add($hboxMargins);

        my $realMain = new Gtk2::VBox(0,0);

#        GCUtils::setWidgetPixmap($self->{mainBox},$ENV{GCS_SHARE_DIR}.'/logos/splash.png');

#        $self->{backgroundPixbuf} = Gtk2::Gdk::Pixbuf->new_from_file($ENV{GCS_SHARE_DIR}.'/logos/splash.png');
#        $self->{mainBox}->signal_connect('expose-event' => sub {
#            $self->paintBg;
#        });

        $realMain->pack_start($self->{mainBox},1,1,0);

        $self->{layoutBox} = $self->{inBox};

        $self->add($realMain);
        $self->show_all;


        $self->{warning} = new Gtk2::Label;
        
        $self->{infoBox}->pack_start($self->{warning},1,1,0);
        $self->{infoBox}->pack_start($self->{viewAllBox},1,1,0);
        $realMain->pack_start($self->{infoBox},1,1,0);

        return $self;
    }
    
    sub loadStyles
    {
        my ($self, $style) = @_;
        
        $self->{styles} = {};
        my $styleFile = $ENV{GCS_SHARE_DIR}.'/panels/'.$style;
        
        open STYLE, $styleFile;
        while (<STYLE>)
        {
            chomp;
            next if !$_;
            m/^(.*?)\s*=\s*(.*)$/;
            my $item = $1;
            (my $value = $2) =~ s/^"(.*?)"$/$1/;
            $item =~ /(.*?)(Bg|Color|Style|Justify)/;
            my $comp = $1;
            my $style = lc $2;
            $self->{styles}->{$comp}->{$style} = $value;
        }            
        close STYLE;
        foreach (keys %{$self->{styles}})
        {
            $self->{styles}->{$_}->{bgColor} = Gtk2::Gdk::Color->parse($self->{styles}->{$_}->{bg});
            $self->{styles}->{$_}->{style} = $self->{styles}->{$_}->{style}
                                           . " foreground='".$self->{styles}->{$_}->{color}."'";
        }
        #$self->{styles}->{invisible}->{}
    }
    
    sub deactivate
	{
        my $self = shift;
	}

    sub getValues
    {
        my ($self, $field) = @_;
        
        return $self->{$field}->getValues;
                
    }
    

    sub selectTitle
    {
        my $self = shift;
        $self->myRealize;
    }

#    sub paintBg
#    {
#        my $self = shift;
#        my $window = $self->{mainBox}->window;
#        $self->{mainBox}->set_app_paintable(1);
#        if (! $self->{gcMain})
#        {
#            $self->{gcMain} = Gtk2::Gdk::GC->new($window);
#        }
#        $window->draw_pixbuf($self->{gcMain}, $self->{backgroundPixbuf}, 0, 0, 0, 0, -1, -1, "none", 5, 5); 
#    }

    sub setBorrowers
    {
        my ($self, $values) = @_;
    }

    sub createSpecial
    {
        my $self = shift;
    }
    
    sub createLabel
    {
        my ($self, $info) = @_;
        
        my $label = $self->{model}->getDisplayedText($info->{label});
        my $widget = new GCColorLabel($self->{styles}->{label}->{bgColor});
        $widget->set_selectable(0);
        $widget->set_alignment(0,0.5);
        $widget->set_padding(5,5);
        $widget->set_justify('fill');
        $widget->set_markup("<span ".$self->{styles}->{label}->{style}.">".$label.'</span>');
        return $widget;
    }
    
    sub setWidgetStyle
    {
        my ($self, $widget, $name, $style) = @_;
        $self->{style}->{$name} = $style;
        if ($self->{styles}->{$style}->{bgColor})
        {
            $widget->modify_bg('normal', $self->{styles}->{$style}->{bgColor});
        }

        if ($widget->isa('GCColorLabel') || $widget->isa('GCColorText'))
        {
            my $justify = $self->{styles}->{$style}->{justify};
            $widget->set_justify($justify) if $justify;
        }
    }
    
    sub createItem
    {
        my ($self, $info) = @_;
        
        my $widget;
        
        if ($info->{type} eq 'long text')
        {
            $widget = new GCColorText($self->{styles}->{field}->{bgColor}, 0);
            $self->{style}->{$info->{value}} = 'field';
        }
        elsif ($info->{type} eq 'image')
        {
            $widget = new GCItemImage($self->{parent}->{options}, 
                                      $self->{model}->{defaultImage});
        }
        elsif ($info->{type} eq 'button')
        {
            $widget = new GCButton($self->{model}->getDisplayedText($info->{label}));
        }
        elsif($info->{type} =~ /list$/o)
        {
            my $number = GCUtils::listNameToNumber($info->{type});
            my @labels;
            for my $i (1..$number)
            {
                push @labels, $self->{model}->getDisplayedText($info->{'label'.$i});
            }
            $widget = new GCColorTable($number, \@labels,
                                       $self->{styles}->{label},
                                       $self->{styles}->{field},
                                       $self->{model}->getDisplayedText($info->{label}));            
        }
        else
        {
            # See GCColorLabel constructor for the meaning of listType;
            if ($info->{value} eq $self->{model}->{commonFields}->{borrower}->{name})
            {
                $self->{prepare}->{$info->{value}}->{func} = 'prepareBorrower';
            }
            $widget = new GCColorLabel($self->{styles}->{field}->{bgColor});
            $widget->set_alignment(0,0.5);
            $self->{style}->{$info->{value}} = 'field';
            $widget->set_selectable(0);
            $widget->set_padding(5,5);
            $widget->set_justify('fill');
        }
        $widget->{name} = $info->{value};        
        return $widget;
    }

    sub getWidget
    {
        my ($self, $field, $info) = @_;

        my $widget = $self->{$field};
        if ($widget->isa('GCColorLabel') && $info->{height})
        {
            my $style = $info->{style};
            $style = 'field' if !$style;
            $widget->destroy;
            $widget = new GCColorText($self->{styles}->{$style}->{bgColor}, 0);
            $widget->{name} = $self->{$field}->{name};
            $self->{$field} = $widget;
        }
        if ($widget->isa('GCColorTable') && ($info->{flat} eq 'true'))
        {
            my $columns = $widget->{number};
            $widget->destroy;
            $self->{prepare}->{$field}->{func} = 'prepareMultipleList';
            $self->{prepare}->{$field}->{extra} = $columns;
            $widget = new GCColorLabel($self->{styles}->{field}->{bgColor}, $columns);
            $widget->set_alignment(0,0.5);
            $self->{style}->{$field} = 'field';
            $widget->set_selectable(0);
            $widget->set_padding(5,5);
            $widget->set_justify('fill');
            $self->{$field} = $widget;
        }

        return $widget;
    }

    sub prepareMultipleList
    {
        my ($self, $value, $extra) = @_;
        return GCPreProcess::multipleList($value, $extra);
    }
    
    sub prepareBorrower
    {
        my ($self, $value) = @_;

        $value = $self->{lang}->{PanelNobody} if $value eq 'none';
        $value = $self->{lang}->{PanelUnknown} if $value eq 'unknown';
        return $value;
    }

    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        if (scalar @_)
        {
            my $text = shift;
            $self->{$name.'Value'} = $text;

            if (my $func = $self->{prepare}->{$name}->{func})
            {
                $text = $self->$func($text, $self->{prepare}->{$name}->{extra});
            }
            $text = ' ' if ! defined $text;
            if ($self->{displayedValues}->{$name})
            {
                $text = $self->{model}->getDisplayedValue($self->{displayedValues}->{$name}, $text);
            }
            
            if ($self->{$name})
            {
                if ($self->{$name}->acceptMarkup)
                {
                    $self->{$name}->setMarkup("<span ".
                                              $self->{styles}->{$self->{style}->{$name}}->{style}.
                                              ">".$text.'</span>');
                }
                else
                {
                    $self->{$name}->setValue($text);
                }
            }
        }
        else
        {
            return $self->{$name.'Value'};
        }
    }
}


{
    package GCFormPanel;

    use base 'GCItemPanel';

    sub createSpecial
    {
        my $self = shift;
    
        $self->{mailButton} = new GCButton($self->{parent}->{lang}->{MailTitle});
        $self->{mailButton}->signal_connect('clicked' => sub {
            $self->sendBorrowerEmail;
        });
        push @{$self->{toDestroy}}, $self->{mailButton};

        $self->{itemBackButton} = new GCButton($self->{parent}->{lang}->{PanelReturned});
        $self->{itemBackButton}->signal_connect('clicked' => sub {
            $self->itemBack;
        });
        push @{$self->{toDestroy}}, $self->{mailButton};

        $self->{searchButton} = GCButton->newFromStock('gtk-jump-to');
        $self->{searchButton}->signal_connect('clicked' => sub {
            $self->searchItem;
        });
        $self->{tooltips}->set_tip($self->{searchButton}, $self->{parent}->{lang}->{PanelSearchTip}, '');
        push @{$self->{toDestroy}}, $self->{searchButton};
        
        $self->{deleteButton} = GCButton->newFromStock('gtk-delete');
        $self->{deleteButton}->signal_connect("clicked" => \&deleteItem, $self);
        $self->{tooltips}->set_tip($self->{deleteButton}, $self->{parent}->{lang}->{PanelRemoveTip}, '');
        push @{$self->{toDestroy}}, $self->{deleteButton};
    }

    sub new
    {
        my ($proto, $parent, $options, $layout) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent, $options, $layout);

        bless ($self, $class);

        $self->{withSeparator} = 1;
        $self->{colSpacing} = $self->{margin};
        $self->{rowSpacing} = $self->{halfMargin};

        $self->{mainBox} = new Gtk2::VBox(0, 0);
        $self->{mainBox}->set_border_width($self->{margin});

        #$self->deactivate if ($readonly ||  $parent->{options}->lockPanel);
        $self->{trailerExtract} = Gtk2::Button->new($parent->{lang}->{ExtractButton});
        $self->{trailerExtract}->signal_connect('clicked' => sub {
            $self->{parent}->extractInfo($self->{trailer}->get_text, $self);
        });
       
        $self->set_name('GCItemFrame');

        my $realMain = new Gtk2::VBox(0,0);

        $realMain->pack_start($self->{mainBox},1,1,0);
        $self->{layoutBox} = $self->{mainBox};

        $self->add($realMain);
        $self->show_all;

        $self->{warning} = new Gtk2::Label;
        
        $self->{infoBox}->pack_start($self->{warning},1,1,0);
        $self->{infoBox}->pack_start($self->{viewAllBox},1,1,0);
        $realMain->pack_start($self->{infoBox},1,1,0);
        
        return $self;
    }
    
    sub isReadOnly
    {
        return 0;
    }

    sub searchItem
    {
        my $self = shift;

        #$self->{parent}->searchItemForPanel($self->{$self->{model}->{commonFields}->{title}}->getValue, $self);
        $self->{parent}->searchItemForPanel($self);

        $self->showMe;
    }

    sub checkBorrowerButtons
    {
        my $self = shift;
        my $borrowerField = $self->{model}->{commonFields}->{borrower}->{name};
        return if ! $self->{$borrowerField};
        my $locked = (($self->{$borrowerField}->getValue eq 'none')
                   || ($self->{$borrowerField}->getValue eq 'unknown'));
        $self->{mailButton}->lock($locked)
            if $self->{mailButton};
        $locked = ($self->{locked} || ($self->{$borrowerField}->getValue eq 'none'));
        $self->{itemBackButton}->lock($locked)
            if $self->{itemBackButton};
    }
    
    sub itemBack
    {
        my $self = shift;

        my $dialog = new GCDateSelectionDialog($self->{parent});
        if ($dialog->show)
        {
            my $borrowerField = $self->{model}->{commonFields}->{borrower}->{name};
            my $lendDateField = $self->{model}->{commonFields}->{borrower}->{date};
            my $borrowingsField = $self->{model}->{commonFields}->{borrower}->{history};
            my @data = ($self->{$borrowerField}->getDisplayedValue,
                        $self->{$lendDateField}->getValue,
                        $dialog->date);
            $self->{$borrowingsField}->addValues(@data);
            my $previous = $self->getAsHash;
            $self->{$borrowerField}->setValue('none');
            $self->{$lendDateField}->clear;
            my $new = $self->getAsHash;           
            $self->{parent}->{itemsList}->changeCurrent($previous, $new) if $self->image;
        }
        $self->showMe;
    }
    
    sub lock
    {
        my ($self, $value) = @_;
        $self->changeState($self, $value);
    }
    
    sub deactivate
	{
        my $self = shift;
		$self->changeState($self,1);
	}

	sub changeState
    {
        my ($caller, $self, $locked)  = @_;
        $locked = 1 if $self->{parent}->{items}->getLock;
        return if $self->{locked} == $locked;
        $self->{locked} = $locked;

        foreach (@{$self->{parent}->{model}->{fieldsNames}})
        {
            $self->{$_}->lock($locked) if ($self->{$_});
        }

        $self->{searchButton}->lock($locked);
        $self->checkBorrowerButtons;
        $self->{deleteButton}->lock($locked);
    }
    
    sub setShowOptionForSpecials
    {
        my ($self, $show, $hasToShow) = @_;

        if (! $show->{$self->{model}->{commonFields}->{borrower}->{name}})
        {
            $self->{mailButton}->hide;
            $self->{itemBackButton}->hide;
        }
   }

    sub sendBorrowerEmail
    {
        use GCDialogs;

        my $self = shift;

        my %info;
        $info{title} = $self->{$self->{model}->{commonFields}->{title}}->getValue;
        $info{borrower} = $self->{$self->{model}->{commonFields}->{borrower}->{name}}->getValue;
        $info{lendDate} = $self->{$self->{model}->{commonFields}->{borrower}->{date}}->getValue;
        $self->{parent}->sendBorrowerEmail(\%info);
#        my $dialog = new GCMailDialog($self->{parent}, \%info);
#
#        $dialog->show;
#        $self->showMe;
    }

    sub selectTitle
    {
        my $self = shift;
        my $titleField = $self->{model}->{commonFields}->{title};
        #$self->{$titleField}->select_region(0, length($self->{$titleField}->getValue));
        $self->{$titleField}->selectAll;
    }

    sub deleteItem
    {
       my ($widget, $self) = @_;

       $self->{parent}->deleteCurrentItem;
    }

    sub setBorrowers
    {
        my $self = shift;

        return if ! $self->{borrowerControl};
  
        my $borrowers = $self->{model}->getValues($self->{model}->{commonFields}->{borrower}->{name});
        shift @{$borrowers} if !$borrowers->[0]->{displayed};
        $self->{borrowerControl}->setValues($borrowers);

        return;

    }
    
    sub createItem
    {
        my ($self, $info) = @_;
        
        my $widget;
        
        if ($info->{type} eq 'short text')
        {
            $widget = new GCShortText;
            $widget->signal_connect('activate', sub {
                $self->searchItem;
            }) if $self->{model}->isSearchField($info->{value});
        }
        elsif ($info->{type} eq 'long text')
        {
            $widget = new GCLongText;
        }
        elsif ($info->{type} eq 'history text')
        {
            $widget = new GCHistoryText;
        }
        elsif ($info->{type} eq 'options')
        {
            $info->{values} = $info->{value} if !$info->{values};
            if ($info->{value} eq $self->{model}->{commonFields}->{borrower}->{name})
            {
                $widget = new GCMenuList;
                #When borrower not found, we use the last choice: unknown
                $widget->setLastForDefault;
                $self->{borrowerControl} = $widget;
                $widget->signal_connect('changed' => sub {
                    $self->checkBorrowerButtons;
                });
                $self->setBorrowers;
            }
            else
            {
                $widget = new GCMenuList($self->{model}->getValues($info->{values}), $info->{separator});
            }
        }
        elsif ($info->{type} =~ /list$/o)
        {
            my $number;
            my $readonly = 0;
            my $withHistory = 1;
            my @labels;
            $withHistory = 0
                if $info->{history} eq 'false';
            $readonly = 1
                if $info->{value} eq $self->{model}->{commonFields}->{borrower}->{history};
            my $number = GCUtils::listNameToNumber($info->{type});
            if ($number == 1)
            {
                @labels = ($self->{model}->getDisplayedText($info->{label}));
            }
            else
            {
                for my $i (1..$number)
                {
                    push @labels, $self->{model}->getDisplayedText($info->{'label'.$i});
                }
            }
            $widget = new GCMultipleList($self, $number, \@labels, $withHistory, $readonly);
        }
		elsif ($info->{type} eq 'checked text')
		{
			$widget = new GCCheckedText($info->{format});
			$widget->signal_connect('activate', sub {
				$self->searchItem;
			}) if $self->{model}->isSearchField($info->{value});
		}
        elsif ($info->{type} eq 'number')
        {
            if ((exists $info->{min}) && ($info->{min} ne ''))
            {
                $widget = new GCNumeric($info->{init}, 
                                        $info->{min},
                                        $info->{max},
                                        $info->{step});
            }
            else
            {
                $widget = new GCCheckedText('0-9.');
            }
            $widget->signal_connect('activate', sub {
                $self->searchItem;
            }) if $self->{model}->isSearchField($info->{value});
        }
        elsif ($info->{type} eq 'image')
        {
            my $img = new GCItemImage($self->{parent}->{options}, 
                                      $self->{model}->{defaultImage});
            my $isCover = 0;
            $isCover = 1 if $info->{value} eq $self->{model}->{commonFields}->{cover};
            $widget = new GCImageButton($self, $img, $isCover, $info->{default});
        }
        elsif ($info->{type} eq 'button')
        {
            $widget = new GCButton($self->{model}->getDisplayedText($info->{label}));
        }
        elsif ($info->{type} eq 'url')
        {
            $widget = new GCUrlButton($self->{model}->getDisplayedText($info->{label}),
                                      $self->{parent});
        }
        elsif ($info->{type} eq 'yesno')
        {
            $widget = new GCCheckBox($self->{model}->getDisplayedText($info->{label}));
        }
        elsif ($info->{type} eq 'file')
        {
            $widget = new GCFile($self);
        }
        elsif ($info->{type} eq 'date')
        {
            $widget = new GCDate($self, $self->{lang});
        }
        else
        {
            print "Invalid type : ",$info->{type}," for ",$info->{value},"\n";
            $widget = 0;
        }
        
        return $widget;
    }
    
    sub getValues
    {
        my ($self, $field) = @_;
        return $self->{$field}->getValues;
    }
    
    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        return if !$name;
        if (scalar @_)
        {
            my $text = shift;
            $self->{$name}->setValue($text);
        }
        else
        {
            return $self->{$name}->getValue;
        }
    }
}

1;
