package GCDisplay;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;
use GCUtils;

my $firstYear = 1895;

{
    package GCFilterSearch;

    sub compareExact
    {
        my ($field, $value) = @_;
        return $field eq $value;
    }
    
    sub compareContain
    {
        my ($field, $value) = @_;
        return $field =~ m/$value/i;
    }

    sub compareLessStrings
    {
        my ($field, $value) = @_;
        return $field lt $value;
    }

    sub compareLessNumbers
    {
        my ($field, $value) = @_;
        return 0 if !defined($field);
        return $field < $value;
    }

    sub compareLessOrEqualStrings
    {
        my ($field, $value) = @_;
        return $field le $value;
    }

    sub compareLessOrEqualNumbers
    {
        my ($field, $value) = @_;
        return 0 if !defined($field);
        return $field <= $value;
    }
    
    sub compareGreaterStrings
    {
        my ($field, $value) = @_;
        return $field gt $value;
    }

    sub compareGreaterNumbers
    {
        my ($field, $value) = @_;
        return 0 if !defined($field);
        return $field > $value;
    }

    sub compareGreaterOrEqualStrings
    {
        my ($field, $value) = @_;
        return $field ge $value;
    }

    sub compareGreaterOrEqualNumbers
    {
        my ($field, $value) = @_;
        return 0 if !defined($field);
        return $field >= $value;
    }

    sub compareRangeStrings
    {
        my ($field, $value) = @_;
        return 1 if $value eq ';';
        my @values = split m/;/, $value;
        return ($field ge $values[0]) && ($field le $values[1]);
    }

    sub compareRangeNumbers
    {
        my ($field, $value) = @_;
        return 1 if $value eq ';';
        return 0 if !defined($field);
        my @values = split m/;/, $value;
        return ($field >= $values[0]) && ($field <= $values[1]);
    }

    sub new
    {
        my ($proto, $info) = @_;
        my $class = ref($proto) || $proto;
        my $self  = {};

        bless ($self, $class);

        $self->{info} = $info if $info;
        $self->clear if !$info;
        $self->setMode;
        return $self;
    }

    sub clear
    {
        my $self = shift;
        
        $self->{cleared} = 1;

        $self->{info} = {};
    }
    
    sub getComparisonFunction
    {
        my ($self, $type) = @_;
        
        my ($comparison, $numeric) = @$type;
        if ($comparison eq 'eq')
        {
            return \&compareExact;
        }
        elsif ($comparison eq 'contain')
        {
            return \&compareContain;
        }
        elsif ($comparison eq 'lt')
        {
            return \&compareLessStrings if ($numeric ne 'true');
            return \&compareLessNumbers if ($numeric eq 'true');
        }
        elsif ($comparison eq 'le')
        {
            return \&compareLessOrEqualStrings if ($numeric ne 'true');
            return \&compareLessOrEqualNumbers if ($numeric eq 'true');
        }
        elsif ($comparison eq 'gt')
        {
            return \&compareGreaterStrings if ($numeric ne 'true');
            return \&compareGreaterNumbers if ($numeric eq 'true');
        }
        elsif ($comparison eq 'ge')
        {
            return \&compareGreaterOrEqualStrings if ($numeric ne 'true');
            return \&compareGreaterOrEqualNumbers if ($numeric eq 'true');
        }
        elsif ($comparison eq 'range')
        {
            return \&compareRangeStrings if ($numeric ne 'true');
            return \&compareRangeNumbers if ($numeric eq 'true');
        }
    }
    
    sub setFilter
    {
        my ($self, $filter, $value, $type, $model, $add) = @_;
        if (!$filter)
        {
            $self->clear;
            return;
        }
        if ($value eq '')
        {
            delete $self->{info}->{$filter};
        }
        else
        {
            $self->{cleared} = 0;
            my $preprocess = $type->[2];
            if (!$preprocess)
            {
                my $fieldType = $model->{fieldsInfo}->{$filter}->{type};
                $preprocess = ($fieldType eq 'date') ? 'reverseDate'
                            : ($fieldType eq 'single list') ? 'singleList'
                            : ($fieldType eq 'double list') ? 'doubleList'
                            : ($fieldType =~ /list$/o) ? 'singleList'
                            : '';
            }
            my $info = {
                            value => $value,
                            comp => $self->getComparisonFunction($type),
                            preprocess => $preprocess,
                            temporary => $add
                         };
            if ($add)
            {
                push @{$self->{info}->{$filter}}, $info;
            }
            else
            {
                $self->{info}->{$filter} = [$info];
            }
        }
    }
    
    sub removeTemporaryFilters
    {
        my $self = shift;
        
        foreach (keys %{$self->{info}})
        {
            foreach my $i(0 .. scalar @{$self->{info}->{$_}} - 1)
            {
                delete $self->{info}->{$_}->[$i] if $self->{info}->{$_}->[$i]->{temporary};
            }
        }
    }

    sub setModel
    {
        my ($self, $model) = @_;
        
        $self->{model} = $model;
    }

    sub testAnd
    {
        my ($self, $info) = @_;
        return 1 if $self->{cleared};
        foreach my $field(keys %{$self->{info}})
        {
            my $value = $info->{$field};
            foreach my $info(@{$self->{info}->{$field}})
            {
                if ($info->{preprocess})
                {
                    my $preProcess = 'GCPreProcess::'.$info->{preprocess};
                    eval {
                        no strict qw/refs/;
                        $value = $preProcess->($value);
                    };
                }
                return 0 if $info
                         && ! $info->{comp}->($value, $info->{value});
            }
        }
        return 1;
    }

    sub testOr
    {
        my ($self, $info) = @_;
        return 1 if $self->{cleared};
        foreach my $field(keys %{$self->{info}})
        {
            my $value = $info->{$field};
            foreach my $info(@{$self->{info}->{$field}})
            {
                if ($info->{preprocess})
                {
                    my $preProcess = 'GCPreProcess::'.$info->{preprocess};
                    eval {
                        no strict qw/refs/;
                        $value = $preProcess->($value);
                    };
                }
                return 1 if $info
                         && $info->{comp}->($value, $info->{value});
            }
        }
        return 0;
    }

    sub setMode
    {
        my ($self, $mode) = @_;
        $mode ||= 'and';
        *test = \&testAnd if $mode eq 'and';
        *test = \&testOr if $mode eq 'or';
    }
}

use Gtk2;

{
    package GCSearchDialog;
    
    use GCGraphicComponents;
    
    use base 'GCModalDialog';
    
    sub initValues
    {
        my $self = shift;
        
        my $info = $self->{parent}->{items}->{filterSearch}->{info};
        
        foreach (@{$self->{fields}})
        {
            if (exists $info->{$_})
            {
                $self->{$_}->setValue($info->{$_}->[0]->{value});
            }
            else
            {
                $self->{$_}->clear if $self->{$_};
            }
            if ($self->{fieldsInfo}->{$_}->{type} eq 'history text')
            {
                $self->{$_}->setValues($self->{parent}->{panel}->{$_}->getValues);
            }
            if (
                 (
                   ($self->{fieldsInfo}->{$_}->{type} eq 'single list')
                   ||
                   ($self->{fieldsInfo}->{$_}->{type} eq 'double list')
                 )
                 &&
                 (
                   $self->{parent}->{panel}->{$_}->{withHistory}
                 )
               )
            {
                my @values;
                foreach ($self->{parent}->{panel}->{$_}->getValues)
                {
                    push @values, $_->[0];
                }
                $self->{$_}->setValues(@values);
            }
        }
    }
    
    sub show
    {
        my $self = shift;

        $self->initValues;
        
        $self->SUPER::show();
        $self->show_all;
        $self->{search} = undef;
        my $ended = 0;
        while (!$ended)
        {
            my $response = $self->run;
            if ($response eq 'ok')
            {
                my %info;
            
                foreach (@{$self->{fields}})
                {
                    $info{$_} = $self->{$_}->getValue
                        if ! $self->{$_}->isEmpty;
                }

                $self->{parent}->{menubar}->initFilters(\%info);

                $self->{search} = \%info;
            }
            $ended = 1 if ($response eq 'ok') || ($response eq 'cancel') || ($response eq 'delete-event');
            $self->clear if ($response eq 'reject');
        }
        $self->hide;
    }
    
    sub clear
    {
        my $self = shift;

        foreach (@{$self->{fields}})
        {
            $self->{$_}->clear;
        }
    }
    
    sub search
    {
        my $self = shift;
        
        return $self->{search};
    }
    
    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent,
                                       $parent->{lang}->{SearchTitle},
                                       'gtk-find',
                                       0,
                                       'gtk-clear' => 'reject',
                                      );
        bless ($self, $class);

        $self->{parent} = $parent;
        $self->{layoutTable} = new Gtk2::Table(1,4,0);
        $self->{layoutTable}->set_row_spacings($GCUtils::halfMargin);
        $self->{layoutTable}->set_col_spacings($GCUtils::margin);
        $self->{layoutTable}->set_border_width($GCUtils::margin);
        
        $self->vbox->pack_start($self->{layoutTable}, 1, 1, 0);
        return $self;
    }
    
    sub setModel
    {
        my ($self, $model) = @_;

        foreach ($self->{layoutTable}->get_children)
        {
            $self->{layoutTable}->remove($_);
            $_->destroy;
        }
        #$self->{layoutTable}->set_size_request(-1,-1);

        my $fieldsInfo = $model->{fieldsInfo};
        $self->{fieldsInfo} = $fieldsInfo;
        my @filtersGroup = @{$model->{filtersGroup}};
        my @filtersTotal = @{$model->{filters}};

        my $row = 0;
        return if ($#filtersTotal + (2 * $#filtersGroup)) <= 0;
        $self->{layoutTable}->resize($#filtersTotal + (2 * $#filtersGroup),4);

        $self->{fields} = [];
        foreach my $group(@filtersGroup)
        {
            $row++;
            my @filters = @{$group->{filter}};
            my $label = new GCHeaderLabel($model->getDisplayedText($group->{label}));
            $self->{layoutTable}->attach($label, 0, 4, $row, $row + 1, 'fill', 'expand', 0, 0);
            $row++;
            foreach my $filter(@filters)
            {
                my $field = $filter->{field};
                if ($field ne 'separator')
                {
                    push @{$self->{fields}}, $field;
                    my $labelText = $fieldsInfo->{$field}->{displayed};
                    $labelText = $model->getDisplayedText($filter->{label}) if $filter->{label};            
                    my $label = new Gtk2::Label($labelText);
                    $label->set_alignment(0,0.5);
                    $self->{layoutTable}->attach($label, 2, 3, $row, $row + 1, 'fill', 'fill', 0, 0);
                    if ($fieldsInfo->{$field}->{type} eq 'short text')
                    {
                        if ($filter->{comparison} eq 'range')
                        {
                            $self->{$field} = new GCRange('text', $self->{parent}->{lang});
                            $self->{$field}->setWidth(16);
                            
                        }
                        else
                        {
                            $self->{$field} = new GCShortText;
                        }
                        $self->{$field}->signal_connect('activate' => sub {$self->response('ok')} );
                    }
                    elsif ($fieldsInfo->{$field}->{type} eq 'number')
                    {
                        #If we want to have values that are less to the specified one,
                        #we use max as default to be sure everything will be returned
                        #in that case.
                        my $default = $fieldsInfo->{$field}->{min};
                        $default = $fieldsInfo->{$field}->{max}
                            if $filter->{comparison} =~ /^l/;
                        if (exists $fieldsInfo->{$field}->{min})
                        {
                            if ($filter->{comparison} eq 'range')
                            {
                                $self->{$field} = new GCRange('number',
                                                              $self->{parent}->{lang},
                                                              $fieldsInfo->{$field}->{min},
                                                              $fieldsInfo->{$field}->{max},
                                                              $fieldsInfo->{$field}->{step});
                                $self->{$field}->setWidth(16);
                            }
                            else
                            {
                                $self->{$field} = new GCNumeric($default,
                                                                $fieldsInfo->{$field}->{min},
                                                                $fieldsInfo->{$field}->{max},
                                                                $fieldsInfo->{$field}->{step});
                            }
                        }
                        else
                        {
                            if ($filter->{comparison} eq 'range')
                            {
                                $self->{$field} = new GCRange('numeric text', $self->{parent}->{lang});
                                $self->{$field}->setWidth(16);
                            }
                            else
                            {
                                $self->{$field} = new GCCheckedText('0-9.');
                            }
                        }
                    }
                    elsif (($fieldsInfo->{$field}->{type} eq 'history text')
                        || (($fieldsInfo->{$field}->{type} =~ /list/)
                         && ($fieldsInfo->{$field}->{history} ne 'false')))
                    {
                        $self->{$field} = new GCHistoryText;
                    }
                    elsif ($fieldsInfo->{$field}->{type} eq 'options')
                    {
                        $self->{$field} = new GCMenuList;
                        $self->{$field}->setValues($model->getValues($fieldsInfo->{$field}->{'values'}), $fieldsInfo->{$field}->{separator});
                    }
                    elsif ($fieldsInfo->{$field}->{type} eq 'yesno')
                    {
                        $self->{$field} = new GCCheckBoxWithIgnore($self->{parent});
                    }
                    elsif ($fieldsInfo->{$field}->{type} eq 'date')
                    {
                        $self->{$field} = new GCDate($self, $self->{parent}->{lang}, 1);
                    }
                    else
                    {
                        $self->{$field} = new GCShortText;
                    }
                    $self->{layoutTable}->attach($self->{$field}, 3, 4, $row, $row + 1, 'fill', 'expand', 0, 0);
                }
                else
                {
                    $self->{layoutTable}->attach(Gtk2::HSeparator->new, 0, 2, $row, $row + 1, 'fill', 'fill', 0, 0);
                }
                $row++;
            }
        }
        #$self->resize(1,1);
    }
}

{
    package GCAdvancedSearchDialog;
    
    use GCGraphicComponents;
    
    use base 'GCSearchDialog';
    
    sub addItem
    {
        my $self = shift;
        $self->{layoutTable}->resize($self->{nbFields} + 1, 3);
        my $field = new GCFieldSelector(0, $self->{model});
        push @{$self->{fields}}, $field;
        $self->{layoutTable}->attach($field, 0, 1, $self->{nbFields}, $self->{nbFields}+1,
                                     ['expand', 'fill'], 'fill', 0, 0);
        $field->show_all;

        my $comp = new GCComparisonSelector($self->{parent});
        push @{$self->{comps}}, $comp;
        $self->{layoutTable}->attach($comp, 1, 2, $self->{nbFields}, $self->{nbFields}+1,
                                     ['expand', 'fill'], 'fill', 0, 0);
        $comp->show_all;

        my $value = new GCShortText;
        push @{$self->{values}}, $value;
        $self->{layoutTable}->attach($value, 2, 3, $self->{nbFields}, $self->{nbFields}+1,
                                     ['expand', 'fill'], 'fill', 0, 0);
        $value->show_all;
        $self->{remove}->set_sensitive(1);
        $self->{nbFields}++;
    }
    
    sub removeItem
    {
        my $self = shift;
        $self->{layoutTable}->remove(pop @{$self->{fields}});
        $self->{layoutTable}->remove(pop @{$self->{comps}});
        $self->{layoutTable}->remove(pop @{$self->{values}});
        $self->{layoutTable}->resize(--$self->{nbFields}, 3);
        $self->{remove}->set_sensitive(0) if $self->{nbFields} < 2;
    }

    sub show
    {
        my $self = shift;
        
        $self->show_all;
        $self->{search} = undef;
        my $ended = 0;
        while (!$ended)
        {
            my $response = $self->run;
            if ($response eq 'ok')
            {
                my @info;
                my $i = 0;
                foreach (@{$self->{fields}})
                {
                    my $field = $_->getValue;
                    push @info, {
                        field => $field,
                        value => $self->{values}->[$i]->getValue,
                        filter => [$self->{comps}->[$i]->getValue,
                                   $self->{model}->{fieldsInfo}->{$field}->{type} eq 'number',
                                   undef]
                    }
                        if ! $self->{values}->[$i]->isEmpty;
                    $i++;
                }
                $self->{search} = \@info;
            }
            $ended = 1 if ($response eq 'ok') || ($response eq 'cancel') || ($response eq 'delete-event');
            $self->clear if ($response eq 'reject');
        }
        $self->hide;
    }

    sub getMode
    {
        my $self = shift;
        
        return ($self->{testAnd}->get_active ? 'and' : 'or');
    }

    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent);
        bless ($self, $class);

        $self->vbox->remove($self->{layoutTable});

        my $allTable = new Gtk2::Table(7,3,0);
        $allTable->set_row_spacings($GCUtils::halfMargin);
        $allTable->set_col_spacings($GCUtils::margin);
        $allTable->set_border_width($GCUtils::margin);
        
        my $labelType = new GCHeaderLabel($parent->{lang}->{AdvancedSearchType});
        $self->{testAnd} = new Gtk2::RadioButton(undef, $parent->{lang}->{AdvancedSearchTypeAnd});
        $self->{testOr} = new Gtk2::RadioButton($self->{testAnd}->get_group, $parent->{lang}->{AdvancedSearchTypeOr});

        $allTable->attach($labelType, 0, 3, 0, 1, 'fill', 'fill', 0, 0);
        $allTable->attach($self->{testAnd}, 2, 3, 1, 2, 'fill', 'fill', 0, 0);
        $allTable->attach($self->{testOr}, 2, 3, 2, 3, 'fill', 'fill', 0, 0);

        my $labelCriteria = new GCHeaderLabel($parent->{lang}->{AdvancedSearchCriteria});
        my $scrolled = new Gtk2::ScrolledWindow;
        $scrolled->set_policy ('never', 'automatic');
        $scrolled->set_border_width(0);
        $scrolled->set_shadow_type('none');
        $scrolled->add_with_viewport($self->{layoutTable});

        $allTable->attach($labelCriteria, 0, 3, 4, 5, 'fill', 'fill', 0, 0);
        $allTable->attach($scrolled, 2, 3, 5, 6, ['expand', 'fill'], ['expand', 'fill'], 0, 0);

        my $hboxAction = new Gtk2::HBox(0,0);
        $self->{add} = Gtk2::Button->new_from_stock('gtk-add');
        $self->{add}->signal_connect('clicked' => sub {
            $self->addItem;
        });
        $hboxAction->pack_start($self->{add}, 0, 0, 0);
        $self->{remove} = Gtk2::Button->new_from_stock('gtk-remove');
        $self->{remove}->signal_connect('clicked' => sub {
            $self->removeItem;
        });
        $hboxAction->pack_start($self->{remove}, 0, 0, $GCUtils::margin);
        $allTable->attach($hboxAction, 2, 3, 6, 7, 'fill', 'fill', 0, 0);

        $self->vbox->pack_start($allTable,1,1,0);
        #$self->vbox->pack_start(Gtk2::HBox->new(0,0), 0, 0, $GCUtils::halfMargin);
        
        $self->set_size_request(-1, 400);
        return $self;
    }
    
    sub clear
    {
        my $self = shift;
        $self->setModel($self->{model});
    }

    sub setModel
    {
        my ($self, $model) = @_;

        $self->{model} = $model;
        $self->{nbFields} = 0;
        $self->{fields} = [];
        $self->{comps} = [];
        $self->{values} = [];
        foreach ($self->{layoutTable}->get_children)
        {
            $self->{layoutTable}->remove($_);
            $_->destroy;
        }
        $self->addItem;
        $self->{remove}->set_sensitive(0)
    }
}

1;
