/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <string.h>
#include <signal.h>
#include <stdlib.h>

#include "vector_commands.h"
#include "contractions.h"
#include "linebuffer.h"
#include "common_gtk.h"
#include "modify_file_set.h"
#include "command.h"
#include "globals.h"


/* I don't know if the setpgid(getpid(), getpid()) calls are necessary any
 * more, they were needed for the kill/waitpid calls at one stage, but I've
 * changed a lot of code since...
 */

pid_t iso = 0, cdr = 0;

/* a readonly popen variant that redirects stderr instead of stdout
 */
FILE *popen_r_stderr(cmd_v *command) {
	int *pfd;

	if (NULL == (pfd = popen_re_unbuffered(command)))
			return NULL;

	return fdopen(pfd[1], "r");
}

/* a readonly popen variant that accepts cmd_v *command
 */
FILE *popen_r_stdout(cmd_v *command) {
	int *pfd;

	if (NULL == (pfd = popen_re_unbuffered(command)))
		return NULL;

	return fdopen(pfd[0], "r");
}

/* assumes a successfull pipe() won't set the fd's to -1 */
static void close_pipe(int *pipe) {
	if (-1 != pipe[0]) {
		close(pipe[0]);
		pipe[0] = -1;
	}
	if (-1 != pipe[1]) {
		close(pipe[1]);
		pipe[1] = -1;
	}
}

/* executes the command stored in command->elemets (which is suitable for execv())
 * returns an int *pfd with file descriptors:
 * pfd[0] STDOUT output of the command and
 * pfd[1] STDERR output of the command
 */

int *popen_re_unbuffered(cmd_v *command) {
	static int p_r[2] = {-1, -1}, p_e[2] = {-1, -1};
	static int *pfd = NULL;

	if (check_if_exe_ok(command->elements[0]))
		return NULL;

	iso = cdr = 0;

	if (command->elements[command->elements_used - 1] != NULL)
		add_option_to_cmd(command, NULL); /* the [] should end with a NULL pointer */

	/* only allocate once */
	if (NULL == pfd)
		pfd = malloc(sizeof(int)*2);

	/* clean up from last command */
	close_pipe(p_r);
	close_pipe(p_e);

	if (pipe(p_r) < 0 || pipe(p_e) < 0) {
		g_warning("popen_rw_unbuffered: Error creating pipe!");
		return NULL;
	}

	if ((cdr = fork()) < 0) {
		g_warning("popen_rw_unbuffered: Error forking!");
		return NULL;
	} else if (cdr == 0) { /* child */
		if (setpgid(getpid(), getpid()) < 0)
			g_warning("popen_rw_unbuffered: setpgid() failed");
		if (close(p_r[0]) < 0)
			g_warning("popen_rw_unbuffered: close(p_r[0]) failed");
		if (p_r[1] != STDOUT_FILENO)
			if (dup2(p_r[1], STDOUT_FILENO) < 0)
				g_warning("popen_rw_unbuffered: child dup2 STDOUT failed!");
		if (close(p_r[1]) < 0)
			g_warning("popen_rw_unbuffered: close(p_r[1]) failed");

		if (close(p_e[0]) < 0)
			g_warning("popen_rw_unbuffered: close(p_e[0]) failed");
		if (p_e[1] != STDERR_FILENO)
			if (dup2(p_e[1], STDERR_FILENO) < 0)
				g_warning("popen_rw_unbuffered: child dup2 STDERR failed!");
		if (close(p_e[1]) < 0)
			g_warning("popen_rw_unbuffered: close(p_e[1]) failed");
		execv(command->elements[0], command->elements);
		g_warning("%s %i: popen_rw_unbuffered: execv() returned", __FILE__, __LINE__);
		_exit(127);
	} else { /* parent */
		if (close(p_r[1]) < 0)
			g_warning("popen_rw_unbuffered: close(p_r[1]) (parent) failed");
		if (close(p_e[1]) < 0)
			g_warning("popen_rw_unbuffered: close(p_e[1]) (parent) failed");
		pfd[0] = p_r[0];
		pfd[1] = p_e[0];
		return pfd;
	}
	g_assert_not_reached();
	return pfd;
}

/* executes the command stored in mkisofs->elemets and cdrecord->elements
 * (which are suitable for execv()), pipeing the stdout from mkisofs to stdin
 * of cdrecord
 * returns an int *pfd with file descriptors:
 * pfd[0] mkisofs's STDERR,
 * pfd[1] cdrecord's STDOUT and
 * pfd[2] cdrecord's STDERR
 */
int *popen_r_err_in(cmd_v *mkisofs, cmd_v *cdrecord) {
	static int cdr_stdout_to_me[2] = {-1, -1}, cdr_stderr_to_me[2] = {-1, -1};
	static int iso_stderr_to_me[2] = {-1, -1};
	int iso_stdout_to_cdr_stdin[2];
	static int *pfd;

	if (check_if_exe_ok(mkisofs->elements[0]))
		return NULL;
	if (check_if_exe_ok(cdrecord->elements[0]))
		return NULL;

	iso = cdr = 0;

	if (mkisofs->elements[mkisofs->elements_used - 1] != NULL)
		add_option_to_cmd(mkisofs, NULL); /* the [] should end with a NULL pointer */
	if (cdrecord->elements[cdrecord->elements_used - 1] != NULL)
		add_option_to_cmd(cdrecord, NULL);

	/* only allocate once */

	if (NULL == pfd)
		pfd = malloc(sizeof(int)*3);

	/* clean up from last command */
	close_pipe(cdr_stdout_to_me);
	close_pipe(cdr_stderr_to_me);
	close_pipe(iso_stderr_to_me);

	if (pipe(cdr_stdout_to_me) < 0 || pipe(cdr_stderr_to_me) < 0
			|| pipe(iso_stderr_to_me) < 0) {
		g_warning("popen_r_err_in: Error creating pipe!");
		return NULL;
	}

	if (pipe(iso_stdout_to_cdr_stdin) < 0) {
		g_warning("popen_r_err_in: Error creating pipe!");
		return NULL;
	}

	if ((cdr = fork()) < 0) {
		g_warning("popen_r_err_in: Error forking!");
		return NULL;
	} else if (cdr == 0) { /* child / cdrecord */

		if (setpgid(getpid(), getpid()) < 0)
			g_warning("popen_r_err_in: setpgid() failed (iso child)");

		if (close(cdr_stdout_to_me[0]) < 0)
			g_warning("popen_r_err_in: close(cdr_stdout_to_me[0] failed");
		if (cdr_stdout_to_me[1] != STDOUT_FILENO)
			if (dup2(cdr_stdout_to_me[1], STDOUT_FILENO) < 0)
				g_warning("popen_r_err_in: dup2 STDOUT failed!");
		if (close(cdr_stdout_to_me[1]) < 0)
			g_warning("popen_r_err_in: close(cdr_stdout_to_me[1]) failed");

		if (close(cdr_stderr_to_me[0]) < 0)
			g_warning("popen_r_err_in: close(cdr_stderr_to_me[0]) failed");
		if (cdr_stderr_to_me[1] != STDERR_FILENO)
			if (dup2(cdr_stderr_to_me[1], STDERR_FILENO) < 0)
				g_warning("popen_r_err_in: dup2 STDERR failed!");
		if (close(cdr_stderr_to_me[1]) < 0)
			g_warning("popen_r_err_in: close(cdr_stderr_to_me[1]) failed");

		if (close(iso_stdout_to_cdr_stdin[1]) < 0)
			g_warning("popen_r_err_in: close(iso_stdout_to_cdr_stdin[1]) failed (cdr)");

		if (iso_stdout_to_cdr_stdin[0]  != STDIN_FILENO)
			if (dup2(iso_stdout_to_cdr_stdin[0], STDIN_FILENO) < 0)
				g_warning("open_r_err_in: dup2 iso_stdout_to_cdr_stdin[0] failed");
		if (close(iso_stdout_to_cdr_stdin[0]) < 0)
			g_warning("popen_r_err_in: close(iso_stdout_to_cdr_stdin[1]) failed");

		execv(cdrecord->elements[0], cdrecord->elements);
		g_warning("%s::%i: popen_r_err_in: execl() returned (cdrecord)", __FILE__, __LINE__);
		_exit(127);
	} else if ((iso = fork()) < 0) {
		g_warning("popen_r_err_in: Error forking (child)!");
		return NULL;
	} else if (iso == 0) { /* child / mkiosfs */
		if (setpgid(getpid(), getpid()) < 0)
			g_warning("popen_r_err_in: setpgid() failed (iso child)");

		if (close(iso_stderr_to_me[0]) < 0)
			g_warning("popen_r_err_in: close(cdr_stderr_to_me[0]) failed");
		if (iso_stderr_to_me[1] != STDERR_FILENO)
			if (dup2(iso_stderr_to_me[1], STDERR_FILENO) < 0)
				g_warning("popen_r_err_in: dup2 STDERR failed!");
		if (close(iso_stderr_to_me[1]))
			g_warning("popen_r_err_in: close(iso_stderr_to_me[1]) failed");

		if (close(iso_stdout_to_cdr_stdin[0]) < 0)
			g_warning("popen_r_err_in: close(iso_stdout_to_cdr_stdin[0]) failed");

		if (iso_stdout_to_cdr_stdin[1]  != STDOUT_FILENO)
			if (dup2(iso_stdout_to_cdr_stdin[1], STDOUT_FILENO) < 0)
				g_warning("open_r_err_in: dup2 iso_stdout_to_cdr_stdin[1] failed");
		if (close(iso_stdout_to_cdr_stdin[1]) < 0)
			g_warning("popen_r_err_in: close(iso_stdout_to_cdr_stdin[1]) failed (mkisofs)");

		execv(mkisofs->elements[0], mkisofs->elements);
		g_warning("%s::%i: popen_r_err_in: execl() returned (mkiosfs)", __FILE__, __LINE__);
		_exit(127);
	} else { /* parent */
		if (close(iso_stdout_to_cdr_stdin[0]) < 0)
			g_warning("%s::%i: close(iso_stdout_to_cdr_stdin[0]) failed",
					__FILE__, __LINE__);
		if (close(iso_stdout_to_cdr_stdin[1]) < 0)
			g_warning("%s::%i: close(iso_stdout_to_cdr_stdin[1]) failed",
					__FILE__, __LINE__);
		pfd[0] = iso_stderr_to_me[0];
		pfd[1] = cdr_stdout_to_me[0];
		pfd[2] = cdr_stderr_to_me[0];
		return pfd;
	}
	g_assert_not_reached();
	return pfd;
}

/* this is from glibc-2.0.6 as libc5 doesn't seem to have it */
char *my_basename(char *filename) {
	char *p;

	p = strrchr(filename, '/');
	return p ? p + 1 : (char *)filename;
}
/* from shellutils-2.0i */
void my_strip_trailing_slashes(char *path) {
  int last;

  last = strlen(path) - 1;
  while (last > 0 && path[last] == '/')
	  path[last--] = '\0';
}

