/* Gnome BibTeX gui.C
 *    Alejandro Aguilar Sierra <asierra@servidor.unam.mx>
 *    Felipe Bergo <bergo@seul.org>
 * 
 *    This program is free software; you can redistribute it and'or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2, or (at your option)
 *    any later version.
 */

#include "gbib.h"

#include <iostream.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "recent.h"

#include "lyx.xpm"
#include "import.xpm"
#include "ascend.xpm"
#include "descend.xpm"
#include "none16.xpm"
#include "gbib48.xpm"

// The number of column to display
#define NBCOLUMN 5

// Name of the different field
const gchar *fields_name[NBCOLUMN] = {"key", "author", "title", "year", "Type"};
const int  column_width[NBCOLUMN] = {100, 150, 200, 80, 100}; 
bool     display_column[NBCOLUMN] = {true, true, true, true, true}; 
GtkJustification column_justification[NBCOLUMN] = {GTK_JUSTIFY_RIGHT, 
						   GTK_JUSTIFY_LEFT, 
						   GTK_JUSTIFY_LEFT,
						   GTK_JUSTIFY_CENTER,
						   GTK_JUSTIFY_CENTER};

GtkWidget *app;
GtkWidget *statusbar;

// List of entries
static GtkWidget *elist = 0;

// Options menu
GnomeUIInfo styleradioitems[16];

// Options
extern bool use_braces;
int bib_style = 0;


// Entries
BibentryTable etable;

//int selected_entry = 0;

// Sorting State
int CurrentSortColumn=-1; // none
int SortAscending[5]={1,1,1,1,1};

// column title widget (to display sorting status)

struct SortColTitle {
  
  GtkWidget *hbox;   // hbox container
  GtkWidget *label;  // GtkLabel
  GtkWidget *icon;   // GtkPixmap
  int SortDirection; // -1 = none, 0=descending (Z-A), 1=ascending (A-Z)
  
} ListColumnTitles[NBCOLUMN];

// Type of picture to display when sorting elements par in the row title
GdkPixmap *gbib_ct_pix[3];
GdkBitmap *gbib_ct_mask[3];

// column title associated funcs

static int gbib_ct_inited=0;
void gbib_column_title_init();
void gbib_column_title_over();
void gbib_column_title_new(SortColTitle *dest,char *name);
void gbib_column_title_set_state(SortColTitle *dest,int state);

void file_new_cb (GtkWidget *widget, void *data);
void file_open_cb (GtkWidget *widget, void *data);
void file_add_cb (GtkWidget *widget, void *data);
void file_saveas_cb (GtkWidget *widget, void *data);
void file_save_cb (GtkWidget *widget, void *data);
void opt_cb (GtkWidget *widget, void *data);
void view_cb (GtkWidget *widget, char *data);
void quit_cb (GtkWidget *widget, void *data);
void about_cb (GtkWidget *widget, void *data);
void edit_cb (GtkWidget *widget, void *data);
void edit_commands_cb (GtkWidget *widget, void *data);
void delete_cb (GtkWidget *widget, void *data);
void new_cb (GtkWidget *widget, void *data);
void lyx_cp(GtkWidget *w, gpointer data);
void sort_cb (GtkWidget *, char *);
void search_cb (GtkWidget *, void *);
void search_next_cb(GtkWidget *, void *);
void add_file(char *s);
void sort_gui_cb(GtkCList *clist,gint column,gpointer data);

void update_elist();
void rellena_menu();
void resetFileName();

void chopDirName(char *dest,char *src);
void GoToOpenDir();
void GoToSaveDir();

ResponseType ask_about_conflict(BibEntry *old, BibEntry *nova);

struct FileselDialog {
  GtkWidget *fs;
  int mode;
  string filename;
  char openDir[512];
  char saveDir[512];
} fs_dialog;


//
void entry_dlg(char *, BibEntry *);
void command_dlg(BibentryTable &table);

static GnomeUIInfo file_menu [] = {
        GNOMEUIINFO_MENU_NEW_ITEM(N_("_New"), N_("Create new database"),
				  GTK_SIGNAL_FUNC(file_new_cb),
				  (void*)1),

        GNOMEUIINFO_MENU_OPEN_ITEM(GTK_SIGNAL_FUNC(file_open_cb),
				   (void*)2),

        GNOMEUIINFO_MENU_SAVE_ITEM(GTK_SIGNAL_FUNC(file_save_cb),
				   NULL),

        GNOMEUIINFO_MENU_SAVE_AS_ITEM(GTK_SIGNAL_FUNC(file_saveas_cb),
				      (void*)4),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_SEPARATOR,

	{ GNOME_APP_UI_ITEM,
	  N_("_Merge BibTeX entries..."),
	  N_("Add BibTeX entries to current database"),
	  (gpointer)file_add_cb, NULL, NULL,
	  GNOME_APP_PIXMAP_NONE, NULL, 0,
	  (GdkModifierType)GDK_CONTROL_MASK, NULL },

	GNOMEUIINFO_SEPARATOR,


        GNOMEUIINFO_MENU_EXIT_ITEM(GTK_SIGNAL_FUNC(quit_cb), NULL),

        GNOMEUIINFO_END
};

GnomeUIInfo empty_menu[] = {
  { GNOME_APP_UI_ENDOFINFO }
};

static GnomeUIInfo edit_menu [] = {
  { GNOME_APP_UI_SUBTREE,
    N_("_Insert entry"), 
    N_("Insert a new entry"), 
    empty_menu,
    NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 
    (GdkModifierType)0, NULL },

  { GNOME_APP_UI_ITEM,
    N_("_Edit entry"),
    N_("Edit the selected entry"),
    (gpointer)edit_cb,
    NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH, 'E', 
    (GdkModifierType)GDK_CONTROL_MASK, NULL },

  { GNOME_APP_UI_ITEM,
    N_("_Delete entry"),
    N_("Delete the selected entry"),
    (gpointer)delete_cb,
    NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CUT, 'D', 
    (GdkModifierType)GDK_CONTROL_MASK, NULL },
  
    
  { GNOME_APP_UI_ITEM,
    N_("Edit _commands"),
    N_("Edit BibTeX commands"),
    (gpointer)edit_commands_cb,
    NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 'C', 
    (GdkModifierType)GDK_CONTROL_MASK, NULL },
   
    GNOMEUIINFO_SEPARATOR,
    
  { GNOME_APP_UI_ITEM,
    N_("_Search"),
    N_("Simple search forward"),
    (gpointer)search_cb,
    NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_SEARCH, 'F', 
    (GdkModifierType)GDK_CONTROL_MASK, NULL },
    
  { GNOME_APP_UI_ITEM,
    N_("Search _next"),
    N_("Search forward again"),
    (gpointer)search_next_cb, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 'G', 
    (GdkModifierType)GDK_CONTROL_MASK, NULL },
    
    GNOMEUIINFO_SEPARATOR,
    
  { GNOME_APP_UI_ITEM,
    N_("Sort by _key"),
    N_("Sort entries by key"),
    (gpointer)sort_cb, "key", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0,
    (GdkModifierType)GDK_CONTROL_MASK, NULL },
    
  { GNOME_APP_UI_ITEM,
    N_("Sort by _author"),
    N_("Sort entries by author"),
    (gpointer)sort_cb, "author", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0,
    (GdkModifierType)GDK_CONTROL_MASK, NULL },   

  { GNOME_APP_UI_ITEM,
    N_("Sort by _title"),
    N_("Sort entries by title"),
    (gpointer) sort_cb, "title", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0,
    (GdkModifierType)GDK_CONTROL_MASK, NULL },
          
  { GNOME_APP_UI_ITEM,
    N_("Sort by _year"),
    N_("Sort entries by year"),
    (gpointer) sort_cb, "year", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0,
    (GdkModifierType)GDK_CONTROL_MASK, NULL },
    
  { GNOME_APP_UI_ITEM,
    N_("Sort by _cardtype"),
    N_("Sort entries by type"),
    (gpointer) sort_cb, "cardtype", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0,
    (GdkModifierType)GDK_CONTROL_MASK, NULL },
   
  GNOMEUIINFO_END
};

GnomeUIInfo delimradioitems[] = {
{ GNOME_APP_UI_ITEM, N_("Use {braces}"), NULL, (gpointer)opt_cb, (void*)-1, NULL,
GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
{ GNOME_APP_UI_ITEM, N_("Use \"quotes\""), NULL, (gpointer)opt_cb, (void*)-2, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },    
{ GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo view_subtree[] = {
  { GNOME_APP_UI_TOGGLEITEM, "Author", NULL, (gpointer)view_cb, "author", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
  { GNOME_APP_UI_TOGGLEITEM, "Title", NULL, (gpointer)view_cb, "title", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
  { GNOME_APP_UI_TOGGLEITEM, "Year", NULL, (gpointer)view_cb,  "year", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
  { GNOME_APP_UI_TOGGLEITEM, "Type", NULL, (gpointer)view_cb, "Type", NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },    
  { GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo styles_subtree[] = {
      { GNOME_APP_UI_RADIOITEMS, NULL, NULL, styleradioitems, NULL, NULL,
	  GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },    
      { GNOME_APP_UI_ENDOFINFO }
};


GnomeUIInfo delim_subtree[] = {
      { GNOME_APP_UI_RADIOITEMS, NULL, NULL, delimradioitems, NULL, NULL,
	  GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },    
      { GNOME_APP_UI_ENDOFINFO }
};


GnomeUIInfo options_menu[] = {
      { GNOME_APP_UI_SUBTREE, N_("Field delimiter"), NULL, delim_subtree, NULL, NULL,
	  GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },  
      { GNOME_APP_UI_SUBTREE, N_("Styles"), NULL, styles_subtree, NULL, NULL,
	  GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },   
      { GNOME_APP_UI_SUBTREE, N_("View"), NULL, view_subtree, NULL, NULL,
	  GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
      { GNOME_APP_UI_ENDOFINFO }
};


GnomeUIInfo helpmenu[] = {
  GNOMEUIINFO_MENU_ABOUT_ITEM(about_cb, NULL),
  GNOMEUIINFO_END
};


GnomeUIInfo mainmenu[] = {
    GNOMEUIINFO_SUBTREE(N_("_File"), file_menu),
    GNOMEUIINFO_SUBTREE(N_("_Edit"), edit_menu),
    GNOMEUIINFO_SUBTREE(N_("_Options"), options_menu),
    GNOMEUIINFO_SUBTREE(N_("_Help"), helpmenu),
    GNOMEUIINFO_END
};


GnomeUIInfo toolbar[] = {    
  GNOMEUIINFO_ITEM_STOCK (N_("New"), N_("Create a new database"), file_new_cb, 
			  GNOME_STOCK_PIXMAP_NEW),
  
  GNOMEUIINFO_ITEM_STOCK (N_("Open"), N_("Open a database file"), 
			  file_open_cb, GNOME_STOCK_PIXMAP_OPEN),

  {GNOME_APP_UI_ITEM, N_("Merge"), 
   N_("Merge/Import from a BibTeX file"),
   (gpointer)file_add_cb, NULL, NULL,
   GNOME_APP_PIXMAP_DATA, import_xpm, 0, (GdkModifierType)0, NULL},

  GNOMEUIINFO_ITEM_STOCK (N_("Save"), N_("Save the database"), 
			  file_save_cb, GNOME_STOCK_PIXMAP_SAVE),

  GNOMEUIINFO_SEPARATOR,

  {GNOME_APP_UI_ITEM, N_("Cites"), 
   N_("Establish communication with a running LyX."),
   (gpointer)lyx_cp, NULL, NULL,
   GNOME_APP_PIXMAP_DATA, lyxico, 0, (GdkModifierType)0, NULL},
    
  GNOMEUIINFO_SEPARATOR,
  
  GNOMEUIINFO_ITEM_STOCK (N_("Search"), N_("Simple search forward"),
			  (gpointer)search_cb, GNOME_STOCK_PIXMAP_SEARCH),
  {GNOME_APP_UI_ENDOFINFO, NULL, NULL,
   NULL, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL}
};


inline
void setTitle(char *s)
{
  if (!s) {
    gtk_window_set_title(GTK_WINDOW(app), "gBib");
    return;
  }
  
  string ntitle = s;
  int k = ntitle.rfind('/');
  
  if (k > 0 && k < 1000) {
    ntitle = ntitle.substr(k+1);
  }
  
  ntitle = "gBib: " + ntitle;
  gtk_window_set_title(GTK_WINDOW(app), const_cast<char*>(ntitle.c_str()));
}

int open_file(char *s)
{    
  char msg[1024];
  
  gtk_clist_clear(GTK_CLIST(elist));
  etable.clear();
  etable.resetUI();
  if (etable.readBibfile(s)==0) {
    chopDirName(fs_dialog.openDir,s);
    fs_dialog.filename = s;
    update_elist();
    setTitle(s);
    return 0;
  } else {
    sprintf(msg,_("Cannot read from %s: %s"),s,strerror(errno));
    gnome_app_flash(GNOME_APP(app),msg);
    resetFileName();
    return -1;
  }
}


void add_file(char *s)
{    
  char msg[1024];
  etable.resetUI();
  if (etable.readBibfile(s)==0) {
    chopDirName(fs_dialog.openDir,s);
    update_elist();
  } else {
    sprintf(msg,_("Cannot read from %s: %s"),s,strerror(errno));
    gnome_app_flash(GNOME_APP(app),msg);
    resetFileName();
  }
}

void file_selected_cb(GtkWidget *widget, void *data)
{
    char *s;
    GtkFileSelection *fs = GTK_FILE_SELECTION(fs_dialog.fs);
    char msg[1024];
    
    s = gtk_file_selection_get_filename(fs);
    gtk_widget_hide (GTK_WIDGET(fs));
        
    switch(fs_dialog.mode) {
    case 1: // new
      fs_dialog.filename = s;
      gtk_clist_clear(GTK_CLIST(elist));
      etable.clear();
      setTitle(s);
      break;
    case 2: //open
      open_file(s);
      add_recent_item(GNOME_APP(app),s);
      break;
    case 3: // import/merge
      add_file(s);
      break;
    case 4: // save as
      
      if (etable.writeBibfile(s)==0) {
	chopDirName(fs_dialog.saveDir,s);
	add_recent_item(GNOME_APP(app),s);
      } else {
	sprintf(msg,_("Cannot write to %s: %s"),s,strerror(errno));
	gnome_app_flash(GNOME_APP(app),msg);
      }
      break;
    case 5: // new filename
      fs_dialog.filename = s;
      setTitle(s);      
      if (etable.writeBibfile(s)==0) {
	chopDirName(fs_dialog.saveDir,s);
	add_recent_item(GNOME_APP(app),s);
      } else {
	sprintf(msg,_("Cannot write to %s: %s"),s,strerror(errno));
	gnome_app_flash(GNOME_APP(app),msg);
      }
      break;
    }
}


void file_cancel_cb(GtkWidget *widget, void *data)
{
    GtkFileSelection *fs = GTK_FILE_SELECTION(fs_dialog.fs);
    gtk_widget_hide (GTK_WIDGET(fs));
}



static string searched_string;


void string_cb(gchar * string, gpointer data)
{
    if (string != NULL) {
	int k = etable.search_string(string, (int)data);
    
	if (k < 0) {
	    gnome_app_flash(GNOME_APP (app), _("Search failed."));
	} else {
	    etable.selected_entry = -1;
	    gtk_clist_select_row (GTK_CLIST(elist), k, 0);
	    if (gtk_clist_row_is_visible(GTK_CLIST(elist), k)!=GTK_VISIBILITY_FULL) 
	      gtk_clist_moveto (GTK_CLIST(elist), k, 0, 0.0, 0.0);
	    GTK_CLIST(elist)->focus_row = k;
	    etable.selected_entry = k;
	    searched_string = string;
	}    
    }
}

void search_cb (GtkWidget *widget, void *)
{
  gnome_request_dialog (FALSE, _("Search string"),
			"",
			60,
			(GnomeStringCallback)string_cb,
			0,
			GTK_WINDOW(app));    
}


void search_next_cb(GtkWidget *widget, void *)
{
    if (etable.selected_entry >= 0) {
	string_cb(const_cast<char*>(searched_string.c_str()),
		  (gpointer)(etable.selected_entry + 1));
    }
}

void sort_cb (GtkWidget *widget, char *s)
{
   for (int i=0; i<NBCOLUMN; i++)
     if (string(fields_name[i])==string(s)){
	CurrentSortColumn = i;
	SortAscending[i] = 1;
     } 

  etable.sort(1,s);
  update_elist();
}



inline void file_cb ()
{
  gtk_signal_connect(GTK_OBJECT (GTK_FILE_SELECTION (fs_dialog.fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_selected_cb), 
		     NULL);
  
  gtk_signal_connect(GTK_OBJECT (GTK_FILE_SELECTION (fs_dialog.fs)->cancel_button),
		     "clicked", (GtkSignalFunc) file_cancel_cb, 
		     NULL);
  
  gtk_widget_show (fs_dialog.fs);
}


void file_new_cb (GtkWidget *widget, void *data)
{
    fs_dialog.mode = 1;
    fs_dialog.fs = gtk_file_selection_new(_("New BibTeX Database"));
    GoToSaveDir();
    file_cb ();
}


void file_open_cb (GtkWidget *widget, void *data)
{
    fs_dialog.mode = 2;
    fs_dialog.fs = gtk_file_selection_new(_("Open BibTeX Database"));
    GoToOpenDir();
    file_cb ();
}

void file_add_cb (GtkWidget *widget, void *data) 
{
    fs_dialog.mode = 3;
    fs_dialog.fs = gtk_file_selection_new(_("Merge with BibTeX Database"));
    GoToOpenDir();
    file_cb ();
}


void file_save_cb (GtkWidget *widget, void *)
{
  char msg[1024];

  if (fs_dialog.filename.empty()) {
    fs_dialog.mode = 5;
    fs_dialog.fs = gtk_file_selection_new(_("Name BibTeX Database as ..."));
    GoToSaveDir();
    file_cb ();
    return;
  }
  // write the backup file
  string fn = fs_dialog.filename;
  fn += ".bak";
  rename(fs_dialog.filename.c_str(), fn.c_str());
  
  if (etable.writeBibfile(const_cast<char*>(fs_dialog.filename.c_str()))<0) {
    sprintf(msg,_("Cannot write to %s: %s"),fs_dialog.filename.c_str(),
	    strerror(errno));
    gnome_app_flash(GNOME_APP(app),msg);
  } else {
    strcpy(msg,fs_dialog.filename.c_str());
    chopDirName(fs_dialog.saveDir,msg);
  }
}


void file_saveas_cb (GtkWidget *widget, void *data)
{
    fs_dialog.mode = 4;
    fs_dialog.fs = gtk_file_selection_new(_("Save BibTeX Database as ..."));
    GoToSaveDir();
    file_cb ();
}


void opt_cb (GtkWidget *widget, void *data)
{
    int i = (int) data;
    
    if (i>=0) {
	bib_style = i;
	return;
    }

    switch(i) {
     case -1:
	use_braces = true;
	break;
     case -2:
	use_braces = false;
	break;
    }
}


void view_cb (GtkWidget *widget, char *data)
{
  for(int i=0; i<NBCOLUMN; i++){
    if (strcmp(data, fields_name[i])==0){
      display_column[i] = !(display_column[i]);     
      gtk_clist_set_column_visibility (GTK_CLIST(elist),i, display_column[i]);
      return;
    }
  }
}


/* TRS: Quit callback */
void quit_cb (GtkWidget *widget, void *data)
{
  gtk_main_quit ();
  return;
}


void
about_cb (GtkWidget *widget, void *data)
{
    GtkWidget *about;
    const gchar *authors[] = {
	"Alejandro Aguilar Sierra",
	"Felipe Paulo Guazzi Bergo",
	"Defour David",
	NULL
    };

    about = gnome_about_new ( _("gBib - Gnome BibTeX Editor"), VERSION,
			      /* copyrigth notice */
			      "(C) 1998-2001 The Free Software Foundation",
			      authors,    
			      _("A user-friendly BibTeX database editor"),
			      NULL);
  gtk_widget_show (about);

  return;
}


void edit_cb (GtkWidget *, void *)
{
    BibEntry *bib = etable.get_entry(etable.selected_entry);
    if (bib) {
	if (bib->isSpecial()) {
	   cerr << "Error: a special entry";
	} else
	  entry_dlg(bib->getEntryType(),  bib);
    }
    return;
}


void edit_commands_cb (GtkWidget *, void *)
{
   command_dlg(etable);

   return;
}


void delete_cb (GtkWidget *widget, void *data)
{
    if (etable.selected_entry >= 0) {
	etable.delete_entry(etable.selected_entry);
	gtk_clist_freeze (GTK_CLIST(elist));
	gtk_clist_remove (GTK_CLIST(elist), etable.selected_entry);
	gtk_clist_thaw   (GTK_CLIST(elist));
    }
    return;
}


void new_cb (GtkWidget *widget, void *data)
{
    char *s = "article";
    char *et;
    int i = (int)data;
    
    if (i >=0 && i < 50) {
	et = get_entrytype_name(i);
    } else {
	i = 0;
	et = s;
    }

    entry_dlg(et, 0);
    return;
}


void selection_made( GtkWidget      *clist,
                     gint            row,
                     gint            column,
                     GdkEventButton *event,
                     gpointer        data )
{
    if (etable.selected_entry==row) 
      edit_cb (0, 0);

    etable.selected_entry = row;
    
    return;
}


void update_entry(bool is_new)
{
  char *data[8];
  int i, j;

    if (etable.selected_entry >= 0) {
	i = etable.selected_entry;
	
	gtk_clist_freeze (GTK_CLIST(elist));
	BibEntry *bib =  etable.get_entry(i);

	for(j=0; j<NBCOLUMN; j++){
	  data[j] = bib->getField(fields_name[j]);
	  if ((strcmp(fields_name[j],"author")==0)&&(!data[j]))
	      data[j] = bib->getField("editor");	  
	}

       	if (is_new)
	  gtk_clist_insert (GTK_CLIST(elist), i, (gchar **)data);
	else {
	  for(j=0; j<NBCOLUMN; j++)
	    gtk_clist_set_text (GTK_CLIST(elist), i, j, data[j]);
	}
	gtk_clist_thaw   (GTK_CLIST(elist));
    }
}
 
void update_elist()
{
    char *data[8];
    BibEntry *bib;
    int i, j;

    if (!elist) {
      // Create "nbcoltodisplay" columns
      elist = gtk_clist_new(NBCOLUMN);

      gtk_clist_set_selection_mode(GTK_CLIST(elist), GTK_SELECTION_SINGLE);

      for(i=0; i<NBCOLUMN; i++){
	  gbib_column_title_new(&ListColumnTitles[i], fields_name[i]);
	  gtk_clist_set_column_widget(GTK_CLIST(elist),i,ListColumnTitles[i].hbox);
	  gtk_clist_set_column_width (GTK_CLIST(elist),i,column_width[i]);
	  gtk_clist_set_column_min_width (GTK_CLIST(elist),i,int(column_width[i]*0.8));
	  gtk_clist_set_column_max_width (GTK_CLIST(elist),i,int(column_width[i]*1.2));
	  gtk_clist_set_column_justification (GTK_CLIST(elist), i, column_justification[i]);   
      }
      
      gtk_clist_column_titles_active(GTK_CLIST(elist));
      gtk_clist_column_titles_show(GTK_CLIST(elist));
      gtk_signal_connect (GTK_OBJECT(elist),"click_column",
			  GTK_SIGNAL_FUNC(sort_gui_cb),NULL);	
      gtk_signal_connect(GTK_OBJECT(elist), "select_row",
			 GTK_SIGNAL_FUNC(selection_made),
			 NULL);
      
      gtk_widget_show(elist);
    }
    
    gtk_clist_freeze (GTK_CLIST(elist));
    gtk_clist_clear(GTK_CLIST(elist));

    int tmp = etable.selected_entry;
    for (i=0; (bib = etable.get_entry(i)); i++) {
	for(j=0; j<NBCOLUMN; j++){
	  data[j] = bib->getField(fields_name[j]);
	    if ((strcmp(fields_name[j],"author")==0)&&(!data[j]))
	      data[j] = bib->getField("editor");	  
	}
	gtk_clist_insert(GTK_CLIST(elist), i,data);
    }

    if (tmp>0) {
	etable.selected_entry = -1;
	gtk_clist_select_row (GTK_CLIST(elist), tmp, 0);
	gtk_clist_moveto (GTK_CLIST(elist), tmp, 0, 0.0, 0.0);
	etable.selected_entry = tmp;
    }

    gtk_clist_columns_autosize(GTK_CLIST(elist));

    for(i=0; i<NBCOLUMN; i++)
      gbib_column_title_set_state(&ListColumnTitles[i],
				  (i==CurrentSortColumn)?SortAscending[i]:-1);

    gtk_clist_thaw   (GTK_CLIST(elist));
}

void rellena_menu ()
{    
    int i = 0;
    char *s;
    GnomeUIInfo *entry = new GnomeUIInfo[32];

    while ((s = get_entrytype_name(i))) {
	entry[i].type = GNOME_APP_UI_ITEM;
	entry[i].label = s;
	entry[i].hint = NULL;
	entry[i].moreinfo = (gpointer)new_cb;
	entry[i].user_data = (gpointer) i;
	entry[i].unused_data = NULL;
	entry[i].pixmap_type = GNOME_APP_PIXMAP_NONE;
	entry[i].pixmap_info = NULL;
	/* TRS: Accelerator for add article */
	if ( strcmp(s,"article") == 0 ) {
	  entry[i].accelerator_key = 'a';
	  entry[i].ac_mods = GDK_MOD1_MASK; 
	} else {
	  entry[i].accelerator_key = 0; 
	  entry[i].ac_mods = (GdkModifierType)0; 
	}
	i++;
    }
    entry[i].type = GNOME_APP_UI_ENDOFINFO;
    
    edit_menu[0].moreinfo = entry;

    return;
}


void fill_bibstyles_menu ()
{
    int i = 0;
    char *s;
    
    while ((s = get_bibtex_style(i))) {
	styleradioitems[i].type = GNOME_APP_UI_ITEM;
	styleradioitems[i].label = s;
	styleradioitems[i].hint = NULL;
	styleradioitems[i].moreinfo = (gpointer)opt_cb;
	styleradioitems[i].user_data = (gpointer) i;
	styleradioitems[i].unused_data = NULL;
	styleradioitems[i].pixmap_type = GNOME_APP_PIXMAP_NONE;
	styleradioitems[i].pixmap_info = NULL;
	styleradioitems[i].accelerator_key = 0;
	i++;
    }
    styleradioitems[i].type = GNOME_APP_UI_ENDOFINFO;
        
    return;
}

static int allow_lyx = 1;

int flag_allow_warnings = 0;

const char **startup_files = NULL;

poptContext ctx;

struct poptOption gbib_popt_options [] = {
        { "no-lyx", '\0',  POPT_ARG_INT, &allow_lyx, 0,
          N_("Set off lyx"),   N_("FILE") },
        { "warnings", '\0',  POPT_ARG_INT, &flag_allow_warnings, 0,
          N_("Whether display warnings"),   "[01]" },
    { NULL, '\0', 0, NULL, 0 }
};


int
gmain(int argc, char *argv[])
{
    GtkWidget *scrolled_window;
    GdkPixmap *pmap;
    GdkBitmap *mask;
    GtkStyle *style;

    bindtextdomain (PACKAGE, PREFIX"/share/locale");
    textdomain (PACKAGE);
 
    gnome_init_with_popt_table (
                "gbib", VERSION, argc, argv, gbib_popt_options, 0, 
				&ctx);

    app = gnome_app_new("GBibTex", _("Gnome BibTeX Editor"));

    // init current directories
    if (!getcwd(fs_dialog.openDir,512))
      strcpy(fs_dialog.openDir,".");
    if (!getcwd(fs_dialog.saveDir,512))
      strcpy(fs_dialog.saveDir,".");    
    
    fill_bibstyles_menu();
    rellena_menu();
    gnome_app_create_menus( GNOME_APP (app), mainmenu);
    
    gnome_app_create_toolbar( GNOME_APP (app), toolbar);

    recent_startup(GNOME_APP(app)); // load recent files
    gtk_widget_realize (app); // must realize to load pixmaps
    update_elist();	

    // set icon
    style=gtk_widget_get_style(app);
    pmap = gdk_pixmap_create_from_xpm_d (app->window, &mask,
					 &style->bg[GTK_STATE_NORMAL],
					 (gchar **) gbib48_xpm);
    gdk_window_set_icon (app->window, NULL, pmap, mask);
    gdk_window_set_icon_name(app->window,"gBib");

    scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_set_border_width (GTK_CONTAINER (scrolled_window), 5);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				   GTK_POLICY_AUTOMATIC,
				   GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(scrolled_window, 600, 400);
    gtk_widget_show(scrolled_window);
    gtk_container_add (GTK_CONTAINER (scrolled_window), elist);
    gnome_app_set_contents ( GNOME_APP (app), scrolled_window);

    statusbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_NEVER); //gtk_statusbar_new();
    gtk_widget_show(statusbar);
    gnome_app_set_statusbar (GNOME_APP (app), statusbar);

    /* TRS: close app from window manager */
    gtk_signal_connect (GTK_OBJECT (app), "destroy",
                        GTK_SIGNAL_FUNC (quit_cb), NULL);

    gtk_widget_show (app);

//    rellena_menu ();
    
    startup_files = poptGetArgs (ctx);
    if (startup_files && startup_files [0]) {
	open_file(const_cast<char*>(startup_files [0]));
    }
    poptFreeContext (ctx);
    
    gtk_main ();

    return 0;
}


ResponseType ask_about_conflict(BibEntry *old, BibEntry *nova) {
  GtkWidget *dlg,*f1,*f2,*t1,*t2,*l[2][2][NBCOLUMN],*always;
  int i,j,k;
  char lbuf[512];
  const char *p;

  dlg=gnome_dialog_new(_("Key Conflict"),
                       _("AutoRename"),
                       _("Ignore Duplicate"),
                       _("Replace Existing"),
                       0);

  gnome_dialog_set_parent(GNOME_DIALOG(dlg),GTK_WINDOW(app));
  
  f1=gtk_frame_new("New Entry");
  f2=gtk_frame_new("Previously Existing Entry");
  
  gtk_frame_set_shadow_type(GTK_FRAME(f1),GTK_SHADOW_ETCHED_IN);
  gtk_frame_set_shadow_type(GTK_FRAME(f2),GTK_SHADOW_ETCHED_IN);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dlg)->vbox),f1,FALSE,TRUE,6);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dlg)->vbox),f2,FALSE,TRUE,6);
  
  t1=gtk_table_new(3,2,FALSE);
  t2=gtk_table_new(3,2,FALSE);

  gtk_container_add(GTK_CONTAINER(f1),t1);
  gtk_container_add(GTK_CONTAINER(f2),t2);

  for(i=0; i<NBCOLUMN; i++){
    l[0][0][i]=gtk_label_new(fields_name[i]);
    l[1][0][i]=gtk_label_new(fields_name[i]);
  }
 
  for(i=0; i<NBCOLUMN; i++){
    memset(lbuf,0,512);
    p=nova->getField(fields_name[i]);
    if (p) strncpy(lbuf,p,511); else strcpy(lbuf,"(blank)");
    l[0][1][i]=gtk_label_new(lbuf);

    memset(lbuf,0,512);
    p=old->getField(fields_name[i]);
    if (p) strncpy(lbuf,p,511); else strcpy(lbuf,"(blank)");
    l[1][1][i]=gtk_label_new(lbuf);
  }

  for(i=0;i<2;i++)
    for(j=0;j<NBCOLUMN;j++)
      gtk_table_attach_defaults(GTK_TABLE(t1),l[0][i][j],
				i,i+1,j,j+1);
  for(i=0;i<2;i++)
    for(j=0;j<NBCOLUMN;j++)
      gtk_table_attach_defaults(GTK_TABLE(t2),l[1][i][j],
				i,i+1,j,j+1);

  for(i=0;i<2;i++)
    for(j=0;j<2;j++)
      for(k=0;k<NBCOLUMN;k++)
	gtk_widget_show(l[i][j][k]);

  always=gtk_check_button_new_with_label("Same answer to all conflicts");
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dlg)->vbox),always,FALSE,TRUE,6);

  gtk_widget_show(always);
  gtk_widget_show(t1);
  gtk_widget_show(t2);
  gtk_widget_show(f1);
  gtk_widget_show(f2);

  gnome_dialog_close_hides (GNOME_DIALOG(dlg), TRUE);
  i=gnome_dialog_run(GNOME_DIALOG(dlg));
  j=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(always));  
  gtk_widget_destroy(dlg);

  switch(i) {
  case 0: return(j?AutoFixAlways:AutoFix);
  case 1: return(j?IgnoreNewAlways:IgnoreNew);
  case 2: return(j?ReplaceOldAlways:ReplaceOld);
  default:
    return(IgnoreNew);
  }
}

void sort_gui_cb (GtkCList *clist, gint column, gpointer data)
{
  // if already sorted on this column, reverse sorting direction
  if (column==CurrentSortColumn)
    SortAscending[column]=SortAscending[column]?0:1;

  CurrentSortColumn=column;
   
  if ((column>=0) && (column<NBCOLUMN))
    etable.sort(SortAscending[column],fields_name[column]);
   
  update_elist();
}

// column title associated funcs

void gbib_column_title_init() {
  GtkStyle *style;
  style=gtk_widget_get_style(app);
  gbib_ct_pix[0]=gdk_pixmap_create_from_xpm_d (app->window, 
					       & gbib_ct_mask[0],
					       &style->bg[GTK_STATE_NORMAL],
					       (gchar **) none16_xpm);
  gbib_ct_pix[1]=gdk_pixmap_create_from_xpm_d (app->window, 
					       & gbib_ct_mask[1],
					       &style->bg[GTK_STATE_NORMAL],
					       (gchar **) descend_xpm);
  gbib_ct_pix[2]=gdk_pixmap_create_from_xpm_d (app->window, 
					       & gbib_ct_mask[2],
					       &style->bg[GTK_STATE_NORMAL],
					       (gchar **) ascend_xpm);
  gbib_ct_inited=1;
}

void gbib_column_title_over() {
  for(int i=0;i<3;i++)
    gdk_pixmap_unref(gbib_ct_pix[i]);
}

void gbib_column_title_new(SortColTitle *dest, const char *name) {
  if (!gbib_ct_inited)
    gbib_column_title_init();

  dest->hbox=gtk_hbox_new(FALSE,4);
  dest->label=gtk_label_new(name);
  dest->icon=gtk_pixmap_new(gbib_ct_pix[0],gbib_ct_mask[0]); 
  gtk_box_pack_start(GTK_BOX(dest->hbox),dest->label,FALSE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(dest->hbox),dest->icon,FALSE,FALSE,2);
  gtk_widget_show(dest->label);
  gtk_widget_show(dest->icon);
  gtk_widget_show(dest->hbox);
  dest->SortDirection=-1;
}

void gbib_column_title_set_state(SortColTitle *dest,int state) {
  int previous;
  previous=dest->SortDirection;
  dest->SortDirection=state;
  if (previous!=state) {
    gtk_pixmap_set(GTK_PIXMAP(dest->icon),
		   gbib_ct_pix[state+1],gbib_ct_mask[state+1]);
    gtk_widget_queue_resize(dest->icon); // force redraw    
  }
}

// called when opening a file fails
// since we dropped the old file and no new file came up,
// we must reset any current file name
void resetFileName() {
  fs_dialog.filename="";
  setTitle(0);
}

void chopDirName(char *dest,char *src) {
  char *p;
  p=rindex(src,'/');
  if (!p) {
    if (!getcwd(dest,512))
      strcpy(dest,".");
    return;
  }
  memset(dest,0,(int)(1+p-src));
  memcpy(dest,src,(int)(p-src));
}

void GoToOpenDir() {
  char dest[513];
  strcpy(dest,fs_dialog.openDir);
  strcat(dest,"/");
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs_dialog.fs),dest);
}

void GoToSaveDir() {
  char dest[513];
  strcpy(dest,fs_dialog.saveDir);
  strcat(dest,"/");
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs_dialog.fs),dest);
}
