/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_GATOS_C 1

#include "gatos.h"
#include "channels.h"
#include "atiregs.h"
#include "pci.h"
#include "i2c.h"
#include "ati.h"
#include "tvout.h"
#include "board.h"
#include "bt829.h"
#include "fi12xx.h"
#include "sound.h"
#include "saa5281.h"
#include "version.h"

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>

#define CONFIGFILE	"/usr/local/lib/gatos.conf"
#define MAXLEVELS	10

/* Private global vars */
static int init=0, debug=0, nosa=0, err=0 ;
static char *pciaddr=NULL ;
static char *gatoscalls[MAXLEVELS+1] ;

/* ------------------------------------------------------------------------ */

#define SUBCALL(FUNC)	\
	gatos_enter(#FUNC,"") ; err = FUNC() ; gatos_leave(err) ; \
	if (err) { fprintf(stderr,"GATOS: %s failed: %s\n", \
	  #FUNC,strerror(err)) ; RETURN(err) ; }
#define INSANE(VAR)	{ \
	fprintf(stderr,"%s value '%d' invalid",#VAR,gatos.##VAR) ; \
	RETURN(EINVAL) ; }
#define FUNC0(TYPE,NAME,FMT,VAR) \
	TYPE NAME(void) { \
	  if (GATOSCALL) { \
	    fprintf(stderr,"GATOS: %s() = ",#NAME) ; \
	    fprintf(stderr,FMT,VAR) ; fprintf(stderr,"\n") ; } \
	  return VAR ; }
#define FUNC1(TYPE,NAME,T1,A1,F1,FMT,VAR) \
	TYPE NAME(T1 A1) { \
	  if (GATOSCALL) { \
	    fprintf(stderr,"GATOS: %s(",#NAME) ; \
	    fprintf(stderr,F1,A1) ; \
	    fprintf(stderr,") = ") ; \
	    fprintf(stderr,FMT,VAR) ; fprintf(stderr,"\n") ; } \
	  return VAR ; }

#define gerr(FMT,ARGS...)	fprintf(stderr,FMT,##ARGS)
#define gout(FMT,ARGS...)	printf(FMT,##ARGS) ; fflush(stdout)

/* ------------------------------------------------------------------------ */

void gatos_enter(char *func, char *fmt, ...) {
  int i ; va_list args ;
  fflush(stdout) ;
  if (GATOSCALL && gatos.pending) fprintf(stderr,"\n") ;
  gatos.level++ ; gatos.pending = 0 ;
  if (gatos.level>MAXLEVELS) {	/* This should not happen */
    if (!gatos.debug&GATOSQUIET)
      fprintf(stderr,"GATOS: gatos_enter(): Fatal: MAXLEVELS exceeded\n") ;
    exit(1) ; }
  gatoscalls[gatos.level] = func ;
  if (!GATOSCALL) return ;
  gatos.pending = 1 ;
  fprintf(stderr,"GATOS:") ;
  for ( i=0 ; i<gatos.level ; i++ ) fprintf(stderr," ") ;
  fprintf(stderr,"%s(",func) ;
  va_start(args, fmt) ;
  if (strlen(fmt)) vfprintf(stderr,fmt,args) ;
  va_end(args) ;
  fprintf(stderr,")") ; }

int gatos_leave(int err) {
  int i, j ;
  if (GATOSCALL) {
    if (err) {
      if (!gatos.pending) {
        fprintf(stderr,"GATOS:") ;
        for ( i=0 ; i<gatos.level ; i++ ) fprintf(stderr," ") ;
        fprintf(stderr,"%s()",gatoscalls[gatos.level]) ; }
      fprintf(stderr,": %s\n",strerror(err)) ; }
    else if (gatos.pending) fprintf(stderr,"\n") ; }
  else if (err && !(gatos.debug&GATOSQUIET)) {
    for ( j=1 ; j<gatos.level ; j++ ) {
      fprintf(stderr,"GATOS:") ;
      for ( i=0 ; i<j ; i++ ) fprintf(stderr," ") ;
      fprintf(stderr,"%s()\n",gatoscalls[j]) ; }
    fprintf(stderr,"GATOS:") ;
    for ( i=0 ; i<gatos.level ; i++ ) fprintf(stderr," ") ;
    fprintf(stderr,"%s(): %s\n",gatoscalls[gatos.level],strerror(err)) ; }
  gatoscalls[gatos.level] = NULL ;
  gatos.level-- ; gatos.pending = 0 ;
  if (gatos.level<0) {
    gatos.level = 0 ; if (!gatos.debug&GATOSQUIET) fprintf(stderr,
      "GATOS: gatos_leave(): Failure to call gatos_enter() on entry\n") ; }
  if (err) return errno = err ; else return 0 ; }

FUNC0(int,gatos_verbosity,"%d",gatos.debug)
void gatos_setverbosity(int value) {
  gatos.debug = debug = value ; }

FUNC0(int,gatos_nosa,"%d",gatos.nosa)
int gatos_setnosa(int value) {
  gatos.nosa = nosa = value ; return 0 ; }

int gatos_setpci(char *addr) {
  pciaddr = addr ; return 0 ; }

/* ------------------------------------------------------------------------ */

int gatos_init(void) {

  int i ; char buf[256], *bufp ; FILE *file ;
  struct sigaction act ;
  uid_t uid, euid ;

  if (VERBOSE) {
    fprintf(stderr, "GATOS version %s\n", GATOS_VERSION_STRING) ;
    fprintf(stderr, "Homepage: http://www.core.binghamton.edu/~insomnia/gatos/\n") ; }

  /* Zero out all fields in struct gatos: Init charstrings to *
   * zero length, ints (all sizes) to zero, pointers to NULL. */
  memset(&gatos,0,sizeof(gatos)) ;
  /* Init fields whose bitpattern is not all zeros */
  gatos.dotclock = 0.0 ; gatos.overclock = 0.0 ;
  gatos.minfreq = 0.0 ; gatos.freq = 0.0 ;
  gatos.maxfreq = 0.0 ; gatos.Fpc = 0.0 ;
  gatos.mux = 2 ; gatos.brightness = 100 ;
  gatos.contrast = 100 ; gatos.saturation = 100 ;
  gatos.audiotype = 1 ; gatos.refclock = 14.31818 ;
  /* Set remembered debug and nosa values */
  gatos.debug = debug ; gatos.nosa = nosa ;
  if (gatos.debug&GATOSQUIET) gatos.debug = GATOSQUIET ;

  /* Must be called *after* init of struct gatos ;-! */
  gatos_enter("gatos_init","") ;
  gatoscalls[0] = " (internal error in GATOS)" ;
  /* gatos_init() is special ... */
  gatos.pending = 0 ; if (GATOSCALL) gerr("\n") ;

  /* Must run as root */
  if (geteuid()) RETURN(EACCES) ;

#ifdef GATOSBUTTONS 
  for ( i=0 ; i<GATOSBUTTONS ; i++ ) gatosbuttons[i] = 0 ;
#endif

  /* Read and parse systemwide config file */
  SUBCALL(gatos_configfile) ;

  /* Command line option "-pci addr" overrides config file */
  if (pciaddr) gatos.pciaddr = pciaddr ;
  if (!gatos.pciaddr) gatos.pciaddr = "" ;

  /* Open /dev/mem for access to real memory (also used by pci_scan()) */
  if ( (gatos.memfd = open("/dev/mem",O_RDWR)) == -1) {
    perror("GATOS: open(/dev/mem) failed") ; RETURN(errno) ; }

  /* Scan all PCI and AGP busses for ATI cards */
  SUBCALL(pci_scan) ;

  /* If only one ATI card, choose it (ignore gatos.conf pciaddr ;-) */
  /* Match gatos.conf pciaddr if more than one card. */
  if (!gatos.aticards) {
    fprintf(stderr,"GATOS: No ATI PCI/AGP Cards ?\n") ; RETURN(EINVAL) ; }
  else if (gatos.aticards==1) gatos.cardidx = 0 ;
  else for ( i=0 ; i<gatos.aticards ; i++ )
    if (!strcmp(gatos.pciaddr,gatos.aticard[i].pciaddr)) gatos.cardidx = i ;

  /* Set card info */
  memcpy(&gatos.ati,&gatos.aticard[gatos.cardidx],sizeof(gatos.ati)) ;
  gatos.fbaddr = gatos.ati.fbaddr ; gatos.regaddr = gatos.ati.regaddr ;
  gatos.biosaddr = gatos.ati.biosaddr ; gatos.refclock = gatos.ati.refclock ;

  /* Verbose card info */
  if (VERBOSE) for ( i=0 ; i<gatos.aticards ; i++ ) {
    gout("%s card @ %s: ATI %s (%s)%s\n",
      (gatos.aticard[i].bus)?"AGP":"PCI",
      gatos.aticard[i].pciaddr,gatos.aticard[i].name,
      gatos.aticard[i].ident,(i == gatos.cardidx)?":":"") ;
    if (i != gatos.cardidx) continue ;
    gout("==> FrameBuffer @ 0x%08X, Registers @ 0x%08X",
      gatos.fbaddr, (gatos.regaddr==0xFFFFFFFF)
        ? gatos.fbaddr+0x007FF800 : gatos.regaddr) ;
    if (gatos.biosaddr != 0xFFFFFFFF) {
      gout(", BIOS @ 0x%08X",gatos.biosaddr) ; }
    gout("\n==> RefClock %f MHz",gatos.refclock) ;
    gout(", Max Scaler Clock %.0f MHz",gatos.ati.dotclock) ;
    gout(", Max Scaler Width %d\n",gatos.ati.width) ; }

  /* Card not found ? */
  if (gatos.cardidx == -1) {
    fprintf(stderr,"GATOS: No ATI PCI/AGP Card @ %s\n",gatos.pciaddr) ;
    RETURN(EINVAL) ; }

  /* Sanity check */
  if (gatos.fbaddr == 0xFFFFFFFF) {
    fprintf(stderr,"GATOS: FrameBuffer Aperture not mapped\n") ;
    RETURN(ENXIO) ; }
  if ( !(gatos.fbaddr & 0xFFC00000) || (gatos.fbaddr & 0x003FFFFF) )
    INSANE(fbaddr) ;
  if (gatos.regaddr != 0xFFFFFFFF &&
      (!(gatos.regaddr & 0xFFFFF000) || (gatos.regaddr & 0x00000FFF)) )
    INSANE(regaddr) ;
  if ( (gatos.videoram < 2048) || (gatos.videoram > 32768) ) INSANE(videoram) ;
  if ( (gatos.buffermem < 300) || (gatos.buffermem > 2048) ) INSANE(buffermem) ;

  /* Memory map ATI FrameBuffer Aperture */
  if ( (int) (ATIFB = (u8*) mmap(0,0x01000000,PROT_READ|PROT_WRITE,MAP_SHARED,
                                 gatos.memfd,gatos.fbaddr)) == -1 ) {
    perror("GATOS: mmap() failed") ; RETURN(errno) ; }

  /* Memory map ATI Register Aperture */
  if (gatos.regaddr == 0xFFFFFFFF) MEM_1 = (u32*)(ATIFB+0x007FF800) ;
  else if ( (int) (MEM_1 = (u32*) mmap(0,4096,PROT_READ|PROT_WRITE,MAP_SHARED,
                                  gatos.memfd,gatos.regaddr)) == -1 ) {
    perror("GATOS: mmap() failed") ; RETURN(errno) ; }
  MEM_0 = MEM_1+256 ;

  /* Setuid(root) privileges no longer needed; give them up for good :-) */
  uid = getuid() ; euid = geteuid() ;
  if (euid != uid && setuid(uid)) { perror("setuid() failed") ; exit(0) ; }

  /* Initalize ATI chip */
  SUBCALL(ati_init) ;

  /* Initalize ImpacTV chip */
  SUBCALL(tvout_init) ;

  /* CPU BogoMIPS value, used by delay.S or i2c_delay() */
  if ( (file=fopen("/proc/cpuinfo","r")) != NULL ) {
    while (fgets(buf,sizeof(buf)-1,file)) {
      if (strncmp(buf,"bogomips",8)) continue ; bufp = buf ;
      while (*bufp && (*bufp != ':') && (*bufp != '\n')) bufp++ ;
      if (*bufp != ':') continue ; bufp++ ;
      while (*bufp == ' ') bufp++ ;
      gatos.bogomips = (int) (atof(bufp)+0.5) ; break ; }
    fclose(file) ;
    if (VERBOSE) gout("CPU speed is %d BogoMIPS\n",gatos.bogomips) ; }
  else {
    /* Roughly a PentiumII/2GHz, should be large enough for a few months ;-) */
    perror("GATOS: Couldn't open /proc/cpuinfo") ; gatos.bogomips = 2000 ;
    gerr("GATOS: Warning: CPU speed assumed to be %d BogoMIPS\n",
      gatos.bogomips) ; }
#ifdef USE_DELAY_S
  /* Used by delay.S */
  loops_per_sec = gatos.bogomips * 500000 ;
#endif

  /* Initialize I2C Driver */
  SUBCALL(i2c_init) ;
  if (VERBOSE && gatos.bugfix_mppread) {
    gout("I2C: MPP read bugfix value is 0x%02X\n",gatos.bugfix_mppread) ; }

  /* Initalize I2C dependant submodules */
  SUBCALL(board_init) ; SUBCALL(bt829_init) ; SUBCALL(fi12xx_init) ;
  SUBCALL(sound_init) ; SUBCALL(saa5281_init) ;

  /* Remaining submodules */
  SUBCALL(chan_init) ;

  /* Load ~/.gatosrc */
  snprintf(gatos.statename, sizeof(gatos.statename),
           "%s/.gatosrc", getenv("HOME")) ;
  if (gatos_loadstate(gatos.statename)) {
    /* It failed, use format from gatos.conf */
    if (gatos.format) gatos_setformat(gatos.format) ;
    else switch (gatos.tunertype) { /* ... or derive it from tuner type */
      case 2:         gatos_setformat(2) ; break ;	/* NTSC Japan */
      case 3: case 4: gatos_setformat(3) ; break ;	/* PAL */
      case 5: case 7: gatos_setformat(6) ; break ;	/* SECAM */
      case 1: case 6: case 8: case 12: case 0x12: default:/* NTSC M/N */
        gatos_setformat(1) ; break ; } }

  /* Correct amount of video memory ? */
  /* If no problems encountered, chip setting can be used to autodetect /AA */
  if (gatos.atiram != gatos.videoram) {
    gerr("%d kb of Video RAM specified in gatos.conf, ATI chip says %d kb\n",
      gatos.videoram,gatos.atiram) ; }

  /* We're initialized */
  init = 1 ;

  /* Desktop panning and mode change support (hackish, but it works ;-) */
  act.sa_flags = 0 ; act.sa_restorer = NULL ; sigemptyset(&act.sa_mask) ;
  act.sa_handler = &gatos_pollscreen ; sigaction(SIGUSR1, &act, NULL) ;
  if ( (gatos.usr1pid=fork()) == -1) {
    gerr("Warning: Desktop panning and Modechange support disabled:\n") ;
    gerr("Warning: Fork() failed: %s\n",strerror(errno)) ;
    gatos.usr1pid = 0 ; RETURN(0) ; }
  if (gatos.usr1pid) RETURN(0) ;

  /* Child process, sends approx. 50 SIGUSR1's per sec. */
  /* Note: Surround *all* sleep() and usleep() calls with stop (SIGSTOP) *
   * and restart (SIGCONT) of this child process, or the u?sleep()       *
   * can end prematurely with EINTR (interrupted system call).      /AA */
  close(0) ; close(1) ; close(2) ;
  while (1) { usleep(10000) ; if (kill(getppid(),SIGUSR1)) exit(0) ; }
  exit(0) ; }

/* ------------------------------------------------------------------------ */

int gatos_terminate(void) {
  gatos_enter("gatos_terminate","") ;
  if (!init) RETURN(0) ;
  gatos_enable_sound(0) ;
  gatos_enable_video(0) ;
  gatos_enable_capture(0) ;
  gatos_setCC(0) ;
  sound_restore() ;
  gatos_savestate(gatos.statename) ;
  if (gatos.usr1pid) kill(gatos.usr1pid,SIGKILL) ;
  init = 0 ; RETURN(0) ; }

/* ------------------------------------------------------------------------ */

static int ival(char *name, char **str, int min, int max) {
  char *tok = strsep(str," \t\n"), *endp ; int value = strtol(tok,&endp,0) ;
  if (*tok==0 || *endp!=0)
    gerr("gatos: Invalid value '%s' for '%s'.\n",tok,name) ;
  if (value < min) { gerr(
    "gatos: %s value '%d' too low, setting to %d.\n",name,value,min) ;
     value = min ; }
  if (value > max) { gerr(
    "gatos: %s value '%d' too high, setting to %d.\n",name,value,min) ;
     value = max ; }
  return value ; }

static char *sval(char *name, char **str, char *min, char *max) {
  char *tok, *buf ;
  if (**str == '\'') { (*str)++ ; tok = strsep(str,"'\n") ; }
  else {
    tok = strsep(str,"\n") ; buf = tok+strlen(tok)-1 ;
    while (buf >= tok && index(" \t",*buf)) *buf-- = 0 ; }
  buf = malloc(strlen(tok)+1) ; if (buf) strcpy(buf,tok) ; return buf ; }

#define TOKEN(NAME,VAR,FUNC,MIN,MAX) \
  if (!strcmp(tok,NAME)) { VAR = FUNC(tok,&bufp,MIN,MAX) ; continue ; }

int gatos_configfile(void) {

  FILE *file ; char buf[256], *tok, *bufp, *endp ; int instr ;

  gatos_enter("gatos_configfile","") ;

  if ((file=fopen(CONFIGFILE,"r")) == NULL) {
    perror("fopen(config) failed") ; RETURN(errno) ; }

  while (fgets(buf,sizeof(buf)-1,file)) {

    bufp = buf ; while (index(" \t",*bufp)) bufp++ ;
    endp = bufp+strlen(bufp) ;
    if (!index(buf,'\n')) { *endp++ = '\n' ; *endp = 0 ; }

    instr = 0 ; tok = bufp ;
    while (tok < endp) {
      tok += strcspn(tok,(instr)?"'":"'#") ;
      if (*tok == '\'') { tok++ ; instr = 1-instr ; continue ; }
      if (*tok == '#') { *tok++ = '\n' ; *tok = 0 ; endp = tok ; break ; } }

    tok = strsep(&bufp," \t\n") ;
    while (index(" \t",*bufp)) bufp++ ;

    if (*bufp=='\n') {
      gerr("gatos: Missing token after '%s'.\n",tok) ; continue ; }

    TOKEN("videoram",gatos.videoram,ival,2048,32768) ;
    TOKEN("buffermem",gatos.buffermem,ival,300,2048) ;
    TOKEN("format",gatos.format,ival,1,7) ;
    TOKEN("i2c_mode",gatos.i2c_mode,ival,1,5) ;
    TOKEN("pciaddr",gatos.pciaddr,sval,"","") ; }

  fclose(file) ; RETURN(0) ; }

/* ------------------------------------------------------------------------ */

int gatos_loadstate(char *value) {
  FILE *cfg ; char buf[256], *ln = buf ;
  gatos_enter("gatos_loadstate","\"%s\"",value) ;
  for(gatos.esn=0; gatos.esn<256; gatos.esn++)
	gatos.extrastring[gatos.esn] = NULL;
  gatos.esn = 0;
  cfg = fopen(value, "r") ;
  if (cfg != NULL) {
    int *var = NULL ;
    ln = fgets(buf, 256, cfg) ;
    while (ln != NULL) {
      if (!strncmp(ln, "ALLCHANNELS", 11)) {
        char *freqstr, *descstr ;
        int ch, vol, br, cnt, hue, sat ;
        ln = fgets(buf, 256, cfg) ;
        while(ln != NULL && strncmp("EOF", ln, 3)) {
          freqstr = ln ; while(*ln == ' ' || *ln == '\t') ++ln ;
          while (*ln != ' ' && *ln != '\t' && *ln != '\n' && *ln != 0) ++ln ;
          *ln = 0 ; ++ln ;
	  vol = strtol(ln, &ln, 10) ;
	  br = strtol(ln, &ln, 10) ;
	  cnt = strtol(ln, &ln, 10) ;
	  hue = strtol(ln, &ln, 10) ;
	  sat = strtol(ln, &ln, 10) ;
	  strtol(ln, &ln, 10) ;	/* Future expansion support */
	  strtol(ln, &ln, 10) ; /* Up to 3 more Numbers */
	  strtol(ln, &ln, 10) ; /* Will be ignores */
          while (*ln == ' ' || *ln == '\t') ++ln ;
	  descstr = ln ;
          while (*ln != '\n' && *ln != 0) ln++ ; *ln = 0 ;
          ch = chan_addchan(strtod(freqstr, NULL), descstr) ;
	  chan_setvolume(ch, vol) ;
	  chan_setbrightness(ch, br) ;
	  chan_setcontrast(ch, cnt) ;
	  chan_sethue(ch, hue) ;
	  chan_setsaturation(ch, sat) ;
          ln = fgets(buf, 256, cfg) ; } }
      else if(!strncmp(ln, "Mute=", 5)) var = &gatos.mute ;
      else if(!strncmp(ln, "SAP=", 4)) var = &gatos.sap ;
      else if(!strncmp(ln, "Stereo=", 7)) var = &gatos.stereo ;
      else if(!strncmp(ln, "Format=", 7)) var = &gatos.format ;
      else if(!strncmp(ln, "Brightness=", 11)) var = &gatos.brightness ;
      else if(!strncmp(ln, "Hue=", 4)) var = &gatos.hue ;
      else if(!strncmp(ln, "Saturation=", 7)) var = &gatos.saturation ;
      else if(!strncmp(ln, "Cold=", 5)) var = &gatos.cold ;
      else if(!strncmp(ln, "Gamma=", 6)) var = &gatos.gamma ;
      else if(!strncmp(ln, "Volume=", 7)) var = &gatos.volume ;
      else if(!strncmp(ln, "Mixer=", 6)) var = &gatos.mixerdev ;
      else if(!strncmp(ln, "Input=", 6)) var = &gatos.mux ;
      else if(!strncmp(ln, "Channel=", 8)) var = &gatos.channel ;
      else if(!strncmp(ln, "Contrast=", 8)) var = &gatos.contrast ;
/* Don't want to save this */
/*		it'll prevent sound from working if it gets fixed! -ISG */
/*      else if(!strncmp(ln, "AudioType=", 10)) var = &gatos.audiotype ; */
		/* audiotype: 0-none, 1-soundcard, 2-board, 3-both */
      else if(!strncmp(ln, "Overclock=", 10)) var = &gatos.overclock ;
      else if(!strncmp(ln, "Sensitive=", 10)) var = &gatos.senscan ;
      else {
	if(ln[strlen(ln)-1] == '\n') ln[strlen(ln)-1] = 0;
	gatos.extrastring[gatos.esn] = (char*)malloc(strlen(ln)+1);
	strcpy(gatos.extrastring[gatos.esn], ln);
/*
	printf("Unclaimed String in ~/.gatosrc = \"%s\"\n",
		gatos.extrastring[gatos.esn]);
*/
	++gatos.esn;
	}
    
      if (var != NULL) {
        char *chr=ln ;
        while (*chr != '=' && *chr != 0 && *chr != '\n') ++chr ;
        if (*chr != 0 && *chr != '\n') chr++ ;
        if (chr != 0) *var = strtol(chr, NULL, 10) ;
        var = NULL ; }  
      ln = fgets(buf, 256, cfg) ; }
    fclose(cfg) ;
    gatos_setformat(gatos.format) ;
    gatos_setmux(gatos.mux) ;
    gatos_setmute(gatos.mute) ;
    gatos_setsap(gatos.sap) ;
    gatos_setstereo(gatos.stereo) ;
    gatos_setbrightness(gatos.brightness) ;
    gatos_setcontrast(gatos.contrast) ;
    gatos_setsaturation(gatos.saturation) ;
    gatos_sethue(gatos.hue) ;
    gatos_setcold(gatos.cold) ;
    gatos_setgamma(gatos.gamma) ;
    gatos_setmixer(gatos.mixerdev) ;
    gatos_setvolume(gatos.volume) ;
    gatos_setchan(gatos.channel) ;
    gatos_setoverclock(gatos.overclock) ;
    RETURN(0) ; }
  else {
    if (!gatos.debug&GATOSQUIET)
      gerr( "WARNING: No ~/.gatosrc file, using default setup.\n");
    return(1) ; } }

int gatos_getintvalue(char *n, int *v) {
  int ctr;
  for(ctr=0; ctr<gatos.esn; ++ctr) {
    if(!strncmp(gatos.extrastring[ctr], n, strlen(n))) {
      *v = strtol(&gatos.extrastring[ctr][strlen(n)], NULL, 10);
      return 0;
      }
    }
  return -1;
  }

int gatos_getstringvalue(char *n, char *v, int m) {
  int ctr;
  for(ctr=0; ctr<gatos.esn; ++ctr) {
    if(!strncmp(gatos.extrastring[ctr], n, strlen(n))) {
      if(strlen(gatos.extrastring[ctr])-strlen(n) >= m) {
	fprintf(stderr, "String token \"%s\" to large!\n",
	  gatos.extrastring[ctr]);
	return -1;
	}
      strcpy(v, &gatos.extrastring[ctr][strlen(n)]);
      return 0;
      }
    }
  return -1;
  }

int gatos_setintvalue(char *n, int v) {
  int ctr;
  for(ctr=0; ctr<gatos.esn; ++ctr) {
    if(!strncmp(gatos.extrastring[ctr], n, strlen(n))) {
      free(gatos.extrastring[ctr]);
      gatos.extrastring[ctr] = (char*)malloc(strlen(n)+20);
      sprintf(gatos.extrastring[ctr], "%s%d%c", n, v, 0);
      return 0;
      }
    }
  gatos.extrastring[ctr] = (char*)malloc(strlen(n)+20);
  sprintf(gatos.extrastring[ctr], "%s%d%c", n, v, 0);
  ++gatos.esn;
  return 0;
  }

int gatos_setstringvalue(char *n, char *v) {
  int ctr;
  for(ctr=0; ctr<gatos.esn; ++ctr) {
    if(!strncmp(gatos.extrastring[ctr], n, strlen(n))) {
      free(gatos.extrastring[ctr]);
      gatos.extrastring[ctr] = (char*)malloc(strlen(n)+strlen(v)+2);
      sprintf(gatos.extrastring[ctr], "%s%s%c", n, v, 0);
      return 0;
      }
    }
  gatos.extrastring[ctr] = (char*)malloc(strlen(n)+strlen(v)+2);
  sprintf(gatos.extrastring[ctr], "%s%s%c", n, v, 0);
  ++gatos.esn;
  return 0;
  }

/* ------------------------------------------------------------------------ */

int gatos_savestate(char *value) {
  int ctr ; FILE *cfg ;
  gatos_enter("gatos_savestate","\"%s\"",value) ;
  cfg = fopen(value, "w") ;
  if (cfg != NULL) {
    fprintf(cfg, "Format=%d\n", gatos.format) ;
    fprintf(cfg, "Mixer=%d\n", gatos.mixerdev) ;
    fprintf(cfg, "Brightness=%d\n", gatos.brightness) ;
    fprintf(cfg, "Hue=%d\n", gatos.hue) ;
    fprintf(cfg, "Saturation=%d\n", gatos.saturation) ;
    fprintf(cfg, "Contrast=%d\n", gatos.contrast) ;
    fprintf(cfg, "Cold=%d\n", gatos.cold) ;
    fprintf(cfg, "Gamma=%d\n", gatos.gamma) ;
    fprintf(cfg, "Volume=%d\n", gatos.volume) ;
    fprintf(cfg, "Mute=%d\n", gatos.mute) ;
    fprintf(cfg, "SAP=%d\n", gatos.sap) ;
    fprintf(cfg, "Stereo=%d\n", gatos.stereo) ;
    fprintf(cfg, "Input=%d\n", gatos.mux) ;
    fprintf(cfg, "Channel=%d\n", gatos.channel) ;
/*    fprintf(cfg, "AudioType=%d\n", gatos.audiotype) ; */
    fprintf(cfg, "Overclock=%d\n", gatos.overclock) ;
    fprintf(cfg, "Sensitive=%d\n", gatos.senscan) ;
    for(ctr=0; ctr<gatos.esn; ctr++)
      fprintf(cfg, "%s\n", gatos.extrastring[ctr]);
    fprintf(cfg, "ALLCHANNELS\n") ;
    for ( ctr=0 ; ctr<gatos.numchannels ; ctr++ )
      fprintf(cfg,"%6.2f\t%d\t%d\t%d\t%d\t%d\t%s\n",
	gatos.channelinfo[ctr].frequency,
	gatos.channelinfo[ctr].mod_volume,
	gatos.channelinfo[ctr].mod_brightness,
	gatos.channelinfo[ctr].mod_contrast,
	gatos.channelinfo[ctr].mod_hue,
	gatos.channelinfo[ctr].mod_saturation,
        gatos.channelinfo[ctr].name) ;
    fprintf(cfg, "EOF\n") ; fclose(cfg) ; RETURN(0) ; }
  else RETURN(errno) ; }

/* ------------------------------------------------------------------------ */

FUNC0(int,gatos_format,"%d",gatos.format)
int gatos_setformat(int value) {

  int i, h, v, m=1024*gatos.captbufsize ;

  gatos_enter("gatos_setformat","%d",value) ;
  if (gatos.pid) RETURN(EBUSY) ;

  if ( (value < 1) || (value > 7) ) RETURN(EINVAL) ; gatos.format = value ;
  switch (gatos.format) {
    case 1: case 2: case 4:					/* NTSC */
      gatos.minfreq = 55.25 ; gatos.maxfreq = 801.25 ;
      hactive = hactive_ntsc ; vactive = vactive_ntsc ; break ;
    case 3: case 5: case 6: case 7:				/* PAL/SECAM */
      gatos.minfreq = 48.25 ; gatos.maxfreq = 855.25 ;
      hactive = hactive_pal  ; vactive = vactive_pal  ; break ;
    default: RETURN(EINVAL) ; }

  /* Choose maximum capture geometry that will fit in capture buffers */
  /* Also limit horizontal capture size to ATI chip limit */

  i = sizeof(capturesizes)/sizeof(char*) - 1 ;
  while (i && (hactive[i]*vactive[i] > m)) i-- ;
  while (i && (hactive[i] > gatos.ati.width)) i-- ;
  gatos.hcaptidx = gatos.hcaptmax = i ; h = hactive[i] ;

  i = sizeof(capturesizes)/sizeof(char*) - 1 ;
  while (i && (h*vactive[i] > m)) i-- ;
  gatos.vcaptidx = gatos.vcaptmax = i ; v = vactive[i] ;

  /* Set default capture geometry to max */
  gatos.xcapt = h ; gatos.ycapt = v ;

  /* Default window size is same */
  if (!gatos.xsize && !gatos.ysize) {
    gatos.xsize = gatos.xcapt ; gatos.ysize = gatos.ycapt ; }

  /* Program the chips */
  SUBCALL(bt829_setformat) ;
  fi12xx_setformat() ;
  gatos_setcapturesize(gatos.xcapt,gatos.ycapt) ;
  if (gatos.donegeometry)
    gatos_setgeometry(gatos.xsize,gatos.ysize,gatos.xpos,gatos.ypos) ;
  RETURN(0) ; }

FUNC0(int,gatos_mux,"%d",gatos.mux)
int gatos_setmux(int value) {
  gatos_enter("gatos_setmux","%d",value) ;
  if (gatos.pid) RETURN(EBUSY) ;
  gatos.mux = LIMIT(value,0,3) ;
  bt829_setmux() ; board_setmux() ; RETURN(0) ; }

FUNC0(u32,gatos_colorkey,"%d",gatos.colorkey)
int gatos_setcolorkey(u32 value) {
  gatos_enter("gatos_setcolorkey","0x%08X",value) ;
  gatos.colorkey = value ; err = ati_setcolorkey() ; RETURN(err) ; }

void gatos_pollscreen(int sig) {
  if (ati_pollscreen(sig)) ati_setgeometry() ; }

int gatos_setgeometry(int xs, int ys, int x0, int y0) {
  gatos_enter("gatos_setgeometry","%d,%d,%d,%d",xs,ys,x0,y0) ;
  gatos.xsize = xs ; gatos.ysize = ys ; gatos.xpos = x0 ; gatos.ypos = y0 ;
  gatos.donegeometry = 1 ;
  ati_setgeometry() ; RETURN(0) ; }

int gatos_setcapturesize(int xc, int yc) {
  gatos_enter("gatos_setcapturesize","%d,%d",xc,yc) ;
  if (gatos.pid) RETURN(EBUSY) ;
  gatos.xcapt = xc ; gatos.ycapt = yc ;
  SUBCALL(ati_setcaptsize) ; SUBCALL(bt829_setcaptsize) ;
  if (gatos.donegeometry)
    gatos_setgeometry(gatos.xsize,gatos.ysize,gatos.xpos,gatos.ypos) ;
  RETURN(0) ; }

FUNC0(int,gatos_xcaptmax,"%d",hactive[gatos.hcaptmax]) ;
FUNC0(int,gatos_ycaptmax,"%d",vactive[gatos.vcaptmax]) ;
FUNC0(int,gatos_xcaptmin,"%d",hactive[0]) ;
FUNC0(int,gatos_ycaptmin,"%d",vactive[0]) ;
FUNC0(int,gatos_xcapt,"%d",gatos.xcapt) ;
FUNC0(int,gatos_ycapt,"%d",gatos.ycapt) ;
FUNC0(int,gatos_xdim,"%d",gatos.xdim) ;
FUNC0(int,gatos_ydim,"%d",gatos.ydim) ;

int gatos_enable_capture(int value) {
  gatos_enter("gatos_enable_capture","%d",value) ;
  if (gatos.pid) RETURN(EBUSY) ;
  ati_enable_capture(value) ; RETURN(0) ; }

int gatos_enable_video(int value) {
  gatos_enter("gatos_enable_video","%d",value) ;
  gatos.video = value ;
  ati_enable_video(gatos.mapped && gatos.video && (gatos.visibility!=2)) ;
  RETURN(0) ; }

int gatos_enable_sound(int value) {
  gatos_enter("gatos_enable_sound","%d",value) ;
  gatos.audio = value ; board_setmute() ; RETURN(0) ; }

int gatos_setmapped(int value) {
  gatos_enter("gatos_setmapped","%d",value) ;
  gatos.mapped = value ;
  ati_enable_video(gatos.mapped && gatos.video && gatos.visibility!=2) ;
  RETURN(0) ; }

int gatos_setvisibility(int state) {
  gatos_enter("gatos_setvisibility","%d",state) ;
  gatos.visibility = state ;
  ati_enable_video(gatos.mapped && gatos.video && (gatos.visibility!=2)) ;
  ati_setgeometry() ; RETURN(0) ; }

/* ------------------------------------------------------------------------ */

FUNC0(int,gatos_mute,"%d",gatos.mute)
int gatos_setmute(int value) {
  int v ;
  gatos_enter("gatos_setmute","%d",value) ;
  gatos.mute = value ; v = (gatos.mute) ? 0 : gatos.volume ;
  sound_setvol(v) ; board_setmute() ; RETURN(0) ; }

FUNC0(int,gatos_volume,"%d",gatos.volume)
int gatos_setvolume(int value) {
  int v ;
  gatos_enter("gatos_setvolume","%d",value) ;
  gatos.volume = LIMIT(value,0,100) ;
  v = (gatos.mute) ? 0 :
    LIMIT(gatos.volume + gatos.channelinfo[gatos.channel].mod_volume,0,100) ;
  sound_setvol(v) ; board_setvol(v) ; RETURN(0) ; }

FUNC0(int,gatos_stereo,"%d",gatos.stereo)
int gatos_setstereo(int value) {
  gatos_enter("gatos_setstereo","%d",value) ;
  gatos.stereo = value ; board_setstereo() ; RETURN(0) ; }

FUNC0(int,gatos_sap,"%d",gatos.sap)
int gatos_setsap(int value) {
  gatos_enter("gatos_setsap","%d",value) ;
  gatos.sap = value ; board_setsap() ; RETURN(0) ; }

/* ------------------------------------------------------------------------ */

FUNC0(double,gatos_minfreq,"%.3f",gatos.minfreq)
FUNC0(double,gatos_maxfreq,"%.3f",gatos.maxfreq)
FUNC0(double,gatos_tuner,"%.3f",gatos.freq)
int gatos_settuner(double value) {
  int mute=gatos.audio ;
  gatos_enter("gatos_settuner","%.3f",value) ;
  if (gatos.pid) RETURN(EBUSY) ;
  if (gatos.usr1pid) kill(gatos.usr1pid,SIGSTOP) ;
  gatos.freq = value ; gatos.tuned = 1 ; gatos.audio = 0 ;
  sound_setvol(0) ; fi12xx_tune() ; usleep(100000) ;
  gatos.audio = mute ; sound_setvol(gatos.volume) ; board_setmute() ;
  if (gatos.usr1pid) kill(gatos.usr1pid,SIGCONT) ; RETURN(0) ; }

double gatos_scan(double min, double max,
  int (*addchan)(double, char*), int (*progress)(double)) {
  return fi12xx_scan(min,max,addchan,progress) ; }

/* ------------------------------------------------------------------------ */

FUNC0(int,gatos_brightness,"%d",gatos.brightness)
int gatos_setbrightness(int value) {
  gatos_enter("gatos_setbrightness","%d",value) ;
  gatos.brightness = LIMIT(value,0,200) ; err = bt829_setbrightness() ;
  RETURN(err) ; }

FUNC0(int,gatos_contrast,"%d",gatos.contrast)
int gatos_setcontrast(int value) {
  gatos_enter("gatos_setcontrast","%d",value) ;
  gatos.contrast = LIMIT(value,0,200) ; err = bt829_setcontrast() ;
  RETURN(err) ; }

FUNC0(int,gatos_saturation,"%d",gatos.saturation)
int gatos_setsaturation(int value) {
  gatos_enter("gatos_setsaturation","%d",value) ;
  gatos.saturation = LIMIT(value,0,200) ; err = bt829_setsaturation() ;
  RETURN(err) ; }

FUNC0(int,gatos_hue,"%d",gatos.hue)
int gatos_sethue(int value) {
  gatos_enter("gatos_sethue","%d",value) ;
  gatos.hue = LIMIT(value,-90,90) ; err = bt829_sethue() ;
  RETURN(err) ; }

FUNC0(int,gatos_gamma,"%d",gatos.gamma)
int gatos_setgamma(int value) {
  gatos_enter("gatos_setgamma","%d",value) ;
  gatos.gamma = LIMIT(value,0,3) ; err = ati_setgamma() ;
  RETURN(err) ; }

FUNC0(int,gatos_cold,"%d",gatos.cold)
int gatos_setcold(int value) {
  gatos_enter("gatos_setcold","%d",value) ;
  gatos.cold = LIMIT(value,0,1) ; err = ati_setcold() ;
  RETURN(err) ; }

FUNC0(int,gatos_mixer,"%d",gatos.mixerdev)
int gatos_setmixer(int value) {
  gatos_enter("gatos_setmixer","%d",value) ;
  if (sound_setmixer(value)) gatos.audiotype &= 0xfe ;
  gatos_setvolume(gatos.volume) ; RETURN(0) ; }

int gatos_nummixers() {
  if (GATOSCALL) fprintf(stderr,"GATOS: gatos_nummixers()\n");
  return(sound_nummixers()); }

char* gatos_mixername(int value) {
  gatos_enter("gatos_mixername","%d",value) ;
  return(sound_mixername(value)); }

char** gatos_mixernames() {
  if (GATOSCALL) fprintf(stderr,"GATOS: gatos_mixernames()\n");
  return(sound_mixernames()); }

/* ------------------------------------------------------------------------ */

FUNC0(int,gatos_CC,"%d",gatos.CCmode)
int gatos_setCC(int value) {
  gatos_enter("gatos_setCC","%d",value) ;
  gatos.CCmode = LIMIT(value,0,3) ; err = bt829_setCC() ;
  RETURN(err) ; }

int gatos_getCCdata(struct CCdata *data) {
  gatos_enter("gatos_getCCmode","%p",data) ;
  if(data==NULL) {
    err=EINVAL; RETURN(err); }
  if(gatos.CCmode == 0) {
    data->num_valid=0; err=EPERM; RETURN(err); }
  err = bt829_getCCdata(data) ;
  RETURN(err) ; }



/* ------------------------------------------------------------------------ */

static FILE *file ;
static struct sigaction act ;

static void gatos_stopchild(int sig) { gatos.stop = 1 ; }

#if 0 /* NIY */
static void gatos_sigchild(int sig) {
  /* TODO: Restore state */
}
#endif

int gatos_stop(void) {
  gatos_enter("gatos_stop","") ;
  if (!gatos.pid) RETURN(ECHILD) ;
  kill(gatos.pid,SIGUSR1) ; gatos.pid = 0 ;
  RETURN(0) ; }

int gatos_capture(char *name, int wait) {

  gatos_enter("gatos_capture","\"%s\",%d",name,wait) ;

  if (gatos.pid) RETURN(EBUSY) ; gatos.stop = 0 ;
  if ( (file = fopen(name,"a") ) == NULL) RETURN(errno) ;

  if (wait<0) gatos.stop = 1 ; /* No child process, only one frame */
  else {
    if ((gatos.pid = fork())) RETURN(0) ; if (errno) RETURN(errno) ;
    /* Child process from here */
    act.sa_flags = 0 ; act.sa_restorer = NULL ; sigemptyset(&act.sa_mask) ;
    act.sa_handler = &gatos_stopchild ; sigaction(SIGUSR1, &act, NULL) ; }

  /* Child process here if wait>=0 */
  ati_capture(file,wait) ; fclose(file) ;
  if (wait>=0) exit(0) ; else RETURN(0) ; }

int gatos_playback(char *name, int wait) {

  gatos_enter("gatos_playback","\"%s\",%d",name,wait) ;

  if (gatos.pid) RETURN(EBUSY) ; gatos.stop = 0 ;
  if ( (file = fopen(name,"r") ) == NULL) RETURN(errno) ;

  /* TODO: Save state */

  if ((gatos.pid = fork())) RETURN(0) ; if (errno) RETURN(errno) ;

  /* Child process from here */
  act.sa_flags = 0 ; act.sa_restorer = NULL ; sigemptyset(&act.sa_mask) ;
  act.sa_handler = &gatos_stopchild ; sigaction(SIGUSR1, &act, NULL) ;

  ati_playback(file,wait) ; fclose(file) ;
  exit(0) ; return 0 ; }

void gatos_delchan(int c) { chan_delchan(c) ; }
void gatos_clearchans() { chan_clearchans() ; }

int gatos_addchan(double f, char *n) {
  if (GATOSCALL)
    gerr("GATOS: gatos_addchan(%.3f,\"%s\")\n",f,n) ;
  if (f < gatos.minfreq) f = gatos.minfreq ;
  if (f > gatos.maxfreq) f = gatos.maxfreq ;
  return chan_addchan(f,n) ; }

FUNC0(int,gatos_numchans,"%d",gatos.numchannels)

double gatos_chanfreq(int c) {
  if (GATOSCALL) gerr("GATOS: gatos_chanfreq(%d) = %.3f\n",
    c,gatos.channelinfo[c].frequency) ;
  return gatos.channelinfo[c].frequency ; }

int gatos_setchanfreq(int c, double f) {
  if (GATOSCALL)
    gerr("GATOS: gatos_setchanfreq(%d,%f)\n",c,f) ;
  if (f < gatos.minfreq) f = gatos.minfreq ;
  if (f > gatos.maxfreq) f = gatos.maxfreq ;
  return chan_setfreq(c, f) ; }

FUNC1(char*,gatos_channame,int,c,"%d","\"%s\"",gatos.channelinfo[c].name)

FUNC1(int,gatos_chanvolume,int,c,"%d","%d",gatos.channelinfo[c].mod_volume)

int gatos_setchanvolume(int c, int v) {
  if (GATOSCALL)
    gerr("GATOS: gatos_setchanvolume(%d,%d)\n",c,v) ;
  return chan_setvolume(c, v) ; }

int gatos_chanbrightness(int c) {
  return chan_brightness(c) ; }

int gatos_setchanbrightness(int c, int v) {
  if (GATOSCALL)
    gerr("GATOS: gatos_setchanbrightness(%d,%d)\n",c,v) ;
  return chan_setbrightness(c, v) ; }

int gatos_chancontrast(int c) {
  return chan_contrast(c) ; }

int gatos_setchancontrast(int c, int v) {
  if (GATOSCALL)
    gerr("GATOS: gatos_setchancontrast(%d,%d)\n",c,v) ;
  return chan_setcontrast(c, v) ; }

int gatos_chanhue(int c) {
  return chan_hue(c) ; }

int gatos_setchanhue(int c, int v) {
  if (GATOSCALL)
    gerr("GATOS: gatos_setchanhue(%d,%d)\n",c,v) ;
  return chan_sethue(c, v) ; }

int gatos_chansaturation(int c) {
  return chan_saturation(c) ; }

int gatos_setchansaturation(int c, int v) {
  if (GATOSCALL)
    gerr("GATOS: gatos_setchansaturation(%d,%d)\n",c,v) ;
  return chan_setsaturation(c, v) ; }

int gatos_setchanname(int c, char *n) {
  if (GATOSCALL)
    gerr("GATOS: gatos_setchanname(%d,\"%s\")\n",c,n) ;
  return chan_setname(c, n) ; }

FUNC0(int,gatos_channel,"%d",gatos.channel)
int gatos_setchan(int c) {
  int v;
  v = gatos_volume();  gatos_setvolume(0);
  if (GATOSCALL) gerr("GATOS: gatos_setchan(%d)\n",c) ;
  if (c >= gatos.numchannels) c = gatos.numchannels-1 ;
  if (c < 0) c = 0 ;
  gatos.channel = c ;
  if(gatos.channelinfo[c].frequency >= gatos.minfreq
	&& gatos.channelinfo[c].frequency <= gatos.maxfreq)
    gatos_settuner(gatos.channelinfo[c].frequency) ;
  gatos_setvolume(v);
  return 0 ; }

double gatos_chanscan(int (*progress)(double)) {
  return fi12xx_scan(gatos.minfreq,gatos.maxfreq,&gatos_addchan,progress) ; }

/* ------------------------------------------------------------------------ */

#ifdef GATOSBUTTONS
int gatos_testbutton(int button) { return gatosbuttons[button] ; }
int gatos_settestbutton(int button, int value) {
  gatosbuttons[button] = value ;
  switch (button) {
    case 0: ati_setcaptsize() ; break ;
    default: break ; }
  return 0 ; }
#endif

void gatos_debug1(void) {
  int d=gatos.debug ;
  gerr("GATOS: Chip Register Dump follows:\n") ;
  ati_dumpregs() ; /*tvout_dumpregs() ;*/ gatos.debug &= ~4 ;
  board_dumpregs() ; bt829_dumpregs() ; fi12xx_dumpregs() ;
  tvout_dumpregs() ;
  gerr("GATOS: End of Register Dump\n") ; gatos.debug = d ; }

void gatos_debug2(void) {
  gerr("GATOS: I2C Bus Info follows:\n") ; i2c_info() ;
  gerr("GATOS: End of I2C Bus Info\n") ; }

void gatos_debug3(void) {}

void gatos_debug4(void) {}

int gatos_setcaptidx(int h, int v) {
  gatos_enter("gatos_setcaptidx","%d,%d",h,v) ;
  gatos.hcaptidx = LIMIT(h,0,gatos.hcaptmax) ;
  gatos.vcaptidx = LIMIT(v,0,gatos.vcaptmax) ;
  gatos_setcapturesize(hactive[gatos.hcaptidx],vactive[gatos.vcaptidx]) ;
  RETURN(0) ; }

FUNC0(int,gatos_tvout,"%d",gatos.tvout)
int gatos_settvout(int value) {
  gatos_enter("gatos_settvout","%d",value) ;
  gatos.tvout = value ;
  /*TVO_CNTL = (gatos.tvout) ? 0x8000000 : 0x00000000 ;*/
  RETURN(0) ; }

FUNC0(int,gatos_overclock,"%d",gatos.overclock)
int gatos_setoverclock(int value) {
  gatos_enter("gatos_setoverclock","%d",value) ;
  gatos.overclock = value ;
  gatos_pollscreen(0) ;
  RETURN(0) ; }

FUNC0(int,gatos_senscan,"%d",gatos.senscan)
int gatos_setsenscan(int value) {
  gatos_enter("gatos_setsenscan","%d",value) ;
  gatos.senscan = value ;
  RETURN(0) ; }

FUNC0(int,gatos_audiotype,"%d",gatos.audiotype)
