/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				cis_trans.c

Purpose:
	Prepare cis-trans flag: 0 = bad/undefined peptide group conform.,
	1 = trans, 2 = cis.

Input:
	(1) Pointer to VectorS structure, with N atom coordinates.
	(2) Pointer to VectorS structure, with CA atom coordinates.
	(3) Pointer to VectorS structure, with C atom coordinates for the
	    previous residue.
	(4) Pointer to VectorS structure, with O atom coordinates for the
	    previous residue.

Output:
	Return value.

Return value:
	(1) 0 for bad conformation of peptide group.
	(2) 1 for trans peptide group.
	(3) 2 for cis peptide group.

========includes:============================================================*/

#include <stdio.h>

#include <math.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

void		VectorProduct_ (VectorS *, VectorS *, VectorS *);
double		ScalarProduct_ (VectorS *, VectorS *);
double		AbsoluteValue_ (VectorS *);

/*======prepare cis-trans flag:==============================================*/

int CisTrans_ (VectorS *N_vectorSP, VectorS *CA_vectorSP,
	       VectorS *C_old_vectorSP, VectorS *O_old_vectorSP)
{
VectorS		C_old_O_old_vectorS, N_CA_vectorS, C_old_N_vectorS;
VectorS		v1S, v2S;
double		scalar_product;
double		denom;
double		cos_angle, abs_cos_angle;
double		abs_minimum;

/* Initialize abs_minimum value: */
abs_minimum = cos (DEG_TO_RAD * OMEGA_TOLERANCE);

/* Prepare three auxiliary vectors: */
C_old_O_old_vectorS.x = O_old_vectorSP->x - C_old_vectorSP->x;
C_old_O_old_vectorS.y = O_old_vectorSP->y - C_old_vectorSP->y;
C_old_O_old_vectorS.z = O_old_vectorSP->z - C_old_vectorSP->z;
N_CA_vectorS.x = CA_vectorSP->x - N_vectorSP->x;
N_CA_vectorS.y = CA_vectorSP->y - N_vectorSP->y;
N_CA_vectorS.z = CA_vectorSP->z - N_vectorSP->z;
C_old_N_vectorS.x = N_vectorSP->x - C_old_vectorSP->x;
C_old_N_vectorS.y = N_vectorSP->y - C_old_vectorSP->y;
C_old_N_vectorS.z = N_vectorSP->z - C_old_vectorSP->z;

/* Vector perpendicular to C_old_O_old_vectorS and C_N_vectorS: */
VectorProduct_ (&v1S, &C_old_O_old_vectorS, &C_old_N_vectorS);

/* Vector perpendicular to N_CA_vectorS and C_old_N_vectorS: */
VectorProduct_ (&v2S, &N_CA_vectorS, &C_old_N_vectorS);

/* The scalar product: */
scalar_product = ScalarProduct_ (&v1S, &v2S);

/* If scalar product is equal to zero, peptide group is bad: */
if (scalar_product == 0.0) return 0;

/* The cosine of the angle and the absolute value: */
denom = AbsoluteValue_ (&v1S) * AbsoluteValue_ (&v2S);
if (denom == 0.0) return 0;
cos_angle = scalar_product / denom;
abs_cos_angle = fabs (cos_angle);

/* Check the absolute cosine of the angle; */
/* the angle should not exceed 20 degrees: */
if (abs_cos_angle < abs_minimum) return 0;

/* The cosine of the angle for the trans peptide should be positive: */
if (cos_angle > 0.0) return 1;

/* If this point is reached, cis peptide is in question: */
return 2;
}

/*===========================================================================*/


