''' -*- python -*-                        vim:set ts=4 sw=4:

    FILE: "/home/life/projects/garchiver/garchiver-0.5/src/garchiver.py"
    LAST MODIFICATION: "Wed, 26 Sep 2001 02:19:26 +0200 (life)"

    (C) 2000 by Danie Roux <droux@tuks.co.za>

$Id: garchiver.py,v 1.79 2001/07/04 12:58:03 uid43216 Exp $

This is the core of garchiver. All shared code/callbacks should be here.
Import from this module, e.g. from garchiver import *
'''

import sys
import os
import popen2

import _load_imlib

from settings import *

def run (args=None, start_with_tree=FALSE, start_with_classic=FALSE):
    ''' This function gets called by the wrapper garchiver. In that case
    it will also pass in the argv of that function

    It also gets called when this script is called directly in which case
    argv has already been checked.
    '''

    interface = None

    # Only import this here, else all sorts of nasty recursive imports
    # happens.
    from tree_interface import *
    from classic_interface import *

    if start_with_tree:
        interface = Tree ()

    elif start_with_classic:
        interface = Classic ()

    elif gnome.config.has_section ('/garchiver/interface'):
        cfg_interface = gnome.config.get_string \
            ('/garchiver/interface/interface')

        if cfg_interface == 'tree':
            interface = Tree ()
        else:
            interface = Classic ()

    else:
        dlg = GnomeMessageBox \
            (_("Welcome to garchiver.\nWe created it to make your life easier.\n\nIf it does: email us. \nIf not: email us.\nIf you want to help: email us.\n\nWhich interface would you like to start with?"), \
            MESSAGE_BOX_QUESTION, _("Classic"), _("Tree"))

        ret = dlg.run ()
        if ret == 0:
            interface = Classic ()
        else: interface = Tree ()

    def _open (interface, file):
        if access (file, R_OK) == 0:
            interface.app.error (_("File specified on command line could not be accessed: " +
                file))
        else:
                interface.open_file (file)

    try:
        _open (interface, args [0])
        del args [0]

        if interface.type == 'Tree':
            for file in args: _open (interface, file)

    except: pass # Ignore it when the files doesn't want to open

    interface.run ()

    # It finished via mainquit ()
    if interface.switch:
        interface.switch = FALSE

        try: archive = interface.active_archive.name
        except: archive = None

        if interface.type == 'Classic':
            inter_string = '--tree'
        else: inter_string = '--classic'

        if archive != None:
            os.execl (sys.argv[0], 'garchiver', inter_string, \
                archive)
        else:
            os.execl (sys.argv[0], 'garchiver', inter_string)

from gnome.ui import *
from gtk import *
from DirListing import *
from os import *

import thread
import threading

import gnome.config
import gnome.mime
import gettext
import string
import libglade
import stat

import zip
import tar
import gzip
import bzip
import compress
import unrar
import arj

from settings import *

# Gets gettext going
gettext.bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR)
gettext.textdomain (PACKAGE)

# XPM's
# This was drawn by Brian Masney <masneyb@seul.org  
open_dir_xpm = [
    "18 14 7 1",
    "   c   None",
    "#  c   #000000",
    "-  c   #ffffff",
    "*  c   gray50",
    "i  c   darkgray",
    "h  c   gray85",
    "j  c   yellow",
    "     ****         ",
    "    *hjhj*        ",
    "   *hjhjhj******* ",
    "  *jijijijijijij*#* ",
    "#############iji*#",
    "#-----------#jij*#",
    " #-hjhjhjhjhj#ji*#",
    " #-jhjhjhjhjh#ij*#",
    " #-hjhjhjhjhj#ji*#",
    "  #-hjhjhjhjhi#j*#",
    "  #-jhjhjhjhjh#i*#",
    "  #-hjhjhjhjhj#j*#",
    "  ***************#",
    "   ###############"]

# This was drawn by Brian Masney <masneyb@seul.org  
closed_dir_xpm = [
    "16 14 6 1",
    "   c   None",
    "#  c   #000000",
    "-  c   #ffffff",
    "*  c   gray50",
    "h  c   gray85",
    "j  c   yellow",
    "   ****         ",
    "  *hjhj*        ",
    " *hjhjhj******* ",
    "*-------------* ",
    "*-jhjhjhjhjhjh*#",
    "*-hjhjhjhjhjhj*#",
    "*-jhjhjhjhjhjh*#",
    "*-hjhjhjhjhjhj*#",
    "*-jhjhjhjhjhjh*#",
    "*-hjhjhjhjhjhj*#",
    "*-jhjhjhjhjhjh*#",
    "*-hjhjhjhjhjhj*#",
    "***************#",
    " ###############"]

delete_xpm = [
    "24 24 85 1",
    "   c None",
    ".  c #000000",
    "+  c #252525",
    "@  c #A8BA9E",
    "#  c #B4C1AB",
    "$  c #E1E7DD",
    "%  c #838A7D",
    "&  c #909F86",
    "*  c #484E43",
    "=  c #A0AD94",
    "-  c #9EAB91",
    ";  c #B6C4AD",
    ">  c #BFCCB7",
    ",  c #A7B69B",
    "'  c #BEC9B5",
    ")  c #656F5E",
    "!  c #2B2E28",
    "~  c #353931",
    "{  c #242622",
    "]  c #9BAA8F",
    "^  c #9AA78E",
    "/  c #98A58C",
    "(  c #96A58D",
    "_  c #D7DED2",
    ":  c #A4B398",
    "<  c #A2B196",
    "[  c #A0B095",
    "}  c #97A38A",
    "|  c #7C9674",
    "1  c #B0C0AB",
    "2  c #CED6C8",
    "3  c #DAE1D6",
    "4  c #BBC4B1",
    "5  c #5B6B57",
    "6  c #637354",
    "7  c #748C6B",
    "8  c #94AA8D",
    "9  c #8FA58A",
    "0  c #6F8668",
    "a  c #667961",
    "b  c #5D6E58",
    "c  c #4D5A4B",
    "d  c #485245",
    "e  c #738B6E",
    "f  c #687B63",
    "g  c #5E6F5A",
    "h  c #576452",
    "i  c #495345",
    "j  c #4F5B4B",
    "k  c #3F453D",
    "l  c #687D63",
    "m  c #B9C6B4",
    "n  c #65785F",
    "o  c #9BAE97",
    "p  c #525F4F",
    "q  c #728A6D",
    "r  c #4F5C4C",
    "s  c #6E8468",
    "t  c #4D5849",
    "u  c #9BAD96",
    "v  c #6E8368",
    "w  c #080808",
    "x  c #6E8367",
    "y  c #B7C6B4",
    "z  c #9AAD96",
    "A  c #525F4E",
    "B  c #4D5848",
    "C  c #687A63",
    "D  c #B7C5B4",
    "E  c #6D8367",
    "F  c #4C5848",
    "G  c #809A78",
    "H  c #A0B29A",
    "I  c #60725C",
    "J  c #728A6B",
    "K  c #4F5B4C",
    "L  c #586853",
    "M  c #677A61",
    "N  c #7D9775",
    "O  c #7A9472",
    "P  c #788F71",
    "Q  c #748D6E",
    "R  c #6F8568",
    "S  c #6D8267",
    "T  c #697D64",
    "                        ",
    "                        ",
    "                        ",
    "                        ",
    "       ........         ",
    "     +.@#$%&*=-..       ",
    "    .;>,')!~{-]^/.      ",
    "    .(_':<[=-]^/}.      ",
    "    .|1234=-]^/|5.      ",
    "    .67899||0abcd.      ",
    "     .+effghijk+.       ",
    "     .l++++++++f.       ",
    "     .lmnopqrstf.       ",
    "     .lmnupqrvtf.www    ",
    "     .lmnupqrxtf.wwww   ",
    "     .lynzAqrxBC.wwww   ",
    "     .lDnzAqrEFC.wwww   ",
    "     .GHnzIJKELM.www    ",
    "      .NOPQJRST.ww      ",
    "       ........w        ",
    "                        ",
    "                        ",
    "                        ",
    "                        "
]

# You can do
# _ = gettext.gettext to use gettext going in a nice way Reason this
# doesn't work in modules imported from this one, is because everything
# that starts with a _ is a private variable is in Python

_ = gettext.gettext

class Interface:
    ''' The base interface class.

    Provides the base for all the 2 interfaces. Stuff like threading for
    example.
    '''

    # The reason we keep this reference is so that we have something
    # non-interactive to grab focus when we block the user
    appbar = GnomeAppBar (1, 1, PREFERENCES_NEVER)

    # Some dnd constants
    TARGET_STRING = 0
    TARGET_URL = 1
    TARGET_ROOTWIN = 2
    TARGET_INTERNAL = 3

    # internal drag target
    internal = [ 
        ('garchiver_internal', 0, TARGET_INTERNAL),
    ]

    # Drag targets
    targets = [
        ('STRING',     0, TARGET_INTERNAL),
        ('text/plain', 0, TARGET_STRING),
        ('text/uri-list', 0, TARGET_URL),
        ('application/x-rootwin-drop', 0, TARGET_ROOTWIN),
        internal [0]
    ]

    # The main app object
    app = GnomeApp ('garchiver', 'garchiver - making life easier')

    # Set the whole app as a destination for drags
    app.drag_dest_set (DEST_DEFAULT_ALL, targets, GDK.ACTION_COPY | \
        GDK.ACTION_MOVE)

    # Gets set to true when the interface must switch to the other one and
    # not quit
    switch = FALSE

    def __init__ (self):

        # Syncs with the settings about what interface this is.
        if self.type == 'Classic':
            gnome.config.set_string ('/garchiver/interface/interface', \
                'classic')
        else:
            gnome.config.set_string ('/garchiver/interface/interface', \
                'tree')

        gnome.config.sync_file ('/garchiver')

        # Variable that holds the current active archive
        self.active_archive = None

        # Errors in a thread gets reported in this string. Assuming that
        # only one other thread of execution exist. Remember that only one
        # thread (main thread in this case) can control the gui.
        self.thread_error = []

        # Create the settings object
        self.Settings = Settings (self)

        # Create the sandbox for the tempfiles
        self.sandbox = Sandbox ()

        # Create the Most Recently Used object
        self.mru = MRU (self.open_file)
        
        # Busy cursor
        self.busy_cursor = cursor_new (150 & 0xfe)
        # Normal cursor
        self.normal_cursor = cursor_new (68 & 0xfe)

    def run (self):
        # Runs the interface

        mainloop ()

    def new (self, archive):
        ''' Gets passed an archive object '''

        archive.new ()
        self.open (archive)

    def common_dnd_data_received (self, x, data):

        if self.active_archive == None:
            self.app.error \
                (_("You must first create/open a new archive!"))
            return

        list = string.split (data, '\n')

        def filter (x):
            if string.find (x, 'file:', 0) != -1:
                x = string.strip (x)
                x = string.replace (x, '\000', '')
                return x[5:]
            else: pass

        list = map (filter, list)

        try:
            while 1:
                list.remove (None)
        except ValueError:
            pass

        try: 
            # The base of the file, which we don't want in the archive
            base = os.path.dirname (list [0])

            # Make it into a list without paths
            # TODO Make it a setting?
            list = map (os.path.basename, list)

        except IndexError:
            # There wasn't a file: in the dropped string, it was probably
            # a http:// string.
            # TODO It would really, really be great if we can download the
            # file and compress it into an archive. Part of version 2?
            # Return no files
            return [], ''

        return (list, base)

    def common_dnd_data_get (self, selection_data, retarget=None):
        ''' Dropped on file manager. This is a function called by both
            interfaces 
        '''

        ''' Information for this is in description.py of the
            gnome-python source
        '''
        
        import tempfile
        self.dnd_extract_folder = tempfile.mktemp () + 'garchiver_dnd/'
        del tempfile

        os.mkdir (self.dnd_extract_folder)
        self.sandbox.add_cleanself (self.dnd_extract_folder)

        files = self.extract (self.dnd_extract_folder, FALSE, TRUE)

        selection = ''
        for file in files:
            selection = selection + 'file:///' + \
                self.dnd_extract_folder + file + '\n'


        if retarget != None:
            selection_data.set (retarget, 8, selection)
        else:
            selection_data.set (selection_data.target, 8, selection)

    def open (self, archive):
        ''' Opens an archive object that should have been created already.
        '''
        raise NotImplementedError, 'Use one of the derived classes!'

    def extract (self, to, with_path, certain_files):
        ''' The abstract extract method.

        to: Where to the files has to be extracted.

        with_path: Whether the files has to extracted with or or without
        path

        certain_files: Whether certain files should be extracted.

        returns: The files extracted
        '''

        raise NotImplementedError, 'Use one of the derived classes!'

    def view_log (self, widget):
        ''' Views the log of the previous action.

        This of course requires the archive type to have the function
        get_log ()
        '''
        if self.active_archive == None: self.app.error ('No log available')
        else:self.__show_log (self.active_archive.get_log ())

    def gui_handle (self, func, args):
        ''' Run the function func with arguments args and call

        This should be used for anything UI that runs in a seperate
        thread.

        i.e.:
        self.gui_handle (my_action_function, (arg1, arg2)))
        '''

        def secret_function (self, event, func, args):
            try: 
                apply (func, args)

                # We're done
                event.set ()
            except:
                event.set ()
                import traceback
                type, message, tb = sys.exc_info ()
                f_tb = traceback.format_tb (tb)
                self.thread_error.append (str (type) + ': ' + str (message))
                self.thread_error.append ('')
                for line in f_tb:
                    self.thread_error [1] = self.thread_error [1] + line

        event = threading.Event ()
        self.app.set_sensitive (FALSE)
        self.app.get_window ().set_cursor (self.busy_cursor)

        thread.start_new_thread (secret_function, (self, event, func, args))

        # Block all user input until the event is set.
        # Do one event loop though
        while not event.isSet ():
            if events_pending (): mainiteration ()

        self.app.get_window().set_cursor (self.normal_cursor)
        self.app.set_sensitive (TRUE)

        if self.thread_error != []:
            self.error_callback (self.thread_error [0], self.thread_error [1], 'Thread error occured')
            self.thread_error = []

    def __on_filesel_ok_button_clicked (self, na, *sel):
        selected = sel[0].get_filename ()

        sel[0].hide ()
        sel[0].destroy ()

        self.open_file (selected)

    def get_archive_type_of_file (self, file):
        ''' This takes a string, and from that return a reference to an
        intitializer that can be used to create an object of that type.

        Reason this is seperated, is because in the dnd section for
        example, it's necessary to know if a file is a valid archive type.
        '''

        archive = None
        string.strip (file)

        type = gnome.mime.type_of_file (file)

        '''
        if type == 'text/plain':
            # Try the magic then
            # TODO This dumps core?
            type = gnome.mime.type_from_magic (file)
        '''

        if type == 'application/zip':
            archive = zip.Zip
        elif type == 'application/x-gtar':
            archive = tar.Tar
        elif type == 'application/x-tar':
            archive = tar.Tar
        elif type == 'application/x-compressed-tar':
            archive = gzip.Gzip
        elif type == 'application/x-bzip2':
            archive = bzip.Bzip
        elif type == 'application/x-rar':
            archive = unrar.Unrar
        elif type == 'application/x-arj':
            archive = arj.Arj
        else:
            # If mime is broken, we have to go manually, now don't we!?
            if string.lower (file [-6:]) == '.tar.z': archive = \
                compress.Compress
            if string.lower (file [-8:]) == '.tar.bz2' : archive = \
                bzip.Bzip
            if string.lower (file [-7:]) == '.tar.gz' : archive = \
                gzip.Gzip
            if string.lower (file [-4:]) == '.zip' : archive = zip.Zip
            if string.lower (file [-4:]) == '.rar' : archive = \
                unrar.Unrar
            if string.lower (file [-4:]) == '.tar' : archive = tar.Tar
            if string.lower (file [-4:]) == '.arj' : archive = arj.Arj
            # Self installing Winzip archives
            if string.lower (file [-4:]) == '.exe' : archive = zip.Zip

        return archive

    def insert_file_menu (self):
        # Menu structures
        file_menu = [
            (APP_UI_ITEM, _("_New Archive"), _("Creates a new archive"), 
                self.on_new, None, APP_PIXMAP_STOCK, STOCK_MENU_NEW, GDK.N,
                GDK.CONTROL_MASK),
            UIINFO_MENU_OPEN_ITEM (self.on_open_activated,
                None)
        ]

        entries = self.mru.get_uiinfo_list ()

        if entries != []:
            file_menu.append (UIINFO_SEPARATOR)
            for entry in entries: file_menu.append (entry)
            file_menu.append (UIINFO_SEPARATOR)

        file_menu.append (UIINFO_MENU_EXIT_ITEM (self.on_delete_event, None))

        #self.app.insert_menus ('', [UIINFO_SUBTREE (_("_File"), file_menu)])
        self.app.insert_menus ('', [UIINFO_SUBTREE ("_File", file_menu)])

    def open_file (self, file):
        ''' This opens a file in the GUI by calling self.open with the
        right arguments
        '''

        archive = self.get_archive_type_of_file (file)

        if archive == None:
            self.app.error (_("File not recognised as a valid archive: " +
                file))
            return

        # Add this file to the Most Recently Used list
        self.mru.add (file)
        self.app.remove_menu_range ("_File", 0, 1)
        self.insert_file_menu ()

        gnome.config.set_string ('/garchiver/accessed/path', \
            file[:string.rfind (file, '/')+1])
        gnome.config.sync_file ('/garchiver')

        instance = archive (file, self.yes_no_callback,
            self.msg_callback, self.error_callback)

        if (instance.available () == 0):
            self.app.error (_("The external program is either not installed or the path to it is incorrect."))
            return

        self.open (instance)

    def __on_filesel_ok_button_clicked (self, na, *sel):

        selected = sel[0].get_filename ()

        sel[0].hide ()
        sel[0].destroy ()

        self.open_file (selected)

    def cancel_clicked (self, widget, data):
        data.destroy ()

    def translate_extract_to_ok_clicked (self, widget, data):
        '''This is an intermediate function called to take the text out of
        the combo_entry, and make a bool from extract_to_path.

        It also sets the setting that gets loaded next time you open the
        dialog
        '''

        (dialog, combo, with_path, only_selected) = data

        gnome.config.set_string ('/garchiver/accessed/extract_to_path', \
            combo.get_text ())
        gnome.config.sync_file ('/garchiver')

        dialog.hide ()
        dialog.destroy ()
        with_path = with_path.get_active ()
        only_selected = only_selected.get_active ()

        if not only_selected:
            self.extract (combo.get_text (), with_path, FALSE)
        else:
            self.extract (combo.get_text (), with_path, TRUE)

    def update_fs_listing (self, widget, path, file_clist):
        '''
        The clist widget that displays the files in the current directory
        that the user clicked on is updated by this callback function
        '''

        dir = path

        list_of_files = []
        for files in os.listdir(dir):
            try:
                file = []
                mode = os.stat('%s/%s' %(dir,files))[stat.ST_MODE]
                size = os.stat('%s/%s' %(dir,files))[stat.ST_SIZE]
                time = os.stat('%s/%s' %(dir,files))[stat.ST_CTIME]
                if stat.S_ISREG(mode):
                    file.append(files)
                    file.append(`size`)
                    list_of_files.append(file)
            except: # A stat failed. Who cares? Go on with life
                pass
        file_clist.clear ()
        map (file_clist.append, list_of_files)

    def on_add (self, widget):

        dlg = GnomeDialog (_("Add files"))
        dlg.set_usize (500, 400)
        dlg.append_button_with_pixmap (_("Ok"), STOCK_BUTTON_OK)
        dlg.append_button_with_pixmap (_("Cancel"), STOCK_BUTTON_CANCEL)

        class add_details:
            file_path = ''

        if gnome.config.has_section ('/garchiver/accessed'):
            add_details.file_path = gnome.config.get_string \
                ('/garchiver/accessed/add_path')
            if add_details.file_path == None: add_details.file_path = '/'
        else: add_details.file_path = '/'

        clist = the_clist (['Name', 'Size'], None)
        clist.set_selection_mode (SELECTION_EXTENDED)
        clist.set_column_width (0, 110)
        clist.set_column_width (1, 80)

        self.update_fs_listing (self, add_details.file_path, clist)

        scrolled = GtkScrolledWindow ()
        scrolled.add (clist)

        def update_path (widget, new_path, add_details):
            add_details.file_path = new_path

        dl = GtkDirListing (add_details.file_path)
        dl.connect('dir_selected', update_path, add_details)
        dl.connect('dir_selected', self.update_fs_listing, clist)

        hbox = GtkHBox ()
        hbox.add (dl)
        hbox.add (scrolled)

        dlg.vbox.add (hbox)

        def clicked_cb (widget, button, (self, clist, add_details)):
            if button == 1: # Cancel
                widget.close ()
            else:
                files = []
                for row in clist.selection:
                    files.append (clist.get_text (row, 0))

                # Close it before the real action begins
                widget.close ()

                if (len(files) == 0):
                    place = string.rfind(add_details.file_path, "/")
                    directory = add_details.file_path[place+1:]
                    temp = add_details.file_path[0:place+1]
                    self.gui_handle (self.active_archive.add,
                        ([directory], temp))
                else:
                    self.gui_handle (self.active_archive.add,
                        (files, add_details.file_path))

                self.open (self.active_archive)

        dlg.connect ('clicked', clicked_cb, (self, clist, add_details))

        dlg.show_all ()
        dlg.run ()

    def __real_new (self, new_file_details):
        ''' According to the selection of the function below, the actual
        archive gets created here. The interface's .new method is called
        to create the object. The tree interface would add it to it's list
        of archives, the classic interface will remove the old one.
        '''

        file = os.path.join (new_file_details.dest_dir,
            new_file_details.file + '.' + new_file_details.file_ext)

        arc = self.get_archive_type_of_file (file)

        inst = arc (file, self.yes_no_callback, self.msg_callback,
            self.error_callback)

        if not inst.available ():
            self.app.error (_("The external program is either not installed or the path to it is incorrect. Which means I can't create a new archive of this type."))
            return

        self.mru.add (file)
        self.app.remove_menu_range ("_File", 0, 1)
        self.insert_file_menu ()

        self.new (inst)

    def on_new (self, widget):
        ''' Callback for new. Creates the whole dialog'''

        class new_file_details:
            file_ext = 'tar.gz'
            dest_dir = '/'
            tname_widget = None
            fext_widget = None

        if gnome.config.has_section ('/garchiver/accessed'):
            path = gnome.config.get_string \
                ('/garchiver/accessed/new_path')
        else: path = '/'

        win = GnomeDialog(_("New Archive"))
        win.set_usize(400, 450)

        win.append_button_with_pixmap (_("Ok"), STOCK_BUTTON_OK)
        win.append_button_with_pixmap (_("Cancel"), STOCK_BUTTON_CANCEL)

        def clicked_cb (widget, button, (self, new_file_details)):

            # Widget is getting closed, no matter if OK or Cancel was
            # clicked
            widget.close ()

            new_file_details.file = new_file_details.tname_widget.get_chars (0, -1)

            if button == 0: # Ok

                if new_file_details.tname_widget.get_chars(0, -1) == '':
                    self.msg_callback (_("Type in a name for the new archive"))
                elif access (new_file_details.dest_dir, W_OK | X_OK) == 0:
                    self.error_callback (_("You are trying to create an archive in a folder you have no permission to"))
                else:
                    try:
                        os.stat (os.path.join (new_file_details.dest_dir,
                            new_file_details.file + '.' + new_file_details.file_ext))
                        self.error_callback (_("File already exists!"))
                    except:
                        gnome.config.set_string ('/garchiver/accessed/new_path',
                            new_file_details.dest_dir)
                        gnome.config.sync_file ('/garchiver')
                        self.__real_new (new_file_details)

        # Don't you just love these functions within functions? Notice
        # that this is part of on_new
        win.connect ('clicked', clicked_cb, (self, new_file_details))

        def update_path(widget, path, new_file_details):
            new_file_details.dest_dir = path

        dl = GtkDirListing(path)
        dl.connect('dir_selected', update_path, new_file_details)
        win.vbox.pack_start(dl, TRUE, TRUE, 5)

        separator = GtkHSeparator()
        win.vbox.pack_start(separator, FALSE, FALSE, 2)

        table = GtkTable (2, 2)

        win.vbox.pack_start (table, FALSE)

        def update_ext(widget, (ext, new_file_details)):
            new_file_details.file_ext = ext

        tar_g = GtkMenuItem('tar.gz')
        tar_b = GtkMenuItem('tar.bz2')
        tar = GtkMenuItem('tar')
        zip = GtkMenuItem('zip')
        tar_z = GtkMenuItem('tar.Z')

        zip.connect("activate", update_ext, ("zip", new_file_details))
        tar.connect("activate", update_ext, ("tar", new_file_details))
        tar_g.connect("activate", update_ext, ("tar.gz", new_file_details))
        tar_b.connect("activate", update_ext, ("tar.bz2", new_file_details))
        tar_z.connect("activate", update_ext, ("tar.Z", new_file_details))

        men = GtkMenu()
        men.append(tar_g)
        men.append(tar_b)
        men.append(tar_z)
        men.append(tar)
        men.append(zip)

        new_file_details.fext_widget = GtkOptionMenu()
        new_file_details.fext_widget.set_menu(men)

        new_file_details.tname_widget = GtkEntry()

        all1 = GtkAlignment (0, 0, 0, 0)
        all1.add (GtkLabel ('Filename:'))

        all2 = GtkAlignment (0, 0, 0, 0)
        all2.add (GtkLabel ('Type:'))

        table.attach (all1, 0, 1, 0, 1, xpadding = 10, ypadding = 3)
        table.attach (new_file_details.tname_widget, 1, 2, 0, 1, ypadding = 3)
        table.attach (all2, 0, 1, 1, 2, xpadding = 10, ypadding = 3)
        table.attach (new_file_details.fext_widget, 1, 2, 1, 2, ypadding = 3)

        win.show_all()
        # After the widget is shown, call cd. cd needs for the widget to
        # be visible
        if path != None: dl.cd (path)

    def on_extract (self, widget):
        ''' Callback for extracting. '''
        if self.active_archive == None:
            self.app.error ('No open Archive')
            return

        if gnome.config.has_section ('/garchiver/accessed'):
            path = gnome.config.get_string \
                ('/garchiver/accessed/extract_to_path')
            if path == None: path = '/'
        else: path = '/'

        path_to_glade = os.path.join \
            (GARCHIVER_DATADIR, 'extract_to_dialog.glade')

        if not os.access (path_to_glade , os.R_OK):
            self.app.error (_("Can't find " + path_to_glade + " - Did you specify the correct prefix when you did ./configure?"))
            return

        glade = libglade.GladeXML (path_to_glade)

        place = glade.get_widget ('placeholder')
        dialog = glade.get_widget ('extract_to_dialog')
        combo = glade.get_widget ('combo_entry')
        ok = glade.get_widget ('ok')
        # Send the dialog, combo_entry, and with_path to
        # intermediate cb
        ok.connect ('clicked', self.translate_extract_to_ok_clicked, \
            (dialog, \
             combo, \
             glade.get_widget ('with_path'), \
             glade.get_widget ('only_selected')))
        cancel = glade.get_widget ('cancel')
        cancel.connect ('clicked', self.cancel_clicked, dialog)

        def update_combo (widget, path, combo):
            combo.set_text (path)

        dl = GtkDirListing (path)
        dl.connect ('dir_selected', update_combo, combo)

        place.add (dl)

        while not os.path.exists (path):
            path, dummy = os.path.split(path)

        combo.set_text (path)

        dialog.set_usize (400, 400)
        dialog.show_all ()

    def on_open_activated (self, item):

        sel = GtkFileSelection (_("Select Archive"))
        path = gnome.config.get_string ('/garchiver/accessed/path')
        if path != None: sel.set_filename (path)

        ok_button = sel.__getattr__ ('ok_button')
        ok_button.connect ('clicked',self.__on_filesel_ok_button_clicked, sel)
        cancel_button = sel.__getattr__ ('cancel_button')
        cancel_button.connect ('clicked', self.cancel_clicked, sel)

        sel.set_modal (TRUE)
        sel.show ()

    def on_view (self, widget, file):

        def secret_function (self, file):
            # Gets a temp file
            tmp = self.active_archive.view (file)
            self.view (file, tmp)

        if PS('confirm_file_view'):
            dlg = GnomeMessageBox \
                (_('Do you want to view the file ' + file + '?'),
                MESSAGE_BOX_QUESTION, STOCK_BUTTON_YES, STOCK_BUTTON_NO)

            ret = dlg.run ()
            if ret == 1:
                return

        timeout_add (50, self.update_progress, self.appbar)

        self.gui_handle (secret_function, (self, file))

    def view (self, file, tmp=None):
        ''' Views the file given with the associate program. file should be
        the whole directory to the file. tmp should be the file to where
        this file was extracted temporarily, see tar.py. If tmp=None this
        is not a tempfile but a real file, like in the file_listing of the
        tree interface
        '''

        # TODO Something is wrong with htm/html files.

        if tmp != None:
            try:
                t = os.stat (tmp)
                if t [6] == 0:
                    # Reason why I'm not using a simple "if then" is because
                    # something else could have gone wrong that raised an
                    # exception
                    raise
            except:
                self.app.error (_("Can't view the file. See the log."))
                return # If the file doesn't exist we can't go on.

        mime = gnome.mime.type (os.path.basename (file))
        keys = gnome.mime.get_keys (mime)

        try:
            keys.index ('view')
            cmd = gnome.mime.get_value (mime, 'view')
        except ValueError:
            try:
                keys.index ('open')
                cmd = gnome.mime.get_value (mime, 'open')
            except ValueError:
                cmd = None

        if cmd != None:
            if tmp != None:
                cmd = string.replace (cmd, '%f', tmp)
            else: cmd = string.replace (cmd, '%f', file)

            def secret_function (self, file, tmp, cmd):
                if tmp != None: self.sandbox.add (file, tmp)
                os.system (cmd)
                if tmp != None: os.remove (tmp)

            # So that all external programs happen on their own.
            thread.start_new_thread (secret_function, (self, file, tmp, cmd))
        else:
            self.app.error (_("Could not find an associated program to open the file: ") +
                file + _(". Check your mime-types."))

    def update_progress (self, bar):
        ''' Update the progress bar until it reach 100%. It then resets
        everything and stop updating the bar.
        '''

        if self.active_archive != None:
            bar.set_progress (self.active_archive.progress)
            if self.active_archive.progress == 1:
                # Resets the progress
                self.active_archive.progress = 0
                self.active_archive.progress_inc = 0
                return FALSE
            elif self.active_archive.progress_inc == 0: return FALSE
            else: return TRUE
        else: return FALSE

    # Pre condition:
    # yes_no_callback has connect this function as the callback function
    # for an app question. And has called mainloop to put the mainloop one
    # level deeper.

    # Post condition:
    # This function set self._answer to whatever was answered and calls
    # mainquit () to let yes_no_callback continue
    def __question_cb (self, button):
        if button == 0:
            self.__answer = TRUE
        else:
            self.__answer = FALSE
        mainquit ()

    def yes_no_callback (self, question):
        ''' A modal yes or no dialog. '''
        dlg = GnomeMessageDialog (question, MESSAGE_BOX_QUESTION, \
            STOCK_BUTTON_YES, STOCK_BUTTON_NO, self.app)
        return dlg.run ()

    def sandbox_warning  (self, files):

        files = reduce ((lambda x,y: x+ '\n' +y), files)

        return self.yes_no_callback (_("These files are still being viewed: \n") +
            files + _(". \nAre you sure you want to quit, and leave the tempfiles lying around?"))

    def msg_callback (self, msg):
        ''' Pops up a modal warning dialog. '''
        dlg = GnomeWarningDialog (msg, self.app)
        dlg.set_modal (TRUE)
        dlg.run ()

    def __show_log (self, log):
        ''' Private function that shows a dialog box with a log of all the
        actions
        '''

        if log != '':
            log_dlg = GnomeDialog (_("Execution log"))
            log_dlg.set_usize (500, 400)

            log_dlg.append_button_with_pixmap ('', STOCK_PIXMAP_BACK)
            log_dlg.append_button_with_pixmap ('', STOCK_PIXMAP_FORWARD)
            log_dlg.append_button_with_pixmap (_("Ok"), STOCK_BUTTON_OK)

            log_dlg.set_sensitive (0, FALSE)
            log_dlg.set_sensitive (1, FALSE)

            if self.active_archive != None:
                if self.active_archive.has_previous_log ():
                    log_dlg.set_sensitive (0, TRUE)

                if self.active_archive.has_next_log ():
                    log_dlg.set_sensitive (1, TRUE)

            def clicked_cb (widget, button, (self, edit)):
                if button == 0:
                    # Previous
                    edit.delete_text (0, -1)
                    edit.insert_defaults \
                        (self.active_archive.get_previous_log ())

                elif button == 1:
                    edit.delete_text (0, -1)
                    edit.insert_defaults \
                        (self.active_archive.get_next_log ())
                elif button == 2:
                    widget.close ()
                    return

                if not self.active_archive.has_previous_log ():
                    widget.set_sensitive (0, FALSE)
                else: widget.set_sensitive (0, TRUE)

                if not self.active_archive.has_next_log ():
                    widget.set_sensitive (1, FALSE)
                else: widget.set_sensitive (1, TRUE)

            edit = GtkText ()
            edit.set_editable (FALSE)
            edit.insert_defaults (log)
            edit.show ()

            log_dlg.connect ('clicked', clicked_cb, (self, edit))

            scrolled = GtkScrolledWindow (edit.get_hadjustment (),
                edit.get_vadjustment ())

            scrolled.add (edit)
            scrolled.show ()
            log_dlg.vbox.add (scrolled)

            log_dlg.run ()

    def error_callback (self, error, error_log = '', dialog_title='Error'):
        ''' Creates a modal error dialog '''
        # TODO Document it a bit better
        if error_log == '':
            dlg = GnomeErrorDialog (error, self.app)
            dlg.run ()
        else:
            dlg = GnomeDialog (dialog_title)
            dlg.append_button_with_pixmap (_("Ok"), STOCK_BUTTON_OK)
            dlg.append_button_with_pixmap (_("View log"),
                STOCK_PIXMAP_BOOK_OPEN)
            dlg.set_modal (TRUE)

            hbox = GtkHBox ()
            hbox.show ()
            dlg.vbox.add (hbox)

            l = GtkLabel (error)
            l.show ()
            hbox.add (l)

            if dlg.run_and_close () == 1: # Clicked on view log
                self.__show_log (error_log)

# This is for the integer sorting
import _garchiver_extra_gtk
from gtk import *

class the_clist (GtkCList):
    ''' Generic clist that has all the nice functions like sort by, any
    number of columns, context menu's.

    This is so that we don't have double code in classic and tree
    '''

    def __init__ (self, title_list, popup_menu, has_pixmap=FALSE):
        ''' Give it a list with what the titles should be. It will figure
        out how many that is and create the clist accordingly

        It also needs a popup_info structure to create the popup from

        It is not nearly complete. I will have little icons saying by
        which column we are sorting, and all that stuff

        I need to get the files` icon before the file. As well. That is
        easy enough
        '''

        GtkCList.__init__ (self, cols = len (title_list), titles = title_list)

        self.set_selection_mode (SELECTION_EXTENDED)

        self.has_pixmap = has_pixmap

        if popup_menu != None:
            self.popup = GnomePopupMenu (popup_menu)
            self.connect ('button_press_event', self._popup_handler, self.popup)

        self.mouse_x = 0
        self.mouse_y = 0

        self.connect ('click-column', self.sort_list)
        self.sortby = -1
        self.sort_type = SORT_ASCENDING

    def _popup_handler (self, widget, event, menu):
        ''' Handle popups for the clist. Too bad gnome-popups are very
        unstable in python-gnome :(
        '''
        if (event.type == GDK.BUTTON_PRESS):
            if (event.button == 3):
                self.mouse_x = event.x
                self.mouse_y = event.y
                self.popup.popup (None, None, None, event.button, event.time)
                return 'true'

    def sort_list (self, dummy=None, col=-2):
        ''' Handles all manner of sorting for the clist. Also uses the
        custom C function in here. if has_pix is true another custom
        function is used to sort columns that has pixmaps as well, such as
        folders
        '''

        if col == self.sortby: # Second click on this column, sort different
            if self.sort_type == SORT_ASCENDING:
                self.set_sort_type (SORT_DESCENDING)
                self.sort_type = SORT_DESCENDING
            else:
                self.set_sort_type (SORT_ASCENDING)
                self.sort_type = SORT_ASCENDING

        if col == -2:
            self.sort_type = SORT_ASCENDING
            self.set_sort_type (SORT_ASCENDING)
            col = 0

        self.sortby = col

        self.set_sort_column (col)

        # If the column look like a bunch of numbers, call the wrapper
        # function with 1 to sort it with a custom function for that.
        try:
            text = self.get_text (0, col)
            int (text)
            # Just for luck try the next file in line
            text = self.get_text (1, col)
            int (text)

            # Conversion was possible so set sort as integer
            _garchiver_extra_gtk.garchiver_set_sort_as (self._o, 1)
        except ValueError:
            # It was not possible to cast it to an integer, so it will be
            # sorted as usual, except if it has pixmaps
            if self.has_pixmap: 
                _garchiver_extra_gtk.garchiver_set_sort_as (self._o, 2)
            else: 
                _garchiver_extra_gtk.garchiver_set_sort_as (self._o, 0)


        self.sort ()

    def append_list (self, list):
        ''' Append a list to the rows '''
        map (self.append, list)

import shutil
class Sandbox:
    ''' All tempfiles gets registered with the sandbox. When the
    application quits, the sandbox is first asked whether it's OK '''

    __litter = {}
    __selfclean_litter = []

    def add (self, file, tmp_name):
        self.__litter [file] = tmp_name

    def add_cleanself (self, folder):
        self.__selfclean_litter.append (folder)

    def cleaned (self):
        for folder in self.__selfclean_litter:
            try: shutil.rmtree (folder)
            except: raise

        for file in self.__litter.keys ():
            try:
                os.stat (self.__litter [file])
                # Oops, file exists! Else an exception would have been
                # thrown
                return FALSE
            except: del self.__litter [file] # Cleared from the sandbox
        # Sandbox is clean
        return TRUE

    def litter (self):
        ''' Returns the list of files still open. '''
        return self.__litter.keys ()

class MRU:
    ''' Provides a convenient interface for Most Recently Used lists '''

    def __init__ (self, open_callback):
        ''' callback_open is the function that has to be called when a
        file wants to be opened
        '''
        self.open_callback = open_callback
        self.list = []
        count = 0
        while 1:
            count = count +1
            entry = gnome.config.get_string ('/garchiver/mru/file' + `count`)
            if entry == None: break # No more entries in config file
            if os.path.exists (entry): # Make sure the file actually is there
                self.list.append (entry)

    def add (self, new_entry):

        try: self.list.remove (new_entry)
        except ValueError: pass # First-time entry

        self.list = [new_entry] + self.list
        self.list = self.list [:10]

        count = 1
        for entry in self.list:
            gnome.config.set_string ('/garchiver/mru/file' + `count`, \
                self.list [count-1])
            count = count +1
            if count > 10:
                break # MRU lists must not be bigger than 10, please!

        gnome.config.sync_file ('/garchiver')

    def get_uiinfo_list (self):
        ''' Returns the mru list as a uiinfo list, ready to add to a menu
        structure. Had to do an ugly hack here, which I'm none the less
        proud of :-)
        Had to do the hack because sending data doesn't work
        '''
        count = 1
        ret = []

        for entry in self.list:
            # The callback data doesn't work!!!!!!
            # This means I have to bloody well hack it.
            # I'm compiling my own dynamic functions that call the
            # real callback with some REAL data.
            codestring = 'def hack' + `count` + '(garbage):\n' + \
                         '    self.open_callback (self.list [' + `count-1` + '])\n'

            code = compile (codestring, '<callback hack>', 'exec')
            # Make part of execution, the locals statement makes it
            # possible for the codestring to get to self
            exec (code, locals ())

            # This would be nice, except that I can't get the menu hints
            # to work, maybe my taskbar is not working right? TODO
            # I want to display the full path as a hint, not as part of
            # the entry
            # ret.append (APP_UI_ITEM, '_' + `count` + ' ' +                 #
            #     os.path.basename (entry), os.path.dirname (entry), eval    #
            #         ('hack'+`count`), entry, APP_PIXMAP_NONE, None, 0, 0), #

            ret.append (APP_UI_ITEM, '_' + `count` + ' ' +
                entry, '', eval ('hack'+`count`), entry, APP_PIXMAP_NONE,
                    None, 0, 0),

            count = count +1

        return ret

if __name__ == '__main__':
    run ()
