/*************************************************
 *
 * $GAMGI/src/gtk/group/gamgi_gtk_group_modify.c
 *
 * Copyright (C) 2004 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_gtk.h"
#include "gamgi_mesa.h"
#include "gamgi_math.h"
#include "gamgi_chem.h"
#include "gamgi_phys.h"
#include "gamgi_io.h"
#include "gamgi_global.h"

#include "gamgi_engine_create.h"
#include "gamgi_engine_start.h"
#include "gamgi_engine_copy.h"
#include "gamgi_engine_copy_out.h"
#include "gamgi_engine_link.h"
#include "gamgi_engine_unlink.h"
#include "gamgi_engine_find.h"
#include "gamgi_engine_list.h"
#include "gamgi_engine_dlist.h"
#include "gamgi_engine_find.h"
#include "gamgi_engine_remove.h"
#include "gamgi_gtk_dialog.h"
#include "gamgi_gtk_history.h"
#include "gamgi_gtk_object.h"
#include "gamgi_gtk_group_link.h"
#include "gamgi_mesa_select.h"
#include "gamgi_chem_polytope.h"
#include "gamgi_chem_atom.h"
#include "gamgi_math_vector.h"
#include "gamgi_math_matrix.h"
#include "gamgi_math_euler.h"
#include "gamgi_math_position.h"
#include "gamgi_io_token.h"

/*****************************************************
 * GTK combo box enumerations: static, starting at 0 *
 *****************************************************/

enum { STATIC_TYPE_CONTAINER = 0, STATIC_TYPE_POLYTOPE, STATIC_TYPE_LOCAL };
enum { STATIC_AUTONOMY_ALL = 0, STATIC_AUTONOMY_PARTIAL, 
STATIC_AUTONOMY_NONE, STATIC_AUTONOMY_LOCAL };
enum { STATIC_NUMBER_NUMBER = 0, STATIC_NUMBER_RADIUS, STATIC_NUMBER_LOCAL };
enum { STATIC_STYLE_WIRED = 0, STATIC_STYLE_SOLID, STATIC_STYLE_LOCAL };

static gamgi_bool static_scan (gamgi_group *group, gamgi_dlist *start,
double *position, gamgi_atom **central)
{
gamgi_atom *atom;
gamgi_dlist *dlist;
double x, y, z;

for (dlist = start; dlist != NULL; dlist = dlist->next)
  {
  atom = GAMGI_CAST_ATOM dlist->data;
  x = atom->position[0] - position[0];
  y = atom->position[1] - position[1];
  z = atom->position[2] - position[2];
  if (x*x + y*y +z*z < GAMGI_MATH_TOLERANCE_LENGTH * GAMGI_MATH_TOLERANCE_LENGTH)
    { *central = atom; return TRUE; }
  }

return FALSE;
}

static gamgi_bool static_find (gamgi_group *group, gamgi_atom **atom)
{
gamgi_dlist *dlist;
double matrix[9], inverse[9];
double position[3], translate[3];
double euler[3];

/*********************************************
 * get list of atoms not linked to planes or *
 * directions, including the current plane   *
 *********************************************/

dlist = gamgi_engine_dlist_atom_object (group->object.object, NULL);
dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_GROUP);
dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_PLANE);
dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_DIRECTION);

/***************************************************
 * get rotation and translation to transform atoms *
 * back to where they would be if plane rotation   *
 * and translation were (0, 0, 0) and (0, 0, 0)    *
 ***************************************************/

gamgi_math_position_group_rotation_get (group, euler);
gamgi_math_euler_to_matrix (euler, matrix);
gamgi_math_matrix_inverse (matrix, inverse);

gamgi_math_matrix_vector (inverse, group->origin, translate);
gamgi_math_position_translation (group->object.object, position);
gamgi_math_vector_sub (position, translate, translate);

/****************************************************************
 * try to find the original atoms of the reference atoms, where *
 * plane rotation and translation are (0, 0, 0) and (0, 0, 0)   *
 ****************************************************************/

gamgi_math_matrix_vector (inverse, (*atom)->position, position);
gamgi_math_vector_add (position, translate, position);

if (static_scan (group, dlist, position, atom) == FALSE) return FALSE;

return TRUE;
}

static void static_remove (gamgi_group *group)
{
gamgi_dlist *dlist;

dlist = group->group_start;
while (dlist != NULL)
  { gamgi_engine_remove_group (GAMGI_CAST_GROUP dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); }
group->group_start = NULL;
group->group_end = NULL;
 
dlist = group->plane_start;
while (dlist != NULL)
  { gamgi_engine_remove_plane (GAMGI_CAST_PLANE dlist->data); 
    dlist = gamgi_engine_dlist_remove_start (dlist); }
group->plane_start = NULL;
group->plane_end = NULL;

dlist = group->direction_start;
while (dlist != NULL)
  { gamgi_engine_remove_direction (GAMGI_CAST_DIRECTION dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); }
group->direction_start = NULL;
group->direction_end = NULL;

dlist = group->atom_start;
while (dlist != NULL)
  { gamgi_engine_remove_atom (GAMGI_CAST_ATOM dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); }
group->atom_start = NULL;
group->atom_end = NULL;

dlist = group->orbital_start;
while (dlist != NULL)
  { gamgi_engine_remove_orbital (GAMGI_CAST_ORBITAL dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); }
group->orbital_start = NULL;
group->orbital_end = NULL;

dlist = group->text_start;
while (dlist != NULL)
  { gamgi_engine_remove_text (GAMGI_CAST_TEXT dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); }
group->text_start = NULL;
group->text_end = NULL;
}

static gamgi_enum static_class (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *combo;
const char *name;
int row;

/***************************************************
 * when the group entry is empty, look for a group *
 ***************************************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_group");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == FALSE) return GAMGI_ENGINE_GROUP;

/*******************************************
 * otherwise select which object to select *
 *******************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
if (row == STATIC_TYPE_CONTAINER) return GAMGI_ENGINE_GROUP;
if (row == STATIC_TYPE_POLYTOPE) return GAMGI_ENGINE_ATOM;

/***********************************************
 * this case can only occur when Type is Local *
 ***********************************************/

return FALSE;
}

static void static_number (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_number");
gtk_entry_set_text (GTK_ENTRY (entry), "");
}

static void static_start_container (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;

/**********************
 * start central Atom *
 **********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom");
gtk_entry_set_text (GTK_ENTRY (entry), "");
}

static void static_start_polytope (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *entry;
GtkWidget *button;

/**********************
 * start central Atom *
 **********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/***************************
 * start polytope Autonomy *
 ***************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), STATIC_AUTONOMY_LOCAL);

/******************************
 * start Number of neighbours *
 ******************************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_number");
gtk_button_clicked (GTK_BUTTON (button));

/*******************************************
 * start face tolerance: Visible, Coplanar *
 *******************************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_visible");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_coplanar");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/*****************************
 * start Style, Color, Scale *
 *****************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_style");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), STATIC_STYLE_LOCAL);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_red");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_green");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_blue");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
gtk_entry_set_text (GTK_ENTRY (entry), "");
}

static void static_start_position (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
char token[GAMGI_ENGINE_TOKEN];

/***********************
 * start Position page *
 ***********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_CELL_X);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_CELL_Y);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_CELL_Z);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e1");
sprintf (token, "%.*f", gamgi->gamgi->angle, GAMGI_PHYS_CELL_E1);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e2");
sprintf (token, "%.*f", gamgi->gamgi->angle, GAMGI_PHYS_CELL_E2);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e3");
sprintf (token, "%.*f", gamgi->gamgi->angle, GAMGI_PHYS_CELL_E3);
gtk_entry_set_text (GTK_ENTRY (entry), token);
}

static void static_autonomy (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *vbox;
GtkWidget *entry;
GtkWidget *label;
GtkWidget *combo;
char token[GAMGI_ENGINE_TOKEN];
int row;

vbox = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_position");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
label = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_scale");

/************************
 * hide,show position   *
 * enable,disable scale *
 ************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
if (row != GAMGI_GTK_ALL - 1 && row != GAMGI_GTK_PARTIAL - 1)
  {
  gtk_widget_hide (vbox);

  gtk_entry_set_text (GTK_ENTRY (entry), "");
  if (row == GAMGI_GTK_NONE - 1)
    {
    gtk_widget_set_sensitive (entry, FALSE);
    gtk_widget_set_sensitive (label, FALSE);
    }
  else
    {
    gtk_widget_set_sensitive (entry, TRUE);
    gtk_widget_set_sensitive (label, TRUE);
    }

  }
else
  {
  if (gtk_widget_get_visible (vbox) == FALSE)
    {
    static_start_position (window);

    gtk_widget_set_sensitive (entry, TRUE);
    gtk_widget_set_sensitive (label, TRUE);
    sprintf (token, "%.*f", GAMGI_MATH_DECIMAL_SCALE, GAMGI_MESA_SCALE);
    gtk_entry_set_text (GTK_ENTRY (entry), token);
    }
  gtk_widget_show (vbox);
  }

}

static void static_type (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *vbox_scope, *vbox_tolerance, *vbox_view;
GtkWidget *combo;
GtkWidget *label;
GtkWidget *entry;
int row;

/************
 * get data *
 ************/

vbox_scope = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_scope");
vbox_tolerance = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_tolerance");
vbox_view = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_view");

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
if (row == STATIC_TYPE_POLYTOPE)
  {
  /*****************
   * Polytope mode *
   *****************/

  label = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_central");
  gtk_widget_set_sensitive (label, TRUE);
  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom");
  gtk_widget_set_sensitive (entry, TRUE);
  label = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_atom");
  gtk_widget_set_sensitive (label, TRUE);

  gtk_widget_show (vbox_scope);
  gtk_widget_show (vbox_tolerance);
  gtk_widget_show (vbox_view);
  static_start_polytope (window);
  }
else
  {
  /***************************
   * Container or Local mode *
   ***************************/

  label = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_central");
  gtk_widget_set_sensitive (label, FALSE);
  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom");
  gtk_widget_set_sensitive (entry, FALSE);
  label = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_atom");
  gtk_widget_set_sensitive (label, FALSE);

  gtk_widget_hide (vbox_scope);
  gtk_widget_hide (vbox_tolerance);
  gtk_widget_hide (vbox_view);
  static_start_container (window);
  }

}

static void static_reset (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *entry;

/******************************
 * clean group and atom names *
 ******************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_group");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/*********************************
 * set autonomy, element, number *
 *********************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), STATIC_AUTONOMY_LOCAL);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_element");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_number");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/*****************************
 * set position, orientation *
 *****************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e1");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e2");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e3");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/**************************************
 * reset visible, coplanar tolerances *
 **************************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_visible");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_coplanar");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/******************************
 * reset style, colors, scale *
 ******************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_style");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), STATIC_STYLE_LOCAL);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_red");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_green");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_blue");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
gtk_entry_set_text (GTK_ENTRY (entry), "");
}

/****************** internal function ***************
 *                                                  *
 *                    STATIC_GLOBAL                 *
 *                                                  *
 * Makes the group entry (but not the table button) *
 * sensitive when the local button is pressed and   *
 * insensitive when the global button is pressed.   *
 *                                                  *
 ****************************************************/

static void static_global (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *hbox, *button;

hbox = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "hbox_group");

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == FALSE)
  {
  /****************************
   * Choose the atom directly *
   ****************************/

  gtk_widget_set_sensitive (hbox, TRUE);
  }
else
  {
  /****************************************
   * Choose the atoms previously selected *
   ****************************************/

  gtk_widget_set_sensitive (hbox, FALSE);
  static_reset (window);
  }
}

static void static_ok_container (gamgi_group *group, char *group_name, gamgi_window *window)
{
/**********************************************
 * Update the group name, which might be new. *
 **********************************************/

strcpy (group->object.name, group_name);
}

static void static_ok_polytope (gamgi_group *group, char *group_name, gamgi_window *window)
{
gamgi_group *group_old;
gamgi_atom *atom;
gamgi_object *parent;
gamgi_dlist *dlist;
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *button;
GtkWidget *combo;
const char *name;
gamgi_bool autonomy, faces, inside, copy;
char token[GAMGI_ENGINE_TOKEN];
double euler[3], origin[3];
double radius, visible, coplanar, scale;
float red, green, blue;
int element, number;
int independent, row;

/**************************
 * get initial parameters *
 **************************/

autonomy = group->autonomy;
element = group->element;
number = group->number;
visible = group->visible;
coplanar = group->coplanar;
faces = group->faces;
red = group->red;
green = group->green;
blue = group->blue;
scale = group->object.scale;

gamgi_math_position_group_rotation_get (group, euler);
gamgi_math_position_group_translation_get (group, origin);

/***************************
 * 1) Get old central atom *
 * 2) Get new central atom *
 ***************************/

atom = gamgi_engine_find_atom_layer (group->atom, window->layer);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE)
  atom = GAMGI_CAST_ATOM gamgi_gtk_object_name_number (dialog,
  "entry_atom", GAMGI_ENGINE_ATOM, window);
if (atom == NULL)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid central atom", window);
  return;
  }

inside = FALSE;
parent = atom->object.object;
if (parent == GAMGI_CAST_OBJECT group) inside = TRUE;
else
  {
  if (parent->class == GAMGI_ENGINE_GROUP || parent->class == GAMGI_ENGINE_PLANE ||
  parent->class == GAMGI_ENGINE_DIRECTION)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid central atom", window);
    return;
    }
  }

/***********************************************
 * get Autonomy: the default is GAMGI_GTK_NONE *
 ***********************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
if (row < STATIC_AUTONOMY_LOCAL) autonomy = row + 1;

/*********************************************************************
 * number of neighbours or maximum radius, counting from the nearest *
 * one, so for example radius = 1.1 means the range [r, 1.1r] where  *
 * r is the distance from central to the nearest neighbour           *
 *********************************************************************/

radius = 0.0;
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_number");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE)
  {
  button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_number");
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE)
    {
    if (gamgi_io_token_int_scan (name, &number, 2, INT_MAX) == FALSE)
      {
      gamgi_gtk_dialog_message_create ("Error", 
      "Invalid neighbours number", window);
      return;
      }
    }
  else
    {
    if (gamgi_io_token_double_scan (name, &radius,
    GAMGI_MATH_TOLERANCE_LENGTH, DBL_MAX) == FALSE)
      {
      gamgi_gtk_dialog_message_create ("Error",
      "Invalid neighbours maximum radius", window);
      return;
      }
    number = 0;
    }
  }

/***********************************
 * get element for neighbour atoms *
 ***********************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_element");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE && (gamgi_io_token_alpha_scan (name, 
token, GAMGI_IO_TEXT, GAMGI_ENGINE_TOKEN) == FALSE || 
gamgi_chem_atom_number (token, &element) == FALSE))
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid neighbours element", window);
  return;
  }

if (autonomy == GAMGI_GTK_ALL)
  {
  /****************************
   * origin x,y,z coordinates *
   ****************************/

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_check (name) == TRUE &&
  gamgi_io_token_double_scan (name, &origin[0], -DBL_MAX, DBL_MAX) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid position data", window);
    return;
    }

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_check (name) == TRUE &&
  gamgi_io_token_double_scan (name, &origin[1], -DBL_MAX, DBL_MAX) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid position data", window);
    return;
    }

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_check (name) == TRUE &&
  gamgi_io_token_double_scan (name, &origin[2], -DBL_MAX, DBL_MAX) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid position data", window);
    return;
    }

  /*************************
   * e1,e2,e3 euler angles *
   *************************/

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e1");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_check (name) == TRUE &&
  gamgi_io_token_double_scan (name, &euler[0], 0.0, 180.0) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid angle data", window);
    return;
    }

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e2");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_check (name) == TRUE &&
  gamgi_io_token_double_scan (name, &euler[1], 0.0, 360.0) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid angle data", window);
    return;
    }

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e3");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_check (name) == TRUE &&
  gamgi_io_token_double_scan (name, &euler[2], 0.0, 360.0) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid angle data", window);
    return;
    }
  }

/*************
 * get Style *
 *************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_style");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
if (row < STATIC_STYLE_LOCAL) faces = row;

/*************
 * get Color *
 *************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_red");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_float_scan (name, &red, 0.0, 1.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid color", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_green");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_float_scan (name, &green, 0.0, 1.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid color", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_blue");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_float_scan (name, &blue, 0.0, 1.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid color", window);
  return;
  }

/*************
 * get Scale *
 *************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE && gamgi_io_token_double_scan (name,
&scale, GAMGI_MESA_SCALE_LOWER, GAMGI_MESA_SCALE_UPPER) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane scale", window);
  return;
  }

/****************
 * global tests *
 ****************/

if (gamgi_engine_find_object (GAMGI_CAST_OBJECT atom, window->focus) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid searching scope for central atom", window);
  return;
  }

/*************************************************************************
 * polytope groups with autonomy all can be modified to autonomy partial *
 * or none, as long as the central atom in the original position can     *
 * replace that entered by the user, later removed with the group        *
 *************************************************************************/

if (group->autonomy == GAMGI_GTK_ALL && autonomy != GAMGI_GTK_ALL && 
inside == TRUE && static_find (group, &atom) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid central atom", window);
  return;
  }

/**********************************
 * input data is ok: save History *
 **********************************/

gamgi_gtk_history_update (window->layer, window->layer->undo, GAMGI_ENGINE_UNDO);

group_old = gamgi_engine_create_group ();
gamgi_engine_copy_out_group (group, group_old);
group_old->object.selection = NULL;
parent = group->object.object;

/**************
 * reset data *
 **************/

free (group->points);
free (group->loops);
group->points = NULL;
group->loops = NULL;
group->n_points = 0;
group->n_loops = 0;

/*******************************
 * prepare building conditions *
 *******************************/

if (autonomy == GAMGI_GTK_NONE)
  {
  gamgi_math_vector_zero (origin);
  gamgi_math_vector_zero (euler);
  scale = 1.0;
  }

copy = (group->autonomy != GAMGI_GTK_ALL || autonomy != GAMGI_GTK_ALL || inside != TRUE);

if (group->autonomy == GAMGI_GTK_ALL && copy == TRUE)
  {
  /*************************************************
   * Before removing the plane, check if the       *
   * current object is affected. In this case, the *
   * current layer becomes the new current object. *
   *************************************************/

  if (gamgi_engine_find_object (window->focus,
  GAMGI_CAST_OBJECT group) == TRUE)
    gamgi_gtk_object_focus_local (GAMGI_CAST_OBJECT window->layer, window);

  static_remove (group);
  }

if (copy == TRUE)
  {
  /*****************************************
   * filter list of atoms in scope, to     *
   * avoid atoms with the same coordinates *
   *****************************************/

  dlist = gamgi_engine_dlist_atom_object (window->focus, NULL);
  dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_GROUP);
  dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_PLANE);
  dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_DIRECTION);

  gamgi_engine_unlink_group (group);
  }
else
  {
  /******************************************
   * get list of atoms already in the group *
   ******************************************/

  dlist = gamgi_engine_dlist_atom_group (group, NULL);
  }

/************************************************************
 * save new data: update the group name, which might be new *
 ************************************************************/

strcpy (group->object.name, group_name);
group->autonomy = autonomy;
group->element = element;
group->number = number;
group->visible = visible;
group->coplanar = coplanar;
group->faces = faces;
group->red = red;
group->green = green;
group->blue = blue;
group->object.scale = scale;

/********************************************************
 * apply Polytope algorithm: independent is the minimum *
 * number of independent atom positions that were found *
 ********************************************************/

independent = gamgi_chem_polytope (dlist, group, atom, FALSE, radius, copy, NULL);
if (independent < 2)
  {
  if (group->object.object == NULL)
    gamgi_engine_link_object_object (GAMGI_CAST_OBJECT group, parent);
  gamgi_engine_copy_out_group (group_old, group);
  gamgi_engine_remove_group (group_old);

  gamgi_gtk_dialog_message_create ("Error", 
  "Insufficient number of valid neighbours", window);
  return;
  }
else if (independent < 4 && copy == TRUE)
  {
  gamgi_engine_remove_group (group);
  gamgi_engine_remove_group (group_old);
  }
else
  {
  gamgi_engine_remove_group (group_old);
  gamgi_math_position_group_rotation_set (group, euler);
  gamgi_math_position_group_translation_set (group, origin);
  }
/*****************************************
 * Redraw gl_area image and reset dialog *
 *****************************************/

gtk_widget_queue_draw (window->area);
static_reset (window);
}

/******************* internal function *****************
 *                                                     *
 *                        STATIC_OK                    *
 *                                                     *
 * This is the function that actually creates a new    *
 * group. It is triggered when the user presses Ok.    *
 *                                                     *
 *******************************************************/

static void static_ok (GtkWidget *widget, void *data)
{
gamgi_group *group;
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *button;
GtkWidget *combo;
char token[GAMGI_ENGINE_TOKEN];
int row;

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == FALSE)
  {
  /****************
   * Local method *
   ****************/

  /**************************************
   * Get name (possibly new) and number *
   **************************************/

  group = GAMGI_CAST_GROUP gamgi_gtk_object_number (dialog,
  "entry_group", GAMGI_ENGINE_GROUP, window, token);
  if (group == NULL)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid Group Object", window);
    return;
    }

  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
  row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  if (row == STATIC_TYPE_POLYTOPE ||
  (row == STATIC_TYPE_LOCAL && group->type == GAMGI_CHEM_POLYTOPE))
    static_ok_polytope (group, token, window);
  else
    static_ok_container (group, token, window);
  }
}

static void static_init (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *notebook;
GtkWidget *combo;
GtkRequisition size;

/***************************************************
 * Set notebook current size as its maximum size:  *
 * this is needed because pages are shown/hidden,  *
 * changing the currently needed size for dialog.  *
 * Fixing the dialog size only partly solves the   *
 * problem because the page size can still change. *
 ***************************************************/

notebook = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "notebook");
gtk_widget_size_request (GTK_WIDGET (notebook), &size);
gtk_widget_set_size_request (notebook, size.width, size.height);

/********************************************
 * hide pages AFTER measuring notebook size *
 ********************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), STATIC_TYPE_LOCAL);
static_type (NULL, window);

static_autonomy (NULL, window);
}

static void static_press (gamgi_object *object, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
gamgi_group *group;
gamgi_atom *atom;
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *combo;
char string[2 * GAMGI_ENGINE_TOKEN];
double origin[3], euler[3];
gamgi_enum class;

class = static_class (window);
if (class == GAMGI_ENGINE_GROUP)
  {
  group = GAMGI_CAST_GROUP object;

  static_reset (window);

  /**********************************
   * get group, atom identification *
   **********************************/

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_group");
  sprintf (string, "%s %d", object->name, object->number);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), group->type - 1);
  if (group->type == GAMGI_CHEM_CONTAINER) return;

  atom = gamgi_engine_find_atom_layer (group->atom, window->layer);
  if (atom != NULL)
    {
    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom");
    sprintf (string, "%s %d", atom->object.name, atom->object.number);
    gtk_entry_set_text (GTK_ENTRY (entry), string);
    }

  /***********************************************
   * set Group autonomy, element, number, radius *
   ***********************************************/

  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), group->autonomy - 1);

  if (group->element != -1)
    {
    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_element");
    gamgi_chem_atom_name (group->element, string);
    gtk_entry_set_text (GTK_ENTRY (entry), string);
    }

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_number");
  sprintf (string, "%d", group->number);
  gtk_entry_set_text (GTK_ENTRY (entry), string);
  
  /************************************
   * get position, orientation, scale *
   ************************************/

  if (group->autonomy != GAMGI_GTK_NONE)
    {
    /************
     * position *
     ************/

    gamgi_math_position_group_translation_get (group, origin);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
    sprintf (string, "%.*f", gamgi->gamgi->length, origin[0]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
    sprintf (string, "%.*f", gamgi->gamgi->length, origin[1]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
    sprintf (string, "%.*f", gamgi->gamgi->length, origin[2]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    /***************
     * orientation *
     ***************/

    gamgi_math_position_group_rotation_get (group, euler);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e1");
    sprintf (string, "%.*f", gamgi->gamgi->angle, euler[0]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e2");
    sprintf (string, "%.*f", gamgi->gamgi->angle, euler[1]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e3");
    sprintf (string, "%.*f", gamgi->gamgi->angle, euler[2]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    /*********
     * scale *
     *********/

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
    sprintf (string, "%.*f", GAMGI_MATH_DECIMAL_SCALE, group->object.scale);
    gtk_entry_set_text (GTK_ENTRY (entry), string);
    }

  /*****************************************
   * get face tolerance: Visible, Coplanar *
   *****************************************/

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_visible");
  sprintf (string, "%.*E", 2, group->visible);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_coplanar");
  sprintf (string, "%.*E", 2, group->coplanar);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  /********************
   * get faces, color *
   ********************/

  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_style");
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), group->faces);

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_red");
  sprintf (string, "%.*f", GAMGI_MATH_DECIMAL_COLOR, group->red);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_green");
  sprintf (string, "%.*f", GAMGI_MATH_DECIMAL_COLOR, group->green);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_blue");
  sprintf (string, "%.*f", GAMGI_MATH_DECIMAL_COLOR, group->blue);
  gtk_entry_set_text (GTK_ENTRY (entry), string);
  }

else if (class == GAMGI_ENGINE_ATOM)
  {
  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom");
  sprintf (string, "%s %d", object->name, object->number);
  gtk_entry_set_text (GTK_ENTRY (entry), string);
  }

}

/*************** external function *************
 *                                             *
 *         GAMGI_GTK_GROUP_MODIFY_PRESS        *
 *                                             *
 *                                             *
 ***********************************************/

void gamgi_gtk_group_modify_press (gamgi_window *window_mouse,
GdkEventButton *event, int x, int y, gamgi_window *window_dialog)
{
GtkWidget *dialog = window_dialog->dialog0;
GtkWidget *button;

/******************************
 * local mouse selection only *
 ******************************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE ||
window_dialog != window_mouse) return;

gamgi_mesa_select_object (window_mouse, x, y,
static_class (window_dialog), TRUE, static_press);
}

/******************* external function **************
 *                                                  *
 *                GAMGI_GTK_GROUP_MODIFY            *
 *                                                  *
 * Creates the dialog window used to create groups. *
 *                                                  *
 ****************************************************/

void gamgi_gtk_group_modify (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog;
GtkWidget *notebook;
GtkWidget *button;
GtkWidget *label;
GtkWidget *entry;
GtkWidget *table;
GtkWidget *hbox_center, *hbox_left, *hbox_left_left;
GtkWidget *vbox_dialog, *vbox_page, *vbox_center, *vbox_top, *vbox_top_top;
GtkWidget *combo;
GtkListStore *store;
GtkCellRenderer *renderer;
GtkTreeIter iter;

/******************
 * dialog level 0 *
 ******************/

dialog = gamgi_gtk_dialog_task0_create ("Group Modify", window);
window->action = GAMGI_GTK_GROUP_MODIFY;
gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);

/********************
 * global container *
 ********************/

vbox_dialog = gtk_vbox_new (FALSE, 5);
gtk_container_add (GTK_CONTAINER (dialog), vbox_dialog);
gtk_widget_show (vbox_dialog);

/*****************
 * Group objects *
 *****************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_dialog), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 10);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

/*********
 * Local *
 *********/

hbox_left_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_left), hbox_left_left, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "hbox_group", hbox_left_left);
gtk_widget_show (hbox_left_left);

label = gtk_label_new ("Group");
gtk_box_pack_start (GTK_BOX (hbox_left_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_object_set_data (G_OBJECT (dialog), "entry_group", entry);
gtk_widget_show (entry);

/**********
 * Global *
 **********/

button = gtk_toggle_button_new_with_label ("Global");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_signal_connect (button, "toggled",
G_CALLBACK (static_global), window);
g_object_set_data (G_OBJECT (dialog), "button_global", button);
gtk_widget_show (button);

/************
 * notebook *
 ************/

notebook = gtk_notebook_new ();
gtk_box_pack_start (GTK_BOX (vbox_dialog), notebook, FALSE, FALSE, 0);
gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
g_object_set_data (G_OBJECT (dialog), "notebook", notebook);
gtk_widget_show (notebook);

/*************
 * Type page *
 *************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new (" Type ");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
g_object_set_data (G_OBJECT (dialog), "vbox_type", vbox_page);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

vbox_top = gtk_vbox_new (FALSE, 20);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/********
 * Type *
 ********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_signal_connect (combo, "changed",
G_CALLBACK (static_type), window);
g_object_set_data (G_OBJECT (dialog), "combo_type", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Container", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Polytope", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/************
 * Polytope *
 ************/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_top), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

/****************
 * Central Atom *
 ****************/

label = gtk_label_new ("Central");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "label_central", label);
gtk_widget_show (label);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Atom");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "label_atom", label);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_object_set_data (G_OBJECT (dialog), "entry_atom", entry);
gtk_widget_show (entry);

/************************
 * Polytope: Scope page *
 ************************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new (" Scope ");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
g_object_set_data (G_OBJECT (dialog), "vbox_scope", vbox_page);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

vbox_top = gtk_vbox_new (FALSE, 20);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/********************************
 * Autonomy: All, Partial, None *
 ********************************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_end (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Autonomy");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 5);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_signal_connect (combo, "changed",
G_CALLBACK (static_autonomy), window);
g_object_set_data (G_OBJECT (dialog), "combo_autonomy", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "All", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Partial", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "None", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/**************
 * Neighbours *
 **************/

vbox_top_top = gtk_vbox_new (FALSE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Neighbours");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (2, 2, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_widget_show (table);

label = gtk_label_new ("Element");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 10, 5);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 1, 2, GTK_SHRINK, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_3);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_3);
g_object_set_data (G_OBJECT (dialog), "entry_element", entry);
gtk_widget_show (entry);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_table_attach (GTK_TABLE (table), hbox_left, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 10, 5);
gtk_widget_show (hbox_left);

button = gtk_radio_button_new_with_label (NULL, "Number");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_signal_connect (button, "clicked",
G_CALLBACK (static_number), window);
g_object_set_data (G_OBJECT (dialog), "button_number", button);
gtk_widget_show (button);

button = gtk_radio_button_new_with_label (
gtk_radio_button_get_group (GTK_RADIO_BUTTON (button)), "Radius");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_signal_connect (button, "clicked",
G_CALLBACK (static_number), window);
gtk_widget_show (button);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 1, 2, GTK_SHRINK, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_number", entry);
gtk_widget_show (entry);

/*****************
 * Position page *
 *****************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new ("Position");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
g_object_set_data (G_OBJECT (dialog), "vbox_position", vbox_page);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 10);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

/**********
 * Origin *
 **********/

vbox_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

label = gtk_label_new ("Translation");
gtk_box_pack_start (GTK_BOX (hbox_center), label, FALSE, FALSE, 0);
gtk_widget_show (label);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 6, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 5);
gtk_widget_show (table);

label = gtk_label_new ("X");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("Y");
gtk_table_attach (GTK_TABLE (table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("Z");
gtk_table_attach (GTK_TABLE (table), label, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_x", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_y", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_z", entry);
gtk_widget_show (entry);

/*********
 * Angle *
 *********/

vbox_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

label = gtk_label_new ("Rotation");
gtk_box_pack_start (GTK_BOX (hbox_center), label, FALSE, FALSE, 0);
gtk_widget_show (label);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 6, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 5);
gtk_widget_show (table);

label = gtk_label_new ("E1");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("E2");
gtk_table_attach (GTK_TABLE (table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("E3");
gtk_table_attach (GTK_TABLE (table), label, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_e1", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_e2", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_e3", entry);
gtk_widget_show (entry);

/****************************
 * Polytope: Tolerance page *
 ****************************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new ("Tolerance");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
g_object_set_data (G_OBJECT (dialog), "vbox_tolerance", vbox_page);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

/*************************************
 * Face Tolerance: Coplanar, Visible *
 *************************************/

hbox_center = gtk_hbox_new (TRUE, 5);
gtk_box_pack_start (GTK_BOX (vbox_center), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (2, 3, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_widget_show (table);

label = gtk_label_new ("Faces");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("Visible");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_EXPAND, GTK_FILL, 0, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 1, 2, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_10);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_10);
g_object_set_data (G_OBJECT (dialog), "entry_visible", entry);
gtk_widget_show (entry);

label = gtk_label_new ("Coplanar");
gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1, GTK_EXPAND, GTK_FILL, 0, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 1, 2, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_10);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_10);
g_object_set_data (G_OBJECT (dialog), "entry_coplanar", entry);
gtk_widget_show (entry);

/***********************
 * Polytope: View page *
 ***********************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new (" View ");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
g_object_set_data (G_OBJECT (dialog), "vbox_view", vbox_page);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

vbox_top = gtk_vbox_new (FALSE, 15);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/*********
 * Style *
 *********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 20);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

hbox_left_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_left), hbox_left_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left_left);

label = gtk_label_new ("Style");
gtk_box_pack_start (GTK_BOX (hbox_left_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left_left), combo, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "combo_style", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Wired", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Solid", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/*********
 * Color *
 *********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 7, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_widget_show (table);

label = gtk_label_new ("Color");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "label_color", label);
gtk_widget_show (label);

label = gtk_label_new ("R");
gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "label_red", label);
gtk_widget_show (label);

label = gtk_label_new ("G");
gtk_table_attach (GTK_TABLE (table), label, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "label_green", label);
gtk_widget_show (label);

label = gtk_label_new ("B");
gtk_table_attach (GTK_TABLE (table), label, 6, 7, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "label_blue", label);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_red", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_green", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_blue", entry);
gtk_widget_show (entry);

/*********
 * Scale *
 *********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Scale");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "label_scale", label);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_scale", entry);
gtk_widget_show (entry);

/*********************
 * Ok/Cancel buttons *
 *********************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_dialog), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

button = gamgi_gtk_dialog_button_create ("Ok", NULL);
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
g_signal_connect (button, "clicked",
G_CALLBACK (static_ok), window);
gtk_widget_show (button);

button = gamgi_gtk_dialog_button_create ("Cancel", "red");
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
gtk_widget_grab_focus (button);
g_signal_connect (button, "clicked",
G_CALLBACK (gamgi_gtk_dialog_task0_remove), window);
gtk_widget_show (button);

static_init (window);
gtk_widget_show (dialog);
}
