# This file was created by Pietro Battiston <toobaz@email.it> with contents from
# the gup.py file, contained in the original gup distribution
# (which can be found at http://code.google.com/p/galleryuploader/).
# Hence, the license and copyright for the original gup distribution apply:
#
#
# GUP is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any
# later version.
#
# GUP is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along
# with GUP; if not, write to the Free Software Foundation, Inc., 51
# Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
#
# Copyright (C) 2007 Julio Biason

import logging

import os
from urllib2 import HTTPError
from albums import AlbumList
from gallery import Gallery

# Exceptions
class NoSuchGalleryError(Exception): 
    '''The requested gallery doesn't exist.'''
    pass

class CantSelectAGalleryError(Exception): 
    '''The system can't come with a gallery because there is more than one
    saved.'''
    pass

def locate_gallery(configuration, name):
    '''Look for a gallery in the configuration, or return the name of one
    if there is only one gallery.'''
    if name:
        if not configuration.has_section(name):
            raise NoSuchGalleryError, name
        return name

    sections = configuration.sections()
    if len(sections) == 0:
        return ''

    if len(sections) > 1:
        raise CantSelectAGalleryError

    return sections[0]

def merge_options(configuration, gallery, command_line_options):
    '''Merge the options from the gallery in the configuration file and the
    ones passed in the command line into a single location.'''
    if configuration.has_option(gallery, 'url'):
        config_url = configuration.get(gallery, 'url')

    if configuration.has_option(gallery, 'user'):
        config_user = configuration.get(gallery, 'user')

    if configuration.has_option(gallery, 'password'):
        config_password = configuration.get(gallery, 'password')

    if not command_line_options.url:
        command_line_options.url = config_url

    if not command_line_options.user:
        command_line_options.user = config_user

    if not command_line_options.password:
        command_line_options.password = config_password

    return command_line_options

def missing_options(options):
    '''Check if the required options are available.'''
    if not options.url:
        logging.error('Missing gallery URL')
        return True

    if not options.user:
        logging.error('Missing user')
        return True

    if not options.password:
        logging.error('Missing password')
        return True

    return False

def create_base_config_dir():
    '''Create the base configuration directory, where configuration and
    cache are stored.'''
    if not os.access(os.path.expanduser('~/.gup'), os.F_OK):
        os.mkdir(os.path.expanduser('~/.gup'))

def save_gallery_info(configuration, gallery_name, options):
    '''Save the gallery information back to the config file.'''
    logging.debug('Saving gallery "%s" information...', gallery_name)
    if not configuration.has_section(gallery_name):
        configuration.add_section(gallery_name)
    configuration.set(gallery_name, 'url'     , options.url)
    configuration.set(gallery_name, 'user'    , options.user)
    configuration.set(gallery_name, 'password', options.password)

    create_base_config_dir()
    contents = file(os.path.expanduser('~/.gup/config.ini'), 'w')
    configuration.write(contents)
    contents.close()

def gallery_cache_name(gallery_name):
    '''Return a string with the name of the gallery cache.'''
    return os.path.expanduser('~/.gup/%s.cache' % gallery_name)

def check_album_cache(gallery_name, options):
    '''Check if the album cache exists and, if it doesn't or the reload
    option was used, load it from the server. Returns a tuple with the
    album cache and a bool indicating if it needs to be saved.'''
    cache_name = gallery_cache_name(gallery_name)

    if not options.reload:
        try:
            os.stat(cache_name)
        except OSError:
            options.reload = True
            logging.info('There isn''t a cache for "%s", downloading...',
                    gallery_name)
    else:
        logging.info('Downloading album information for "%s"...',
                gallery_name)

    albums = AlbumList()
    if not options.reload:
        albums.load(cache_name)
        return (albums, False)

    # load from server
    connection = Gallery(options.url, options.user,
            options.password)
    for album in connection.fetch_albums_prune():
        albums.add(int(album[0]), album[1], int(album[2]))
    
    return (albums, True)

def display_albums(tree):
    '''Display the albums in a tree format.'''
    for (level, name, album_id) in tree.tree():
        logging.info('%s%s (%s)', '  ' * level, name, album_id)

def upload(options, files):
    '''Upload files to the gallery.'''
    connection = Gallery(options.url, options.user, options.password)
    total   = len(files)
    current = 0
    for image in files:
        current += 1
        logging.info('Uploading "%s" (%d/%d)...',
                image, current, total)
        try:
            connection.add_item(str(options.album), image)
            if options.delete:
                logging.info('Removing "%s"...',
                        image)
                os.remove(image)
        except HTTPError, e:
            logging.error('HTTP error: %s' % (e))

def save_album_cache(gallery_name, albums):
    '''Save the gallery cache.'''
    cache_name = gallery_cache_name(gallery_name)
    create_base_config_dir()
    albums.save(cache_name)

def make_name(album_list, stop_id):
    '''From a tree of albums, make a flat name for it.

    album_list - the tree of the album
    stop_id - id of the album (top stop generating the list, since the
        search_id function will return the children of the id.'''

    names = []
    tree = album_list.tree()
    tree.next() # skip the root node ('/')

    for (_, name, album_id) in tree:
        names.append(name)
        if album_id == stop_id:
            break

    return '/'.join(names)

