/*
 *  Copyright (C) 2001, 2002 Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* FIXME: finish me! */

/* CREDITS:
 * This file was based initially in nautilus-mozilla-content-view.h, from Nautilus */

#include <config.h>

#include "galeon-nautilus-view.h"
#include "print.h"
#include "find.h"
#include "galeon.h"
#include "embed.h"
#include "eel-gtk-macros.h"
#include "mozilla_i18n.h"
#include "mozilla.h"
#include "misc_string.h"

// FIXME: I guess we don't need all this
#include <bonobo/bonobo-control.h>
#include <bonobo/bonobo-ui-component.h>
#include <bonobo/bonobo-ui-util.h>
#include <bonobo/bonobo-zoomable.h>
#include <gtk/gtksignal.h>
#include <libgnome/gnome-i18n.h>
#include "gtkmozembed.h"
#include <gtk/gtkwindow.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomevfs/gnome-vfs.h>
#include <stdlib.h>

//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

#define MENU_VIEW_CHARSET_ENCODING_PATH "/menu/View/Encoding"

/* Buffer size for streaming contents from gnome-vfs into mozilla */
#define VFS_READ_BUFFER_SIZE	(40 * 1024)

/* property bag properties */
enum {
	ICON_NAME,
	COMPONENT_INFO
};

struct GaleonNautilusViewPrivate {
	GaleonEmbed         *embed;
	char 		    *uri;	/* The URI stored here is nautilus's
					   idea of the URI */
	
	NautilusView 	    *nautilus_view;
	BonoboPropertyBag   *property_bag;
	
	char		    *vfs_read_buffer;
	GnomeVFSAsyncHandle *vfs_handle;

	BonoboUIComponent *ui;

	BonoboZoomable *zoomable;

};

/* GTK Type System */
static void galeon_nautilus_view_initialize_class (GaleonNautilusViewClass *klass);
static void galeon_nautilus_view_initialize       (GaleonNautilusView      *view);
static void galeon_nautilus_view_destroy          (GtkObject               *object);


/* Gnome VFS callback functions */
static void	vfs_open_cb			(GnomeVFSAsyncHandle 		*handle,
						 GnomeVFSResult			result,
						 gpointer			data);

static void	vfs_read_cb			(GnomeVFSAsyncHandle		*handle,
						 GnomeVFSResult			result,
						 gpointer			buffer,
						 GnomeVFSFileSize		bytes_requested,
						 GnomeVFSFileSize		bytes_read,
						 gpointer			data);

/* NautilusView callback functions */
static void	view_load_location_cb		(NautilusView 			*nautilus_view,
						 const char 			*location,
						 gpointer 			data);

/* GtkEmbedMoz callback functions */

static void	mozilla_realize_cb		(GtkWidget 			*mozilla,
						 gpointer			user_data);

/* Private GaleonNautilusView functions */ 

static void	navigate_mozilla_to_nautilus_uri	(GaleonNautilusView     *view,
							 const char		*uri);

static void	cancel_pending_vfs_operation		(GaleonNautilusView	*view);

static void     galeon_nautilus_view_merge_menus        (GaleonNautilusView *view);
static gint     strcasestr_cb                           (gconstpointer a, gconstpointer b);

static void     charset_encoding_changed_cb             (BonoboUIComponent *component,
							 gpointer callback_data,
							 const char *path);
/* functions taken from bonobo-extensions.c */
static void     nautilus_bonobo_add_submenu         (BonoboUIComponent *ui,
						     const char *path,
						     const char *label);
static void     nautilus_bonobo_set_label           (BonoboUIComponent *ui,
						     const char *path,
						     const char *label);
static void     add_numbered_menu_item_internal (BonoboUIComponent *ui,
						 const char *container_path,
						 guint index,
						 const char *label,
						 gboolean is_toggle,
						 GdkPixbuf *pixbuf);
static void     nautilus_bonobo_add_numbered_menu_item (BonoboUIComponent *ui, 
							const char *container_path, 
							guint index,
							const char *label,
							GdkPixbuf *pixbuf);


static char *   get_numbered_menu_item_name     (BonoboUIComponent *ui,
						 const char *container_path,
						 guint index);
static char *   nautilus_bonobo_get_numbered_menu_item_path (BonoboUIComponent *ui,
							     const char *container_path, 
							     guint index);
static char *   nautilus_bonobo_get_numbered_menu_item_command (BonoboUIComponent *ui,
								const char *container_path, 
								guint index);
static void     nautilus_bonobo_set_tip          (BonoboUIComponent *ui,
						  const char *path,
						  const char *tip);

/* Utility functions */

static gboolean uris_identical				(const char		*uri1,
							 const char		*uri2);

static gboolean	should_mozilla_load_uri_directly 	(const char		*uri);

/* zoomable */
static void zoomable_set_zoom_level_cb    (BonoboZoomable        *zoomable,
					   float                  level,
					   GaleonNautilusView    *view);
static void zoomable_zoom_in_cb           (BonoboZoomable        *zoomable,
					   GaleonNautilusView    *view);
static void zoomable_zoom_out_cb          (BonoboZoomable        *zoomable,
					   GaleonNautilusView    *view);
static void zoomable_zoom_to_fit_cb       (BonoboZoomable        *zoomable,
					   GaleonNautilusView    *view);
static void zoomable_zoom_to_default_cb   (BonoboZoomable        *zoomable,
					   GaleonNautilusView    *view);

static float preferred_zoom_levels[] = { .25, .50, .75, 1.0, 1.1, 1.25, 1.5, 2.0, 3.0, 4.0 };
static const int preferred_zoom_levels_nelems = (sizeof (preferred_zoom_levels) /
						 sizeof (preferred_zoom_levels[0]));

/* BonoboControl callbacks */
static void bonobo_control_activate_cb 		(BonoboObject *control, gboolean state, 
						 gpointer callback_data);
/***********************************************************************************/

EEL_DEFINE_CLASS_BOILERPLATE (GaleonNautilusView,
			      galeon_nautilus_view,
			      GTK_TYPE_VBOX);

static void
galeon_nautilus_view_initialize_class (GaleonNautilusViewClass *klass)
{
	GtkObjectClass *object_class;
	object_class = GTK_OBJECT_CLASS (klass);
	object_class->destroy = galeon_nautilus_view_destroy;
}


/* property bag property access routines to return sidebar icon */
static void
get_bonobo_properties (BonoboPropertyBag *bag,
			BonoboArg *arg,
			guint arg_id,
			CORBA_Environment *ev,
			gpointer callback_data)
{
	GaleonNautilusView *content_view;
	
	content_view = (GaleonNautilusView*) callback_data;

	switch (arg_id) {
        	case ICON_NAME:	
			if (!strncmp (content_view->priv->uri, "man:", 4)) {
                   		BONOBO_ARG_SET_STRING (arg, "manual");					
			} else if (!strncmp (content_view->priv->uri, "http:", 5)) {
                		BONOBO_ARG_SET_STRING (arg, "i-web");					
			} else if (!strncmp (content_view->priv->uri, "https:", 6)) {
				/* FIXME: put a nice icon for secure sites */
                		BONOBO_ARG_SET_STRING (arg, "i-web");					
			} else {
                		BONOBO_ARG_SET_STRING (arg, "");					
                	}
                	break;

        	case COMPONENT_INFO:
               		BONOBO_ARG_SET_STRING (arg, "");					
                 	break;
        		
        	default:
                	g_warning ("Unhandled arg %d", arg_id);
                	break;
	}
}

/* there are no settable properties, so complain if someone tries to set one */
static void
set_bonobo_properties (BonoboPropertyBag *bag,
			const BonoboArg *arg,
			guint arg_id,
			CORBA_Environment *ev,
			gpointer callback_data)
{
                g_warning ("Bad Property set on view: property ID %d",
			   arg_id);
}

static void
galeon_nautilus_view_initialize (GaleonNautilusView *view)
{
	view->priv = g_new0 (GaleonNautilusViewPrivate, 1);

	DEBUG_MSG (("+%s\n", G_GNUC_FUNCTION));

	/* Conjure up the beast.  May God have mercy on our souls. */
	view->priv->embed = embed_create_no_window ();
	view->priv->embed->nautilus_view = view;
	gtk_widget_show (view->priv->embed->mozembed);

	/* Add callbacks to the beast */ // FIXME: we don't need all of them
	gtk_signal_connect (GTK_OBJECT (view->priv->embed->mozembed), 
				"realize",
				GTK_SIGNAL_FUNC (mozilla_realize_cb),
				view);

	gtk_box_pack_start (GTK_BOX (view), 
			    GTK_WIDGET (view->priv->embed->mozembed),
			    TRUE, TRUE, 1);
	
	gtk_widget_show (GTK_WIDGET (view->priv->embed->mozembed));
	
	view->priv->nautilus_view = nautilus_view_new (GTK_WIDGET (view));
	
	gtk_signal_connect (GTK_OBJECT (view->priv->nautilus_view), 
				"load_location",
				GTK_SIGNAL_FUNC (view_load_location_cb), 
				view);

	/* Connect to the active signal of the view to merge our menus */
        gtk_signal_connect (GTK_OBJECT (nautilus_view_get_bonobo_control (view->priv->nautilus_view)),
                            "activate",
                            bonobo_control_activate_cb,
                            view);
	
	/* zoomable */
	view->priv->zoomable = bonobo_zoomable_new ();
	
	gtk_signal_connect (GTK_OBJECT (view->priv->zoomable), "set_zoom_level",
			    GTK_SIGNAL_FUNC (zoomable_set_zoom_level_cb), view);
	gtk_signal_connect (GTK_OBJECT (view->priv->zoomable), "zoom_in",
			    GTK_SIGNAL_FUNC (zoomable_zoom_in_cb), view);
	gtk_signal_connect (GTK_OBJECT (view->priv->zoomable), "zoom_out",
			    GTK_SIGNAL_FUNC (zoomable_zoom_out_cb), view);
	gtk_signal_connect (GTK_OBJECT (view->priv->zoomable), "zoom_to_fit",
			    GTK_SIGNAL_FUNC (zoomable_zoom_to_fit_cb), view);
	gtk_signal_connect (GTK_OBJECT (view->priv->zoomable), "zoom_to_default",
			    GTK_SIGNAL_FUNC (zoomable_zoom_to_default_cb), view);
	
	bonobo_zoomable_set_parameters_full (view->priv->zoomable,
					     1.0, .10, 8.0, FALSE, FALSE, TRUE,
					     preferred_zoom_levels, NULL,
					     preferred_zoom_levels_nelems);
	
	bonobo_object_add_interface (BONOBO_OBJECT (nautilus_view_get_bonobo_control (view->priv->nautilus_view)),
				     BONOBO_OBJECT (view->priv->zoomable));

 	/* allocate a property bag to specify the name of the icon for this component */
	view->priv->property_bag = bonobo_property_bag_new (get_bonobo_properties,  set_bonobo_properties, view);
	bonobo_control_set_properties (nautilus_view_get_bonobo_control (view->priv->nautilus_view),
				       view->priv->property_bag);
	bonobo_property_bag_add (view->priv->property_bag, "icon_name", ICON_NAME, BONOBO_ARG_STRING, NULL,
				 _("name of icon for the mozilla view"), 0);
	bonobo_property_bag_add (view->priv->property_bag, "summary_info", COMPONENT_INFO, BONOBO_ARG_STRING, NULL,
				 _("mozilla summary info"), 0);
	
	gtk_widget_show_all (GTK_WIDGET (view));

	DEBUG_MSG (("-%s\n", G_GNUC_FUNCTION));
}

static void
galeon_nautilus_view_destroy (GtkObject *object)
{
	GaleonNautilusView *view;

	DEBUG_MSG (("+%s\n", G_GNUC_FUNCTION));

	view = GALEON_NAUTILUS_VIEW (object);

	embed_close (view->priv->embed);

	g_free (view->priv->uri);
	view->priv->uri = NULL;

	cancel_pending_vfs_operation (view);

	/* free the property bag */
	if (view->priv->property_bag != NULL) {
		bonobo_object_unref (BONOBO_OBJECT (view->priv->property_bag));
		view->priv->property_bag = NULL;
	}

	g_free (view->priv);

	EEL_CALL_PARENT (GTK_OBJECT_CLASS, destroy, (object));
	
	DEBUG_MSG (("-%s\n", G_GNUC_FUNCTION));
}




/**
 * Returns a new GaleonNautilusView as a BonoboObject 
 **/
BonoboObject *
galeon_nautilus_view_new_component (void)
{
	GaleonNautilusView *view;
	view = GALEON_NAUTILUS_VIEW (gtk_object_new (GALEON_TYPE_NAUTILUS_VIEW, NULL));
	return BONOBO_OBJECT (view->priv->nautilus_view);
}

/***********************************************************************************/

static void
view_load_location_cb (NautilusView *nautilus_view,
		       const char *location,
		       gpointer data)
{
	GaleonNautilusView *view = GALEON_NAUTILUS_VIEW (data);

	DEBUG_MSG (("+%s\n", G_GNUC_FUNCTION));

	nautilus_view_report_load_underway (nautilus_view);
	navigate_mozilla_to_nautilus_uri (view, location);
	
	DEBUG_MSG (("-%s\n", G_GNUC_FUNCTION));
}

/***********************************************************************************/

typedef struct
{
	char *encoding;
	GaleonNautilusView *view;
} EncodingMenuData;

static void
charset_encoding_changed_cb (BonoboUIComponent *component,
			     gpointer callback_data,
			     const char *path)
{
	EncodingMenuData *data;

	g_return_if_fail (callback_data != NULL);
	data = callback_data;
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (data->view));

	/* Change the encoding and reload the page */
	mozilla_force_character_set (data->view->priv->embed,
				     data->encoding);
	embed_reload (data->view->priv->embed, GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}

static void
galeon_nautilus_view_create_charset_encoding_submenu (GaleonNautilusView *view)
{
	gint i, j;
	gint lgroups_count;
	gint translated_cscount;
	GList *tl;
	gint index;

	/* get the charset titles and names from mozilla, if still not done */
	if (charsets == NULL)
	{
		mozilla_get_charsets (&charsets, &sorted_charset_titles);
	}

	lgroups_count = get_lgroups_count ();
	translated_cscount = get_translated_cscount ();
	
	tl = g_list_copy (sorted_charset_titles);
	
	for (i = 0; i < lgroups_count; i++) {
		gchar *escaped_encoding_group = bonobo_ui_util_encode_str (_(lgroups[i]));
		gchar *path;

		nautilus_bonobo_add_submenu (view->priv->ui,
					     MENU_VIEW_CHARSET_ENCODING_PATH,
					     escaped_encoding_group);
		path = g_strdup_printf ("%s/%s", MENU_VIEW_CHARSET_ENCODING_PATH, escaped_encoding_group);
		nautilus_bonobo_set_label (view->priv->ui,
					   path,
					   _(lgroups[i]));
		index = 0;
		for (j = 0; j < translated_cscount; j++) {
			if ((gint)charset_trans_array[j].lgroup == i) {
				gchar *ui_path;
				gchar *verb_name;
				EncodingMenuData *data;
				GList *tl2 = g_list_find_custom (
					tl, 
					_(charset_trans_array[j].charset_title),
					strcasestr_cb);
				if (tl2 != NULL) {
					tl = g_list_remove_link (tl, tl2);
					g_list_free_1 (tl2);
				}
				else /* we dont want to add menuitems for
				      * charset titles not in mozilla */
					continue;
				
				nautilus_bonobo_add_numbered_menu_item 
					(view->priv->ui, path, index,
					 _(charset_trans_array[j].charset_title),
					 NULL);

				/* Add the status tip */
				ui_path = nautilus_bonobo_get_numbered_menu_item_path 
					(view->priv->ui, path, index);

				nautilus_bonobo_set_tip (view->priv->ui,
							 ui_path,
							 _(charset_trans_array[j].charset_title));
				g_free (ui_path);
		
				/* Add verb to new menu item */
				verb_name = nautilus_bonobo_get_numbered_menu_item_command (view->priv->ui,
											    path,
											    index);
				
				data = g_new0 (EncodingMenuData, 1);
				data->encoding = charset_trans_array[j].charset_name;
				data->view = view;
				
				bonobo_ui_component_add_verb_full (view->priv->ui,
								   verb_name, 
								   charset_encoding_changed_cb,
								   data,
								   g_free);
				g_free (verb_name);
				index++;
			}
		}
		
		g_free (path);
		g_free (escaped_encoding_group);

	}
}

static void
file_print_cmd (BonoboUIComponent *uic,
		gpointer data,
		const char *verbname)
{
	GaleonNautilusView *view = data;
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));

	print_show_dialog (view->priv->embed);
}

static void
file_find_cmd (BonoboUIComponent *uic, 
	       gpointer data,
	       const char *verbname)
		
{
	GaleonNautilusView *view = data;
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));

	find_show_dialog (view->priv->embed);
}

static void
nop_verb (BonoboUIComponent *uic, gpointer data, const char *verbname)
{
	/* This is just to avoid warnings about not found verbs */
}

static void
view_encoding_build_cmd (BonoboUIComponent *uic,
			 gpointer data,
			 const char *verbname)
		
{
	GaleonNautilusView *view = data;
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));

	bonobo_ui_component_remove_verb (uic, "EncodingMenuBuild");
	bonobo_ui_component_add_verb (uic, "EncodingMenuBuild", nop_verb, NULL);

	/* Create the charset encodings submenu */
	bonobo_ui_component_freeze (view->priv->ui, NULL);
	galeon_nautilus_view_create_charset_encoding_submenu (view);
	bonobo_ui_component_thaw (view->priv->ui, NULL);
}

BonoboUIVerb verbs [] = {
        BONOBO_UI_VERB ("FilePrint", file_print_cmd),
        BONOBO_UI_VERB ("EditFind", file_find_cmd),
        BONOBO_UI_VERB ("EncodingMenuBuild", view_encoding_build_cmd),
        BONOBO_UI_VERB_END
};

static void
galeon_nautilus_view_merge_menus (GaleonNautilusView *view)
{
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));

        view->priv->ui = nautilus_view_set_up_ui (view->priv->nautilus_view,
						  SHARE_DIR,
						  "nautilus-galeon-view-ui.xml",
						  "galeon");

	bonobo_ui_component_add_verb_list_with_data (
                view->priv->ui, verbs, view);

}

/* BonoboControl callbacks */
static void
bonobo_control_activate_cb (BonoboObject *control, gboolean state, gpointer callback_data)
{
        GaleonNautilusView *view;

	g_return_if_fail (BONOBO_IS_CONTROL (control));
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (callback_data));
	
	view = GALEON_NAUTILUS_VIEW (callback_data);

	if (state) {
		galeon_nautilus_view_merge_menus (view);
	}

	view->priv->embed->is_active = state;
}

/***********************************************************************************/

static void
mozilla_realize_cb (GtkWidget *mozilla, gpointer user_data)
{
	GaleonNautilusView *view = GALEON_NAUTILUS_VIEW (user_data);
	char *uri;

	DEBUG_MSG (("+%s\n", G_GNUC_FUNCTION));
	g_assert (mozilla == view->priv->embed->mozembed);

	if (view->priv->uri != NULL) {
		DEBUG_MSG (("=%s navigating to uri after realize '%s'\n", G_GNUC_FUNCTION, view->priv->uri));

		uri = g_strdup (view->priv->uri); /* yes, this needs to be strduped */
		navigate_mozilla_to_nautilus_uri (view, uri);
		g_free (uri);
	}

	DEBUG_MSG (("-%s\n", G_GNUC_FUNCTION));
}

void 
galeon_nautilus_view_set_title (GaleonNautilusView *view,
 				const gchar *title)
{
 	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
 	g_return_if_fail (title != NULL);

 	nautilus_view_set_title (view->priv->nautilus_view, title);
}

void
galeon_nautilus_view_set_location (GaleonNautilusView *view,
				   const gchar *uri)
{
	const gchar *prefixes_to_ignore[] = 
		{
			"about:",
			"javascript:",
			NULL 
		};
	int i = 0;
 	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
 	g_return_if_fail (uri != NULL);

 	g_free (view->priv->uri);
 	view->priv->uri = g_strdup (uri);

	/* don't inform nautilus about uris that it can't understand */
	while (prefixes_to_ignore[i] != NULL)
	{
		if (!strncmp (prefixes_to_ignore[i], uri, 
			     strlen (prefixes_to_ignore[i])))
			return;
		i++;
	}

	nautilus_view_report_location_change (view->priv->nautilus_view,
					      uri, NULL, uri);
 	/* TODO, FIXME
 	nautilus_view_report_redirect (view->priv->nautilus_view, 
 				       view->priv->uri, new_location,
 				       NULL, new_location);
 	*/
}

void
galeon_nautilus_view_report_load_underway (GaleonNautilusView *view)
{
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	nautilus_view_report_load_underway (view->priv->nautilus_view);
}

void
galeon_nautilus_view_report_load_complete (GaleonNautilusView *view)
{
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	nautilus_view_report_load_complete (view->priv->nautilus_view);
}

void
galeon_nautilus_view_set_statusbar (GaleonNautilusView *view, 
				    const gchar *message)
{
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	
	if (message == NULL)
	{
		message = "";
	}

	nautilus_view_report_status (view->priv->nautilus_view, message);
}

void
galeon_nautilus_view_report_load_progress (GaleonNautilusView *view,
					   double value)
{
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));

	if (value < 0.0) value = 0.0;
	if (value > 1.0) value = 1.0;
	
	nautilus_view_report_load_progress (view->priv->nautilus_view, value);
}

void
galeon_nautilus_view_open_in_new_window (GaleonNautilusView *view,
					 const gchar *url)
{
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	g_return_if_fail (url != NULL);
	
	nautilus_view_open_location_force_new_window (view->priv->nautilus_view, 
						      url, NULL);
}

/***********************************************************************************/

/**
 * vfs_open_cb
 *
 * Callback for gnome_vfs_async_open. Attempt to read data from handle
 * and pass to mozilla streaming callback.
 * 
 **/
static void
vfs_open_cb (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer data)
{
	GaleonNautilusView *view = data;

	DEBUG_MSG (("+%s GnomeVFSResult: %u\n", G_GNUC_FUNCTION, (unsigned)result));

	if (result != GNOME_VFS_OK)
	{
		gtk_moz_embed_close_stream (GTK_MOZ_EMBED (view->priv->embed->mozembed));
		/* NOTE: the view may go away after a call to report_load_failed */
		DEBUG_MSG ((">nautilus_view_report_load_failed\n"));
		nautilus_view_report_load_failed (view->priv->nautilus_view);
	} else {
		if (view->priv->vfs_read_buffer == NULL) {
			view->priv->vfs_read_buffer = g_malloc (VFS_READ_BUFFER_SIZE);
		}
		gtk_moz_embed_open_stream (GTK_MOZ_EMBED (view->priv->embed->mozembed), "file:///", "text/html");
		gnome_vfs_async_read (handle, view->priv->vfs_read_buffer, VFS_READ_BUFFER_SIZE, vfs_read_cb, view);
	}
	DEBUG_MSG (("-%s\n", G_GNUC_FUNCTION));
}

/**
 * vfs_read_cb:
 *
 * Read data from buffer and copy into mozilla stream.
 **/

static void
vfs_read_cb (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer buffer,
		   GnomeVFSFileSize bytes_requested,
		   GnomeVFSFileSize bytes_read,
		   gpointer data)
{
	GaleonNautilusView *view = data;

	DEBUG_MSG (("+%s %ld/%ld bytes\n", G_GNUC_FUNCTION, (long)bytes_requested, (long) bytes_read));

	if (bytes_read != 0) {
		gtk_moz_embed_append_data (GTK_MOZ_EMBED (view->priv->embed->mozembed), buffer, bytes_read);
	}

	if (bytes_read == 0 || result != GNOME_VFS_OK) {
		gtk_moz_embed_close_stream (GTK_MOZ_EMBED (view->priv->embed->mozembed));
		view->priv->vfs_handle = NULL;
		g_free (view->priv->vfs_read_buffer);
		view->priv->vfs_read_buffer = NULL;
		
		gnome_vfs_async_close (handle, (GnomeVFSAsyncCloseCallback) gtk_true, NULL);

		DEBUG_MSG ((">nautilus_view_report_load_complete\n"));
		nautilus_view_report_load_complete (view->priv->nautilus_view);

		DEBUG_MSG (("=%s load complete\n", G_GNUC_FUNCTION));
    	} else {
		gnome_vfs_async_read (handle, view->priv->vfs_read_buffer, VFS_READ_BUFFER_SIZE, vfs_read_cb, view);
	}

	DEBUG_MSG (("-%s\n", G_GNUC_FUNCTION));
}

/***********************************************************************************/

static void
cancel_pending_vfs_operation (GaleonNautilusView *view)
{
	if (view->priv->vfs_handle != NULL) {
		gnome_vfs_async_cancel (view->priv->vfs_handle);
		gtk_moz_embed_close_stream (GTK_MOZ_EMBED (view->priv->embed->mozembed));
	}

	view->priv->vfs_handle = NULL;
	g_free (view->priv->vfs_read_buffer);
	view->priv->vfs_read_buffer = NULL;
}


/* this takes a "nautilus" uri, not a "mozilla" uri and uses (sometimes) GnomeVFS */
static void
navigate_mozilla_to_nautilus_uri (GaleonNautilusView *view,
			 	  const char *uri)
{
	char *old_uri;

	cancel_pending_vfs_operation (view);
	
	if (!GTK_WIDGET_REALIZED (view->priv->embed->mozembed)) {
		
		/* Doing certain things to gtkmozembed before
		 * the widget has realized (specifically, opening
		 * content streams) can cause crashes.  To avoid
		 * this, we postpone all navigations
		 * until the widget has realized (we believe
		 * premature realization may cause other issues)
		 */
		
		DEBUG_MSG (("=%s: Postponing navigation request to widget realization\n", G_GNUC_FUNCTION));
		/* Note that view->priv->uri is still set below */
	} else {
		if (should_mozilla_load_uri_directly (uri)) {

			/* See if the current URI is the same as what mozilla already
			 * has.  If so, issue a reload rather than a load.
			 * We ask mozilla for it's uri rather than using view->priv->uri because,
			 * from time to time, our understanding of mozilla's URI can become inaccurate
			 * (in particular, certain errors may cause embedded mozilla to not change
			 * locations)
			 */

			old_uri = view->priv->embed->location;

			if (old_uri != NULL && uris_identical (uri, old_uri)) {
				DEBUG_MSG (("=%s uri's identical, telling galeon to reload\n", G_GNUC_FUNCTION));
				embed_reload (view->priv->embed,
					      GTK_MOZ_EMBED_FLAG_RELOADBYPASSCACHE);
			} else {
				embed_load_url (view->priv->embed, uri);
			}

		} else {
			DEBUG_MSG (("=%s loading URI via gnome-vfs\n", G_GNUC_FUNCTION));
			gnome_vfs_async_open (&(view->priv->vfs_handle), uri,
					      GNOME_VFS_OPEN_READ, vfs_open_cb, view);
		}
	}

	g_free (view->priv->uri);
	view->priv->uri = g_strdup (uri);

	DEBUG_MSG (("=%s current URI is now '%s'\n", G_GNUC_FUNCTION, view->priv->uri));
}

/***********************************************************************************/

static gboolean
uris_identical (const char *uri1, const char *uri2)
{
	/*
	 * FIXME: the dns portion of the authority is case-insensitive,
	 * as is the scheme, but the rest of the URI is case-sensitive.  We just
	 * treat the whole thing as case-sensitive, which is mostly
	 * OK, especially since false negatives here aren't the end
	 * of the world
	 */
	return (strcmp (uri1, uri2) == 0);
}

/*
 * This a list of URI schemes that mozilla should load directly, rather than load through gnome-vfs
 */
static gboolean
should_mozilla_load_uri_directly (const char *uri)
{
	static const char *handled_by_mozilla[] =
	{
		"http",
		"file",
		"toc",
		"man",
		"info",
		"ghelp",
		"gnome-help",
		"https",
		NULL
	};
	gint i;
	gint uri_length;

	if (uri == NULL) return FALSE;

	uri_length = strlen (uri);

	for (i = 0; handled_by_mozilla[i] != NULL; i++)
	{
		const gchar *current = handled_by_mozilla[i];
		gint current_length = strlen (current);
		if ((uri_length >= current_length) 
		    && (!strncasecmp (uri, current, current_length))) 
			return TRUE;
	}
	return FALSE;
}

/******************************************/

static void
zoomable_set_zoom_level_cb    (BonoboZoomable        *zoomable,
					   float                  level,
					   GaleonNautilusView    *view)
{
	gint new_zoom_level = level * 100;
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	if (new_zoom_level < 10) return;
	if (new_zoom_level > 800) return;
	embed_set_zoom (view->priv->embed, new_zoom_level, TRUE);

}

static void
zoomable_zoom_in_cb           (BonoboZoomable        *zoomable,
					   GaleonNautilusView    *view)
{
	gint new_zoom_level;
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	new_zoom_level = view->priv->embed->zoom + 20;
	if (new_zoom_level > 800) return;
	embed_set_zoom (view->priv->embed, new_zoom_level, TRUE);
}

static void
zoomable_zoom_out_cb          (BonoboZoomable        *zoomable,
					   GaleonNautilusView    *view)
{
	gint new_zoom_level;
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	new_zoom_level = view->priv->embed->zoom - 20;
	if (new_zoom_level < 10) return;
	embed_set_zoom (view->priv->embed, new_zoom_level, TRUE);
}

static void
zoomable_zoom_to_fit_cb       (BonoboZoomable        *zoomable,
					   GaleonNautilusView    *view)
{
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	embed_set_zoom (view->priv->embed, 100, TRUE);
}

static void
zoomable_zoom_to_default_cb   (BonoboZoomable        *zoomable,
					   GaleonNautilusView    *view)
{
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	embed_set_zoom (view->priv->embed, 100, TRUE);
}

void
galeon_nautilus_view_report_zoom (GaleonNautilusView *view,
					      gint level)
{
	float flevel;
	g_return_if_fail (GALEON_IS_NAUTILUS_VIEW (view));
	
	flevel = ((float) level) / 100.0;
	
	bonobo_zoomable_report_zoom_level_changed (view->priv->zoomable,
						   flevel);
}

/****************************************************/

/* Helper functions copied from bonobo-extensions.c
   bonobo should have this, wtf were they thinking? */

static void
nautilus_bonobo_add_submenu (BonoboUIComponent *ui,
			     const char *path,
			     const char *label)
{
	char *xml_string, *encoded_label, *name;

	/* Because we are constructing the XML ourselves, we need to
         * encode the label.
	 */
	encoded_label = bonobo_ui_util_encode_str (label);

	/* Labels may contain characters that are illegal in names. So
	 * we create the name by URI-encoding the label.
	 */
	name = gnome_vfs_escape_string (label);
	
	xml_string = g_strdup_printf ("<submenu name=\"%s\" label=\"%s\"/>\n", 
				      name, encoded_label);
	bonobo_ui_component_set (ui, path, xml_string, NULL);

	g_free (encoded_label);
	g_free (name);
	g_free (xml_string);
}

static void
nautilus_bonobo_set_label (BonoboUIComponent *ui,
			   const char *path,
			   const char *label)
{
	g_return_if_fail (BONOBO_IS_UI_COMPONENT (ui));
	bonobo_ui_component_set_prop (ui, path,
				      "label",
				      label,
				      NULL);
}

static void
add_numbered_menu_item_internal (BonoboUIComponent *ui,
				 const char *container_path,
				 guint index,
				 const char *label,
				 gboolean is_toggle,
				 GdkPixbuf *pixbuf)
{
	char *xml_item, *xml_command; 
	char *encoded_label, *command_name;
	char *item_name, *pixbuf_data;

	g_assert (BONOBO_IS_UI_COMPONENT (ui)); 
	g_assert (container_path != NULL);
	g_assert (label != NULL);
	g_assert (!is_toggle || pixbuf == NULL);
	
	/* Because we are constructing the XML ourselves, we need to
         * encode the label.
	 */
	encoded_label = bonobo_ui_util_encode_str (label);
	
	item_name = get_numbered_menu_item_name 
		(ui, container_path, index);
	command_name = nautilus_bonobo_get_numbered_menu_item_command 
		(ui, container_path, index);
	
	/* Note: we ignore the pixbuf for toggle items. This could be changed
	 * if we ever want a toggle item that also has a pixbuf.
	 */
	if (is_toggle) {
		xml_item = g_strdup_printf ("<menuitem name=\"%s\" label=\"%s\" id=\"%s\" type=\"toggle\"/>\n", 
					    item_name, encoded_label, command_name);
	} else if (pixbuf != NULL) {
		/* Encode pixbuf type and data into XML string */			
		pixbuf_data = bonobo_ui_util_pixbuf_to_xml (pixbuf);
		
		xml_item = g_strdup_printf ("<menuitem name=\"%s\" label=\"%s\" verb=\"%s\" "
					    "pixtype=\"pixbuf\" pixname=\"%s\"/>\n", 
					    item_name, encoded_label, command_name, pixbuf_data);	
		g_free (pixbuf_data);
	} else {
		xml_item = g_strdup_printf ("<menuitem name=\"%s\" label=\"%s\" verb=\"%s\"/>\n", 
						item_name, encoded_label, command_name);
	}
	g_free (encoded_label);
	g_free (item_name);
	
	bonobo_ui_component_set (ui, container_path, xml_item, NULL);
	g_free (xml_item);

	/* Make the command node here too, so callers can immediately set
	 * properties on it (otherwise it doesn't get created until some
	 * time later).
	 */
	xml_command = g_strdup_printf ("<cmd name=\"%s\"/>\n", command_name);
	bonobo_ui_component_set (ui, "/commands", xml_command, NULL);
	g_free (xml_command);

	g_free (command_name);
}			 

/* Add a menu item specified by number into a given path. Used for
 * dynamically creating a related series of menu items. Each index
 * must be unique (normal use is to call this in a loop, and
 * increment the index for each item).
 */
static void
nautilus_bonobo_add_numbered_menu_item (BonoboUIComponent *ui, 
					 const char *container_path, 
					 guint index,
			       		 const char *label, 
			       		 GdkPixbuf *pixbuf)
{
	g_return_if_fail (BONOBO_IS_UI_COMPONENT (ui)); 
	g_return_if_fail (container_path != NULL);
	g_return_if_fail (label != NULL);

	add_numbered_menu_item_internal (ui, container_path, index, label, FALSE, pixbuf);
}

static char *
get_numbered_menu_item_name (BonoboUIComponent *ui,
			      const char *container_path,
			      guint index)
{
	return g_strdup_printf ("%u", index);
}			      

static char *
nautilus_bonobo_get_numbered_menu_item_path (BonoboUIComponent *ui,
					      const char *container_path, 
					      guint index)
{
	char *item_name;
	char *item_path;

	g_return_val_if_fail (BONOBO_IS_UI_COMPONENT (ui), NULL); 
	g_return_val_if_fail (container_path != NULL, NULL);

	item_name = get_numbered_menu_item_name (ui, container_path, index);
	item_path = g_strconcat (container_path, "/", item_name, NULL);
	g_free (item_name);

	return item_path;
}

static char *
nautilus_bonobo_get_numbered_menu_item_command (BonoboUIComponent *ui,
						 const char *container_path, 
						 guint index)
{
	char *command_name;
	char *path;

	g_return_val_if_fail (BONOBO_IS_UI_COMPONENT (ui), NULL); 
	g_return_val_if_fail (container_path != NULL, NULL);

	path = nautilus_bonobo_get_numbered_menu_item_path (ui, container_path, index);
	command_name = gnome_vfs_escape_string (path);
	g_free (path);
	
	return command_name;
}

static void
nautilus_bonobo_set_tip (BonoboUIComponent *ui,
			 const char *path,
			 const char *tip)
{
	g_return_if_fail (ui != NULL);
	bonobo_ui_component_set_prop (ui, path,
				      "tip",
				      tip,
				      NULL);
}

/*****************************************/

/* From window.c */
static gint 
strcasestr_cb (gconstpointer a, gconstpointer b)
{
	return misc_string_strcasestr (a, b) == NULL ? 1 : 0;
}
