/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "bookmarks.h"
#include "embed.h"
#include "window.h"
#include "misc_callbacks.h"
#include "misc_gui.h"
#include "misc_string.h"
#include "find.h"
#include "prefs.h"
#include "context.h"
#include "glade.h"
#include "dnd-hints.h"
#include "favicon.h"

#include <string.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnome/gnome-util.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtkbox.h>
#include <gdk/gdkkeysyms.h>

extern const GtkTargetEntry bookmarks_dnd_targets[];

typedef struct
{ 
	GtkWidget *dialog;
	GtkWidget *entry;
	gchar *gconfkey;
} SmartBookmarksEntrySizeEditor; 

static const char invalid_chars[] = " \t\r\n\"$&<>,+=#!()'|{}[]?~`;%\\" "/:";
static const char digits_base16[] = "0123456789abcdef";

/* local function prototypes */
static void bookmarks_toolbars_create_from_root (GaleonWindow *window, 
						 BookmarkItem *bi);
static void  bookmarks_toolbar_create_from_folder (GaleonWindow *window,
						     BookmarkItem *bi);
static GtkWidget *bookmarks_toolbar_create_from_bookmark (GaleonWindow *window, 
						    GtkToolbar *tb,
						    BookmarkItem *b,
						    ToolbarStyle style);
gboolean bookmarks_nick_entry_key_press_event_cb (GtkEntry *entry,
						  GdkEventKey *event,
						  BookmarkItem *bi);
static void bookmarks_toolbar_drag_data_received_cb 
                                       (GtkToolbar *tb, 
					GdkDragContext *drag_context,
					gint x, gint y,
					GtkSelectionData *selection_data,
					guint info, guint time,
					BookmarkItem *bm);
static gboolean bookmarks_toolbar_drag_motion_cb (GtkToolbar *tb, 
						  GdkDragContext *context,
						  gint x, gint y, guint time,
						  BookmarkItem *bi);

static gboolean bookmarks_toolbar_button_press_event_cb (GtkWidget *item,
							GdkEventButton *event,
							BookmarkItem *bi);
static gboolean folder_button_press_event_cb (GtkWidget *item, 
						GdkEventButton *event,
						BookmarkItem *bi);
static gboolean bookmark_button_release_event_cb (GtkWidget *item,
						  GdkEventButton *event,
						  BookmarkItem *bi);
static gboolean bookmark_button_create_normal_menu (GtkWidget *item, 
						    GdkEventButton *event,
						    BookmarkItem *bi);
static gboolean bookmark_button_create_smart_menu (GtkWidget *item, 
						   GdkEventButton *event,
						   BookmarkItem *bi);
static gboolean bookmark_button_create_menu (GtkWidget *item, 
					     GdkEventButton *event,
					     BookmarkItem *bi,
					     gboolean smart,
					     gboolean smart_expanded);
static gboolean smartbookmark_tray_button_release_event_cb (GtkWidget *item, 
						          GdkEventButton *event,
						          BookmarkItem *bi);
static gboolean smartbookmark_fold_button_release_event_cb (GtkWidget *item, 
						          GdkEventButton *event,
						          BookmarkItem *bi);
static void bookmarks_toolbar_item_menu_hide_cb (GtkMenuItem *menuitem, 
						 BookmarkItem *bm);
static void bookmarks_resize_entry_cb (GtkMenuItem *menuitem, 
				       BookmarkItem *bi);

static void smartbookmark_setup_tray (GtkWidget *tray,
				      gboolean visiblity);
static void smartbookmark_setup_history (GtkCombo *combo,
				         GSList *history);
static void smartbookmark_add_to_history (GtkWidget *widget,
					  char *text);

static void smartbookmark_visibility_changed (GConfClient *client,
					      guint cnxn_id,
					      GConfEntry *entry,
					      gpointer user_data);
static void smartbookmark_entry_size_changed (GConfClient *client,
					      guint cnxn_id,
					      GConfEntry *entry,
					      gpointer user_data);
static void smartbookmark_history_changed (GConfClient *client,
					   guint cnxn_id,
					   GConfEntry *entry,
					   gpointer user_data);
static void smartbookmark_remove_notifier (GtkWidget *widget,
					   guint noid);
static gint smartbookmark_get_size (BookmarkItem *b);
static GtkWidget *toolbar_get_dnd_info (GtkToolbar *tb, gint desp_x,
					gint desp_y, 
					BookmarkItem **b, 
					GtkCListDragPos *type);
static void bookmarks_toolbars_orientation_changed_cb (GtkToolbar *tb, 
						       GtkOrientation o,
						       GtkTable *t);

static void toolbar_show_dnd_hint (GtkToolbar *tb, GtkWidget *w,
				   GtkCListDragPos type);
static void bookmarks_fold_cb (GtkMenuItem *menuitem, BookmarkItem *bi);
static void bookmarks_clear_smart_history_cb (GtkWidget *menuitem, 
					      BookmarkItem *bi);

void smart_bookmarks_entry_size_changed_cb (GtkWidget *w,
					    SmartBookmarksEntrySizeEditor *e);
void smart_bookmarks_default_clicked_cb (GtkWidget *w,
					 SmartBookmarksEntrySizeEditor *e);
void smart_bookmarks_close_clicked_cb (GtkWidget *w,
				       SmartBookmarksEntrySizeEditor *e);
static guint prefs_key_len (gchar const * str);
static gchar *prefs_key_new (gchar const * str);


/* this magic number is used to hack around a problem with
 * gnome-libs (I think) which causes multiple toolbars with
 * the same number not to display properly -- MattA
 * This number will also be appended to toolbar names to make sure no two
 * toolbars have the same name.  This keeps the gnome layout routines from
 * getting confused when restoring toolbar positions  --Josh */
static gint magic_number;

/**
 * bookmarks_toolbars_create: create personal toolbars
 */
void 
bookmarks_toolbars_create (GaleonWindow *window)
{
	gboolean visible = TRUE;

	/* get visibility */
	if (window->view_bookmarks)
	{
		visible = GTK_CHECK_MENU_ITEM (window->view_bookmarks)->active;
	}

	/* reset the magic number */
	magic_number = 99;

	/* build all the toolbars */
	bookmarks_toolbars_create_from_root (window, bookmarks_root);

	/* setup visibility of them all */
	bookmarks_toolbars_set_visibility (window, visible);
}

/**
 * bookmarks_toolbars_recreate: recreate personal toolbars
 */
void 
bookmarks_toolbars_recreate (GaleonWindow *window)
{
	GList *l;
	gboolean visible;

	return_if_not_window (window);
	g_return_if_fail (window->view_bookmarks != NULL);

	/* get visibility */
	visible = GTK_CHECK_MENU_ITEM (window->view_bookmarks)->active;
			
	/* destroy all the toolbars */
	for (l = window->bookmarks_toolbars; l != NULL; l = g_list_next (l))
	{
		GtkToolbar *toolbar = GTK_TOOLBAR (l->data);
		gtk_widget_destroy (GTK_WIDGET (toolbar)->parent);
	}

	/* list is now empty */
	g_list_free (window->bookmarks_toolbars);
	window->bookmarks_toolbars = NULL;

	/* now create toolbars for all of these */
	bookmarks_toolbars_create (window);

	/* update visibility */
	bookmarks_toolbars_set_visibility (window, visible);
}

/**
 * bookmarks_toolbars_create_from_root: create toolbars from a given
 * root bookmark
 */
static void
bookmarks_toolbars_create_from_root (GaleonWindow *window, BookmarkItem *bi)
{
	GList *l;

	switch (bi->type)
	{
	case BM_SITE:
	case BM_SEPARATOR:
		/* can't create a toolbar from these */
		return;

	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		/* continue... */
		break;
	}

	/* if we're meant to create a toolbar... */
	if (bi->create_toolbar)
	{
		/* do so */
		bookmarks_toolbar_create_from_folder (window, bi);
	}
	
	/* now iterate to find any marked subfolders, 
	 * unless this is an alias */
	if (!(bi->alias_of))
	{
		for (l = bi->list; l != NULL; l = g_list_next (l))
		{
			bookmarks_toolbars_create_from_root (window, l->data);
		}
	}
}

/**
 * bookmarks_toolbar_create_from_folder: create a toolbar from a folder
 */
static void 
bookmarks_toolbar_create_from_folder (GaleonWindow *window, BookmarkItem *bi)
{
	gboolean toolbar_relief;
	gboolean toolbar_detachable;
	GnomeDockItemBehavior props;
	GtkWidget *toolbar, *dummy;
	GtkRequisition r;
	gchar *unique_name;
	GList *l; 
	GList *seps = NULL, *others = NULL;
	gint height = 0;

	/* read gnome prefs */
	toolbar_relief = gnome_preferences_get_toolbar_relief ();
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();

	/* make the toolbar */
	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_TEXT);
	
	/* check whether it should be detachable or not */
	props = (GNOME_DOCK_ITEM_BEH_EXCLUSIVE |
		 (toolbar_detachable ? 0 : GNOME_DOCK_ITEM_BEH_LOCKED));

	/* Make sure each toolbar has a unique name */
	/* don't try to use the bookmark id to create an unique name that
	 * can be used to save and restore the layout, because the id will 
	 * change and you won't be able to restore the layout */
	/* the following name will only change if the name of the toolbar
	   changes or the relative order of the toolbars change */
	/* I hash the string instead of uding it directly because I'm not
	   sure if having spaces and "weird" chars in the toolbar name is OK.
	   --- Ricardo */
	 
	magic_number++;
	unique_name = g_strdup_printf ("%d_%d", g_str_hash (bi->name),
				       magic_number);

	/* add the toolbar it to the main window */
	gnome_app_add_docked (GNOME_APP (window->wmain), toolbar, unique_name,
			      props, GNOME_DOCK_TOP, magic_number, 0, 0);
	gtk_container_set_border_width (GTK_CONTAINER (toolbar->parent), 2);
	gnome_dock_item_set_shadow_type (GNOME_DOCK_ITEM (toolbar->parent), 
					 toolbar_relief ? 
					 GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	g_free (unique_name);

	/* override global setting */
	gtk_toolbar_set_space_style (GTK_TOOLBAR (toolbar),
				     GTK_TOOLBAR_SPACE_EMPTY);
	gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar), 6);

	/* iterate over the contents of the folder */
	for (l = bi->list; l != NULL; l = g_list_next (l))
	{
		BookmarkItem *bm = l->data;

		/* build the appropriate element */
		if (bm->type != BM_SEPARATOR)
		{
			others = g_list_append (others,
					bookmarks_toolbar_create_from_bookmark 
						(window, GTK_TOOLBAR (toolbar),
						 bm, bi->toolbar_style));
		} 
		else 
		{
			if (!gnome_preferences_get_toolbar_lines ())
			{
				gtk_toolbar_append_space
					(GTK_TOOLBAR (toolbar));
			}
			else
			{
				GtkWidget *sep =
					misc_gui_toolbar_append_separator (
							GTK_TOOLBAR (toolbar));
				seps = g_list_append (seps, sep);
			}
		}
	}

	/* set separator sizes */
	for (l = seps; l != NULL; l = g_list_next (l))	
	{
		separator_toolbar_orientation_changed_cb (GTK_TOOLBAR (toolbar),
					GTK_TOOLBAR (toolbar)->orientation,
					GTK_WIDGET (l->data));
	}

	/* get bar height, the entry stuff is needed for when the 
         * smartbookmarks are not visible */
        dummy = gtk_entry_new ();                 
        gtk_widget_size_request (dummy, &r);      
        gtk_widget_destroy (dummy);
        height = r.height;             
        gtk_widget_size_request (toolbar, &r);
        if (r.height > height) height = r.height;
	
	/* update all buttons to match toolbar height */
	for (l = others; l != NULL; l = g_list_next (l))
	{
		GtkRequisition r2;

		gtk_widget_size_request (GTK_WIDGET (l->data), &r2);

		if (GTK_TOOLBAR (toolbar)->orientation ==
                    GTK_ORIENTATION_HORIZONTAL)
		{
			gtk_widget_set_usize (GTK_WIDGET (l->data), r2.width,
					      height);
		}
	}

	gtk_signal_connect (GTK_OBJECT (toolbar), "drag_data_received",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_toolbar_drag_data_received_cb), bi);
	gtk_drag_dest_set (toolbar,
			   GTK_DEST_DEFAULT_DROP,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);
	gtk_signal_connect (GTK_OBJECT (toolbar), "drag_motion",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_toolbar_drag_motion_cb), bi);
	gtk_signal_connect (GTK_OBJECT (toolbar), "drag_leave",
			    GTK_SIGNAL_FUNC (dnd_hints_hide_all), NULL);

	/* setup the toolbar context menu */
	gtk_signal_connect (GTK_OBJECT (toolbar->parent), "button-press-event",
			    GTK_SIGNAL_FUNC 
				(bookmarks_toolbar_button_press_event_cb), bi);

	/* show it */
	gtk_widget_show (GTK_WIDGET (toolbar));

	/* add it to the per-window list of toolbars */
	window->bookmarks_toolbars = 
		g_list_append (window->bookmarks_toolbars,
			       GTK_TOOLBAR (toolbar));

}

/**
 * bookmarks_toolbar_create_from_bookmark:
 */
static GtkWidget* 
bookmarks_toolbar_create_from_bookmark (GaleonWindow *window, 
					GtkToolbar *tb,
					BookmarkItem *b, 
					ToolbarStyle style)
{
        GtkWidget *button, *box, *pixmap;
	const PixmapData *bm_icon = NULL;
	GtkWidget *dnd_dest;
	gchar *name = NULL;
	gchar *tip = NULL;
	gchar *short_url = NULL;
	GtkRequisition req;

	/* build a box, and pack the pixmap and label inside */
	if (style == TOOLBAR_STYLE_VERTICAL)
		box = gtk_vbox_new (FALSE, 2);
	else
		box = gtk_hbox_new (FALSE, 2);

	if (style != TOOLBAR_STYLE_TEXT_ONLY)
	{
		if (b->pixmap_data == NULL)
		{
			if (b->type == BM_FOLDER || 
			    b->type == BM_AUTOBOOKMARKS)
			{
				if (default_bookmarks_root == b)
					pixmap = gtk_pixmap_new 
						(default_folder_pixmap_data->
						 pixmap,
					 	 default_folder_pixmap_data->
						 mask);
				else
					pixmap = gtk_pixmap_new 
						(folder_pixmap_data->pixmap,
					 	 folder_pixmap_data->mask);
			}
			else
			{
				bm_icon = favicon_get_pixmap (b->url);
				pixmap = gtk_pixmap_new (bm_icon->pixmap, 
							 bm_icon->mask);
			}
			gtk_box_pack_start (GTK_BOX (box), pixmap, 
					    FALSE, FALSE, 0);
		}
		else
		{
			pixmap = gtk_pixmap_new (b->pixmap_data->pixmap, 
						 b->pixmap_data->mask);
			gtk_box_pack_start (GTK_BOX (box), GTK_WIDGET (pixmap),
					    TRUE, FALSE, 0);
		}
	}

	if (((b->pixmap_data == NULL || style == TOOLBAR_STYLE_TEXT_ONLY) &&
	    style != TOOLBAR_STYLE_ICONS_ONLY) ||
	    (style == TOOLBAR_STYLE_ICONS_ONLY &&
	     (b->type == BM_FOLDER ||
	      (bm_icon == site_pixmap_data && b->pixmap_data == NULL))))
	{
		gchar *loc_str = mozilla_utf8_to_locale (b->name);
		GtkWidget *dummy = gtk_button_new_with_label ("sdgs");
		GtkStyle *rcstyle = gtk_rc_get_style (dummy);
		GtkWidget *l;

		gtk_widget_destroy (dummy);

		if (rcstyle == NULL)
		{
			rcstyle = gtk_style_new ();
		}
		else
		{
			gtk_style_ref (rcstyle);
		}

		name = misc_string_strip_uline_accel (loc_str);
		g_free (loc_str);

		l = gtk_label_new (name);
		gtk_widget_set_style (l, rcstyle);
		gtk_box_pack_start (GTK_BOX (box), l, TRUE, TRUE, 0);

		g_free (name);
		gtk_style_unref (rcstyle);
	}

	/* is this a normal (i.e. non-smart) bookmark */
	if (!b->smarturl || strstr (b->smarturl, "%s") == NULL)
	{
		/* yes, build a button and pack it */
		dnd_dest = button = (b->type == BM_FOLDER) ?
			gtk_toggle_button_new ():
			gtk_button_new ();
		gtk_container_add (GTK_CONTAINER(button), box);

		if ((b->url != NULL)
		    && strlen (b->url) > BOOKMARKS_TOOLTIP_MAX_LENGTH) {
			gchar *s = g_strndup 
				(b->url, BOOKMARKS_TOOLTIP_MAX_LENGTH);
			short_url = g_strconcat (s, "...", NULL);
			g_free (s);
		} 
		else 
		{
			short_url = b->url;
		}

		if (b->notes == NULL || b->notes[0] == '\0')
		{
			tip = short_url;
		}
		else
		{
			gchar *loc_str = mozilla_utf8_to_locale (b->notes);
			if (b->type == BM_FOLDER) {
				tip = g_strdup_printf ("%s", loc_str);
			}
			else
			{
				tip = g_strdup_printf ("%s\n%s", short_url,
						       loc_str);
			}
			g_free (loc_str);
		}
		
		gtk_toolbar_append_widget (tb, button, tip, NULL);
		if (short_url != b->url) g_free (short_url);
		if (tip != short_url) g_free (tip);

		gtk_button_set_relief (GTK_BUTTON (button),
				gnome_preferences_get_toolbar_relief_btn () ?
					GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);

		/* the click handlers */
		if (b->type == BM_SITE)
		{
			gtk_signal_connect
				(GTK_OBJECT (dnd_dest), 
				 "button-press-event",
				 GTK_SIGNAL_FUNC (
					 bookmark_button_create_normal_menu),
				 b);
			gtk_signal_connect 
				(GTK_OBJECT (dnd_dest), 
				 "button-release-event",
				 GTK_SIGNAL_FUNC (
					 bookmark_button_release_event_cb),
				 b);
		}
		else
		{
			gtk_signal_connect 
				(GTK_OBJECT (dnd_dest), 
				 "button-press-event",
				 GTK_SIGNAL_FUNC (
					 folder_button_press_event_cb),
				 b);
		}

		gtk_widget_show_all (button);
	}
	else
	{
		GtkWidget *tray, *combo = NULL, *entry, *hbox;
		GSList *history, *item;
		gboolean enable_history;
		GtkRequisition r;

		gchar *key, *str, *short_url, *tip;
		gint entry_width;
		guint noid;

		GtkWidget *tray_button;

		GtkWidget *fold_state, *unfold_state, *fold_button;

		key = prefs_key_new (b->smarturl);

		fold_state = gtk_pixmap_new 
			(smart_bm_fold_pixmap_data->pixmap,
			 smart_bm_fold_pixmap_data->mask);
		unfold_state = gtk_pixmap_new 
			(smart_bm_unfold_pixmap_data->pixmap,
			 smart_bm_unfold_pixmap_data->mask);
		
		/* the tray */
		tray = gtk_hbox_new (FALSE, 0);
		
		/* the tray button */
		button = tray_button = gtk_button_new ();
		gtk_object_set_data (GTK_OBJECT (button), "bookmark", b);
		GTK_WIDGET_UNSET_FLAGS (tray_button, GTK_CAN_FOCUS);
		gtk_button_set_relief (GTK_BUTTON (tray_button),
				gnome_preferences_get_toolbar_relief_btn () ?
					GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);
	
		/* the tooltip */
		if ((b->url != NULL)
		    && strlen (b->url) > BOOKMARKS_TOOLTIP_MAX_LENGTH) {
			gchar *s = g_strndup 
				(b->url, BOOKMARKS_TOOLTIP_MAX_LENGTH);
			short_url = g_strconcat (s, "...", NULL);
			g_free (s);
		} 
		else 
		{
			short_url = b->url;
		}

		if (b->notes == NULL || b->notes[0] == '\0')
		{
			tip = short_url;
		}
		else
		{
			gchar *loc_str = mozilla_utf8_to_locale (b->notes);
			if (b->type == BM_FOLDER) {
				tip = g_strdup_printf ("%s", loc_str);
			}
			else
			{
				tip = g_strdup_printf ("%s\n%s", short_url,
						       loc_str);
			}
			g_free (loc_str);
		}
		
       		gtk_tooltips_set_tip (tb->tooltips, button, tip, NULL);

		if (short_url != b->url) g_free (short_url);
		if (tip != short_url) g_free (tip);
	
		/* fill the button */
		hbox = gtk_hbox_new (FALSE, 2);
		gtk_box_pack_start (GTK_BOX (hbox), box, FALSE, TRUE, 0);
		if (eel_gconf_get_boolean (CONF_BOOKMARKS_SMARTBOOKMARK_HIDE_ARROWS))
			gtk_box_pack_end (GTK_BOX (hbox), unfold_state, FALSE, TRUE, 0);
		gtk_container_set_border_width (GTK_CONTAINER (box), 0);
		gtk_container_add (GTK_CONTAINER (tray_button), hbox);
		gtk_widget_show_all (tray_button);

		dnd_dest = gtk_table_new (2, 2, FALSE);
		gtk_table_attach_defaults (GTK_TABLE (dnd_dest), tray_button,
					   0, 1, 0, 1);
		gtk_table_attach_defaults (GTK_TABLE (dnd_dest), tray,
					   1, 2, 0, 1);
		gtk_object_set_data (GTK_OBJECT (dnd_dest), "tray", tray);

		/* when the orientation of the toolbar changes, the widgets 
		   need to be moved in the table */
		gtk_signal_connect 
			(GTK_OBJECT (tb),
			 "orientation_changed", 
			 GTK_SIGNAL_FUNC 
			 (bookmarks_toolbars_orientation_changed_cb),
			 dnd_dest);

		/* set the correct orientation for the first time... */
		bookmarks_toolbars_orientation_changed_cb 
			(tb, tb->orientation, GTK_TABLE (dnd_dest));

		gtk_widget_size_request (tray_button, &r);

		gtk_object_set_data (GTK_OBJECT (tray_button), "tray", tray);
		gtk_object_set_data (GTK_OBJECT (tray), "button", tray_button);
		gtk_object_set_data (GTK_OBJECT (tray), "origsize",
				     GINT_TO_POINTER ((gint) r.width));
		gtk_object_set_data_full (GTK_OBJECT (tray), "key", key, 
					  g_free);

		gtk_widget_show_all (dnd_dest);
		gtk_toolbar_append_widget (tb, dnd_dest, NULL, NULL);

		gtk_signal_connect (GTK_OBJECT (tray_button),
				    "button-press-event", GTK_SIGNAL_FUNC
				    (bookmark_button_create_smart_menu), b);
		gtk_signal_connect 
			(GTK_OBJECT (tray_button),
			 "button-release-event", GTK_SIGNAL_FUNC
			 (smartbookmark_tray_button_release_event_cb), b);

		/* the entry field */
		enable_history = eel_gconf_get_boolean 
			(CONF_BOOKMARKS_HISTORY);
		
		if (enable_history)
		{
			combo = gtk_combo_new ();
			gtk_combo_disable_activate (GTK_COMBO (combo));
			entry = GTK_COMBO (combo)->entry;
			gtk_signal_connect 
				(GTK_OBJECT (GTK_COMBO (combo)->button),
				 "button-press-event", 
				 GTK_SIGNAL_FUNC
				 (bookmark_button_create_smart_menu), 
				 b);
		}
		else
		{
			entry = gtk_entry_new ();
		}

		gtk_object_set_data (GTK_OBJECT (tray_button), "entry", entry);
		gtk_object_set_data (GTK_OBJECT (dnd_dest), "entry", entry);
		gtk_object_set_data (GTK_OBJECT (tray), "entry", entry);
		gtk_object_set_data (GTK_OBJECT (tray), "combo",
				     enable_history ? combo : entry);
		gtk_object_set_data (GTK_OBJECT (tray), "fold", fold_state);
		gtk_object_set_data (GTK_OBJECT (tray), "unfold", unfold_state);

		entry_width = smartbookmark_get_size (b);

		gtk_widget_set_usize(entry, entry_width, -2);
		gtk_widget_show_all (enable_history ? combo : entry);

		gtk_object_set_data (GTK_OBJECT (entry),
				     "GaleonWindow", window);
		gtk_box_pack_start (GTK_BOX (tray), 
				enable_history ? combo : entry, TRUE, TRUE, 0);

		gtk_signal_connect (GTK_OBJECT (entry),
				    "key-press-event", GTK_SIGNAL_FUNC
				    (bookmarks_nick_entry_key_press_event_cb),
				    b);
		gtk_signal_connect (GTK_OBJECT (entry),
				    "button-press-event", GTK_SIGNAL_FUNC
				    (bookmark_button_create_smart_menu), b);
		
		/* folding button */
		if (!eel_gconf_get_boolean (CONF_BOOKMARKS_SMARTBOOKMARK_HIDE_ARROWS))
		{
			fold_button = gtk_button_new ();
			gtk_object_set_data (GTK_OBJECT (fold_button), "bookmark", b);
			GTK_WIDGET_UNSET_FLAGS (fold_button, GTK_CAN_FOCUS);
			gtk_button_set_relief (GTK_BUTTON (fold_button),
					gnome_preferences_get_toolbar_relief_btn () ?
						GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);

			gtk_box_pack_end (GTK_BOX (tray), 
					fold_button, TRUE, TRUE, 0);

	
     	  		gtk_tooltips_set_tip (tb->tooltips, fold_button,
					      _("Click to (un)fold"), NULL);

			hbox = gtk_hbox_new (FALSE, 0);
			gtk_box_pack_start (GTK_BOX (hbox), fold_state,
					    TRUE, TRUE, 0);
			gtk_box_pack_start (GTK_BOX (hbox), unfold_state,
					    TRUE, TRUE, 0);
			gtk_container_add (GTK_CONTAINER (fold_button), hbox);
			gtk_widget_show (hbox);
			gtk_widget_show (fold_button);
	
			gtk_object_set_data (GTK_OBJECT (fold_button), "key", key);

			gtk_signal_connect (GTK_OBJECT (fold_button),
					    "button-press-event", GTK_SIGNAL_FUNC
					    (bookmark_button_create_smart_menu), b);
			gtk_signal_connect (GTK_OBJECT (fold_button),
					    "button-release-event", GTK_SIGNAL_FUNC
					    (smartbookmark_fold_button_release_event_cb), b);
		}

		/* append some space */
		gtk_toolbar_append_space (tb);

		/* tray visibility notifier */
		str = CONF_STATE_SMARTBOOKMARK_VISIBILITY (key);
		
		noid = eel_gconf_notification_add
			(str, smartbookmark_visibility_changed, tray);
		gtk_signal_connect (GTK_OBJECT (tray),
				    "destroy", GTK_SIGNAL_FUNC
				    (smartbookmark_remove_notifier),
				    GINT_TO_POINTER (noid));
		
		smartbookmark_setup_tray (tray, !eel_gconf_get_boolean (str));
		
		g_free (str);


		/* entry size notifier */
		str = CONF_STATE_SMARTBOOKMARK_ENTRY_SIZE (key);
		
		noid = eel_gconf_notification_add
			(str, smartbookmark_entry_size_changed, tray);
		gtk_signal_connect (GTK_OBJECT (tray),
				    "destroy", GTK_SIGNAL_FUNC
				    (smartbookmark_remove_notifier),
				    GINT_TO_POINTER (noid));
		
		g_free (str);

		
		if (enable_history)
		{
			g_assert (combo != entry);

			/* history notifier */
			str = CONF_STATE_SMARTBOOKMARK_HISTORY (key);
		
			noid = eel_gconf_notification_add
				(str, smartbookmark_history_changed, combo);
			gtk_signal_connect (GTK_OBJECT (combo),
				    "destroy", GTK_SIGNAL_FUNC
				    (smartbookmark_remove_notifier),
				    GINT_TO_POINTER (noid));
		
			history = eel_gconf_get_string_list (str);
			smartbookmark_setup_history (GTK_COMBO (combo),
						     history);
			for (item = history; item; item = item->next) 
				g_free (item->data);
			g_slist_free (history);
				    
			g_free (str);
		}
	}

	gtk_object_set_data (GTK_OBJECT (dnd_dest), "bookmark", b);

	gtk_widget_size_request (dnd_dest, &req);

	return button;
}

void 
bookmarks_toolbars_set_visibility (GaleonWindow *window, gboolean visible)
{
	GList *l;
	return_if_not_window (window);	
	for (l = window->bookmarks_toolbars; l != NULL; l = g_list_next (l))
	{
		GtkWidget *toolbar = l->data;
		if (!GTK_IS_WIDGET (toolbar))
			continue;
		if (GTK_IS_WIDGET (toolbar->parent))
		{
			GtkWidget *w = toolbar->parent;
			if (visible)
			{
				gtk_widget_show (w);
			}
			else
			{
				gtk_widget_hide (w);
			}
		}
	}

	if (GTK_IS_CHECK_MENU_ITEM (window->view_fullscreen)
	    && !GTK_CHECK_MENU_ITEM (window->view_fullscreen)->active)
	{
		window->show_bookmarks = visible;
 	}
}

/** 
 * FIXME: comments
 */
static gboolean
folder_button_press_event_cb (GtkWidget *item, GdkEventButton *event,
				BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);

	if (event->window != item->window) return FALSE;
	if (event->button > 3) return FALSE;
	
	/* create the folder menu */
	if (event->button == 1)
	{
		GtkWidget *menu = gtk_menu_new();
		GtkTooltips *tooltips = gtk_tooltips_new ();

		bookmarks_menu_create_recursively
			(bi, GTK_MENU (menu), NULL, tooltips, 
			 TRUE, TRUE, TRUE);
		
		/* attach "GaleonWindow" to the menu so it can be looked up */
		gtk_object_set_data (GTK_OBJECT (menu), "GaleonWindow",
				     window);
		
		gtk_signal_emit_stop_by_name (GTK_OBJECT (item),
					      "button-press-event");

		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (item), TRUE);
		gnome_popup_menu_do_popup_modal (menu, 
					misc_gui_menu_position_under_widget,
					item, event, NULL);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (item), FALSE);
		gtk_widget_unref (GTK_WIDGET(menu));

		if (tooltips)
		{
			gtk_object_destroy (GTK_OBJECT (tooltips));
		}
	}
	else if (event->button == 2)
	{
		/* Load the bookmark(s) */
		gboolean tabbed_mode;

		tabbed_mode = eel_gconf_get_boolean (CONF_TABS_TABBED);
		if (event->state & GDK_SHIFT_MASK)
			tabbed_mode = !tabbed_mode;

		/* Load the bookmark in a new tab/window */
		bookmarks_folder_open_all_items (window->active_embed,
						 bi, !tabbed_mode, FALSE);
	}
	else if (event->button == 3)
	{
		GtkMenu *popup = GTK_MENU (gtk_menu_new ());


		bookmarks_menu_create_extra_items (popup,
				bi, EXTRA_ITEM_LOCATION_EXCLUSIVE);

		context_menu_add_seperator (popup);
		context_show_appearance_menu (window, popup, bi, TRUE, TRUE);
		context_menu_add_seperator (popup);

		context_menu_add_item (popup, _("Hide toolbar"),
				bookmarks_toolbar_item_menu_hide_cb, bi, 
				NULL, TRUE);

		/* warning: ugly code */
		gtk_object_set_data(GTK_OBJECT(popup), "GaleonWindow",
				    window);

		gtk_signal_emit_stop_by_name (GTK_OBJECT(item),
				"button-press-event");

		gnome_popup_menu_do_popup_modal (GTK_WIDGET (popup), NULL,
						 NULL, event, NULL);
		gtk_widget_unref (GTK_WIDGET (popup));
	}
	return TRUE;
}

/**
 * FIXME: comments
 */
static gboolean
bookmark_button_create_normal_menu (GtkWidget *item, GdkEventButton *event,
			     BookmarkItem *bi)
{
	return bookmark_button_create_menu (item, event, bi, FALSE, FALSE);
}

static gboolean
bookmark_button_create_smart_menu (GtkWidget *item, GdkEventButton *event,
			     BookmarkItem *bi)
{
	gchar *key, *str;
	gboolean expanded;
	
	key = prefs_key_new (bi->smarturl);
	str = CONF_STATE_SMARTBOOKMARK_VISIBILITY (key);
	expanded = eel_gconf_get_boolean (str);
	g_free (str);
	g_free (key);

	return bookmark_button_create_menu (item, event, bi, TRUE, !expanded);
}

static gboolean
bookmark_button_create_menu (GtkWidget *item, GdkEventButton *event,
			     BookmarkItem *bi, gboolean smart,
			     gboolean smart_expanded)
{
	GaleonWindow *window = window_from_widget (item);

	if ((event->button == 3) 
	    && (BOOKMARK_ITEM_IS_FOLDER (bi)
		|| bi->parent))
	{
		GtkMenu *popup = GTK_MENU (gtk_menu_new ());

		bookmarks_item_create_extra_items (popup, bi);
		
		if (smart) 
		{
			context_menu_add_item (popup, _("_Resize entry..."), 
					       bookmarks_resize_entry_cb, bi,
					       NULL, TRUE);
			if (smart_expanded)
			{
				context_menu_add_item (popup,
						       _("_Fold bookmark"), 
						       bookmarks_fold_cb, bi,
						       NULL, TRUE);
			}
			else
			{
				context_menu_add_item (popup,
						       _("_Unfold bookmark"), 
						       bookmarks_fold_cb, bi,
						       NULL, TRUE);
			}
			if (eel_gconf_get_boolean (CONF_BOOKMARKS_HISTORY)) 
			{
				context_menu_add_item
					(popup, _("_Clear history"),
					 bookmarks_clear_smart_history_cb, bi,
					 NULL, TRUE);
			}
		}

		context_menu_add_seperator (popup);
		context_show_appearance_menu (window, popup, bi, TRUE, TRUE);
		context_menu_add_seperator (popup);

		context_menu_add_item (popup, _("Hide toolbar"),
				bookmarks_toolbar_item_menu_hide_cb, bi,
				NULL, TRUE);

		/* warning: ugly code */
		gtk_object_set_data(GTK_OBJECT(popup), "GaleonWindow",
				    window);

		gtk_signal_emit_stop_by_name (GTK_OBJECT(item),
				"button-press-event");

		gnome_popup_menu_do_popup_modal (GTK_WIDGET (popup), NULL,
						 NULL, event, NULL);
		gtk_widget_unref (GTK_WIDGET (popup));
	}
	return TRUE;
}

/**
 * FIXME: comments
 */
static gboolean
bookmark_button_release_event_cb (GtkWidget *item, GdkEventButton *event,
				  BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);

	return_if_not_sane_click (item, event);

	/* Load the bookmark(s) */
	if (!GTK_IS_EVENT_BOX(item))
	{
		/* normal bookmark */
		bookmarks_set_visited (bi);
		embed_activate_link_mouse (window->active_embed, NULL, 
					   bi->url, event);
		embed_grab_focus (window->active_embed);
	}
	return TRUE;
}

/**
 * FIXME: comments
 */
static gboolean
bookmarks_toolbar_button_press_event_cb (GtkWidget *item,
					 GdkEventButton *event,
					 BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);

	if (event->button == 3)
	{
		if (BOOKMARK_ITEM_IS_FOLDER (bi) || (bi->parent != NULL))
		{
			GtkMenu *popup = GTK_MENU (gtk_menu_new ());
						
			bookmarks_menu_create_extra_items 
				(popup, bi, EXTRA_ITEM_LOCATION_EXCLUSIVE);
			context_menu_add_seperator (popup);
			context_show_appearance_menu (window, popup,
						      bi, FALSE, TRUE);
			
			/* warning: ugly code */
			gtk_object_set_data (GTK_OBJECT (popup),
					     "GaleonWindow",
					     window);
			
			gtk_signal_emit_stop_by_name (GTK_OBJECT (item),
						      "button-press-event");

			gnome_popup_menu_do_popup_modal (GTK_WIDGET (popup),
							 NULL, NULL,
							 event, NULL);
			gtk_widget_unref (GTK_WIDGET (popup));
		}
	}
	return TRUE;
}

/**
 * smartbookmark_tray_button_release_event_cb: The tray button of a smart
 * bookmark was clicked. Open/close the tray, or open in new tab/window
 * when middleclicked.
 */
static gboolean
smartbookmark_tray_button_release_event_cb (GtkWidget *item,
				            GdkEventButton *event,
				            BookmarkItem *bi)
{
	return_if_not_sane_click (item, event);

	if (GTK_IS_EVENT_BOX (item))
		return FALSE;

	/* decide what to do */
	if (event->button != 3 && !(event->state & GDK_CONTROL_MASK))
	{
		GaleonWindow *window = window_from_widget (item);
		GaleonEmbed *embed = window->active_embed;
		gchar *text, *translated_text, *url;
		GtkWidget *entry;

		/* smart bookmark, do search for phrase */
		entry = gtk_object_get_data (GTK_OBJECT (item), "entry");
		text = gtk_entry_get_text (GTK_ENTRY (entry));

		if (text != NULL && strlen (text) > 0)
		{
			/* translate non-alphanumeric characters+into %{}
			 * format */
			translated_text = bookmarks_encode_smb_string (text);
		
	                /* get a completed url */
			url = bookmarks_substitute_argument
				(bi, translated_text);

			/* if the search was already performed... */
			if (g_strcasecmp (url, embed->location) == 0 &&
			    event->button == 1)
			{
				/* search in the current page for the entered
				 * text */
				find_next (embed, text);
			}
			else
			{
				/* otherwise activate normal behaviour */
				bookmarks_set_visited (bi);
				if (eel_gconf_get_boolean
				    (CONF_BOOKMARKS_HISTORY))
				{
					smartbookmark_add_to_history
						(entry, text);
				}
				embed_activate_link_mouse
					(window->active_embed, NULL, url,
					 event);
				embed_grab_focus (window->active_embed);
			}

			g_free (url);
			g_free (translated_text);
		}
		else if (bi->url)
		{
			/* if search field is empty, go to the standard url */
			bookmarks_set_visited (bi);
			embed_activate_link_mouse (window->active_embed, NULL,
						   bi->url, event);
			embed_grab_focus (window->active_embed);
		}

		/* clear entry if requested */
		if (eel_gconf_get_boolean (CONF_BOOKMARKS_CLEAR))
			gtk_entry_set_text (GTK_ENTRY (entry), "");
		return TRUE;
	}
	else if (event->button != 3 && (event->state & GDK_CONTROL_MASK))
	{
		gchar *key, *str;
		GtkWidget *tray;
	
		tray = gtk_object_get_data (GTK_OBJECT (item), "tray");
		key = gtk_object_get_data (GTK_OBJECT (tray), "key");
		str = CONF_STATE_SMARTBOOKMARK_VISIBILITY (key);
		eel_gconf_set_boolean (str, !eel_gconf_get_boolean (str));
		g_free (str);
	}

	return FALSE;
}

/**
 * smartbookmark_fold_button_release_event_cb: (un)fold smart bookmark
 */
static gboolean
smartbookmark_fold_button_release_event_cb (GtkWidget *item,
				            GdkEventButton *event,
				            BookmarkItem *bi)
{
	return_if_not_sane_click (item, event);

	if (event->button != 3)
	{
		gchar *str, *key;
	
		key = gtk_object_get_data (GTK_OBJECT (item), "key");
		str = CONF_STATE_SMARTBOOKMARK_VISIBILITY (key);
		eel_gconf_set_boolean (str, !eel_gconf_get_boolean (str));
		g_free (str);

		return TRUE;
	}

	return FALSE;
}

/**
 * bookmarks_toolbar_item_menu_hide_cb: called by the hide context menu on the
 * toolbar items. Hide the toolbar.
 */
static void
bookmarks_toolbar_item_menu_hide_cb (GtkMenuItem *menuitem, 
				     BookmarkItem *bm)
{
	if (!bm->parent) return;
	bookmarks_toolbars_check_update (bm);
	bm->parent->create_toolbar = FALSE;
	bookmarks_updated ();
}

/**
 * bookmarks_nick_entry_key_press_event_cb: called when the user hits
 * return on an entry field on the toolbar (i.e. one created by a bookmark
 * which has %s in the url)
 */
gboolean
bookmarks_nick_entry_key_press_event_cb (GtkEntry *entry, GdkEventKey *event,
					 BookmarkItem *bi)
{
	gchar *text, *url, *translated_text;
	GaleonWindow *window;
	gint state;

	if (!(event->keyval == GDK_Return || event->keyval == GDK_KP_Enter))
		return FALSE;

	window = gtk_object_get_data (GTK_OBJECT (entry),
				      "GaleonWindow");

	/* get the entry text: DON'T free the returned text! */
	text = gtk_entry_get_text (entry);

	if (text != NULL && strlen (text) > 0)
	{
		/* translate non-alphanumeric characters into %{} format */
		translated_text = bookmarks_encode_smb_string (text);

		/* get a completed url */
		url = bookmarks_substitute_argument (bi, translated_text);

		/* update history */
		bookmarks_set_visited (bi);
		if (eel_gconf_get_boolean (CONF_BOOKMARKS_HISTORY))
			smartbookmark_add_to_history (GTK_WIDGET (entry), text);
	
		/* clear entry if requested */
		if (eel_gconf_get_boolean (CONF_BOOKMARKS_CLEAR))
			gtk_entry_set_text (entry, "");

		/* ignore the shift key */
		state = event->state & GDK_SHIFT_MASK ? 0 : event->state;

		/* load */
		embed_activate_link_keyboard (window->active_embed, NULL, url,
					      state);
		embed_grab_focus (window->active_embed);
	
		/* free all allocated strings */
		g_free (url);
		g_free (translated_text);
	}
	else if (bi->url)
	{
		/* if search field is empty, go to the standard url */
		bookmarks_set_visited (bi);
		embed_activate_link_keyboard (window->active_embed, NULL,
					      bi->url, event->state);
		embed_grab_focus (window->active_embed);
	}
	
	return TRUE;
}

/**
 * smartbookmark_setup_tray: setup the tray according to the @visiblity flag.
 */
static void
smartbookmark_setup_tray (GtkWidget *tray,
			  gboolean visiblity)
{
	GtkWidget *button, *fold, *unfold, *combo;
	gint origsize;
	GtkRequisition r, r2;

	g_return_if_fail (tray != NULL);

	fold = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (tray), "fold"));
	unfold = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (tray), "unfold"));
	button = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (tray), "button"));
	origsize = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (tray),
							 "origsize"));
	combo = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (tray), "combo"));

	gtk_widget_size_request (unfold, &r);
	gtk_widget_size_request (button, &r2);

	if (visiblity)
	{
		gtk_widget_show (combo);
		gtk_widget_hide (unfold);
		gtk_widget_show (fold);
		if (eel_gconf_get_boolean (CONF_BOOKMARKS_SMARTBOOKMARK_HIDE_ARROWS))
			gtk_widget_set_usize (button, origsize - r.width, r2.height);
	}
	else
	{
		gtk_widget_hide (combo);
		gtk_widget_show (unfold);
		gtk_widget_hide (fold);
		if (eel_gconf_get_boolean (CONF_BOOKMARKS_SMARTBOOKMARK_HIDE_ARROWS))
			gtk_widget_set_usize (button, origsize, r2.height);
	}

	if (tray->parent && GTK_IS_WIDGET (tray->parent)
	    && GTK_IS_WIDGET (tray->parent->parent)) 
	{
		gtk_widget_queue_resize (tray->parent->parent);
	}
}

/**
 * smartbookmark_setup_tray: setup the history combo according to the
 * @history list.
 */
static void
smartbookmark_setup_history (GtkCombo *combo,
			     GSList *history)
{
	if (NULL != history)
	{
		GSList *item;
		GList *strings = NULL;

		for (item = history; item; item = item->next)
		{
			gchar *locale = mozilla_utf8_to_locale (item->data);
			strings = g_list_prepend (strings, locale);
		}
		
		strings = g_list_reverse (strings);

		if (eel_gconf_get_boolean (CONF_BOOKMARKS_CLEAR))
			strings = g_list_prepend (strings, NULL);

		gtk_combo_set_popdown_strings (combo, strings);

		g_list_foreach (strings, (GFunc) g_free, NULL);
		g_list_free (strings);
	}
	else
		gtk_list_clear_items (GTK_LIST (combo->list), 0, -1);
}

/**
 * smartbookmark_add_to_history: add a value to the smartbookmarks history
 */
static void
smartbookmark_add_to_history (GtkWidget *widget, char *text)
{
	GtkCombo *combo;
	GtkWidget *tray;
	char *key, *str;
	GSList *history, *item;
	gint count = 0;
	gchar *unicode;

	if (!text || (0 == strlen (text))) return;

	unicode = mozilla_locale_to_utf8 (text);
	combo = GTK_COMBO (widget->parent);
	tray = GTK_WIDGET (combo)->parent;
	key = gtk_object_get_data (GTK_OBJECT (tray), "key");
	str = CONF_STATE_SMARTBOOKMARK_HISTORY (key);

	g_assert (unicode != NULL);
	history = eel_gconf_get_string_list (str);
	item = g_slist_find_custom (history, unicode, (GCompareFunc) strcmp);

	if (item)
	{
		history = g_slist_remove_link (history, item);
		history = g_slist_concat (item, history);
	}
	else
	{
		history = g_slist_prepend (history, strdup (unicode));
	}

	/* Trunc the slist to a maximum of 25 items. Yes, this maximum is 
	   hardcoded. If you need more than that for this, you are prbably on 
	   crack */
	for (item = history; item; item = item->next)
	{
		count++;
		if (count == 25)
		{
			GSList *rest = item->next;
			item->next = NULL;
			g_slist_foreach (rest, (GFunc) g_free, NULL);
			g_slist_free (rest);
		}
	}

	eel_gconf_set_string_list (str, history);

	g_slist_foreach (history, (GFunc) g_free, NULL);
	g_slist_free (history);

	g_free (unicode);
	g_free (str);
}

/**
 * smartbookmark_visibility_changed: modification monitor for the
 * smartbookmark's tray visiblity.
 */
static void
smartbookmark_visibility_changed (GConfClient *client,
                                  guint cnxn_id,
                                  GConfEntry *entry,
                                  gpointer user_data)
{
	smartbookmark_setup_tray (GTK_WIDGET (user_data),
				  !gconf_value_get_bool (entry->value));
}

/**
 * smartbookmark_history_changed: modification monitor for the
 * smartbookmark's history.
 */
static void
smartbookmark_history_changed (GConfClient *client,
                               guint cnxn_id,
                               GConfEntry *entry,
                               gpointer user_data)
{
	GSList *history, *item;

	history = eel_gconf_get_string_list (entry->key);

	smartbookmark_setup_history (GTK_COMBO (user_data), history);

	for (item = history; item; item = item->next) g_free (item->data);
	g_slist_free (history);
}

/**
 * smartbookmark_entry_size_changed: modification monitor for the
 * smartbookmark's entry size.
 */
static void
smartbookmark_entry_size_changed (GConfClient *client,
                                  guint cnxn_id,
                                  GConfEntry *entry,
                                  gpointer user_data)
{
	GtkWidget *tray = GTK_WIDGET (user_data);
	GtkWidget *w = gtk_object_get_data (GTK_OBJECT (tray),
					    "entry");
	gint entry_width = eel_gconf_get_integer (entry->key);
	gtk_widget_set_usize (w, entry_width, -2);

	if (tray->parent && GTK_IS_WIDGET (tray->parent)
	    && GTK_IS_WIDGET (tray->parent->parent)) 
	{
		gtk_widget_queue_resize (tray->parent->parent);
	}

}


/**
 * smartbookmark_remove_notifier: kill the associated notifier
 */
static void
smartbookmark_remove_notifier (GtkWidget *widget, guint noid)
{
	eel_gconf_notification_remove (noid);
}

static void 
bookmarks_clear_smart_history_cb (GtkWidget *menuitem, BookmarkItem *bi)
{
	gchar *key, *str;
	
	key = prefs_key_new (bi->smarturl);
	str = CONF_STATE_SMARTBOOKMARK_HISTORY (key);

	eel_gconf_set_string_list (str, NULL);

	g_free (str);
	g_free (key);
}

static void 
bookmarks_resize_entry_cb (GtkMenuItem *menuitem, BookmarkItem *bi)
{
	gchar *key, *str;
	SmartBookmarksEntrySizeEditor *editor = g_new0 
		(SmartBookmarksEntrySizeEditor, 1);
	GladeXML *gxml;
	
	/* make sure that the entry is visible */
	key = prefs_key_new (bi->smarturl);
	str = CONF_STATE_SMARTBOOKMARK_VISIBILITY (key);
	eel_gconf_set_boolean (str, FALSE);
	g_free (str);
	
	str = CONF_STATE_SMARTBOOKMARK_ENTRY_SIZE (key);
	editor->gconfkey = str;

	gxml = glade_widget_new ("bookmarks.glade", 
				 "smart_bookmarks_entry_size_editor",
				 &(editor->dialog), editor);
	
	editor->entry = glade_xml_get_widget 
		(gxml, "smart_bookmark_size_editor_entry");
	
	gtk_spin_button_set_value
		(GTK_SPIN_BUTTON (editor->entry), 
		 (gint) smartbookmark_get_size (bi));

	gtk_object_unref (GTK_OBJECT (gxml));
	g_free (key);
}

static void 
bookmarks_fold_cb (GtkMenuItem *menuitem, BookmarkItem *bi)
{
	gchar *key, *str;
	
	key = prefs_key_new (bi->smarturl);
	str = CONF_STATE_SMARTBOOKMARK_VISIBILITY (key);
	eel_gconf_set_boolean (str, !eel_gconf_get_boolean (str));
	g_free (str);
	g_free (key);
}

void
smart_bookmarks_entry_size_changed_cb (GtkWidget *w,
				       SmartBookmarksEntrySizeEditor *e)
{
	gint value = gtk_spin_button_get_value_as_int 
		(GTK_SPIN_BUTTON (e->entry));
	eel_gconf_set_integer (e->gconfkey, value);
}

void
smart_bookmarks_default_clicked_cb (GtkWidget *w,
				    SmartBookmarksEntrySizeEditor *e)
{
	gtk_spin_button_set_value
		(GTK_SPIN_BUTTON (e->entry), 
		 (gint) eel_gconf_get_integer (CONF_BOOKMARKS_ENTRY_WIDTH));
}

void
smart_bookmarks_close_clicked_cb (GtkWidget *w,
				  SmartBookmarksEntrySizeEditor *e)
{
	gtk_widget_destroy (e->dialog);
	g_free (e->gconfkey);
	g_free (e);
}

static gint 
smartbookmark_get_size (BookmarkItem *b)
{
	gchar *key, *str;
	gint size;

	key = prefs_key_new (b->smarturl);
	str = CONF_STATE_SMARTBOOKMARK_ENTRY_SIZE (key);
	size = eel_gconf_get_integer (str);

	if (size == 0)
	{
		size = eel_gconf_get_integer
			(CONF_BOOKMARKS_ENTRY_WIDTH);
	}

	g_free (str);
	g_free (key);
	return size;
}


static gboolean
bookmarks_toolbar_drag_motion_cb (GtkToolbar *tb, GdkDragContext *context,
				  gint x, gint y, guint time, BookmarkItem *bi)
{
	BookmarkItem *b;
	GtkCListDragPos drag_pos;
	GtkWidget *w;

	w = toolbar_get_dnd_info (tb, x, y, &b, &drag_pos);
	toolbar_show_dnd_hint (tb, w, drag_pos);

	gdk_drag_status (context, context->suggested_action, time);
	return TRUE;
}

/** 
 * bookmarks_toolbar_drag_data_received_cb:
 */
static void
bookmarks_toolbar_drag_data_received_cb (GtkToolbar *tb, 
					 GdkDragContext *drag_context,
					 gint x, gint y,
					 GtkSelectionData *selection_data,
					 guint info, guint time,
					 BookmarkItem *drag_bm)
{
	BookmarkItem *b = NULL; 
	gchar *mem = selection_data->data;
	gchar **tmp;

	g_return_if_fail (drag_bm != NULL);

	dnd_hints_hide_all ();

	switch (info)
	{
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_item_from_string (mem);
		/* This is a bit hacky... */
		if (BOOKMARK_ITEM_IS_FOLDER (b) 
		    && (g_list_length (b->list) == 1))
		{
			BookmarkItem *c = b->list->data;
			bookmark_unparent (c);
			bookmarks_free_bookmark (b);
			b = c;
		}
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
                /* netscape format is: url \n title */
		tmp = g_strsplit(mem, "\n", 2);
		if (tmp)
		{
			b = bookmarks_new_bookmark (BM_SITE, TRUE, tmp[1],
						    tmp[0], NULL, NULL, NULL);
		}
		else
		{
			b = bookmarks_new_bookmark (BM_SITE, TRUE, NULL, mem, 
						    NULL, NULL, NULL);
		}
		g_strfreev (tmp);
		break;
	case DND_TARGET_GALEON_URL:
		b = bookmarks_new_bookmark (BM_SITE, TRUE, NULL, mem, 
					    NULL, NULL, NULL);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	if (b)
	{
		BookmarkItem *bi;
		GtkCListDragPos drag_pos;

		toolbar_get_dnd_info (tb, x, y, &bi, &drag_pos);

		if (bi)
			drag_bm = bi;
		else /* the destination bookmark is the toolbar itself */
			drag_pos = GTK_CLIST_DRAG_INTO;
		
		bookmarks_insert_bookmark (b, drag_bm, drag_pos);
		bookmarks_tb_dirty = TRUE;
		bookmarks_updated ();
	 }
}

static GtkWidget *
toolbar_get_dnd_info (GtkToolbar *tb, gint desp_x, gint desp_y,
		      BookmarkItem **b,
		      GtkCListDragPos *type)
{
	GtkWidget *ret = NULL;
	GtkWidget *prev = NULL;
	GList *children = gtk_container_children (GTK_CONTAINER (tb));
	GList *li;
	gint x;
	gint desp;
	GtkOrientation o = tb->orientation;
	
	if (o == GTK_ORIENTATION_HORIZONTAL) 
	{
		x = GTK_WIDGET (tb)->allocation.x;
		desp = desp_x;
	}
	else
	{
		x = GTK_WIDGET (tb)->allocation.y;
		desp = desp_y;
	}
		

	if (b) *b = NULL;
	if (type) *type = GTK_CLIST_DRAG_AFTER;

	for (li = children; li != NULL; li = li->next)
	{
		GtkWidget *w = li->data;
		BookmarkItem *bi;
		gint size;
		if (!GTK_IS_WIDGET (w)) continue;
		if (o == GTK_ORIENTATION_HORIZONTAL)
			size = w->allocation.width;
		else
			size = w->allocation.height;
		bi = gtk_object_get_data (GTK_OBJECT (w), "bookmark");
		if (!bi) continue;
		if ((bi->type == BM_FOLDER)
		    || (bi->type == BM_AUTOBOOKMARKS))
		{
			if ((x <= desp) && ((x + size) >= desp))
			{
				ret = w;
				if (type) *type = GTK_CLIST_DRAG_INTO;
			}
		}
		else 
		{
			if ((x <= desp) && ((x + (size / 2)) >= desp))
			{
				ret = prev;
				if (type) *type = GTK_CLIST_DRAG_AFTER;
			}
			else if (((x + (size /2)) <= desp) 
				 && ((x + size ) >= desp))
			{
				ret = w;
				if (type) *type = GTK_CLIST_DRAG_AFTER;
			}
		}
		x += size;
		if ((li->next == NULL) && (x <= desp)) 
		{
			ret = w;
			if (type) *type = GTK_CLIST_DRAG_AFTER;
		}
		prev = w;
	}	
	
	g_list_free (children);

	if (ret && b)
	{
		*b = gtk_object_get_data (GTK_OBJECT (ret), "bookmark");
	}

	return ret;
}

static void
toolbar_show_dnd_hint (GtkToolbar *tb, GtkWidget *w,
		       GtkCListDragPos type)
{
	GtkOrientation o = tb->orientation;

	if (type == GTK_CLIST_DRAG_INTO) 
	{	
		dnd_hints_show_relative (HINT_ARROW_UP, w,
					 HINT_POSITION_CENTER, 
					 HINT_POSITION_BOTTOM);
		dnd_hints_show_relative (HINT_ARROW_DOWN, w,
					 HINT_POSITION_CENTER, 
					 HINT_POSITION_TOP);
		dnd_hints_show_relative (HINT_ARROW_LEFT, w,
					 HINT_POSITION_RIGHT, 
					 HINT_POSITION_CENTER);
		dnd_hints_show_relative (HINT_ARROW_RIGHT, w,
					 HINT_POSITION_LEFT, 
					 HINT_POSITION_CENTER);
	}
	else
	{
		if (w == NULL)
		{
			if (o == GTK_ORIENTATION_HORIZONTAL) 
			{
				dnd_hints_show_relative (HINT_ARROW_UP, 
							 GTK_WIDGET (tb),
							 HINT_POSITION_LEFT,
							 HINT_POSITION_BOTTOM);
				dnd_hints_show_relative (HINT_ARROW_DOWN, 
							 GTK_WIDGET (tb),
							 HINT_POSITION_LEFT,
							 HINT_POSITION_TOP);
				dnd_hints_hide (HINT_ARROW_LEFT);
				dnd_hints_hide (HINT_ARROW_RIGHT);
			}
			else
			{
				dnd_hints_show_relative (HINT_ARROW_LEFT, 
							 GTK_WIDGET (tb),
							 HINT_POSITION_RIGHT,
							 HINT_POSITION_TOP);
				dnd_hints_show_relative (HINT_ARROW_RIGHT, 
							 GTK_WIDGET (tb),
							 HINT_POSITION_LEFT,
							 HINT_POSITION_TOP);
				dnd_hints_hide (HINT_ARROW_UP);
				dnd_hints_hide (HINT_ARROW_DOWN);
			}
		}
		else 
		{
			if (o == GTK_ORIENTATION_HORIZONTAL)
			{
				dnd_hints_show_relative (HINT_ARROW_UP,
							 w,
							 HINT_POSITION_RIGHT,
							 HINT_POSITION_BOTTOM);
				dnd_hints_show_relative (HINT_ARROW_DOWN,
							 w,
							 HINT_POSITION_RIGHT,
							 HINT_POSITION_TOP);
				dnd_hints_hide (HINT_ARROW_LEFT);
				dnd_hints_hide (HINT_ARROW_RIGHT);
			}
			else
			{
				dnd_hints_show_relative (HINT_ARROW_LEFT,
							 w,
							 HINT_POSITION_RIGHT,
							 HINT_POSITION_BOTTOM);
				dnd_hints_show_relative (HINT_ARROW_RIGHT,
							 w,
							 HINT_POSITION_LEFT,
							 HINT_POSITION_BOTTOM);
				dnd_hints_hide (HINT_ARROW_UP);
				dnd_hints_hide (HINT_ARROW_DOWN);
			}
		}
	}
}

void
bookmarks_toolbars_check_update (BookmarkItem *b)
{
	g_return_if_fail (b != NULL);
	b = bookmarks_find_real_bookmark (b);
	do {
		if (b->create_toolbar)
			bookmarks_tb_dirty = TRUE;
		if (b->parent && b->parent->create_toolbar)
			bookmarks_tb_dirty = TRUE;
		b = b->alias;
	} while (b != NULL);
}

static void 
bookmarks_toolbars_orientation_changed_cb (GtkToolbar *tb, 
					   GtkOrientation o,
					   GtkTable *t)

{
	GtkWidget *tray = gtk_object_get_data (GTK_OBJECT (t), "tray");
	g_return_if_fail (tray != NULL);
	g_return_if_fail (GTK_IS_TABLE (t));
	g_return_if_fail (GTK_IS_WIDGET (tray));

	gtk_widget_ref (tray);
	gtk_container_remove (GTK_CONTAINER (t), tray);

	switch (o) {
	case GTK_ORIENTATION_HORIZONTAL:
		gtk_table_attach_defaults (t, tray, 1, 2, 0, 1);
		break;
	default:
		gtk_table_attach_defaults (t, tray, 0, 1, 1, 2);
		break;
	}

	gtk_widget_unref (tray);
	gtk_table_set_homogeneous (t, FALSE);
}

/**
 * prefs_key_len: length of @str when converted to a gconf key
 */
static guint 
prefs_key_len (gchar const * str)
{
        size_t len = 1;

        g_return_val_if_fail (str != NULL, 0);
        while (*str) len+= (strchr (invalid_chars, *str++) ? 3 : 1);

        return len;
}

/**
 * prefs_key_new: allocates a gconf key corresponding to @str
 */
static gchar * 
prefs_key_new (gchar const * str)
{
        gchar const * in = str;
        gchar * key, * out;

        g_return_val_if_fail (str != NULL, NULL);
        out = key = g_new0 (gchar, prefs_key_len (in));

        while (*in)
                if (strchr (invalid_chars, *in))
                        *out++ = ':',
                        *out++ = digits_base16 [*in >> 4],
                        *out++ = digits_base16 [*in++ & 15];
                else
                        *out++ = *in++;

        return key;
}

