/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: ObjectTreeView.cpp,v 1.25 2005/04/17 22:34:07 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"

#include "FWBTree.h"
#include "ObjectTreeView.h"
#include "ObjectTreeViewItem.h"
#include "ObjectManipulator.h"
#include "FWObjectDrag.h"

#include "fwbuilder/FWObject.h"
#include "fwbuilder/Resources.h"
#include "fwbuilder/Group.h"

#include <qdragobject.h>
#include <qlistview.h>
#include <qheader.h>
#include <qpainter.h>

#include <iostream>
#include <algorithm>

using namespace std;
using namespace libfwbuilder;

ObjectTreeView* ObjectTreeViewItem::getTree()
{
    return dynamic_cast<ObjectTreeView*>(listView());
}

/****************************************************************************
 *
 *    class ObjectTreeView
 *
 ****************************************************************************/

ObjectTreeView::ObjectTreeView(QWidget* parent, const char * name, WFlags f) :
    QListView(parent,name,f),
    singleClickTimer(this)
{
//    setAcceptDrops( TRUE );
    item_before_drag_started=NULL;
    lastSelected = NULL;
    second_click = false;
    selectionFrozen = false;

    connect( this, SIGNAL(currentChanged(QListViewItem*)),
             this, SLOT(currentChanged(QListViewItem*)) );

    connect( this, SIGNAL( selectionChanged() ),
             this, SLOT( selectionChanged() ) );

    connect( &singleClickTimer, SIGNAL( timeout() ),
             this, SLOT( resetSelection() ) );

    addColumn( tr( "Object" ) );

    header()->hide();

    setMinimumSize( QSize( 200, 0 ) );

//    QFont objTreeView_font(  font() );
//    setFont( objTreeView_font ); 
//    setCursor( QCursor( 0 ) );
    setResizePolicy( QListView::AutoOneFit );

    setColumnWidthMode(0, QListView::Maximum);
    setItemMargin( 2 );

    setDragAutoScroll( TRUE );
    setSelectionMode( QListView::Extended );
    setShowSortIndicator( FALSE );
    setRootIsDecorated( TRUE );
    setResizeMode( QListView::AllColumns );
    setTreeStepSize( 15 );

    setSorting(0,true);
}

void ObjectTreeView::currentChanged(QListViewItem *ovi)
{
    lastSelected = ovi;
}

/*
 * This method makes list selectedObjects flat. If user selects
 * several objects in the tree, and some of them have children, QT
 * puts all the children in the selected objects list even if
 * corresponding subtrees are collapsed. This method eliminates these
 * selected children objects.
 *
 */
std::vector<libfwbuilder::FWObject*> ObjectTreeView::getSimplifiedSelection()
{
    vector<FWObject*> so  = selectedObjects;
    vector<FWObject*> so2 = selectedObjects;
    for (vector<FWObject*>::iterator i=so2.begin();  i!=so2.end(); ++i)
    {
        for (vector<FWObject*>::iterator j=i;  j!=so2.end(); ++j)
        {
            vector<FWObject*>::iterator k=std::find(so.begin(),so.end(),*j);
            if ( (*j)->isChildOf( *i ) && k!=so.end())
                so.erase( k );
        }
    }
    return so;
}


QDragObject* ObjectTreeView::dragObject()
{
    QListViewItem *ovi = currentItem();
    item_before_drag_started = ovi;
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
    assert(otvi!=NULL);

    FWObject *obj = otvi->getFWObject();

/* can't drag system folders 

    in fact, I have to allow to drag system folders because otherwise
    QListView triggers highlighting of objects in the tree when user
    drags mouse cursor across them. This is weird behavior and there
    does not seem to be any way to turn it off. It happens close to
    the end of void QListView::contentsMouseMoveEvent( QMouseEvent * e)
    (See code after they decided that they do not need to call startDrag())

    if (FWBTree::isSystem(obj)) return NULL;
*/
    QString icn_filename =
        Resources::global_res->getObjResourceStr(obj, "icon-ref").c_str();

    vector<FWObject*> so = getSimplifiedSelection();

    list<FWObject*> dragobj;
    for (vector<FWObject*>::iterator v=so.begin(); v!=so.end(); v++)
        dragobj.push_back( *v );

    FWObjectDrag    *drag = new FWObjectDrag(dragobj, this);
    QPixmap          pm   = QPixmap::fromMimeSource( icn_filename );

    if (dragobj.size()>1)
    {
        QPixmap npm(32,32);
        QPainter p( &npm );
        p.fillRect( 0,0,32,32, QBrush( QColor("white"),Qt::SolidPattern ) );
        p.setBackgroundMode( TransparentMode );
        p.drawPixmap( 0, 32-pm.rect().height(), pm);
        p.setPen( QColor("red") );
        p.setBrush( QBrush( QColor("red"),Qt::SolidPattern ) );
        p.drawPie( 16, 0, 16,16, 0, 5760 );
        QString txt;
        txt.setNum(dragobj.size());
        QRect br=p.boundingRect(0, 0, 1000, 1000,
                                Qt::AlignLeft|Qt::AlignVCenter,
                                txt );
        p.setPen( QColor("white") );
        p.drawText( 24-br.width()/2 , 4+br.height()/2, txt );
        npm.setMask( npm.createHeuristicMask() );
        drag->setPixmap( npm,
                         QPoint( npm.rect().width() / 2,
                                 npm.rect().height() / 2 ) );
    } else
        drag->setPixmap( pm,
                         QPoint( pm.rect().width() / 2,
                                 pm.rect().height() / 2 ) );

    resetSelection();

    if (fwbdebug) qDebug("ObjectTreeView::dragObject()  returns !NULL");

    return drag;
}

void ObjectTreeView::dragEnterEvent( QDragEnterEvent *ev)
{
    ev->accept( FWObjectDrag::canDecode(ev) );
    QListViewItem *ovi = currentItem();
    item_before_drag_started = ovi;
}

void ObjectTreeView::dragMoveEvent( QDragMoveEvent *ev)
{
    bool    acceptE = false;
    
    if (FWObjectDrag::canDecode(ev))
    {
        int hy;

//        hy=header()->height();    // if header is shown
        hy=0;

        QListViewItem *ovi = itemAt( ev->pos() );

        ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
        if (otvi==NULL)
        {   
            ev->accept(acceptE);
            return;
        }

        FWObject *trobj = otvi->getFWObject();

/* the tree can accept drop only if it goes into a group and if that group
 * validates the object and tree is not read-only
 */
        if (Group::cast(trobj)!=NULL  && 
            !FWBTree::isSystem(trobj) &&
            !trobj->isReadOnly()
        )
        {
            acceptE = true;

            Group    *g     = Group::cast(trobj);
            list<FWObject*> dragol;
            if (FWObjectDrag::decode(ev, dragol))
            {
                for (list<FWObject*>::iterator i=dragol.begin();
                     i!=dragol.end(); ++i)
                {
                    FWObject *dragobj = *i;
                    assert(dragobj!=NULL);

                    if (FWBTree::isSystem(dragobj))
                    {
/* can not drop system folder anywhere */
                        acceptE = false;
                        break;
                    }

                    bool t= g->validateChild(dragobj);
                    if (!t)
                    {
                        acceptE = false;
                        break;
                    }

                    if (g->getPath(true) == "Services/Groups" && t)
                        setOpen(ovi,true);

                    if (g->getPath(true) == "Objects/Groups" && t)
                        setOpen(ovi,true);
                }
            }
        }
    }
    ev->accept(acceptE);
}

void ObjectTreeView::dropEvent(QDropEvent *ev)
{
    QListViewItem *ovi = itemAt( ev->pos() );
    
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
    if (otvi==NULL) return;


    FWObject *trobj = otvi->getFWObject();

/* the tree can accept drop only if it goes into a group and if that group
 * validates the object and the tree is not read-only
 */
    if (Group::cast(trobj)!=NULL  && 
        !FWBTree::isSystem(trobj) &&
        !trobj->isReadOnly()
    )
    {
        Group *g=Group::cast(trobj);

        item_before_drag_started=NULL;

        list<FWObject*> dragol;
        if (FWObjectDrag::decode(ev, dragol))
        {
            for (list<FWObject*>::iterator i=dragol.begin();
                 i!=dragol.end(); ++i)
            {
                FWObject *dragobj = *i;
                assert(dragobj!=NULL);

/* check for duplicates */
                string cp_id=dragobj->getId();
                list<FWObject*>::iterator j;
                for(j=g->begin(); j!=g->end(); ++j)     
                {
                    FWObject *o1=*j;
                    if(cp_id==o1->getId()) continue;

                    FWReference *ref;
                    if( (ref=FWReference::cast(o1))!=NULL &&
                        cp_id==ref->getPointerId()) return;
                }

                g->addRef(dragobj);
            }
            setCurrentItem(ovi);
            setSelected(ovi, true);

//            emit objectDropped_sign(g);
        }
    }
}

void ObjectTreeView::dragLeaveEvent( QDragLeaveEvent *ev)
{
    QListView::dragLeaveEvent(ev);

    if (item_before_drag_started!=NULL)
    {
        setCurrentItem(item_before_drag_started);
        setSelected(item_before_drag_started, true);
    }
}

void ObjectTreeView::contentsMouseMoveEvent( QMouseEvent * e )
{
    if (e==NULL)  return;

    QListView::contentsMouseMoveEvent(e);
}


void ObjectTreeView::contentsMousePressEvent( QMouseEvent *e )
{
    second_click = false;

    lastSelected = currentItem();

    QListView::contentsMousePressEvent(e);
}

/*
 * Two modes of operation of this widget:
 *
 * 1.  this widget can intercept single mouse click and return
 * selection back to the object that was current before it. If user
 * double ckicks mouse button, then this reset is not done and new
 * object is selected. This is done using timer.
 *
 * 2. this widget can act as usual QListView does, that is, select an object
 * on a single click.
 *
 * uncomment the line that starts timer for mode #1.
 */
void ObjectTreeView::contentsMouseReleaseEvent( QMouseEvent *e )
{
//    if (!second_click) singleClickTimer.start( 200 , true );

    QListView::contentsMouseReleaseEvent(e);
}


void ObjectTreeView::contentsMouseDoubleClickEvent( QMouseEvent *e )
{
    second_click=true;
    singleClickTimer.stop();

    QListViewItem *ovi = currentItem();
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
    assert(otvi!=NULL);
    FWObject *obj = otvi->getFWObject();

/* system folders open on doubleclick, while for regular objects it
 * opens an editor 
 */
    if (FWBTree::isSystem(obj))
        QListView::contentsMouseDoubleClickEvent(e);
    else
        emit openObjectEditor_sign(obj);
}

void ObjectTreeView::keyPressEvent( QKeyEvent* ev )
{
    QListViewItem *ovi = currentItem();
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
    assert(otvi!=NULL);
    FWObject *obj = otvi->getFWObject();

    if (ev->key()==Qt::Key_Enter || ev->key()==Qt::Key_Return)
    {
        emit openObjectEditor_sign(obj);
        ev->accept();
        return;
    }
    if (ev->key()==Qt::Key_Delete)
    {
        emit deleteObject_sign(obj);
        ev->accept();
        return;
    }    
    QListView::keyPressEvent(ev);
}

void ObjectTreeView::returnPressed(QListViewItem *itm)
{
    QListViewItem *ovi = currentItem();
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
    assert(otvi!=NULL);
    FWObject *obj = otvi->getFWObject();

    emit openObjectEditor_sign(obj);
}

void ObjectTreeView::resetSelection()
{
    setSelected( lastSelected , true);
}

void ObjectTreeView::selectionChanged()
{
    if (fwbdebug)
        qDebug("ObjectTreeView::selectionChanged selectionFrozen=%d",
               selectionFrozen);

    if (selectionFrozen) return;

/* in extended selection mode there may be several selected items */

    selectedObjects.clear();

    QListViewItemIterator it(this);
    while ( it.current() )
    {
        if (it.current()->isSelected())
        {
            QListViewItem *itm= it.current();
            ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(itm);
            selectedObjects.push_back(otvi->getFWObject());

            if (fwbdebug)
                qDebug("ObjectTreeView::selectionChanged: selected otvi=%p object %s", otvi, otvi->getFWObject()->getName().c_str());
        }
        ++it;
    }

/* now list  selectedObjects   holds all selected items */
}

bool ObjectTreeView::isSelected(FWObject* obj)
{
    for (vector<FWObject*>::iterator i=selectedObjects.begin();
         i!=selectedObjects.end(); ++i)
    {
        if ( (*i)==obj)  return true;
    }
    return false;
}

int  ObjectTreeView::getNumSelected()
{
    return selectedObjects.size();
}

