/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: PolicyList.cc,v 1.28 2001/12/27 06:48:47 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include <iostream.h>

#include "glademm_support.hh"
#include "helpers.hh"
#include "BuiltinDialog.hh"
#include "gen_popup_menu.hh"
#include "Resources.hh"

#include "fwbuilder/Policy.hh"
#include "fwbuilder/NAT.hh"
#include "fwbuilder/Rule.hh"

#include "PolicyListItem.hh"
#include "PolicyListElement.hh"
#include "PolicyList.hh"

#include "main_window.hh"



using namespace libfwbuilder;

PolicyList::PolicyList(RuleSet *p) : Gtk::Table(1,2)
{
    ruleset=p;
    current_selected_row=0;
    current_selected=0;
    rebuild=false;
    width=height=0;

    set_flags( GTK_CAN_FOCUS );

    realize.connect(SigC::slot(this, &PolicyList::on_realize_event));
    size_allocate.connect(SigC::slot(this, &PolicyList::on_size_allocate));

    button_release_event.connect_after(
	SigC::slot(this,&PolicyList::on_button_release_event));

    show();
}

PolicyList::~PolicyList()
{
//  Clear();
}

/*
 *   We keep track of our physical size
 */
void PolicyList::on_size_allocate(GtkAllocation *all)
{
    x=all->x;
    y=all->y;
    width=all->width;
    height=all->height;
}

gint PolicyList::on_button_release_event(GdkEventButton *ev)
{
    if ( ev->type == GDK_BUTTON_RELEASE && ev->button==3 ) {

	gen_popup_menu         *gpm;

	const char *menu_items0[] = 
	    { "Insert rule at the top",
	      "Append rule at the bottom",
	      NULL,
	      NULL } ;

	if ( ruleset->size()==0 ) {
	    menu_items0[0]="Insert New rule";
	    menu_items0[1]="";
	    menu_items0[2]="Paste Rule";
	    menu_items0[3]=NULL;
	}

	gpm=new gen_popup_menu( menu_items0 );
	gpm->popup(0,0);
	gint menu_choice=gpm->run();
	delete gpm;

	switch (menu_choice) {
	case 0:
	    insertRuleAtTop();
	    break;

	case 1:
	    appendRuleAtBottom();
	    break;

	case 2:
	{
	    FWObject *obj;
	    if ( (obj=FWObjectClipboard::obj_clipboard->getObject())!=NULL &&
		 dynamic_cast<Rule*>(obj)!=NULL ) {
		appendRuleAtBottom( dynamic_cast<Rule*>(obj) );
	    }
	}
	break;
	}
    }
    return( true );
}


void PolicyList::move_focus(GtkDirectionType dir)
{
    gint r,c;

    if (current_selected!=0) {

//	current_selected->deselect();

	r=current_selected->get_row();
	c=current_selected->get_col();

	switch (dir ) {
	case GTK_DIR_UP:     r-=1;    break;
	case GTK_DIR_DOWN:   r+=1;    break;
	case GTK_DIR_LEFT:   c-=1;    break;
	case GTK_DIR_RIGHT:  c+=1;    break;
	default:         return;
	}
	select_element(r,c);
    } else
	select_element(0,0);
}

void PolicyList::select_element(gint r,gint c)
{
    bool going_up=false;

    PolicyListElement *pe;
    if ( (pe=get_element(r,c))!=0 ) {

	if ( current_selected!=0) {
	    int cr=current_selected->get_row();
	    current_selected->deselect();
	    current_selected=NULL;
	    if (cr!=r) {
		deactivate_row(cr);
		activate_row(r);
		going_up= (r<cr);
	    }
	}

	set_current_selected(pe);
	pe->select(false);
	if (going_up) pe->select_last_child();
	else          pe->select_first_child();
    }
}

void PolicyList::deselect_current()
{
    if ( current_selected!=0) {
	current_selected->deselect();
	deactivate_row(current_selected->get_row());
	current_selected=NULL;
    }
}

void PolicyList::set_current_selected(PolicyListElement *pe)
{
    current_selected=pe;
    current_selected_row=pe->get_row();
}

void PolicyList::request_focus()
{
    if (!has_focus()) {
	grab_focus();
	draw_focus();
    }
}


void PolicyList::activate_row(gint row)
{
    PolicyListElement *pe;

    deactivate_row( current_selected_row );
    current_selected_row=row;

    vector<Gtk::Widget*>::iterator k;
    if (!table_children.empty()) {
	for (k=table_children.begin(); k!=table_children.end(); ++k) {
	    if (dynamic_cast<PolicyListElement*>(*k)!=NULL) {
		pe=(PolicyListElement*)(*k);
		if (pe->get_row()==row) pe->activate();
	    }
	}
    }
}

void PolicyList::activate_rule(gint r)
{
    activate_row( get_row_by_rule_num(r) );
}

void PolicyList::deactivate_row(gint row)
{
    PolicyListElement *pe;

    vector<Gtk::Widget*>::iterator k;
    if (!table_children.empty()) {
	for (k=table_children.begin(); k!=table_children.end(); ++k) {
	    if (dynamic_cast<PolicyListElement*>(*k)!=NULL) {
		pe=(PolicyListElement*)(*k);
		if (pe->get_row()==row) pe->deactivate();
	    }
	}
    }
}

void PolicyList::deactivate_rule(gint r)
{
    deactivate_row( get_row_by_rule_num(r) );
}


void PolicyList::schedule_rebuild(int r)
{
    activate_r=r;
    Gtk::Main::idle.connect(slot(this,&PolicyList::rebuild_when_idle));
}

gint PolicyList::rebuild_when_idle()
{
    Build();
    activate_rule(activate_r);
    return false;
}


void PolicyList::on_realize_event()
{
}


void PolicyList::set_data_changed_flag(bool flag)
{
    BuiltinDialog *b= (BuiltinDialog *)find_widget("BuiltinDialog",this);
    b->data_changed_flag(flag);
}

void PolicyList::__print_children()
{
    vector<Gtk::Widget*>::iterator k;
    if (!table_children.empty()) {
	cerr << "  Table size: " << table_children.size() << endl;

	cerr << "  Table: " << this << " gtkobj=" << gtkobj() <<  endl;
	for (k=table_children.begin(); k!=table_children.end(); ++k)
	    print_widget_tree( 0 , (*k) );
	cerr << endl;
    }
}

void PolicyList::Clear()
{
    vector<Gtk::Widget*>::iterator k;

    if (!table_children.empty()) {
	for (k=table_children.begin(); k!=table_children.end(); ++k) {
	    Gtk::Widget *w=(*k);
	    g_assert( w!=NULL );
//      w->destroy();
	    w->hide();
	    remove(*w);
	}
    }
    table_children.clear();
    current_selected=0;
}

Gtk::Widget* PolicyList::_register_child(Gtk::Widget *c)
{
    table_children.push_back(c);
    return c;
}


/************************************************************
 *                                                          *
 *     Rebuild the list                                     *
 *                                                          *
 ************************************************************/

void PolicyList::Build()
{
    gint                row, col, rule_n;
    gint                tbl_size;
    Gtk::Button        *btn;
    string              s1;
    PolicyListElement  *pe;


    hide();

    //
    // destroy current list table_children
    //

    Clear();

    //
    //   Cosntruct new one
    //
    if (ruleset==NULL) {
	return;
    } else {
	tbl_size=ruleset->size()*2+1 +1;
    }


    resize( tbl_size , (ruleset->countElements()+1)*2 );

    row=0;
    col=0;

    FWObject *o;

    btn= manage(new Gtk::Button( "Num" ));
    btn->set_name("Num");
    btn->set_state(GTK_STATE_NORMAL);
    attach( *btn , col , col+1 , row , row+1 , GTK_FILL, 0, 0, 1);
    _register_child(btn);

    btn->show();

    col++;

//    beginTimestamp();

    list<const  char*> titles;
    getTitles(*ruleset, titles);
    for(list<const char*>::iterator ci=titles.begin(); ci!=titles.end(); ci++){
	if ( (*ci) ) {

	    btn= manage(new Gtk::Button( (*ci) ));

	    btn->set_state(GTK_STATE_NORMAL);

	    if ( strcmp( (*ci) , "Source")==0 ||
		 strcmp( (*ci) , "Destination")==0 ||
		 strcmp( (*ci) , "Service")==0 ||
		 strcmp( (*ci) , "Comment")==0 ) 
		attach( *btn , col , col+1 , row , row+1 , 
			GTK_FILL|GTK_EXPAND, 0, 0, 1);
	    else
		attach( *btn , col , col+1 , row , row+1 , 
			GTK_FILL, 0, 0, 1);


	    _register_child(btn);

	    btn->show();

	    col++;
	}
    }

    row++;

//    printTimestamp("Stop1");

    rule_n=0;
    /*
     *   this outer loop choses policy rule
     */
    vector<FWObject*>::iterator m;
    for (m=ruleset->begin(); m!=ruleset->end(); ++m,++rule_n ) {
//	if ( (o=(*m))!=NULL && o->getTypeName()=="RULE") {

	o=(*m);
	if (Rule::cast(o)!=NULL) {
	    Rule *rule=Rule::cast(o);

	    col=0;

	    pe=manage(new PolicyListRuleNum(row,col,rule_n,rule->isDisabled()));
	    pe->set_user_data(this);

	    attach( *pe , col , col+1 , row , row+1 , 
		    GTK_FILL, GTK_FILL, 0, 0);
	    pe->show();

	    _register_child(pe);

	    col++;

//	    printTimestamp("Stop1.1");

/*
 *  this inner loop choses rule element
 */
	    list<const char*> elements;
	    ruleset->getElements(elements);
	    for(list<const char*>::iterator ci=elements.begin(); 
                ci!=elements.end(); 
                ci++)  {
		pe=manage(createPolicyListElement(row,col,(*ci),rule));
		pe->set_user_data(this);
		attach( *pe , col , col+1 , row , row+1 , 
			GTK_FILL, GTK_FILL, 0, 0);
		pe->deactivate();
		pe->show();

		_register_child(pe);

		col++;
	    }
	    row++;

//	    printTimestamp("Stop1.2");
	}
    }

//    printTimestamp("Stop2");


    // this adds row of event boxes at the very bottom - EXPERIMENTAL

    Gtk::EventBox *eb = manage(new Gtk::EventBox());
    attach( *eb , 0 , (ruleset->countElements()+1)*2 , row , row+1 , 
	    GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 1);
    _register_child(eb);
    eb->show();

//    printTimestamp("Stop3");

    show_all();

//    printTimestamp("Stop4");
	    
    activate_rule(0);
    select_element(0,0);
    
    request_focus();
}

void PolicyList::_add_hseparator(gint row,gint col)
{
    Gtk::HSeparator *sep= manage(new Gtk::HSeparator());
    attach( *sep , 0 , col , row , row+1 , GTK_FILL, GTK_FILL, 0, 0);
    _register_child(sep);
}

void PolicyList::_add_vseparator(gint row,gint col)
{
    Gtk::VSeparator *sep= manage(new Gtk::VSeparator());
    attach( *sep , col , col+1 , row , row+1 , 0, GTK_FILL, 0, 0);
    _register_child(sep);
}


gint PolicyList::get_row_by_rule_num(gint rule_n)
{
    return( rule_n+1 );     // if there are no separators
}

gint PolicyList::get_rule_by_row(gint row_n)
{
    return( row_n-1 );       // if there are no separators
}

void PolicyList::copyRuleContent(Rule *dst, Rule *src)
{
    string id=dst->getId();
    int     p=dst->getPosition();

    if ( src->isDisabled() ) dst->disable();
    else                     dst->enable();

    map<string, string>::iterator i;
    for(i=dst->dataBegin(); i!=dst->dataEnd(); ++i) {
	string f= (*i).first;
	dst->setStr(f, src->getStr(f) );
    }

    vector<FWObject*>::iterator j;
    for(j=dst->begin(); j!=dst->end(); ++j) {
	string    dtype= (*j)->getTypeName();
	FWObject *selem= src->getFirstByType(dtype);
	if (selem!=NULL) 
	    (*j)->duplicate(selem);
    }

    if (id!="")	dst->setId(id);
    dst->setPosition(p);
}


void PolicyList::insertRuleAtTop(Rule *r)
{
    if (r!=NULL && 
	ruleset->getTypeName()==Policy::TYPENAME && 
	r->getTypeName()!=PolicyRule::TYPENAME)  return;
    if (r!=NULL && 
	ruleset->getTypeName()==NAT::TYPENAME    && 
	r->getTypeName()!=NATRule::TYPENAME   )  return;

    Rule *newrule;
    if ( (newrule=ruleset->insertRuleAtTop())!=NULL ) {
	if (r!=NULL) {
	    copyRuleContent(newrule,r);
	}
	schedule_rebuild(0);
	set_data_changed_flag(true);
    }
}

void PolicyList::insertRuleBefore(gint rule_n,Rule *r)
{
    if (r!=NULL && 
	ruleset->getTypeName()==Policy::TYPENAME && 
	r->getTypeName()!=PolicyRule::TYPENAME)  return;
    if (r!=NULL && 
	ruleset->getTypeName()==NAT::TYPENAME    && 
	r->getTypeName()!=NATRule::TYPENAME   )  return;

    Rule *newrule;
    int   nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( (newrule=ruleset->insertRuleBefore(nr))!=NULL ) {
	if (r!=NULL) {
	    copyRuleContent(newrule,r);
	}
	schedule_rebuild(nr);
	set_data_changed_flag(true);
    }
}

void PolicyList::appendRuleAfter(gint rule_n,Rule *r)
{
    if (r!=NULL && 
	ruleset->getTypeName()==Policy::TYPENAME && 
	r->getTypeName()!=PolicyRule::TYPENAME)  return;
    if (r!=NULL && 
	ruleset->getTypeName()==NAT::TYPENAME    && 
	r->getTypeName()!=NATRule::TYPENAME   )  return;

    Rule *newrule;
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( (newrule=ruleset->appendRuleAfter( nr ))!=NULL ) {
	if (r!=NULL) {
	    copyRuleContent(newrule,r);
	}
	schedule_rebuild(nr);
	set_data_changed_flag(true);
    }
}

void PolicyList::appendRuleAtBottom(Rule *r)
{
    if (r!=NULL && 
	ruleset->getTypeName()==Policy::TYPENAME && 
	r->getTypeName()!=PolicyRule::TYPENAME)  return;
    if (r!=NULL && 
	ruleset->getTypeName()==NAT::TYPENAME    && 
	r->getTypeName()!=NATRule::TYPENAME   )  return;

    Rule *newrule;
    if ( (newrule=ruleset->appendRuleAtBottom())!=NULL ) {
	if (r!=NULL) {
	    copyRuleContent(newrule,r);
	}
	schedule_rebuild(ruleset->getRuleSetSize() - 1);
	set_data_changed_flag(true);
    }
}

void PolicyList::delRule(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->deleteRule(nr) ) {
	schedule_rebuild(nr-1);
	set_data_changed_flag(true);
    }
}

void PolicyList::moveRuleUp(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->moveRuleUp(nr) ) {
	schedule_rebuild(nr);
	set_data_changed_flag(true);
    }
}

void PolicyList::moveRuleDown(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->moveRuleDown(nr) ) {
	schedule_rebuild(nr);
	set_data_changed_flag(true);
    }
}

void PolicyList::disableRule(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->disableRule(nr) ) {
	schedule_rebuild(nr);
	set_data_changed_flag(true);
    }
}

void PolicyList::enableRule(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->enableRule(nr) ) {
	schedule_rebuild(nr);
	set_data_changed_flag(true);
    }
}

bool  PolicyList::isRuleDisabled(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    return ( ruleset->isRuleDisabled(nr) );
}

PolicyListElement* PolicyList::get_element(gint r,gint c)
{
    PolicyListElement    *pe;
    vector<Gtk::Widget*>::iterator k;
    if (!table_children.empty()) {
	for (k=table_children.begin(); k!=table_children.end(); ++k) {
	    if (dynamic_cast<PolicyListElement*>(*k)!=NULL) {
		pe=(PolicyListElement*)(*k);
		if (pe->get_row()==r && pe->get_col()==c) 
		    return(pe);
	    }
	}
    }
    return(0);
}


PolicyListElement* PolicyList::createPolicyListElement(int row,int col,
						   const char* element_name,
						   Rule *rule)
{
    PolicyListElement *pe;
    RuleElement       *rel;
    FWObject          *o;

    if (strcmp(element_name,"Act")==SAME) 
	pe = new PolicyListRuleAction(row,col,rule);
    else if (strcmp(element_name,"Dir")==SAME) 
	pe = new PolicyListRuleDir(row,col,rule);
    else if (strcmp(element_name,"Log")==SAME) 
	pe = new PolicyListRuleLog(row,col,rule);
    else if (strcmp(element_name,"Opt")==SAME) 
	pe = new PolicyListRuleOpt(row,col,rule);
    else if (strcmp(element_name,"Com")==SAME) 
	pe = new PolicyListRuleComment(row,col,rule);
    else {

	o=rule->getFirstByType( element_name );
	rel=RuleElement::cast(o) ;
	g_assert(rel!=NULL);
		    
	pe=new PolicyListElement(row,col,rel);
    }
    return pe;
}

void PolicyList::getTitles(RuleSet &rs, list<const char*>& titles)
{
    list<const char*> elements;
    rs.getElements(elements);
    
    for(list<const char*>::iterator ci=elements.begin(); 
        ci!=elements.end(); ci++)
    {
	const char *s = Resources::global_res->getRuleElementResourceStr( *ci , "label" );

        /*
         *  TODO: throw exception for global catch for broken resources (vk)
         */
	assert(s!=NULL);
	titles.push_back(s);
    }
}

