/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: ObjectTree.cc,v 1.67 2001/12/29 10:06:35 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include "glademm_support.hh"

#include "fwbuilder/FWObject.hh"
#include "FWObjectDatabaseGUI.hh"
#include "ObjectTree.hh"
#include "fwbuilder/Group.hh"
#include "fwbuilder/ObjectGroup.hh"
#include "fwbuilder/ServiceGroup.hh"

#include "GroupDialog.hh"

#include "FindDialog.hh"

#include "fwbuilder/Host.hh"
#include "fwbuilder/Network.hh"
#include "fwbuilder/Firewall.hh"

#include "fwbuilder/FWOptions.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/NAT.hh"

#include "fwbuilder/IPService.hh"
#include "fwbuilder/ICMPService.hh"
#include "fwbuilder/TCPService.hh"
#include "fwbuilder/UDPService.hh"

#include "FWObjectBook.hh"
#include "fwbuilder/Tools.hh"

#include "main_window.hh"
#include "gen_popup_menu.hh"
#include "helpers.hh"

#include <gdk/gdk.h>


#include <list>
#include <vector>
#include <map>

#include <stdlib.h>
#include <stdio.h>
#include <algorithm>
#include <functional>

using namespace libfwbuilder;


static char drag_obj_id[64];


class cmp_ctree_row_id {
    string row_id;
    public: 
    cmp_ctree_row_id(const string &_id) : row_id(_id) {}
    bool operator()(Gtk::CTree_Helpers::Row r) {
	char *c=(char*)(r.get_data());
 	return ( c && row_id==c );
    }
};

class cmp_clist_row_id {
    string row_id;
    public: 
    cmp_clist_row_id(const string &_id) : row_id(_id) {}
    bool operator()(Gtk::CList_Helpers::Row r) {
	char *c=(char*)(r.get_data());
 	return ( c && row_id==c );
    }
};

const char* titles1[2]={ "Name" , NULL };
const char* titles2[3]={ "Name" , "Properties" , NULL };

static void free_row_data(gpointer data)
{
    delete ((char*)data);
}



ObjectTree::ObjectTree(bool show_p) : 
    CTree( show_p ? titles2 : titles1 ) 
{
    lib="All";
    show_properties=show_p;
    constructor();
}

ObjectTree::ObjectTree(const string &_l,bool show_p) : 
    CTree( show_p ? titles2 : titles1 ) 
{
    lib=_l;
    show_properties=show_p;
    constructor();
}

void ObjectTree::constructor()
{
    show_all=false;
    rebuild_scheduled=false;


    set_selection_mode( GTK_SELECTION_SINGLE );

    selected_object="";
    last_clicked_object="";
    drag_object="";
    show_object="";

    set_name("ObjectTree");
    glademm_set_Widget("ObjectTree", this);

    //  set_expander_style( GTK_CTREE_EXPANDER_SQUARE );

    set_expander_style( GTK_CTREE_EXPANDER_CIRCULAR );
    set_line_style( GTK_CTREE_LINES_SOLID  );

    column(0).set_auto_resize(true);
    if (show_properties) column(1).set_auto_resize(true);

    button_press_event.connect(
	slot(this, &ObjectTree::on_button_press_event));
    
    button_release_event.connect_after(
	slot(this, &ObjectTree::on_button_release_event));


    /*****************************************************************
     *  Initialize Drag&DRop mechanism 
     */
    drag_source_set ( 
	static_cast<GdkModifierType>(GDK_BUTTON1_MASK) ,
	target_table, n_targets, 
	static_cast<GdkDragAction>(GDK_ACTION_COPY) );

    drag_begin.connect(
	slot(this,&ObjectTree::source_drag_begin));
    drag_data_get.connect(
	slot(this,&ObjectTree::source_drag_data_get));
    drag_data_delete.connect(
	slot(this,&ObjectTree::source_drag_data_delete));

    tree_select_row.connect( slot(this,&ObjectTree::selection_made));

    key_press_event.connect(SigC::slot((ObjectTree*)this, 
				       &ObjectTree::on_key_press_event));

}

gint ObjectTree::on_key_press_event(GdkEventKey *ev)
{
    gchar  str[32];
    FWObject  *o=NULL;
    guint      n;
    gchar      c;
    string     id;

    o=FWObjectDatabaseGUI::ScratchPad->getById( selected_object, true ) ;
    assert(o!=NULL);

/*
    cerr << "--------------------\n";
    cerr << "send_event=" << ev->send_event << endl;
    cerr << "time=" << ev->time << endl;
    cerr << "state=" << ev->state << endl;
    cerr << "keyval=" << ev->keyval << endl;
    cerr << "length=" << ev->length << endl;
    cerr << "string=" << ev->string << endl;
*/

    if ( ev->keyval == GDK_F1 ) {   
// F1, just print objects address,ID, name etc.
	o->dump(true,true);
	return(0);
    }

    if ( ev->keyval == GDK_F6 ) {   
// F6, toggle show_all flag and refresh the tree
	show_all= ! show_all;
	scheduleRebuild();
	return(0);
    }

    if ( ev->keyval == GDK_F7 ) {   
// F7, dump selected object
	o->dump(true,false);
	return(0);
    }

    if ( ev->keyval == GDK_F8 ) {   
// F8, dump scratch pad
	std::ofstream out("fwbuilder_scratchpad.dump");
	FWObjectDatabaseGUI::ScratchPad->dump(out,true,false);
	return(0);
    }

    if ( ! main_w->safe_to_close_right_pane() ) {
	/*
	 *  All key presses change selection in the tree. If it is not safe
	 *  to close right pane and show another object there, then igore it.
	 */
	gtk_signal_emit_stop_by_name( GTK_OBJECT( gtkobj() ),
				      "key_press_event" );
 
	return(1);
    }


    Gtk::CTree_Helpers::RowList          rl=rows();
    if (rl.empty()) return(0);
  

    if ( ev->keyval == GDK_Return ) {   // ENTER

	RowList::iterator k;
	for (n=0,k=rl.begin(); k!=rl.end(); ++k,++n) {
	    id=(char*)(k->get_data());
	    if (id==selected_object) {

		//    n = current_selected_row;
		// subtree ?
		Gtk::CTree_Helpers::Row  row=rows()[n];
		Gtk::CTree_Helpers::RowList srl = row.subtree();
		if (! srl.empty()) {
		    row.expand();
		    return(0);
		}
//		if (main_w->isDialogDocked()) {
		    main_w->setCurrentLibrary(lib);
		    main_w->schedule_open_object(id);
//		}
		return(0);
	    }
	}
	return(0);
    }

    if (ev->string!=NULL && ev->length!=0) {
	strncpy(str,ev->string,ev->length);
	c=str[0];
  
	RowList::iterator k;
	for (n=0,k=rl.begin(); k!=rl.end(); ++k,++n) {
	    id=(char*)(k->get_data());
	    if (id==selected_object) break;
	}
	++k; ++n;
	for (; k!=rl.end(); ++k,++n) {
	    id=(char*)(k->get_data());
	    o=FWObjectDatabaseGUI::ScratchPad->getById( id , true );
	    assert(o!=NULL);
	    if (tolower( o->getName()[0] )==c ) {
		selectRow(n);
		selected_object=id;

		gtk_signal_emit_stop_by_name( GTK_OBJECT( gtkobj() ),
					      "key_press_event" );

		return(0);
	    }
	}
    }
    return 0;
}

void ObjectTree::showObject(const string &id)
{
    if (id=="")	clearSelection();
    else {
	if (selected_object!=id) {
	    expand_to_row(id);
	    selectObject(id);
	}
    }
}

void ObjectTree::showObject(FWObject *o)
{
    showObject(o->getId());
}

void ObjectTree::selectObject(const string &id)
{
/*
 *  CTree does not have method to focus row, so we have to work with
 *  CList
 */
    Gtk::CList::RowList::iterator m=find_if( Gtk::CList::rows().begin(),
					     Gtk::CList::rows().end(),
					     cmp_clist_row_id(id) );
    if (m!=Gtk::CList::rows().end()) {
	selected_object=id;
	m->select();
	m->focus();
    } else {
	clearSelection();
    }
}

void ObjectTree::clearSelection()
{
    selected_object="";

    Gtk::CTree_Helpers::RowList          rl=rows();
    if (rl.empty()) return;
  
    RowList::iterator k;
    for (k=rl.begin(); k!=rl.end(); ++k) {
	(*k).unselect();
    }
}

void ObjectTree::selection_made(Gtk::CTree::Row r,gint col)
{
    char  *c=(char*)r.get_data();
    assert (c!=NULL);

    string id(c);

    if (id!=selected_object) {
//	if ( main_w->isDialogDocked() ) {
	    if ( main_w->safe_to_close_right_pane() ){
		main_w->setCurrentLibrary(lib);
		main_w->schedule_open_object(id);
		FWObjectBook *w=(FWObjectBook*)glademm_get_Widget("FWObjectBook");
		if (w!=NULL)  w->clearSelectionOnAllPages(this);
	    } else {
		selectObject(selected_object);
	    }
//	}
    }
}


void ObjectTree::selectRow(gint row_n)
{
    /*
     * calling "select()" for CTree_Helpers::Row
     * does not move focus (but moves selection). There is no "focus()"
     * call for CTree_Helpers::Row, so we have to dig out its 
     * equivalentfrom CList_Helpers and call focus() for it
     *
     * It does not really matter which one we use here to call select()
     */
    Gtk::CList_Helpers::Row r = Gtk::CList::row(row_n);
    r->focus();
    r->select();
}

void ObjectTree::rebuildTreeAndShow(const string &id)
{
    rebuildTree();
    showObject(id);
//    if (main_w->isDialogDocked()) {
	main_w->setCurrentLibrary(lib);
	main_w->schedule_open_object(id);
//    }
}

void ObjectTree::scheduleRebuild()
{
    if ( !rebuild_scheduled ) {
	Gtk::Main::idle.connect(slot(this,&ObjectTree::rebuild_when_idle));
	rebuild_scheduled=true;
    }
}

gint ObjectTree::rebuild_when_idle()
{
    rebuildTree();
    return false;
}

void ObjectTree::scheduleRebuildAndShow(const string &id)
{
    if (id=="") return;
    show_object=id;
    Gtk::Main::idle.connect(slot(this,&ObjectTree::rebuild_and_show_when_idle));
}

gint ObjectTree::rebuild_and_show_when_idle()
{
    if (show_object=="") return false;
    rebuildTreeAndShow(show_object);
    show_object="";
    return false;
}

string ObjectTree::get_properties(libfwbuilder::FWObject *o)
{
    if (Host::cast(o)!=NULL) return string("  ")+o->getStr("address");

    if (Network::cast(o)!=NULL) {
	Network *n=Network::cast(o);
	return string("  ")+n->getAddress().toString()+string("/")+n->getNetmask().toString();
    }

    if (TCPService::cast(o)!=NULL || UDPService::cast(o)!=NULL) {
	string dps,dpe;

	dps=o->getStr("dst_range_start");
	dpe=o->getStr("dst_range_end");

	if (dps==dpe)	return string("  ")+dps;
	else	        return string("  ")+dps+string("-")+dpe;
    }

    if (ICMPService::cast(o)!=NULL) 
	return string("  ")+o->getStr("code")+string("/")+o->getStr("type");

    if (IPService::cast(o)!=NULL) 
	return string("  ")+o->getStr("protocol_num");

    return string();
}


void ObjectTree::insertSubtree(  Gtk::CTree_Helpers::RowList::iterator &i , 
				  FWObject *obj )
{
    RowList::iterator         j;
    vector<FWObject*>::iterator m;
    string                    name;
    string                    properties;
    FWObject                 *o;

    if (obj==NULL) return;

    if (FWObjectReference::isA(obj) && ! show_all) return;

    if (Resources::global_res->getObjResourceBool(obj,"hidden") &&
	! show_all ) return;

    if (lib=="All" || lib==obj->getLibrary() || Resources::isSystem(obj) ) {


	name=obj->getName();
	if (name=="") {  // no name, use type description string instead
	    name=Resources::global_res->getObjResourceStr(obj,"description");
	}

//	column(0).set_auto_resize(true);
	properties="";
	if (show_properties) 
	    properties= get_properties(obj);

	j=_insert_row( i->subtree(),  i->subtree().end(), 
		       name, obj->getId(), properties );

	for (m=obj->begin(); m!=obj->end(); ++m) {
	    o=(*m);
	    if (o==NULL) continue;
	    insertSubtree( j , o );
	}
    }

}

void ObjectTree::rebuildTree()
{
    string cur=selected_object;

    using namespace Gtk::CTree_Helpers;

    clear();

    vector<FWObject*>::iterator m;
    RowList::iterator j;

    j=--rows().end();

    for (m=FWObjectDatabaseGUI::ScratchPad->begin(); 
	 m!=FWObjectDatabaseGUI::ScratchPad->end(); 
	 ++m) {
	insertSubtree( j , (*m) );
    }

    rebuild_scheduled=false;
/*
    main_w->setCurrentLibrary(lib);
    if (cur!="") 
	main_w->schedule_open_object(cur);
    else
	main_w->schedule_open_object(FWObjectDatabaseGUI::ScratchPad->getId());
*/
}

class traverse_tree_and_find_id {
    string row_id;
    Gtk::CTree_Helpers::Row res;
    public:
    traverse_tree_and_find_id(const string &_id):row_id(_id),res() {}
    operator Gtk::CTree_Helpers::Row() const { return res; }
    const traverse_tree_and_find_id& operator()(Gtk::CTree_Helpers::Row r) {
	if (res==NULL) {
	    char *c=(char*)(r.get_data());
	    if ( c && row_id==c ) {
		res=r; r.expand(); return *this; 
	    } else {
		res=std::for_each(r.subtree().begin(),r.subtree().end(),
				  traverse_tree_and_find_id(row_id));
		if (res!=NULL) r.expand();
	    }
	} 

	return *this;
    }
};

/*
 *  returns Row. Since Gtk::CTree_Helpers::Row has special
 *  operator==(gpointer) we can check whether search was sucessful by
 *  comparing returned Row with NULL
 */
Gtk::CTree_Helpers::Row ObjectTree::expand_to_row(const string &id)
{
    return std::for_each(rows().begin(), rows().end(), 
			 traverse_tree_and_find_id(id) );
}


Gtk::CTree_Helpers::RowList::iterator ObjectTree::_insert_row(
    Gtk::CTree_Helpers::RowList           rl,
    Gtk::CTree_Helpers::RowList::iterator i,  
    const string &lbl, const string &id, const string& properties)
{
    vector<const gchar*>      item;
    item.push_back( lbl.c_str() );
    if ( ! properties.empty() ) item.push_back( properties.c_str() );

    RowList::iterator j=rl.insert(i, Element(item));

    char *id_storage=cxx_strdup( id.c_str() );
    j->set_data( id_storage , &free_row_data );
    return j;
}


void ObjectTree::insertObject( FWObject *obj )
{
    FWObject *par=obj->getParent();

    expand_to_row( par->getId() );
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
					      cmp_ctree_row_id(par->getId()) );
    insertSubtree(oi, obj);

    main_w->setCurrentLibrary(lib);
    main_w->schedule_open_object(obj->getId());
    selectObject(obj->getId());
}


void ObjectTree::insertObject( const string &id )
{
    FWObject *obj=FWObjectDatabaseGUI::ScratchPad->getById(id,true);
    assert (obj!=NULL);

    insertObject(obj);
}

void   ObjectTree::removeObject( const string &id )
{
    FWObject *obj=FWObjectDatabaseGUI::ScratchPad->getById(id,true);
    assert (obj!=NULL);

    removeObject(obj);
}

void   ObjectTree::removeObject( libfwbuilder::FWObject *o )
{
    expand_to_row(o->getId());
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
					      cmp_ctree_row_id(o->getId()) );
    Gtk::CTree::RowList::iterator pi=find_if( rows().begin(),
					      rows().end(),
					      cmp_ctree_row_id(o->getParent()->getId()) );
    pi->subtree().erase(oi);
}

class  cmp_rows {
    FWObject *o;
    public:
    cmp_rows(FWObject *p) { o=p; }
    int operator()(const Gtk::CTree_Helpers::Row &r1,
		   const Gtk::CTree_Helpers::Row &r2) {

	return r1[0].get_text() < r2[0].get_text();
    }
};

void   ObjectTree::sortSubtree(const string &id)
{
    FWObject *o=FWObjectDatabaseGUI::ScratchPad->getById(id,true);
    assert (o!=NULL);

    sortSubtree(o);
}

void   ObjectTree::sortSubtree(FWObject *o)
{
    Gtk::CTree_Helpers::Row r=expand_to_row(o->getId());

//    std::sort(rl.begin(),rl.end(),cmp_rows(o) );

    r.sort_recursive();
}

class find_and_replace_label {

    string id;
    string new_lbl;
    string properties;

    public:
    find_and_replace_label(const string &_id,
			   const string _new_lbl,
			   const string _i )
    { id=_id; new_lbl=_new_lbl; properties=_i; }

    void operator()(Gtk::CTree_Helpers::Row r) {
	if ( id==(char*)(r.get_data()) ) {
	    r[0].set_text(new_lbl);
	    if (!properties.empty()) 
		r[1].set_text(properties);
	} else {
	    Gtk::CTree_Helpers::RowList srl = r.subtree();
	    std::for_each( srl.begin() , srl.end() , 
			   find_and_replace_label(id,new_lbl,properties) ); 
	}
    }
};

void ObjectTree::changeTreeLabel(const string &id)
{
    FWObject *obj=FWObjectDatabaseGUI::ScratchPad->getById(id,true);
    assert (obj!=NULL);

    string new_lbl   = obj->getName();
    if (new_lbl=="") new_lbl=obj->getTypeName();

    string properties = (show_properties)?get_properties(obj):"";

    std::for_each( rows().begin() , rows().end() , 
		   find_and_replace_label(id,new_lbl,properties) ); 
}


void ObjectTree::source_drag_begin(GdkDragContext     *context)
{
    drag_object=last_clicked_object;

    Gdk_Colormap cmap ( get_colormap () );
    string    icn=Resources::global_res->getIconPath("Drag");

    Gtk::Pixmap pix(icn);
    Gdk_Pixmap drag_icon;
    Gdk_Bitmap drag_mask;
    pix.get(drag_icon,drag_mask);
    drag_source_set_icon(cmap, drag_icon, drag_mask);
}


void ObjectTree::source_drag_data_get  ( GdkDragContext     *context,
					  GtkSelectionData   *selection_data,
					  guint               info,
					  guint32             time )
{
  const guchar    *idptr;
  
  strcpy(drag_obj_id , drag_object.c_str() );
  idptr= (const guchar*)drag_obj_id;

  gtk_selection_data_set (selection_data,
			  gdk_atom_intern("ObjectTree",TRUE),
			  8,
			  (const guchar*)drag_obj_id,
			  strlen(drag_obj_id)+1 );
}

void ObjectTree::source_drag_data_delete ( GdkDragContext     *context )
{
}



gint ObjectTree::on_button_press_event(GdkEventButton *ev)
{
    gint              r,c;
    char             *oid;

    get_selection_info( int(ev->x) , int(ev->y) , &r , &c );

    if (r<0 || r>get_rows()) return(0);

    oid=(char*)(row(r).get_data());
    if (oid==NULL) return(0);

    if (ev->button != 1) selectRow(r);

//    if (ev->type==GDK_2BUTTON_PRESS && ! main_w->isDialogDocked() &&
//	main_w->safe_to_close_right_pane() ) {
//	main_w->setCurrentLibrary(lib);
//	main_w->schedule_open_object(oid);
//    }

    last_clicked_object=oid;

    return(0);
}

gint ObjectTree::on_button_release_event(GdkEventButton *ev)
{
    gint              r,c;
    char             *oid;

    get_selection_info( int(ev->x) , int(ev->y) , &r , &c );

    if (r<0 || r>get_rows()) return(0);

    oid=(char*)(row(r).get_data());
    if (oid==NULL) return(0);

    if (ev->type == GDK_BUTTON_RELEASE && ev->button==3  ) {
	TreePopupMenu();
    }

    return(0);
}


string ObjectTree::getCurrentSelection()
{
    return( selected_object );
}


void ObjectTree::TreePopupMenu()
{
    FWObject *obj=
	FWObjectDatabaseGUI::ScratchPad->getById( selected_object , true);
    assert(obj!=NULL);

    if (obj->getTypeName()==Policy::TYPENAME ||
	obj->getTypeName()==NAT::TYPENAME ) return;

    if ( Resources::isSystem(obj) ) {

	const char  *menu_items[] ={ "Sort subtree", "Paste", NULL};
	int          menu_choice;
	gen_popup_menu *gpm;

	gpm=new gen_popup_menu( menu_items );

	Gtk::Widget *itm;
	itm=find_widget("Paste",gpm);   
	if (itm) itm->set_sensitive(
	    (obj->getTypeName()==ObjectGroup::TYPENAME ||
	     obj->getTypeName()==ServiceGroup::TYPENAME )  );

	gpm->popup(0,0);
	menu_choice=gpm->run();
	delete gpm;

	switch (menu_choice) {
	case 0: main_w->on_sort();	  break;
	case 1:	main_w->on_pasteobj();	  break;
	}

    } else {

	const char  *menu_items[] ={ "Duplicate", "", "Copy","Cut","Paste","","Delete","","Where used",NULL};
	int          menu_choice;
	gen_popup_menu *gpm;

	gpm=new gen_popup_menu( menu_items );

	Gtk::Widget *itm;
	itm=find_widget("Paste",gpm);   
	if (itm) itm->set_sensitive(
	    (obj->getTypeName()==ObjectGroup::TYPENAME ||
	     obj->getTypeName()==ServiceGroup::TYPENAME )  );

	gpm->popup(0,0);
	menu_choice=gpm->run();
	delete gpm;

	switch (menu_choice) {
	case 0:  main_w->on_duplicate();  break;  // Duplicate
	case 2:  main_w->on_copyobj();    break;  // Copy
	case 3:  main_w->on_cutobj();	  break;  // Cut
	case 4:  main_w->on_pasteobj();	  break;  // Paste
	case 6:  main_w->on_delobj();     break;  // Delete
	case 8:  main_w->on_where_used(); break;  // Where Used
	}
    }
}

string   ObjectTree::getNextId()
{
    expand_to_row( selected_object );
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
				      cmp_ctree_row_id(selected_object) );
    ++oi;
    if (oi!=rows().end()) {
	char *c=(char*)(oi->get_data());
	if (c)
	    return string(c);
    }
    return "";
}

string   ObjectTree::getPrevId()
{
    expand_to_row( selected_object );
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
				      cmp_ctree_row_id(selected_object) );
    --oi;
    if (oi!=rows().end()) {
	char *c=(char*)(oi->get_data());
	if (c)
	    return string(c);
    }
    return "";
}

string   ObjectTree::getParentId()
{
    expand_to_row( selected_object );
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
				      cmp_ctree_row_id(selected_object) );

    if (oi!=rows().end()) {
	Gtk::CTree::Row r=oi->get_parent();
	char *c=(char*)(r.get_data());
	if (c)
	    return string(c);
    }
    return "";
}






