/* 

                          Firewall Builder

                 Copyright (C) 2001 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: IconList.cc,v 1.9 2001/12/29 10:06:35 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/




#include "IconList.hh"
#include "ObjectQuickView.hh"

#include <algorithm>
#include <stdexcept>
#include <iostream.h>

Icon::Icon(const string& icon_filename,const string& _t) : Gtk::EventBox()
{
    text=_t;

    x=0;
    y=0;
    width=0;
    height=0;

    set_name("IconList_Icon");

    vbox=manage(new Gtk::VBox());
    add(*vbox);

    pixmap= manage(new Gtk::Pixmap( icon_filename ));
    pixmap->set_name("IconList_Pixmap");

    bgr   =manage(new Gtk::EventBox());
    bgr->set_name("IconList_LabelBgr");

    label =manage(new Gtk::Label(text));
    label->set_name("IconList_Label");
    label->set_line_wrap(true);
    label->set_alignment(0.5,0.5);

    bgr->add(*label);

    vbox->pack_start(*pixmap,false,false,false);
    vbox->pack_start(*bgr   ,false,false,false);

    vbox->show();
    pixmap->show();
    bgr->show();
    label->show();

    add_events(  GDK_ENTER_NOTIFY_MASK |
		 GDK_LEAVE_NOTIFY_MASK );
}

Icon::~Icon()
{
    deactivateObjectQuickView();
}

gint Icon::button_press_event_impl(GdkEventButton* ev)
{
    Gtk::Widget *w=get_parent();
    IconList *il=dynamic_cast<IconList*>(w);
    if (il) il->icon_clicked(this);
    return false;   // return false so parent widget will get this event, too
}

/**
 *  this method claculates real width of the label, taking into account
 *  that label may contain multiple lines of text separates with '\n'
 */
unsigned Icon::get_label_width()
{
    Gtk::Style *st=get_style();
    Gdk_Font    fn=st->get_font();

    unsigned i,j,n;
    unsigned str_w;

    i=0;
    str_w=0;
    while ( (j=text.find("\n",i))!=string::npos ) {
	n=fn.string_measure(text.substr(i,j-i)); 
	if (str_w<n) str_w=n;
	i=j+1;
    }
    n=fn.string_measure(text.substr(i)); 
    if (str_w<n) str_w=n;
    return str_w;
}

void Icon::size_allocate_impl(GtkAllocation *all)
{
    x=all->x;
    y=all->y;
    width=all->width;
    height=all->height;

    Gtk::EventBox::size_allocate_impl(all);

    Gtk::Style *st=get_style();
    Gdk_Font    fn=st->get_font();

    unsigned i,j;
    unsigned lbl_w=get_label_width();

    if (lbl_w>width) {

	while ( (j=text.find("\n"))!=string::npos ) {
	    text.erase(j,1);
	}

	i=0;
	while (i<text.size() && get_label_width()>width) {
	    j=1;
	    while (i+j<text.size() ) {
		if ( unsigned(fn.string_measure( text.substr(i,j) ))>width ) {
		    j-=2;
		    text.insert(i+j,"\n");
		    break;
		}
		++j;
	    }
	    i=i+j+1;
	} 
	label->set(text);
    }
}

void Icon::state_changed_impl(GtkStateType p0)
{
    deactivateObjectQuickView();
    Gtk::EventBox::state_changed_impl(p0);
}


void Icon::activateObjectQuickView()
{
    string   *id=(string*)(get_user_data());

    if ( id!=NULL ) {

	ObjectQuickView* ov=ObjectQuickView::getInstance();
	ov->setObjectById( *id );
	ov->attachTo(this);
	ov->activate();
    }
}

void Icon::deactivateObjectQuickView()
{
    ObjectQuickView* ov=ObjectQuickView::getInstance();
    ov->deactivate();
}

gint Icon::enter_notify_event_impl(GdkEventCrossing* ev)
{
    activateObjectQuickView();
    return( Gtk::EventBox::enter_notify_event_impl(ev) );
}

gint Icon::leave_notify_event_impl(GdkEventCrossing* ev)
{
    deactivateObjectQuickView();
    return( Gtk::EventBox::enter_notify_event_impl(ev) );
}

/********************************************************************/
IconList::IconList()
{
    x=y=width=height=0;
    row_spacing=0;
    col_spacing=0;
    text_spacing=0;
    border=0;

    icon_width=0;
    icon_height=0;

    current_selected=-1;

    fisrt_expose_event=true;

    set_flags( GTK_CAN_FOCUS );

    add_events(  GDK_EXPOSURE_MASK |
                 GDK_BUTTON_PRESS_MASK |
                 GDK_BUTTON_RELEASE_MASK |
                 GDK_KEY_PRESS_MASK |
                 GDK_KEY_RELEASE_MASK);

    set_name("IconList");
}

IconList::~IconList() 
{
}

void IconList::size_request_impl(GtkRequisition* req)
{
    Gtk::Layout::size_request_impl(req);

    int resw=0;
    int resh=0;

    for (unsigned i=0; i<icons.size(); ++i) {
	Icon *icn=icons[i];
	if (icn==NULL) break;
	if (resw<icn->get_x()+icn->get_width()) 
	    resw=icn->get_x()+icn->get_width();
	if (resh<icn->get_y()+icn->get_height()) 
	    resh=icn->get_y()+icn->get_height();
    }
    req->width   = resw;
    req->height  = resh;
}

void IconList::size_allocate_impl(GtkAllocation* all)
{
    x=all->x;
    y=all->y;

    Gtk::Layout::size_allocate_impl(all);

    if (width!=all->width || height!=all->height) {
	width=all->width;
	height=all->height;

	if (icons.size()!=0) {
	    if (icon_width==0)	 icon_width =icons[0]->get_width();
	    if (icon_height==0)	 icon_height=icons[0]->get_height();
	}

	arrangeIcons();
    }
}

gint IconList::expose_event_impl(GdkEventExpose* ex)
{
    int res;

    res= Layout::expose_event_impl(ex);

    if (fisrt_expose_event && ex->count==0) {
	arrangeIcons(0);
    }
    fisrt_expose_event=false;

    return res;
}

void IconList::icon_clicked(Icon *icn)
{
    for (unsigned i=0; i<icons.size(); i++) {
	if (icn==icons[i]) {

	    grab_focus();

	    unselect_icon(current_selected);

	    icn->set_state(GTK_STATE_SELECTED);
	    current_selected=i;

	    return;
	}
    }
}

/**
 *  IconList does not have to do anything with mouse button event, but
 *  widget derived from it might (See ListOfIcons, where it uses this
 *  callback method for double click event).
 */
gint IconList::button_press_event_impl(GdkEventButton* ev)
{
    return true;
}


gint IconList::key_press_event_impl(GdkEventKey* ev)
{
    /*
     * Keys we are looking for:
     * GDK_Tab
     * GDK_Up
     * GDK_Down
     *   etc.
     */

    if (current_selected==-1) { select_icon(0); }

    switch (ev->keyval ) {
        //  case GDK_Tab:
        //    cerr << "               Tab\n";
        //    break;
    case GDK_Up:
        move_focus(GTK_DIR_UP);
        break;
    case GDK_Down:
        move_focus(GTK_DIR_DOWN);
        break;
    case GDK_Left:
        move_focus(GTK_DIR_LEFT);
        break;
    case GDK_Right:
        move_focus(GTK_DIR_RIGHT);
        break;

    default:
        return(false);
    }
    return (false);
}

void IconList::move_focus(GtkDirectionType dir)
{
    int  n=get_items_per_row();
    unsigned  icn;

    if (current_selected==-1) 
        select_icon(0);

    icn=current_selected;

    unselect_icon(current_selected);

    switch (dir) {
    case GTK_DIR_UP:     icn-=n;        break;
    case GTK_DIR_DOWN:   icn+=n;        break;
    case GTK_DIR_LEFT:    --icn;        break;
    case GTK_DIR_RIGHT:   ++icn;        break;
    default:             return;
    }
    if (icn<0)             icn=0;
    if (icn>=icons.size()) icn=icons.size()-1;

    select_icon(icn);
}

void IconList::getNextPosition(int prev_idx, Icon *new_icn, int &x,int &y)
{
    if (prev_idx<0) { x=border; y=border; return; }

    int cx,cy;

    int prev_x=icons_x[prev_idx];
    int prev_y=icons_y[prev_idx];

    if (prev_x==0) prev_x=border;
    if (prev_y==0) prev_y=border;

    cx=prev_x+icon_width+col_spacing;
    cy=prev_y;

    if (cx+new_icn->get_width()+border>int(width) ) {
	cx=border;
	cy=prev_y+icon_height+row_spacing;
    }
    x=cx; y=cy;

}


void IconList::arrangeIcons(unsigned start_with)
{
    int cx,cy;


    if (width==0 || height==0) return;
    if (start_with<0 || start_with>=icons.size()) return;

    unsigned prev,cur;

    prev=start_with;

    freeze();
    Gtk::Main::grab_add(*this);

    for ( cur=prev+1; cur<icons.size(); ++cur ) {

	Icon *icn=icons[cur];
	if (icn!=NULL) {
	    getNextPosition(prev,icn,cx,cy);
	    icons_x[cur]=cx;
	    icons_y[cur]=cy;
	    move(*icn,cx,cy);
	}
	prev=cur;
    }

    Gtk::Main::grab_remove(*this);
    thaw();
}



int IconList::append(const string& icon_filename,const string& text)
{
    int cx,cy;
    Icon *icn=manage(new Icon(icon_filename,text));

    icn->set_usize( icon_width , icon_height );

    int n= icons.size();

    icons.push_back(icn);

    getNextPosition(n-1,icn,cx,cy);
    icons_x.push_back(cx);
    icons_y.push_back(cy);

    put(*icn,cx,cy);
    icn->show();

    return n;
}



void IconList::remove(int pos)
{
    Icon *icn=icons[pos];
    Gtk::Layout::remove(*icn);
    int n;
    vector<Icon*>::iterator i;
    for (n=0,i=icons.begin(); i!=icons.end(); ++n,++i)
	if (n==pos) {
	    icons.erase(i);
	    break;
	}
    arrangeIcons(pos-1);
}

void IconList::set_border(int bdr)
{
    border=bdr;
}

void IconList::set_row_spacing(int spacing)
{
    row_spacing=spacing;
}

void IconList::set_col_spacing(int spacing)
{
    col_spacing=spacing;
}

void IconList::set_text_spacing(int spacing)
{
    text_spacing=spacing;
}

void IconList::set_icon_width(int w)
{
    icon_width=w;
}

void IconList::set_icon_height(int h)
{
    icon_height=h;
}


void IconList::clear()
{
    vector<Icon*>::iterator r;
    for (r=icons.begin(); r!=icons.end(); ++r)
	Gtk::Layout::remove(**r);

    icons.clear();
}

void IconList::set_icon_data(int pos,gpointer data)
{
    assert (pos>=0 && pos<int(icons.size()));
    Icon *icn=icons[pos];
    if (icn!=NULL)
	icn->set_user_data(data);
}

gpointer IconList::get_icon_data(int pos)
{
    assert (pos>=0 && pos<int(icons.size()));
    return     icons[pos]->get_user_data();
}

void IconList::freeze()
{
    Gtk::Layout::freeze();
}

void IconList::thaw()
{
    Gtk::Layout::thaw();
}


void IconList::moveto(int pos,double yalign)
{
    cerr << "IconList: method 'moveto' not implemented\n";
}


int  IconList::get_icon_at(int x,int y)
{
    vector<Icon*>::const_iterator r;
    int i=0;
    for (r=icons.begin(); r!=icons.end(); ++r,++i) {
	Icon *icn = *r;
	if ( x>=icn->get_x() && x<=icn->get_x()+icn->get_width() &&
	     y>=icn->get_y() && y<=icn->get_y()+icn->get_height() ) {
	    return i;
	}
    }
    return -1;
}

/**
 * this algorithm would work even if this widget supported icons of
 * different size and different number of icons in different rows
 */

int  IconList::get_items_per_row(unsigned idx)
{
    if (idx<0 || idx>=icons.size()) return 0;

    int  i,n=0;
    int  cy=icons[idx]->get_y();
    for (i=idx; i<int(icons.size()) && icons[i]->get_y()==cy; ++i) ++n;
    for (i=idx; i>=0                && icons[i]->get_y()==cy; --i) ++n;

    --n;
    return n;
}


void IconList::select_icon(unsigned idx)
{
    if (idx<0 || idx>=icons.size()) return;
    Icon *icn=icons[idx];
    if (icn!=NULL) icn->set_state(GTK_STATE_SELECTED);
    current_selected=idx;
}


void IconList::unselect_icon(unsigned idx)
{
    if (idx<0 || idx>=icons.size()) return;
    Icon *icn=icons[idx];
    if (icn!=NULL) icn->set_state( get_state() );
    current_selected=-1;
}


int  IconList::unselect_all(GdkEvent* event,gpointer keep)
{
    cerr << "IconList: method 'unselect_all' not implemented\n";
    return 0;
}

