package freenet.support.io;
import freenet.support.Logger;
import freenet.Core;
import java.io.*;

/**
 * An InputStream which limits (or "throttles") the bandwidth of bytes written to it.
 */

public class ThrottledInputStream extends FilterInputStream {
    
     private static Bandwidth bandwidth;

    /**
     * Activate throttling on InputStreams.
     *
     * @param bandwidth the available bandwidth for InputStreams.
     */

    public static void setThrottle(Bandwidth bandwidth) {
        ThrottledInputStream.bandwidth = bandwidth;
    }

    /**
     * Gets a ThrottledInputStream based on the given InputStream.
     * If throttling is turned off (by calling {@ref #setThrottle setThrottle}
     * with a zero or negative argument) then the given InputStream is
     * simply returned.
     *
     * @param in the InputStream to throttle.
     * @returns an InputStream which is either the original InputStream if
     * throttling is turned off, or a new ThrottledInputStream if not. 
     */
        
    public static InputStream throttledStream(InputStream in) {
        if (bandwidth == null || bandwidth.bandwidthPerTick <= 0)
            return in;
        Core.logger.log(ThrottledInputStream.class, 
                        "ThrottledInput, creating new stream, bpt = " +
                        bandwidth.bandwidthPerTick, Logger.DEBUG);
        return new ThrottledInputStream(in);
    }

    // Force the use of a static method that checks if throttling
    // is on before creating a throttled stream.
    private ThrottledInputStream(InputStream in) {
        super(in);
    }

    /**
     * Write a single byte to this InputStream.
     * 
     * @param b the byte to read.
     * @throws IOException if an I/O error occurs on the InputStream.
     */
        
    public int read() throws IOException {
        int rv = super.read();
        if (rv != -1)
            bandwidth.chargeBandwidth(1);
        return rv;
    }
    
    /**
     * Write an array of bytes to this InputStream.
     *
     * @param data the bytes to read.
     * @param offset the index in the array to start at.
     * @param length the number of bytes to read.
     * @throws IOException if an I/O error occurs on the InputStream.
     */
        
    public int read(byte[] data, int offset, int length)
        throws IOException
    {
        int rv = in.read(data, offset, length);
        if (rv > 0) {
            bandwidth.chargeBandwidth(rv);
        }
        return rv;
    }
}





