package freenet.support;

import java.util.Vector;

/**
 * Does binary searches and insertions on a vector
 * that is assumed to be sorted.
 * @author tavin
 */
public class OrderedVector extends Vector {

    protected final Comparator c;

    public OrderedVector() {
        this(ObjectComparator.instance);
    }

    public OrderedVector(Comparator c) {
        super();
        this.c = c;
    }

    public OrderedVector(int initialCapacity) {
        this(initialCapacity, ObjectComparator.instance);
    }

    public OrderedVector(int initialCapacity, Comparator c) {
        super(initialCapacity);
        this.c = c;
    }

    public OrderedVector(int initialCapacity, int capacityIncrement) {
        this(initialCapacity, capacityIncrement, ObjectComparator.instance);
    }

    public OrderedVector(int initialCapacity, int capacityIncrement, Comparator c) {
        super(initialCapacity, capacityIncrement);
        this.c = c;
    }

    public synchronized final int search(Object o) {
        return search(this, o, c);
    }

    public static final int search(Vector v, Object o) {
        return search(v, o, ObjectComparator.instance);
    }

    public static final int search(Vector v, Object o, Comparator c) {
        synchronized (v) {
            int lo = 0, hi = v.size() - 1;
            while (lo <= hi) {
                int m = (lo + hi) >> 1;
                int cmp = c.compare(o, v.elementAt(m));
                if (cmp > 0)
                    lo = m+1;
                else if (cmp < 0)
                    hi = m-1;
                else
                    return m;
            }
            return -1;
        }
    }

    public synchronized final int insert(Object o) {
        return insert(this, o, c);
    }
    
    public static final int insert(Vector v, Object o) {
        return insert(v, o, ObjectComparator.instance);
    }

    public static final int insert(Vector v, Object o, Comparator c) {
        synchronized (v) {
            int m = -1, lo = 0, hi = v.size() - 1;
            int cmp = 0;
            while (lo <= hi) {
                m = (lo + hi) >> 1;
                cmp = c.compare(o, v.elementAt(m));
                if (cmp > 0)
                    lo = m+1;
                else if (cmp < 0)
                    hi = m-1;
                else
                    break;
            }
            if (cmp >= 0) ++m;
            v.insertElementAt(o, m);
            return m;
        }
    }

    public synchronized final int insert(Object o, boolean replace) {
        return insert(this, o, c, replace);
    }

    public static final int insert(Vector v, Object o, boolean replace) {
        return insert(v, o, ObjectComparator.instance, replace);
    }
    
    public static final int insert(Vector v, Object o, Comparator c, boolean replace) {
        synchronized (v) {
            int m = -1, lo = 0, hi = v.size() - 1;
            int cmp = 1;
            while (lo <= hi) {
                m = (lo + hi) >> 1;
                cmp = c.compare(o, v.elementAt(m));
                if (cmp > 0)
                    lo = m+1;
                else if (cmp < 0)
                    hi = m-1;
                else {
                    if (replace) v.setElementAt(o, m);
                    return m;
                }
            }
            if (cmp > 0) ++m;
            v.insertElementAt(o, m);
            return m;
        }
    }
}


