package freenet.support;

import freenet.Core;
import freenet.support.io.NullInputStream;
import java.io.*;

/**
 * A file Bucket is an implementation of Bucket that writes to a file.
 * 
 * @author oskar
 **/
public class FileBucket implements Bucket {

    protected File file;
    protected boolean restart = true;
    protected boolean newFile;	// hack to get around deletes

    protected static String tempDir = null;

    /**
     * Creates a new FileBucket.
     * @param file  The File to read and write to.
     **/
    public FileBucket(File file) {
		this.file = file;
		this.newFile = false;
    }

    /**
     * Creates a new FileBucket in a random temporary file in the temporary directory.
     */

    public FileBucket() {
	file = new File(tempDir, "t" + Fields.longToString(Math.abs(Core.randSource.nextInt())));
	newFile = true;
        OnExitCleanUp.deleteOnExit(file);
    }

    public OutputStream getOutputStream() throws IOException {
	boolean append = !restart;
	restart = false;
	return new FileOutputStream(file.getPath(),append);
    }

    public InputStream getInputStream() throws IOException {
        return file.exists() ? (InputStream) new FileInputStream(file)
                             : (InputStream) new NullInputStream();
    }

    /** @return  the name of the file.
      */
    public String getName() {
        return file.getName();
    }

    public void resetWrite() {
	restart = true;
    }

    public long size() {
	return file.length();
    }


    /**
     * Returns the file object this buckets data is kept in.
     **/
    public File getFile() {
	return file;
    }

    public void finalize() throws Throwable
    {
	if(newFile && file.exists()) {
	    Core.logger.log(this, "Deleting bucket "+file.getName(), Logger.DEBUGGING);
            file.delete();
            if (file.exists())
		Core.logger.log(this, "Delete failed on bucket "+file.getName(), Logger.NORMAL);
	}
    }

    /**
     * Return directory used for temp files.
     **/
    public final synchronized static String getTempDir() {
        return tempDir;
    }

    /**
     * Set temp file directory.
     * <p> 
     * The directory must exist.
     **/
    public final synchronized static void setTempDir(String dirName) {
        File dir = new File(dirName);
        if (!(dir.exists() && dir.isDirectory() && dir.canWrite())) {
            throw new IllegalArgumentException("Bad Temp Directory: " + dir.getAbsolutePath());
        }
        tempDir = dirName;
    }

    // determine the temp directory in one of several ways

    static {
	// Try the Java property (1.2 and above)
	tempDir = System.getProperty("java.io.tmpdir");

	// Deprecated calls removed.

	// Try TEMP and TMP
	//	if (tempDir == null) {
	//	    tempDir = System.getenv("TEMP");
	//	}

	//	if (tempDir == null) {
	//	    tempDir = System.getenv("TMP");
	//	}

	// make some semi-educated guesses based on OS.

	if (tempDir == null) {
	    String os = System.getProperty("os.name");
	    if (os != null) {

		String[] candidates = null; 

		// XXX: Add more possible OSes here.
		if (os.equalsIgnoreCase("Linux")) {
		    String[] linuxCandidates = { "/tmp", "/var/tmp" };
		    candidates = linuxCandidates;
		} else if (os.equalsIgnoreCase("Windows")) {
		    String[] windowsCandidates = { "C:\\TEMP", "C:\\WINDOWS\\TEMP" };
		    candidates = windowsCandidates;
		}

		if (candidates != null) {
		    for (int i = 0; i < candidates.length; i++) {
			File path = new File(candidates[i]);
			if (path.exists() &&
			    path.isDirectory() &&
			    path.canWrite())
			{
			    tempDir = candidates[i];
			    break;
			}
		    }
		}
	    }
	}

	// last resort -- use current working directory

	if (tempDir == null) {
	    // This can be null -- but that's OK, null => cwd for File constructor, anyways.
	    tempDir = System.getProperty("user.dir");
	}
    }
}
