package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.node.states.data.*;
import freenet.node.ds.KeyCollisionException;
import freenet.message.*;
import freenet.support.Logger;
import java.io.IOException;

/**
 * The states for inserts at the terminal node that are waiting for a 
 * DataInsert or NoInsert message.
 */
public class AwaitingInsert extends RequestState {

    /** the auxiliary chain that will read the data into the store */
    ReceiveData receivingData;

    /** the MO that times out the DataInsert we are waiting on */
    NoInsert ni;

    AwaitingInsert(InsertPending ancestor) {
        super(ancestor);
        ni = ancestor.ni;
    }

    public final String getName() {
        return "Awaiting Insert";
    }

    public final void lost(Node n) {
        Core.diagnostics.occurrenceCounting("lostRequestState", 1);
        ni.cancel();
        fail(n, "State lost while waiting for your DataInsert");
    }

    public State receivedMessage(Node n, DataInsert dim) throws StateException {
        if (!fromOrigPeer(dim)) {
            throw new BadStateException("DataInsert from the wrong peer!");
        }
        ni.cancel();
        try {
            receivingData = dim.cacheData(n, searchKey);
        }
        catch (KeyCollisionException e) {
            // we've already sent the Accepted so we should try..
            dim.eatData(n);
            
            scheduleRestart(n, 0);
            // this bloody well ought to work
            return new DataPending(this);
        }
        catch (IOException e) {
            fail(n, "I/O error receiving insert");
            n.logger.log(this, "Failed to cache insert on chain " 
                               + Long.toHexString(id), e, Logger.ERROR);
            return new RequestDone(this);
        }
        receivingData.schedule(n);
        return new ReceivingInsert(this);
    }

    public State receivedMessage(Node n, NoInsert ni) throws StateException {
        if (this.ni != ni) {
            throw new BadStateException("Not my NoInsert: "+ni);
        }
        n.logger.log(this, "Did not receive expected DataInsert on chain " 
                           + Long.toHexString(id), Logger.MINOR);
        fail(n, "DataInsert never received");
        return new RequestDone(this);
    } 
}


