package freenet.node.states.data;

import freenet.node.*;
import freenet.node.ds.*;
import freenet.*;
import freenet.support.io.*;
import freenet.support.Logger;
import freenet.support.Fields;
import java.io.*;

/**
 * Reads data from a stream and feeds it to a data object.
 * @author Oskar
 */
public class ReceiveData extends DataState {

    private final VerifyingInputStream data;
    private final KeyOutputStream out;
    
    private final long length;

    private volatile int result = -1;
    private boolean silent = false;


    /**
     * @param id      the id this data chain will run under
     * @param parent  the id of the chain the DataStateReply will go to
     * @param data    stream containing the key data
     * @param length  number of bytes to receive
     * @param out     stream to cache the data in the store
     */ 
    public ReceiveData(long id, long parent, VerifyingInputStream data,
                       KeyOutputStream out, long length) {
        super(id, parent);
        this.length = length;
        this.out    = out;
        this.data   = data;
    }

    public final String getName() {
        return "Receiving Data";
    }

    public final int result() {
        return result;
    }

    public final void cancel() {
        silent = true;
        result = Presentation.CB_CANCELLED;
        out.rollback();
    }

    public final void commit() throws IOException, KeyCollisionException {
        out.commit();
    }

    public final KeyInputStream getKeyInputStream() throws IOException {
        return out.getKeyInputStream();
    }

    /**
     * If the state is lost, we're too overworked to try to eat the data anyway.
     */
    public final void lost(Node n) {
        try {
            //out.rollback();
            out.close();
        }
        catch (IOException e) {
            n.logger.log(this, "I/O error closing KeyOutputStream",
                         e, Logger.ERROR);
        }
        try {
            data.close();
        }
        catch (IOException e) {
            n.logger.log(this, "I/O error closing data receive stream",
                         e, Logger.MINOR);
        }
    }
    

    
    public State received(Node n, MessageObject mo) throws BadStateException {
        if (!(mo instanceof DataStateInitiator))
            throw new BadStateException("expecting DataStateInitiator");

        // if there is an IOException, this says
        // whether it was while writing to the store
        boolean inWrite = false;
        
        byte[] buffer = new byte[Core.blockSize];
        long moved = 0;
        
        try {
            while (moved < length) {
                inWrite = false;
                int m = data.read(buffer, 0, (int) Math.min(length - moved, buffer.length));
                if (m == -1) {
                    throw new IOException("Could not read all the expected data, read "
                                          +moved+" of "+length);
                }
                moved += m;
                inWrite = true;
                if (result != -1) throw new CancelledIOException();
                out.write(buffer, 0, m);
            }
            out.close();
            result = Presentation.CB_OK;
        }
        catch (DataNotValidIOException e) {
            result = e.getCode();
            n.logger.log(this, "Got DNV: "+Presentation.getCBdescription(result),
                         Logger.DEBUG);
            if (result == Presentation.CB_OK || result >= 0x80) {
                result = Presentation.CB_BAD_DATA;  // if peer claimed ok, replace
            }
        }
        catch (CancelledIOException e) {}  // already set result
        catch (IOException e) {
            // because we should be using a pad on error stream
            // (and hopefully a working disk for the Cache), we
            // should not see any of these, so they are really an
            // error condition.
            result = (inWrite ? Presentation.CB_CACHE_FAILED
                              : Presentation.CB_RECV_CONN_DIED);
            n.logger.log(this,
                "I/O error while moving data: "+Presentation.getCBname(result),
                e, Logger.ERROR);
        }
        finally {

            if (result != Presentation.CB_OK) {
                out.fail(result);
                //out.rollback();
                try {
                    out.close();
                }
                catch (IOException e) {
                    n.logger.log(this, "I/O error closing KeyOutputStream",
                                 e, Logger.ERROR);
                }
            }

            // we could be exiting with an uncaught exception or something..
            if (result == -1) result = Presentation.CB_RECV_CONN_DIED;
            
            if (!silent) n.schedule(new DataReceived(this));
            
            // eat the rest if necessary
            if (moved < length && inWrite) {
                DataState eatData = new EatData(id, data, length - moved);
                try {
                    eatData.received(n, new DataStateInitiator(eatData));
                    return null;  // EatData closed the stream.
                }
                catch (Throwable e) {}
            }
    
            try {
                if (result == Presentation.CB_RESTARTED
                           || result == Presentation.CB_ABORTED)
                    data.discontinue();
                else
                    data.close();
            }
            catch (IOException e) {
                n.logger.log(this, "I/O error closing data receive stream",
                             e, Logger.MINOR);
            }
        }
        
        return null;
    }
}



