package freenet.node.states.announcing;

import freenet.node.states.announcement.*;
import freenet.*;
import freenet.node.*;
import freenet.message.NodeAnnouncement;
import freenet.message.AnnouncementFailed;
import freenet.crypt.SHA1;
import freenet.crypt.Digest;
import freenet.support.Logger;
import java.math.BigInteger;

/**
 * The state that an announcement starts off with at Alice. 
 *
 * @author oskar
 */

public class SendAnnouncement extends AnnouncingState {
    
    /**
     * @param n       The node to schedule with.
     * @param id      Aggregate chain ID
     * @param tryN    The number of the attempt.
     * @param target  The node to send the announcement to.
     * @param htl     The number of hops that the announcement should make,
     *                make sure it is not more than what is tolerated by the 
     *                network.
     */
    static State makeTry(Node n, long id, int tryN, Peer target, int htl) {

        if (tryN > Node.announcementAttempts) {
            n.logger.log(SendAnnouncement.class,
                "Giving up after "+(tryN-1)+" failed attempts"
                +" to announce to node: "+target, Logger.NORMAL);
            return null;
        }
        
        if (htl > Node.maxHopsToLive) {
            n.logger.log(SendAnnouncement.class,
                "Reducing announcement HTL from "+htl
                +" to max of "+Node.maxHopsToLive, Logger.DEBUG);
                
            htl = Node.maxHopsToLive;
        }
        
        SendAnnouncement sa = new SendAnnouncement(id, htl, tryN, target);
        sa.schedule(n);
        return sa;
    }

    private class SendAnnouncementMessage implements NodeMessageObject {
        public long id() {
            return id;
        }
        public boolean isExternal() {
            return true;
        }
        public void drop(Node n) {
            // hmm..
        }
        public State getInitialState() {
            return SendAnnouncement.this;
        }
        private boolean belongsTo(SendAnnouncement sa) {
            return sa == SendAnnouncement.this;
        }

        public String toString() {
            return "Announce to: " + target + " try #" + tryNumber;
        }
    }



    private SendAnnouncement(long id, int htl, int tryN, Peer target) {
        super(id, htl, tryN, target);
    }

    public String getName() {
        return "Send My Node Announcement";
    }

    private void schedule(Node n) {
        
        long t = Node.announcementDelay 
                 * (long) Math.pow(Node.announcementDelayBase, tryNumber-1);
        
        n.logger.log(this,
          "Scheduling announcement attempt #"+tryNumber+" in "+(t/1000)+"s "
                     +" to node: "+ target, Logger.MINOR);

        n.schedule(t, new SendAnnouncementMessage());
    }
    
    public boolean receives(MessageObject mo) {
        return mo instanceof SendAnnouncementMessage
            && ((SendAnnouncementMessage) mo).belongsTo(this);
    }

    public State receivedMessage(Node n, SendAnnouncementMessage mo)
                                            throws BadStateException {
        if (!mo.belongsTo(this))
            throw new BadStateException("Not my SendAnnouncementMessage");
                                                
        // Our random commit value
        myVal = new byte[20];
        n.randSource.nextBytes(myVal);

        byte[] commitVal;
        
        synchronized (ctx) {
            ctx.update(myVal);
            commitVal = ctx.digest();
        }
        
        try {
            n.sendMessage(new NodeAnnouncement(id, hopsToLive, 0, n.myRef,
                                               n.myRef, commitVal), target);

            
            // Count outbound requests.
            n.diagnostics.occurrenceBinomial("outboundAggregateRequests", 1, 1);
            if (n.outboundRequests != null) {
                n.outboundRequests.incTotal(target.getAddress().toString());
            }

            // Keep track of outbound requests for rate limiting.
            n.outboundRequestLimit.inc();

            NoReply nr = new NoReply(id);
            n.schedule(getTime(2), nr);
            return new ExecuteAnnouncement(this, nr);
        } catch (CommunicationException e) {
            n.diagnostics.occurrenceBinomial("outboundAggregateRequests", 1, 0);
            n.logger.log(this, "Sending NodeAnnouncement failed: "+e,
                         Logger.MINOR);
                         //e, Logger.MINOR);
            return e.isTerminal()
                ? null
                : makeTry(n, id, tryNumber+1, target, hopsToLive);
        }
    }
}


