package freenet.node.states.announcing;

import freenet.*;
import freenet.node.*;
import freenet.support.Fields;
import freenet.support.Logger;

/**
 * This is the envelope state around the sub-chain to each target node.
 */
public class Announcing extends AggregatingState {

    /**
     * Schedules announcements to a set of target nodes.
     */
    public static void schedule(Node n, Peer[] targets) {
        long id = n.randSource.nextLong();
        n.logger.log(Announcing.class,
            "Scheduling announcements to "+targets.length
            +" target nodes on chain "+Fields.longToHex(id),
            Logger.MINOR);
        n.schedule(new ScheduleAnnouncing(id, targets, n.announcementPeers));
    }

    private static class ScheduleAnnouncing extends EventMessageObject {
        private Peer[] targets;
        private int totalHops;
        private ScheduleAnnouncing(long id, Peer[] targets, int totalHops) {
            super(id, true);
            this.targets = targets;
            this.totalHops = totalHops;
        }
        public State getInitialState() {
            return new Announcing(this.id, targets.length);
        }
    }

    
    
    private Announcing(long id, int slots) {
        super(id, slots);
    }

    public String getName() {
        return "Announcing Chains Aggregate";
    }

    public State received(Node n, MessageObject mo) throws StateException {
        if (mo instanceof ScheduleAnnouncing) {
            Peer[] targets = ((ScheduleAnnouncing) mo).targets;
            int hopsLeft = ((ScheduleAnnouncing) mo).totalHops;
            for (int i=0; i<targets.length; ++i) {
                int htl = Math.max(1, hopsLeft / (targets.length - i));
                hopsLeft -= htl;
                SendAnnouncement.makeTry(n, id, 1, targets[i], htl);
            }
            return this;
        }
        return super.received(n, mo);
    }
}


