package freenet.node.rt;

import freenet.Core;
import freenet.Identity;
import freenet.node.NodeReference;
import freenet.node.BadReferenceException;
import freenet.fs.dir.*;
import freenet.support.*;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.IOException;


/**
 * Uses a pair of DataObjectStores to hold node refs and node properties
 * needed to implement RoutingStore.
 * @author tavin
 */
public class DataObjectRoutingStore implements RoutingStore, Checkpointed {
    
    final DataObjectStore rtNodes, rtProps;

    private int count = 0;  // number of nodes


    public DataObjectRoutingStore(DataObjectStore rtNodes,
                                  DataObjectStore rtProps) {
        this.rtNodes = rtNodes;
        this.rtProps = rtProps;
        // initial count: number of node file entries
        Enumeration rme = rtNodes.keys(true);
        while (rme.hasMoreElements()) {
            ++count;
            rme.nextElement();
        }
    }


    public final String getCheckpointName() {
        return "Saving routing table changes.";
    }

    public final long nextCheckpoint() {
        return System.currentTimeMillis() + 1000 * 120;  // 2 minutes from now
    }

    public final void checkpoint() {
        try {
            rtNodes.flush();
            rtProps.flush();
        }
        catch (IOException e) {
            // hmz..
            Core.logger.log(this, "I/O error flushing routing table data",
                            e, Core.logger.ERROR);
        }
    }
    

    public final int size() {
        return count;
    }
    

    public boolean remove(Identity ident) {
        FileNumber fn = new FileNumber(ident.fingerprint());
        if (rtNodes.remove(fn)) {
            --count;
            // clear out properties entries
            Enumeration pk = rtProps.keys(new PrefixFilePattern(fn, true));
            while (pk.hasMoreElements()) {
                fn = (FileNumber) pk.nextElement();
                rtProps.remove(fn);
            }
            return true;
        }
        return false;
    }
    
    public final boolean contains(Identity ident) {
        return rtNodes.contains(new FileNumber(ident.fingerprint()));
    }

    
    public final Enumeration elements() {
        return new RoutingMemoryEnumeration(rtNodes.keys(true));
    }

    private final class RoutingMemoryEnumeration implements Enumeration {
        
        private final Enumeration keys;
        private Object next;
        
        RoutingMemoryEnumeration(Enumeration keys) {
            this.keys = keys;
            next = step();
        }

        private Object step() {
            while (keys.hasMoreElements()) {
                FileNumber fn = (FileNumber) keys.nextElement();
                Object o = getNode(fn);
                if (o != null)
                    return o;
            }
            return null;
        }

        public final boolean hasMoreElements() {
            return next != null;
        }

        public final Object nextElement() {
            if (next == null) throw new NoSuchElementException();
            try {
                return next;
            }
            finally {
                next = step();
            }
        }
    }
    
    

    private RoutingMemory getNode(FileNumber fn) {
        try {
            return (DataObjectRoutingMemory) rtNodes.get(fn);
        }
        catch (DataObjectUnloadedException dop) {
            try {
                return new DataObjectRoutingMemory(this, dop);
            }
            catch (BadReferenceException e) {
                Core.logger.log(this, "bad reference while resolving: "+fn,
                                e, Core.logger.ERROR);
                return null;
            }
            catch (IOException e) {
                Core.logger.log(this, "I/O error while resolving: "+fn,
                                e, Core.logger.ERROR);
                return null;
            }
        }
    }

    public final RoutingMemory getNode(Identity ident) {
        return getNode(new FileNumber(ident.fingerprint()));
    }
    
    
    public RoutingMemory putNode(NodeReference nr) {
        
        Identity ident = nr.getIdentity();
        FileNumber fn = new FileNumber(ident.fingerprint());

        boolean extant = rtNodes.contains(fn);
        
        DataObjectRoutingMemory mem = (DataObjectRoutingMemory) getNode(fn);

        if (mem == null)
            mem = new DataObjectRoutingMemory(this, nr);
        else if (mem.noderef == null || nr.supersedes(mem.noderef))
            mem.noderef = nr;

        rtNodes.set(fn, mem);

        if (!extant)
            ++count;

        return mem;
    }
}



