package freenet.node.http;

import freenet.Version;
import freenet.Core;
import freenet.node.Node;
import freenet.node.rt.RoutingStore;
import freenet.node.rt.RoutingMemory;
import freenet.support.Fields;
import freenet.support.URLDecoder;
import freenet.support.URLEncodedFormatException;
import freenet.support.io.WriteOutputStream;
import freenet.support.sort.QuickSorter;
import freenet.support.sort.VectorSorter;
import freenet.client.Base64;

import java.util.zip.ZipOutputStream;
import java.util.zip.ZipEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.jar.JarEntry;
import java.util.jar.Attributes;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Date;
import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;

public class DistributionServlet extends HttpServlet {

    private static final long LIFETIME = 24 * 60 * 60 * 1000;
    private static final long MAXHITS = 100;
    private static final int SEEDREFS = 10;

    private static final Object tablelock = new Object();

    private Node node;

    private File jarFreenet;
    private File jarFreenetExt;

    private String[] freenetResources;
    private String[] freenetExtResources;
    

    public void init() {
        ServletContext context = getServletContext();
        node = (Node)context.getAttribute("freenet.node.Node");
        getPages();

        freenetResources = fileList("freenet.files");
        freenetExtResources = fileList("freenet-ext.files");
        
    }

    private Hashtable getPages() {
        ServletContext context = getServletContext();
        Hashtable rt = getPagesOrNull(context);
        if (rt == null) {
            synchronized(tablelock) {
                // do it again synchronized
                rt = getPagesOrNull(context);
                if (rt == null) {
                    rt = new Hashtable();
                    context.setAttribute("freenet.distribution.pages", rt);
                }
            }
        }
        return rt;
    }

    private Hashtable getPagesOrNull(ServletContext context) {
        Object o = context.getAttribute("freenet.distribution.pages");
        return (o != null && (o instanceof Hashtable) ?
                (Hashtable) o : null);
    }
        

    private String[] fileList(String res) {
        try {
            InputStream in = getClass().getResourceAsStream(res);
            BufferedReader br = 
                new BufferedReader(new InputStreamReader(in, "UTF8"));

            Vector v = new Vector(1000);
            String s;
            while ((s = br.readLine()) != null) {
                v.addElement(s);
            }
            String[] rs = new String[v.size()];
            v.copyInto(rs);
            return rs;
        } catch (IOException e) {
            Core.logger.log(this, "Could no read filelist: " + res,
                            e, Core.logger.ERROR);
            return new String[0]; // FIXME
        }
    }

    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException {

        try {
            Hashtable pages = getPages();

            long threshTime = System.currentTimeMillis() - LIFETIME;
            Vector remove = new Vector();
            for (Enumeration e = pages.elements() ; e.hasMoreElements() ;) {
                DistributionPage dp = (DistributionPage) e.nextElement();
                if (dp.creationTime < threshTime) {
                    //                    pages.removeElementAt(i);
                    remove.addElement(dp.name);
                }
            }

            for (Enumeration e = remove.elements() ; e.hasMoreElements() ;) {
                pages.remove((String) e.nextElement());
            }

            String pi = req.getPathInfo();
            String uri = pi == null ? "" : URLDecoder.decode(pi);

            //System.err.println(uri);
            if (uri.equals("") || uri.equals("index.html")) {
                sendIndex(resp, pages);
            } else if (uri.endsWith("addDistPage.html")) {
                addDistPage(req, resp, pages);
            } else if (pages.containsKey(uri)) {
                sendDistPage((DistributionPage) pages.get(uri), resp);
            } else if (uri.endsWith(".zip") && 
                       pages.containsKey(uri.substring(0, uri.length() - 4))) {
                sendDistro((DistributionPage) pages.get(uri.substring(0, uri.length() - 4)), resp);
            } else {
                resp.setStatus(HttpServletResponse.SC_NOT_FOUND);
                resp.getWriter().print("Error");
            }
        } catch (URLEncodedFormatException e) {
            throw new ServletException(e.toString());
        } catch (RuntimeException e) {
            e.printStackTrace();
        }
        
    }


    public void sendIndex(HttpServletResponse resp,
                          Hashtable pages) throws IOException {

        // can we restrict this based on IP?

        PrintWriter pw = resp.getWriter();
        resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/html");

        pw.println("<html><head><title></title></head><body>");
        pw.println("<br><h2>Node distribution system</h2>");
        pw.println("<p><form action=\"./addDistPage.html\" method=\"Get\">" +
                   "<input type=\"Submit\" value=\"Make a distribution page.\""
                   + "</form></p>");
        pw.println("<p><a href=\"./references.ref\">Get a reference file</a>" + 
                   "</p>");
        pw.println("<p>This node's reference:</p><pre>\n\n");

        ByteArrayOutputStream buf = new ByteArrayOutputStream();

        node.myRef.getFieldSet().writeFields(new WriteOutputStream(buf));
        
        String ref = new String(buf.toByteArray());
        pw.println(ref);
        pw.println("\n\n</pre><b>Existing pages:</b><br><ul>");
        for (Enumeration e = pages.elements() ; e.hasMoreElements() ;) {
            DistributionPage dp = (DistributionPage) e.nextElement();
            pw.println("<li><a href=\"./" + dp.name + "\"> Created " + 
                       new Date(dp.creationTime) + "</li>");
        }
        pw.println("</ul>");

        pw.println("</body><html>");

    }


    public void addDistPage(HttpServletRequest req, 
                            HttpServletResponse resp,
                            Hashtable pages) throws IOException {

        // also IP restricted.

        PrintWriter pw = resp.getWriter();
        resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/html");
        
        byte[] bs = new byte[8];
        node.randSource.nextBytes(bs);
        String name = Base64.encode(bs);
        
        pages.put(name, new DistributionPage(name, 
                                             System.currentTimeMillis()));
        pw.println("<html><head><title></title></head><body>");
        pw.println("<h2>New distribution page added.</h2>");
        pw.println("<p>This URL will contain a freenet distribution for the next 24 hours:</p>");
        
        StringBuffer ruri = HttpUtils.getRequestURL(req);
        ruri.setLength(ruri.length() - ("addDistPage.html").length());
        String file = ruri.append(name).toString();
                                                    
        pw.println("<p><b><a href=\"./" + name + 
                   "\">" + file + "</a></p></b>");
        pw.println("</body></html>");
    }


    public void sendDistPage(DistributionPage dp, 
                             HttpServletResponse resp) throws IOException {

        if (dp.hits >= MAXHITS) {
            resp.setStatus(resp.SC_FORBIDDEN);
            resp.setContentType("text/pain");
            PrintWriter pw = resp.getWriter();
            pw.println("Hits on this distribution above limit, ask operator to open another.");
        } else {
            PrintWriter pw = resp.getWriter();
            resp.setStatus(HttpServletResponse.SC_OK);
            resp.setContentType("text/html");
            
            pw.println("<html><head><title></title></head><body>");
            pw.println("<br><h2>Node distribution system</h2>");
            pw.println("<p>This is a distribution of the Freenet node, version: "+
                       Version.getVersionString()+" automatically generated by the"
                       + " this Freenet node.</p>");
            pw.println("<p><a href=\"./" + dp.name + ".zip\">Download</a>" + 
                       "</p>");
            pw.println("<p>This node's reference:</p><pre>\n\n");
            
            ByteArrayOutputStream buf = new ByteArrayOutputStream();
            
            node.myRef.getFieldSet().writeFields(new WriteOutputStream(buf));
            
            String ref = buf.toString("UTF8");
            pw.println(ref);
            pw.println("\n\n</pre></body><html>");
        }
    }

    public void sendDistro(DistributionPage dp,
                           HttpServletResponse resp) throws IOException {

        if (dp.hits > MAXHITS) {
            resp.setStatus(resp.SC_FORBIDDEN);
            resp.setContentType("text/plain");
            PrintWriter pw = resp.getWriter();
            pw.println("Hits on this distribution above limit, ask operator to open another.");
        } else {
            resp.setStatus(HttpServletResponse.SC_OK);
            resp.setContentType("application/zip");
            
            ZipOutputStream out = new ZipOutputStream(resp.getOutputStream());
            out.putNextEntry(new ZipEntry("seednodes.ref"));
            writeSeedNodes(out);
            out.putNextEntry(new ZipEntry("lib/freenet.jar"));
            writeFreenetJar(out);
            out.putNextEntry(new ZipEntry("lib/freenet-ext.jar"));
            writeFreenetExtJar(out);
            out.finish();
            out.flush();
        }

    }

    private void writeSeedNodes(OutputStream out) throws IOException {
        WriteOutputStream wout = new WriteOutputStream(out);
        // start with me
        node.myRef.getFieldSet().writeFields(wout, "End");

        // now grab some from DS
        synchronized(node.rt.semaphore()) {
            RoutingStore rs = node.rt.getRoutingStore();
            int size = rs.size();

            Vector selected = new Vector(SEEDREFS);
            Integer s;
            for (int i = 0 ; i < Math.min(SEEDREFS, size) ; i++) {
                do {
                    s = new Integer(node.randSource.nextInt(size));
                } while(selected.contains(s));
                selected.add(s);
            }
            QuickSorter.quickSort(new VectorSorter(selected));
            Enumeration memories = rs.elements();
            int j = 0, next;

            for (Enumeration e = selected.elements() ; e.hasMoreElements();) {
                next = ((Integer) e.nextElement()).intValue();
                RoutingMemory rm = null;
                while (j <= next) {
                    rm = (RoutingMemory) memories.nextElement();
                    j++;
                }
                rm.getNodeReference().getFieldSet().writeFields(wout,
                                                                "End");
            }
        }
    }

    private void writeFreenetJar(OutputStream out) throws IOException {
        if (jarFreenet != null && !jarFreenet.equals("")) {
            try {
                copyFile(jarFreenet, out);
            } catch (FileNotFoundException e) {
            }
        }

        // Still here means no file.

        Manifest man = new Manifest();
        Manifest mf = new Manifest();
        Attributes att = mf.getMainAttributes();
        att.putValue("Main-class","freenet.node.Main");

        writeJarStream(freenetResources, mf, out);
    }

    private void writeFreenetExtJar(OutputStream out) throws IOException {
        if (jarFreenetExt != null && !jarFreenetExt.equals("")) {
            try {
                copyFile(jarFreenetExt, out);
            } catch (FileNotFoundException e) {
            }
        }

        // Still here means no file.

        writeJarStream(freenetExtResources, new Manifest(), out);
    }

    static void writeJarStream(String[] fileList, Manifest man,
                               OutputStream out) throws IOException {
        JarOutputStream jout = new JarOutputStream(out, man);

        byte[] buf = new byte[Core.blockSize];

        for (int i = 0 ; i < fileList.length ; i++) {

            ZipEntry ze = new ZipEntry(fileList[i].startsWith("/") ?
                                       fileList[i].substring(1) :
                                       fileList[i]);
            jout.putNextEntry(ze);
            InputStream in = Core.class.getResourceAsStream(fileList[i]);
            if (in == null) {
                Core.logger.log(DistributionServlet.class, 
                                "Could not find file " + fileList[i] +
                                " when making distribution.", 
                                Core.logger.ERROR);
                throw new FileNotFoundException();
            }
            int read;
            // copy, note no input buffering since the blocks are large.
            while ((read = in.read(buf)) != -1) {
                jout.write(buf, 0, read);
            }
        }
        jout.finish();
        jout.flush();
    }

    private static void copyFile(File from, 
                                 OutputStream to) throws IOException {
        byte[] buf = new byte[Core.blockSize];
        InputStream in = new FileInputStream(from);
        int read;
        while ((read = in.read(buf)) != -1) {
            to.write(buf, 0, read);
        }
    }

    private class DistributionPage {

        private String name;
        private long creationTime;
        private int hits = 0;

        public DistributionPage(String name, long creationTime) {
            this.name = name;
            this.creationTime = creationTime;
        }

    }


}

