package freenet.node.http;

import freenet.Core;
import freenet.diagnostics.*;
import freenet.support.URLDecoder;
import freenet.support.URLEncodedFormatException;

import javax.servlet.http.*;
import javax.servlet.ServletException;
import javax.servlet.ServletContext;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.NoSuchElementException;

public class DiagnosticsServlet extends HttpServlet {

    public void init() {
    }


    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException {
        
        String uri;
        try {
            String pi = req.getPathInfo();
            uri = pi == null ? "" : URLDecoder.decode(pi);
        } catch (URLEncodedFormatException e) {
            resp.sendError(resp.SC_BAD_REQUEST, "Could not decode url.");
            return;
        }

        if (uri.startsWith("/"))
            uri = uri.substring(1);

        if (uri.equalsIgnoreCase("index.html") || uri.equals("")) {
            sendDiagnosticsIndex(req, resp);
        } else {
            int sep = uri.indexOf('/');
            if (sep == -1)
                resp.sendError(resp.SC_NOT_FOUND, "Unrecognized URL.");
            String varName = uri.substring(0, sep);
            String period = uri.substring(sep + 1);
            sendVarData(req, resp, 
                        varName, period);
        }
    }

    public String getServletInfo() {
        return "Node diagnostics information.";
    }

    public String getServletName() {
        return "Diagnostics";
    }


    private final void sendDiagnosticsIndex(HttpServletRequest req, 
                                            HttpServletResponse resp)
        throws IOException {

        PrintWriter pw = resp.getWriter();
        resp.setContentType("text/html");

        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Freenet Node Diagnostics Variables");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        pw.println();
        pw.println();

        DiagnosticsFormat indexFormat = new HtmlIndexFormat();
        Core.diagnostics.writeVars(pw, indexFormat);

        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer();
    }


    private final void sendVarData(HttpServletRequest req, 
                                   HttpServletResponse resp, 
                                   String varName, 
                                   String period) throws IOException {


        PrintWriter pw = resp.getWriter();

        DiagnosticsFormat format;
        boolean html = false;
        if (period.equalsIgnoreCase("occurrences")) {
            html = true;
            resp.setContentType("text/html");
            format =  new HtmlDiagnosticsFormat(-1);
        } else if (period.equalsIgnoreCase("raw")) {
            resp.setContentType("text/plain");
            format = new RowDiagnosticsFormat();
        } else if (period.startsWith("raw")) {
            resp.setContentType("text/plain");
            if (period.substring(3).equalsIgnoreCase("occurences"))
                format = new RowDiagnosticsFormat(-1);
            else
                format = 
                    new RowDiagnosticsFormat(Diagnostics.getPeriod(period.substring(3)));

        } else if (period.equalsIgnoreCase("fieldset")) {
            resp.setContentType("text/plain");
            format = new FieldSetFormat();
        } else {
            try {
                resp.setContentType("text/html");
                html = true;
                format = 
                    new HtmlDiagnosticsFormat(Diagnostics.getPeriod(period));
            } catch (IllegalArgumentException e) {
                resp.sendError(resp.SC_NOT_FOUND, 
                               "Unknown period type given.");
                return;
            }
        }

        try {
            if (html)
                pw.println("<html><body>");
            pw.println(Core.diagnostics.writeVar(varName, format));
            if (html)
                pw.println("</body></html>");
        } catch (NoSuchElementException e) {
            resp.sendError(404, "No such diagnostics field");
            return;
        }

        resp.flushBuffer();
    }

    

}

