package freenet.node;

import freenet.*;
import freenet.support.*;
import freenet.thread.ThreadFactory;
import java.util.Hashtable;

/**
 * This class handles incoming messages.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author oskar
 * @author tavin
 */
public class StandardMessageHandler extends MessageHandler {

    public final Node node;
    
    private Hashtable ticketTable;
    private DoublyLinkedList ticketLadder;

    private int limit;
    
    //private ThreadFactory threadFactory;

    /**
     * @param n The node that should be used to send messages
     *          and for which the datastore should be updated
     * @param m The number of message objects that will be
     *          remembered
     */
    public StandardMessageHandler(Node n, int m) {
        //this.threadFactory = threadFactory;
        node = n;
        ticketTable  = new Hashtable(m);
        ticketLadder = new DoublyLinkedListImpl();
        limit = m;
    }
    
    /**
     * Handle a message
     * @param mo The message to handle
     */
    public void handle(MessageObject mo) {
        
        if (!(mo instanceof NodeMessageObject)) {
            node.logger.log(this,
                "Received a MessageObject that the node cannot handle: "+mo,
                Logger.ERROR);
            return;
        }
        NodeMessageObject message = (NodeMessageObject) mo;

        TicketIndex ti = new TicketIndex(message.id(), message.isExternal());
        Ticket ticket, lostTicket = null;
        
        // check out the ticket for this id
        synchronized (ticketTable) {
            ticket = (Ticket) ticketTable.get(ti);
            if (ticket == null) {
                ticket = new Ticket(ti);        // no two threads can make
                ticketTable.put(ti, ticket);    // duplicate tickets
            }
        }

        ticket.received(node, message);

    }

    /** Container for a 64-bit unique id that prevents internal chains
      * from colliding with external ones.
      */
    private final static class TicketIndex {
        private final long id;
        private final boolean external;
        public TicketIndex(long id, boolean external) {
            this.id = id;
            this.external = external;
        }

        public final long id() {
            return id;
        }

        public final boolean equals(Object o) {
            TicketIndex ot = (TicketIndex) o;
            return ot.id == id && ot.external == external;
        }

        public final int hashCode() {
            return (int) id ^ (int) (id >> 8)  ^ (external ? 0xffffffff : 0);
        }

        public final String toString() {
            return (external ? "ext:" : "int:") + Fields.longToHex(id);
        }
    }

    private final class Ticket extends DoublyLinkedListImpl.Item implements Runnable {
        
        private final TicketIndex index;
        private final StateChain chain = new StateChain();

        private DoublyLinkedList workList;
        
        private boolean working;

        private Ticket(TicketIndex index) {
            this.index = index;
            this.working = false;
            workList = new DoublyLinkedListImpl();
        }

        private final void received(Node node, NodeMessageObject mo) {

            boolean shouldrun = false;
            synchronized (workList) {
                workList.push(new TicketEntry(node, mo));

                if (!working) {
                    // it cannot be forgotten while using it
                    synchronized (ticketTable) {
                        ticketLadder.remove(this);
                    }

                    working = true;
                    //threadFactory.getThread(this, true).start();
                    shouldrun = true;
                } 
            }

            if (shouldrun)
                run();
        }

        private final void stopWorking() {
            Ticket lostTicket = null;
            working = false;
               
            // check the ticket back in
            synchronized (ticketTable) {
                if (chain.alive())
                    ticketLadder.push(this);
                else 
                    ticketTable.remove(index);
            
                // forget old states if necessary
                if (ticketLadder.size() > limit) {
                    lostTicket = (Ticket) ticketLadder.shift();
                    ticketTable.remove(lostTicket.index);
                }
            }
                
            // clean up lost states.  no need to synchronize here,
            // we know we are the only one holding this lost ticket
            if (lostTicket != null)
                lostTicket.lost(node);        
        }

        private TicketEntry getNextEntry() {
            synchronized (workList) {
                if (workList.isEmpty()) {
                    stopWorking();
                    return null;
                }

                return (TicketEntry) workList.shift();
            }
        }

        public void run() {
            TicketEntry entry;

            while (null != (entry = getNextEntry())) {
                boolean wasAlive, isAlive;   
     
                synchronized (chain) {
                    Node node = entry.getNode();
                    NodeMessageObject mo = entry.getMessageObject();

                    wasAlive = chain.alive();
                    isAlive  = chain.received(node, mo);
                    if (!wasAlive && isAlive)
                        node.diagnostics.occurrenceCounting("liveChains", 1);
                    else if (wasAlive && !isAlive)
                        node.diagnostics.occurrenceCounting("liveChains", -1);
                }
            }
        }

        private final void lost(Node node) {
            if (chain.lost(node)) {
                node.logger.log(node,
                    "States overflow, discarding: "+chain,
                    Logger.DEBUG);
                node.diagnostics.occurrenceCounting("liveChains", -1);
            }
        }
    }

    private static final class TicketEntry extends DoublyLinkedListImpl.Item  {
        private Node node;
        private NodeMessageObject mo;

        public TicketEntry(Node node, NodeMessageObject mo) {
            this.node = node;
            this.mo = mo;
        }

        public Node getNode() {
            return node;
        }

        public NodeMessageObject getMessageObject() {
            return mo;
        }
    }
}


