package freenet.node;

import freenet.*;
import freenet.diagnostics.*;
import freenet.thread.*;
import freenet.interfaces.*;
import freenet.interfaces.servlet.*;
import freenet.config.*;
import freenet.crypt.*;
import freenet.fs.*;
import freenet.fs.FileSystem;
import freenet.fs.dir.*;
import freenet.support.*;
import freenet.support.io.*;
import freenet.node.ds.*;
import freenet.node.rt.*;
import freenet.node.states.maintenance.*;
import freenet.node.states.announcing.Announcing;
import freenet.message.*;
import freenet.message.client.*;
import freenet.presentation.*;
import freenet.session.*;
import freenet.transport.TCP;
import java.util.*;
import java.net.*;
import java.io.*;
import java.lang.reflect.Constructor;
import javax.servlet.Servlet;
import javax.servlet.http.HttpServlet;

/**
 * This contains the execution code for starting a node. Moved from Node.java
 */

public class Main {

    public static final String[] defaultRCfiles =
        new String[] { "freenet.conf", "freenet.ini", ".freenetrc" };

    private static final Config switches = new Config();
    static {
        Node.class.toString();
        
        String dc = defaultRCfiles[0];
        
        switches.addOption("help",       'h', 0, null, 10);
        switches.addOption("system",          0, null, 11);
        switches.addOption("version",    'v', 0, null, 12);
        switches.addOption("manual",          0, null, 13);
        switches.addOption("export",     'x', 1, "-",  20);
        switches.addOption("seed",       's', 1, "-",  21);
        switches.addOption("config",     'c', 1, dc,   40);
        switches.addOption("paramFile",  'p', 1, null, 41);
 
        switches.shortDesc ("help", "prints this help message");
        switches.shortDesc ("system", "prints JVM properties");
        switches.shortDesc ("version", "prints out version info");
        switches.shortDesc ("manual", "prints a manual in HTML");

        switches.argDesc   ("export", "<file>|-");
        switches.shortDesc ("export", "exports a signed NodeReference");

        switches.argDesc   ("seed", "<file>|-");
        switches.shortDesc ("seed", "seeds routing table with refs");

        switches.argDesc   ("config", "<file>");
        switches.shortDesc ("config", "generates or updates config file");

        switches.argDesc   ("paramFile", "<file>");
        switches.shortDesc ("paramFile", "path to a config file in a non-default location");
    }

    /**
     * Start Fred on his journey to world dominatrix^H^H^Hion
     */
    public static void main(String[] args) throws Throwable {
        try {
            // process command line
            Params sw = new Params(switches.getOptions());
            sw.readArgs(args);
            if (sw.getParam("help") != null) {
                usage();
                return;
            }
            if (sw.getParam("system") != null) {
                System.getProperties().list(System.out);
                return;
            }
            if (sw.getParam("version") != null) {
                version();
                return;
            }
            if (sw.getParam("manual") != null) {
                manual();
                return;
            }

            args = sw.getArgs();  // remove switches recognized so far

            Params params = new Params(Node.config.getOptions());
            
            // attempt to load config file
            String paramFile = sw.getParam("paramFile");
            try {
                if (paramFile == null)
                    params.readParams(defaultRCfiles);
                else
                    params.readParams(paramFile);
            }
            catch (FileNotFoundException e) {
                if (sw.getParam("config") == null) {
                    if (paramFile == null) {
                        System.err.println("Couldn't find any of the following configuration files:");
                        System.err.println("    " + Fields.commaList(defaultRCfiles));
                        System.err.println("If you have just installed this node, use --config with no");
                        System.err.println("arguments to create a config file in the current directory,");
                        System.err.println("or --config <file> to create one at a specific location.");
                    }
                    else {
                        System.err.println("Couldn't find configuration file: "+paramFile);
                    }
                    return;
                }
            }

            params.readArgs(args);
            // I want config after readArgs, which must be after readParams
            // which mandates the hack in the catch block above.
            if (sw.getParam("config") != null) {
                try {
                    Setup set = new Setup(System.in, System.out, 
                                          new File(sw.getString("config")),
                                          false, params);
                    set.dumpConfig();
                } catch (IOException e) {
                    System.err.println("Error while creating config: " + e);
                }
                return;
            }


            // bail out if there were unrecognized switches
            // (we take no args)
            if (params.getNumArgs() > 0) {
                usage();
                return;
            }

            
            // set up runtime logging
            int thresh = Logger.priorityOf(params.getString("logLevel"));
            Core.logger.setThreshold(thresh);

            String fname     = params.getString("logFile");
            String logFormat = params.getString("logFormat");
            String logDate   = params.getString("logDate");
        
            try {
                if (!fname.equalsIgnoreCase("NO"))
                    Core.logger.addHook(new FileLoggerHook(fname, 
                                                           logFormat, 
                                                           logDate,
                                                           thresh));
                else
                    Core.logger.addHook(new FileLoggerHook(System.err,
                                                           logFormat, 
                                                           logDate,
                                                           thresh));
            }
            catch (IOException e) {
                System.err.println("Opening log file failed!");
            }
            // Add a buffering hook for the last couple of entries as well
            Core.logger.addHook(new BufferLoggerHook(20));
            

            // NOTE:  we are slowly migrating stuff related to setting up
            //        the runtime environment for the node into the Main
            //        class (and out of Core and Node)
            //
            //        this includes stuff like registering different
            //        message types, key types, diagnostics counters, etc.,
            //        and anything that involves the config params
            

            // set up static parameters in Core and Node
            Node.init(params);

            // enable diagnostics
            if (params.getBoolean("doDiagnostics"))
                initDiagnostics(params, Core.logger);

            
            // load node secret keys file
            
            Core.logger.log(Main.class, "loading node keys: "+Node.nodeFile, Logger.NORMAL);

            BlockCipher cipher = (Node.storeCipherName == null
                                  ? null
                                  : Util.getCipherByName(Node.storeCipherName,
                                                         Node.storeCipherWidth));
            
            Authentity privateKey;
            byte[] cipherKey = null;
            byte[] baseIV = null;
            
            try {
                InputStream in = new FileInputStream(Node.nodeFile);
                try {
                    FieldSet fs = new FieldSet(new CommentedReadInputStream(in, "#"));
                    privateKey = new DSAAuthentity(fs.getSet("authentity"));
                    if (cipher != null) {
                        cipherKey = Fields.hexToBytes(fs.get("cipherKey"));
                        baseIV = Fields.hexToBytes(fs.get("baseIV"));
                    }
                }
                catch (NumberFormatException e) {
                    throw new ParseIOException("bad node file: "+e);
                }
                finally {
                    in.close();
                }
            }
            catch (FileNotFoundException e) {
                
                Core.logger.log(Main.class, "creating node keys: "+Node.nodeFile, Logger.NORMAL);
                
                FieldSet fs = new FieldSet();
                
                // FIXME: nodes should generate their own DSA group
                privateKey = new DSAAuthentity(Global.DSAgroupC, Node.randSource);
                fs.put("authentity", privateKey.getFieldSet());
                
                if (cipher != null) {
                    cipherKey = new byte[cipher.getKeySize() >> 3];
                    baseIV = new byte[cipher.getBlockSize() >> 3];
                    Node.randSource.nextBytes(cipherKey);
                    Node.randSource.nextBytes(baseIV);
                    fs.put("cipherKey", Fields.bytesToHex(cipherKey));
                    fs.put("baseIV", Fields.bytesToHex(baseIV));
                }
                
                OutputStream out = new FileOutputStream(Node.nodeFile);
                try {
                    fs.writeFields(new WriteOutputStream(out));
                }
                finally {
                    out.close();
                }
            }
            
            
            // are we transient?
            boolean isTransient = params.getBoolean("transient");


            // set up stuff for handling the node's transports,
            // sessions, and presentations
            TransportHandler th    = new TransportHandler();
            SessionHandler sh      = new SessionHandler();
            PresentationHandler ph = new PresentationHandler();
        
            th.register(new TCP(100));
            sh.register(new FnpLinkManager(), 100);
            ph.register(new FreenetProtocol(), 100);

            // get the node's physical addresses
            Address[] addr;
            try {
                addr = getAddresses(th, params);
            }
            catch (BadAddressException e) {
                if (!isTransient) {
                    System.err.println("There was an error determining this node's physical address(es).");
                    System.err.println("Please make sure <ipAddress> and <listenPort> are correctly set.");
                    System.err.println("Note that you may put a host name in the <ipAddress> field if you");
                    System.err.println("have a dynamic IP and are using a dynamic DNS service.");
                    throw e;
                }
                else addr = new Address[0];  // we're transient, screw it..
            }


            // we have enough to generate the node reference now
            NodeReference myRef = Node.makeNodeRef(privateKey, addr, sh, ph);
            
        
            // --export
            // (no need to load up the full node)
            if (sw.getParam("export") != null) {
                writeFieldSet(myRef.getFieldSet(),
                              sw.getString("export"));
                return;
            }

            
            // load up the FS
            
            Core.logger.log(Main.class, "starting filesystem", Logger.NORMAL);

                
            File[] storeFiles = new File[Node.storeFile.length];
            for (int i=0; i<storeFiles.length; ++i)
                storeFiles[i] = new File(Node.storeFile[i]);

            Storage storage = new RAFStorage(storeFiles, Node.storeSize);

            FileSystem fs;
            if (cipher == null) {
                fs = new FileSystem(storage);
            }
            else {
                cipher.initialize(cipherKey);
                fs = new EncryptedFileSystem(storage, cipher, baseIV);
            }
            
            // FIXME:  there need to be some sanity checks on the
            //         ranges for the directory root, accounting tables,
            //         and files, to see if any of them were truncated.
            if (fs.truncation() < fs.size()) {
                FSInitializer fsini = new FSInitializer(fs);
                synchronized (fsini) {
                    try {
                        fsini.start();
                        fsini.wait();
                    }
                    catch (InterruptedException e) {}
                }
                if (fsini.delayedThrow != null)
                    throw fsini.delayedThrow; //.fillInStackTrace();
            }

            //
            // FIXME: replace literals with config values
            //

            Directory dir = new FSDirectory(fs, new SHA1(), 10000, 10000, 1000, 100);
        
            // load DS
            Core.logger.log(Main.class, "loading data store", Logger.NORMAL);
            
            LossyDirectory dsDir = new LossyDirectory(0x0001, dir);
            dsDir.forceFlush();
            DataStore ds = new FSDataStore(dsDir, fs.size() / 50);

            // load RT
            Core.logger.log(Main.class, "loading routing table", Logger.NORMAL);
            
            LossyDirectory rtNodesDir = new LossyDirectory(0x0101, dir, 0x0001);
            LossyDirectory rtPropsDir = new LossyDirectory(0x0102, dir, 0x0001);
            DataObjectStore rtNodes = new FSDataObjectStore(rtNodesDir);
            DataObjectStore rtProps = new FSDataObjectStore(rtPropsDir);
            DataObjectRoutingStore routingStore =
                new DataObjectRoutingStore(rtNodes, rtProps);

            RoutingTable rt = new CPAlgoRoutingTable(routingStore,
                                                     Node.rtMaxNodes,
                                                     Node.rtMaxRefs,
                                                     Core.randSource);

            //              RoutingTable rt = new ProbabilityRoutingTable(routingStore,
            //                                                            Node.rtMaxNodes,
            //                                                            Node.rtMaxRefs,
            //                                                            Core.randSource);

            rt = new FilterRoutingTable(rt, privateKey.getIdentity());

            // load FT

            FailureTable ft = 
                new FailureTable(params.getInt("failureTableSize"),
                                 params.getLong("failureTableTime"));

            

            
            // load temp bucket factory
            Core.logger.log(Main.class, "loading temp bucket factory", Logger.NORMAL);

            LossyDirectory bfDir = new LossyDirectory(0x0002, dir, 0x0001);
            BucketFactory bf = new FSBucketFactory(bfDir);
            

            try {
                // --seed
                // (now that we have the routing table)
                if (sw.getParam("seed") != null) {
                    NodeReference[] seedNodes = readSeedNodes(sw.getString("seed"));
                    seedRoutingTable(rt, seedNodes, true);
                }
                
                // run node
                else {
            
                    // load seed nodes
                    NodeReference[] seedNodes = null;
                    String seedNodesFile = params.getString("seedFile");
                    if (seedNodesFile != null && !seedNodesFile.equals("")) {
                        try {
                            seedNodes = readSeedNodes(seedNodesFile);
                            seedRoutingTable(rt, seedNodes, false);
                            routingStore.checkpoint();  // save data
                        }
                        catch (FileNotFoundException e) {
                            Core.logger.log(Main.class,
                                            "Seed file not found: " + 
                                            seedNodesFile, Logger.NORMAL);
                        }
                    }
            
                    Core.logger.log(Main.class, "starting node", 
                                    Logger.NORMAL);
                    
                    Node node = new Node(privateKey, myRef, dir, bf, ds, rt,
                                         ft, th, sh, ph, isTransient);
                    startNode(node, addr, params);  // run Core

                    // schedule checkpoints for link cleanup,
                    // routing table saves, and diagnostics aggregation

                    DiagnosticsCheckpoint diagnosticsCheckpoint =
                        new DiagnosticsCheckpoint(node.autoPoll, node.diagnostics);
    
                    new Checkpoint(sh).schedule(node);
                    new Checkpoint(routingStore).schedule(node);
                    new Checkpoint(diagnosticsCheckpoint).schedule(node);
                    new Checkpoint(ft).schedule(node);

                    // schedule announcements
                    if (params.getBoolean("doAnnounce")) {
                        if (node.isTransient()) {
                            node.logger.log(Main.class,
                                "Not announcing because I am transient.",
                                Logger.NORMAL);
                        }
                        else if (seedNodes == null || seedNodes.length == 0) {
                            node.logger.log(Main.class,
                                "Cannot announce with no seed nodes!",
                                Logger.ERROR);
                        }
                        else {
                            Peer[] targets;
                            Vector v = new Vector();
                            synchronized (v) {
                                for (int i=0; i<seedNodes.length
                                              && i < 10 ; ++i) {
                                    Peer p = node.getPeer(seedNodes[i]);
                                    if (p == null) {
                                        node.logger.log(Main.class,
                                            "Skipped unresolvable seed node while"
                                            +" scheduling announcements: "+seedNodes[i],
                                            Logger.ERROR);
                                        continue;
                                    }
                                    v.addElement(p);
                                }
                                targets = new Peer[v.size()];
                                v.copyInto(targets);
                            }
                            Announcing.schedule(node, targets);
                        }
		    }

                    node.join();  // join all interface threads
                }
            }
            finally {
                // save the latest state information
                routingStore.checkpoint();
                dsDir.forceFlush();
            }
        }
        catch (DiagnosticsException e) {
            System.err.println("ERROR: "+e.getMessage());
            System.err.println("The internal diagnostics could not be initialized.");
            System.err.println("Set doDiagnostics=no in the configuration to disable them.");
            System.exit(1);
        }
        catch (ListenException e) {
            System.err.println("ERROR: "+e.getMessage());
            System.err.println("Could not bind to listening port(s)"
                               +" - maybe another node is running?");
            System.err.println("Or, you might not have privileges"
                               +" to bind to a port < 1024.");
            System.exit(1);
        }
        finally {
            Core.randSource.close();
        }
    }

    private static final class FSInitializer extends Thread {

        final FileSystem fs;
        Throwable delayedThrow = null;

        FSInitializer(FileSystem fs) {
            super("store-initializer");
            setDaemon(true);
            this.fs = fs;
        }

        public void run() {
            Node.logger.log(this,
                            "initializing data store ("+fs.size()+" bytes)",
                            Logger.NORMAL);
            try {
                long t = System.currentTimeMillis();
                fs.initialize(Node.randSource, this);
                t = System.currentTimeMillis() - t;
                Node.logger.log(this, "finished initializing store ("+(t/1000)+" sec)",
                                Logger.NORMAL);
            }
            catch (IOException e) {
                delayedThrow = e;
                Node.logger.log(this, "I/O error while initializing datastore",
                                e, Logger.ERROR);
            }
            catch (Throwable e) {
                delayedThrow = e;
                Node.logger.log(this, "Unhandled error while initializing datastore",
                                e, Logger.ERROR);
            }
            finally {
                // in case it wasn't reached by fs.initialize()
                synchronized (this) {
                    notify();
                }
            }
        }
    }

    /**
     * @return  an array of physical addresses this node will listen on
     */
    private static Address[] getAddresses(TransportHandler th, Params params)
                                                    throws BadAddressException {
        // for now, we are going to be sloppy and just assume we can get
        // "tcp" out of the TransportHandler and look for "ipAddress" and
        // "listenPort" in the config params.  what we should be doing is
        // pulling a generalized list of addresses out of the config params
        // and resolving them against the TransportHandler based on transport
        // name and transport-specific address string
        String ipAddress = params.getString("ipAddress");
        int listenPort   = params.getInt("listenPort");
        return new Address[] { th.get("tcp").getAddress(ipAddress+':'+listenPort) };
    }

    /**
     * Actually sets up the interfaces and starts running the Core.
     * @throws IOException          for an I/O error reading seed nodes or
     *                              config files
     * @throws BadAddressException  if there was a problem getting listening
     *                              addresses or allowed host addresses
     * @throws ListenException      if one of the interfaces couldn't listen
     */
    private static void startNode(Node node, Address[] addr, Params params)
                    throws IOException, BadAddressException, ListenException {
        // set up a thread group, the threadpool, and the OCM
        ThreadGroup tg = new ThreadGroup(node.toString());
        ThreadFactory tf = (node.maximumThreads > 0
                            ? (ThreadFactory) new PThreadFactory(tg, node.maximumThreads)
                            : (ThreadFactory) new SimpleThreadFactory(tg));

        // Keep a hold of the the ThreadManager so that we can use
        // it for rate limiting in Node.acceptRequest().
        node.threadFactory = tf;

        int maxConn = params.getInt("maxNodeConnections");

        OpenConnectionManager ocm = new OpenConnectionManager(tf, maxConn);
                
        // set up interfaces
        Vector iv = new Vector();
         
        // we start an FNP interface for each of our external addresses
        ConnectionRunner fnp = new FreenetConnectionRunner(node, node.sessions,
                                                           node.presentations, ocm, 3);
        for (int i=0; i<addr.length; ++i) {
            ContactCounter contactCounter = null;
            if (params.getBoolean("logInboundContacts") && 
                (Core.inboundContacts == null)) {
                // REDFLAG: This works because getAddresses() 
                //          only returns one address which is
                //          assumed to be tcp. If getAddresses() 
                //          is ever fixed, this code may break.
                //
                // Enable monitoring of inbound contact addresses
                // via NodeStatusServlet.
                Core.inboundContacts = new ContactCounter();
                contactCounter = Core.inboundContacts;
            }

            iv.addElement(new PublicInterface(addr[i].listenPart(),
                                              maxConn, tf, ocm, fnp, contactCounter));
        }

        if (params.getBoolean("logOutboundContacts")) {
            // Enable monitoring of outbound contact addresses
            // via NodeStatusServlet.
            Core.outboundContacts = new ContactCounter();
        }

        if (params.getBoolean("logInboundRequests")) {
            // Enable monitoring of per host inbound  
            // request stats via NodeStatusServlet
            Core.inboundRequests = new ContactCounter();
        }

        if (params.getBoolean("logOutboundRequests")) {
            // Enable monitoring of per host outbound  
            // request stats via NodeStatusServlet
            Core.outboundRequests = new ContactCounter();
        }

        // the FCP interface
        SessionHandler clientSh = new SessionHandler();
        clientSh.register(new FnpLinkManager(), 10);
        clientSh.register(new PlainLinkManager(), 20);
        
        PresentationHandler clientPh = new PresentationHandler();
        clientPh.register(new ClientProtocol(), 100);
        
        String[] fcpHosts = params.getList("fcpHosts");
        Address[] fcpHostsAddr = null;
        TCP ltcp;

        // if fcpHosts is specified, we'll listen on all interfaces
        // otherwise we listen only on loopback
        if (fcpHosts == null || fcpHosts.length == 0) {
            ltcp = new TCP(InetAddress.getByName("127.0.0.1"), 1);
        }
        else {
            ltcp = new TCP(1);
            fcpHostsAddr = new Address[fcpHosts.length];
            for (int i=0; i<fcpHosts.length; ++i) {
                if (fcpHosts[i].trim().equals("*")) {
                    fcpHostsAddr = null;
                    break;
                }
                fcpHostsAddr[i] = ltcp.getAddress(fcpHosts[i]+":0");
            }
        }

        ConnectionRunner fcp =
            new FreenetConnectionRunner(node, clientSh, clientPh, ocm, 1);
        
        ListeningAddress la =
            ltcp.getListeningAddress(""+params.getInt("clientPort"));
        
        iv.addElement(new LocalInterface(la, tf, fcp, fcpHostsAddr));

        // plus services
        String[] services = params.getList("services");
        if (services != null && !services[0].equals("none")) {
            for (int i=0; i<services.length; ++i) {
		node.logger.log(Main.class, "loading service: " + services[i],
                                Logger.NORMAL);
                FieldSet fs = params.getSet(services[i]);
                if (fs == null) {
                    node.logger.log(Main.class,
                        "No configuration parameters found for: "+services[i],
                        Logger.ERROR);
                    continue;
                }
                try {
                    Service svc = loadService(fs, node);
                    iv.addElement(LocalInterface.make(fs, tf, svc));
                }
                //catch (Exception e) {
                // Need to catch link errors.
                catch (Throwable e) {
                    node.logger.log(Main.class,
                        "Failed to load service: "+services[i],
                        e, Logger.ERROR);
                }
            }
        }
        
        
        // start the Core
        
        Interface[] interfaces = new Interface[iv.size()];
        iv.copyInto(interfaces);

        MessageHandler mh =
            new StandardMessageHandler(node, 
                                       params.getInt("messageStoreSize"));
        initMessageHandler(mh, params);

        Ticker ticker = new Ticker(mh, tf);

        node.begin(new SimpleThreadFactory(tg),
                   ticker, ocm, interfaces, false);
    }

    private static Service loadService(FieldSet fs, Node node)
                            throws IOException, ServiceException {
        Service service;
        
        String className = fs.get("class");
        if (className == null || className.trim().equals(""))
            throw new ServiceException("No class given");
        Class cls;
        try {
            cls = Class.forName(className.trim());
        }
        catch (ClassNotFoundException e) {
            throw new ServiceException(""+e);
        }

        if (Servlet.class.isAssignableFrom(cls)) {
            if (HttpServlet.class.isAssignableFrom(cls))
                service = new SingleHttpServletContainer(node, cls);
            else
                throw new ServiceException("I'm too dumb for: "+cls);
        }
        else {
            service = ServiceLoader.load(cls, node);
        }

        Config serviceConf = service.getConfig();
        if (serviceConf != null) {
            Params params;
            if (fs.getSet("params") != null) {  // read from FieldSet
                params = new Params(serviceConf.getOptions(),
                                    fs.getSet("params"));
            }
            else if (fs.get("params") != null) {  // or external file
                params = new Params(serviceConf.getOptions());
                params.readParams(fs.get("params"));
            }
            else {
                params = new Params(serviceConf.getOptions());
            }
            service.init(params);
        }

        return service;
    }
    
    
    private static void initDiagnostics(Params params, Logger logger)
                                        throws DiagnosticsException {

        // FIXME:  we now have the capability to store
        //         diagnostics data in the datastore

        // set up diagnostics
        String statsDir = params.getString("diagnosticsPath");

        Diagnostics d = new StandardDiagnostics(logger, statsDir);

        // Categories
        DiagnosticsCategory connections =
            d.addCategory("Connections", "Data regarding the connections.",
                          null);
        DiagnosticsCategory outConn =
            d.addCategory("Outbound", "Connections established.",
                          connections);
        DiagnosticsCategory inConn = 
            d.addCategory("Inbound", "Connections accepted.",
                          connections);

        DiagnosticsCategory messages = 
            d.addCategory("Messages", "Data regarding the routing of messages",
                          null);

        DiagnosticsCategory threading = 
            d.addCategory("Threading",
                          "Data regarding the thread pool, job scheduling, " +
                          "and job execution", null);

        
        // connections
        
        d.registerContinuous("connectionLifeTime", d.HOUR, 
                             "The amount of time that connections stay open."
                             + " In ms.", connections);
        d.registerCounting("liveConnections", d.MINUTE,
                           "The number of connections established minus " +
                           "the number closed.", connections);
        d.registerCounting("liveLinks", d.MINUTE,
                           "The number of cached links established minus " +
                           "the number forgotten.", connections);
        d.registerContinuous("connectionMessages", d.MINUTE, 
                             "The number of messages sent over each open " +
                             "connection. In ms.", connections);
        d.registerCounting("readLockedConnections", d.MINUTE,
                           "The number of connections that start waiting for "+
                           "the trailing fields to be read, minus those that "+
                           "finish.", connections);
        
        
        // outbound connections

        d.registerContinuous("connectingTime", d.MINUTE,
                             "The amount of time it takes to establish a " +
                             "new connection fully. In ms.", outConn);
        d.registerContinuous("socketTime", d.MINUTE,
                             "The amount of time it takes to open a socket " +
                             "to other nodes. In ms.", outConn);
        d.registerContinuous("authorizeTime", d.MINUTE, 
                             "The amount of time it takes to authorize new " +
                             "connections. In ms.", outConn);
        d.registerBinomial("connectionRatio", d.MINUTE,
                           "The successrate of new connections.", outConn);

        // inbound connections

        d.registerCounting("incomingConnections", d.MINUTE,
                           "The number of incoming connections.", inConn);
        d.registerCounting("inboundConnectionsDispatched", d.MINUTE,
                           "The number of inbound connection dispatch attempts"
                           + " via PublicInterfaces.",
                           inConn);
        d.registerCounting("inboundConnectionsConnLimitRej", d.MINUTE,
                           "The number of inbound connection's rejected " + 
                           "because of connection limit.", inConn);
        d.registerCounting("inboundConnectionsThreadLimitRej", d.MINUTE,
                           "The  number of inbound connection's rejected " + 
                           "because of thread limit.", inConn);
        d.registerCounting("inboundConnectionsAccepted", d.MINUTE,
                           "The  number of inbound connection's accepted.",
                           inConn);
        
        // messages

        d.registerContinuous("hopTime", d.MINUTE,
                             "The time taken per hop on requests that " +
                             "reach termination. Note that the values are " +
                             "means of the remaining hops, so deviation will "
                             + "be incorrect. In ms.", messages);
        // FIXME:  need to think about this:
        //         is there a good way to capture statistics on this?
        //d.registerContinuous("storeDataTime", d.HOUR,
        //                     "The amount time each piece of data is kept in" 
        //                     + " the cache.");
        //
        // ^^^ this was supposed to be the amount of time to receive the
        //     StoreData message after the data is received

        d.registerCounting("liveChains", d.HOUR,
                           "The number of request chains started minus " +
                           "the number completed.", messages);
        d.registerCounting("incomingRequests", d.MINUTE,
                           "The number of DataRequest queries received.", messages);
        d.registerCounting("incomingInserts", d.MINUTE,
                           "The number of InsertRequest queries received.",
                           messages);
        d.registerCounting("lostRequestState", d.HOUR,
                           "The number of live, pending requests that " +
                           "were dropped due to resource limits, not " +
                           "counting those awaiting the final StoreData",
                           messages);
        d.registerCounting("lostAwaitingStoreData", d.HOUR,
                           "The number of live, pending requests that were " +
                           "dropped due to resource limits, but were in " +
                           "the final phase of waiting for the StoreData",
                           messages);
        d.registerContinuous("timeBetweenFailedRequests", d.MINUTE,
                             "The time between hits to the same entry " +
                             "in the table of recently failed keys.",
                             messages);
        d.registerContinuous("failureTableBlocks", d.MINUTE,
                             "The number of requests that are blocked by " +
                             "each entry on the failure table.",
                             messages);

        // FIXME:  use a Binomial
        d.registerCounting("inboundAggregateRequests", d.MINUTE,
                           "The number of incoming queries of all types.",
                           messages);
        d.registerCounting("inboundAggregateRequestsHandled", d.MINUTE,
                           "The number of incoming queries that are not " + 
                           "rejected", messages);
        
        d.registerBinomial("outboundAggregateRequests", d.MINUTE,
                           "The number of outbound queries of all types.",
                           messages);

        d.registerCounting("inboundClientRequests", d.MINUTE,
                          "The number of client (FCP, InternalClient for fproxy) " +
                          "data and insert requests." ,
                           messages);

        d.registerCounting("announcedTo", d.HOUR,
                           "The of other nodes that the node was successfully "
                           + "announced to.", messages);


        
        // threading

        d.registerCounting("jobsExecuted", d.MINUTE,
                           "The number of jobs executed by the threadpool",
                           threading);
        d.registerCounting("overflowThreads", d.MINUTE,
                           "The number of overflow threads spawned " +
                           "by the threadpool", threading);
        d.registerCounting("insufficientThreads", d.MINUTE,
                           "The number of times the threadpool rejected " +
                           "a job due to thread scarcity", threading);
        d.registerContinuous("tickerDelay", d.MINUTE,
                             "The delay between the time an MO is scheduled " +
                             "to execute on the ticker and the time it " +
                             "actually starts executing, in milliseconds.  " +
                             "It's a very rough measurement, but large " +
                             "values are an indicator of either a bug or a " +
                             "very heavily overloaded node.", threading);
        
        Core.autoPoll = new AutoPoll(d, logger);
        Core.diagnostics = d;
    }
    
    
    private static void initMessageHandler(MessageHandler mh, Params params) {
        // FNP messages
        mh.addType( FNPRawMessage.class, 
                    VoidMessage.messageName, 
                    VoidMessage.class);
        mh.addType( FNPRawMessage.class, 
                    DataRequest.messageName,    
                    DataRequest.class    );
        mh.addType( FNPRawMessage.class, 
                    DataReply.messageName,      
                    DataReply.class      );
        mh.addType( FNPRawMessage.class, 
                    freenet.message.DataNotFound.messageName,   
                    freenet.message.DataNotFound.class );
        mh.addType( FNPRawMessage.class, 
                    QueryRejected.messageName,  
                    QueryRejected.class  );
        mh.addType( FNPRawMessage.class, 
                    QueryAborted.messageName,   
                    QueryAborted.class   );
        mh.addType( FNPRawMessage.class, 
                    QueryRestarted.messageName, 
                    QueryRestarted.class );
        mh.addType( FNPRawMessage.class, 
                    StoreData.messageName,      
                    StoreData.class      );
        mh.addType( FNPRawMessage.class, 
                    InsertRequest.messageName,  
                    InsertRequest.class  );
        mh.addType( FNPRawMessage.class, 
                    InsertReply.messageName,    
                    InsertReply.class    );
        mh.addType( FNPRawMessage.class, 
                    Accepted.messageName,       
                    Accepted.class       );
        mh.addType( FNPRawMessage.class, 
                    DataInsert.messageName,     
                    DataInsert.class     );
        mh.addType( FNPRawMessage.class, 
                    NodeAnnouncement.messageName,     
                    NodeAnnouncement.class     );
        mh.addType( FNPRawMessage.class,
                    AnnouncementReply.messageName,
                    AnnouncementReply.class    );
        mh.addType( FNPRawMessage.class,
                    AnnouncementExecute.messageName,
                    AnnouncementExecute.class  );
        mh.addType( FNPRawMessage.class,
                    AnnouncementComplete.messageName,
                    AnnouncementComplete.class );
        mh.addType( FNPRawMessage.class,
                    AnnouncementFailed.messageName,
                    AnnouncementFailed.class );
   
        // FCP messages
        mh.addType( FCPRawMessage.class, ClientHello.messageName,     ClientHello.class     );
        mh.addType( FCPRawMessage.class, GenerateSVKPair.messageName, GenerateSVKPair.class );
        mh.addType( FCPRawMessage.class, GenerateCHK.messageName,     GenerateCHK.class     );
        mh.addType( FCPRawMessage.class, ClientGet.messageName,       ClientGet.class       );
        mh.addType( FCPRawMessage.class, ClientPut.messageName,       ClientPut.class       );
        mh.addType( FCPRawMessage.class, GetDiagnostics.messageName,  GetDiagnostics.class  );
        mh.addType( FCPRawMessage.class,                              Illegal.class         );

        // additional messages given in the configuration
        String[] messageList = params.getList("messageTypes");
        if (messageList != null) {
            for (int i=0; i+2 < messageList.length; i+=3) {
                try {
                    mh.addType(
                        Class.forName(messageList[i]),
                        messageList[i+1],
                        Class.forName(messageList[i+2])
                    );
                }
                catch (ClassNotFoundException e) {
                    Core.logger.log(Main.class, 
                                    "Cannot register message type", 
                                    e, Logger.ERROR);
                }
            }
        }
    }
    


    private static void copyStream(InputStream in, OutputStream out, long length)
                                                                throws IOException {
        byte[] buf = new byte[Core.blockSize];
        while (length > 0) {
            int n = in.read(buf, 0, (int) Math.min(length, buf.length));
            if (n == -1) throw new EOFException();
            out.write(buf, 0, n);
            length -= n;
        }
    }


    private static final void copyBuffer(Buffer src, Buffer dst) throws IOException {
        copyStream(src.getInputStream(), dst.getOutputStream(), src.length());
    }



    
    /**
     * Write a FieldSet to a file or stdout ("-")
     */
    private static void writeFieldSet(FieldSet fs, String file) throws IOException {
        if (file.equals("-")) {
            fs.writeFields(new WriteOutputStream(System.out));
        }
        else {
            FileOutputStream out = new FileOutputStream(file);
            try {
                fs.writeFields(new WriteOutputStream(out));
            }
            finally {
                out.close();
            }
        }
    }

    
    /**
     * Read NodeReferences from a file or stdin ("-")
     */
    private static NodeReference[] readSeedNodes(String file) throws IOException {
        InputStream in = (file.equals("-") ? System.in : new FileInputStream(file));
        try {
            ReadInputStream rin = new CommentedReadInputStream(in, "#");
            Vector v = new Vector();
            synchronized (v) {
                try {
                    while (true) {
                        //FieldSet fs = new FieldSet(rin);
                        // FIXME:
                        // shouldn't the FieldSet constructor and the parseFields
                        // methods throw the EOFException instead of eating it?
                        
                        FieldSet fs = new FieldSet();
                        if (null == fs.parseFields(rin))
                            throw new EOFException();
                        
                        try {
                            v.addElement(new NodeReference(fs));
                        }
                        catch (BadReferenceException e) {
                            Core.logger.log(Main.class,
                                "Skipped bad NodeReference while reading seed nodes",
                                e, Logger.ERROR);
                        }
                    }
                }
                catch (EOFException e) {}
                
                NodeReference[] ret = new NodeReference[v.size()];
                v.copyInto(ret);
                return ret;
            }
        }
        finally {
            in.close();
        }
    }


    /**
     * Crams a set of node refs into the routing table with fake keys.
     * The fake keys end in 0x0000.
     * @param force  true means to add a ref even if the node already
     *               exists in the routing table
     */
    private static void seedRoutingTable(RoutingTable rt,
                                         NodeReference[] nodes,
                                         boolean force) {
        for (int i=0; i<nodes.length; ++i) {
            if (force || !rt.references(nodes[i].getIdentity())) {
                int r = Core.randSource.nextInt();
                int c = nodes[i].hashCode();
                byte[] k = new byte[18];
                for (int j=0; j<16; ++j)
                    k[j] = (byte) (0xff & ( r>>(8*(j/4)) ^ c>>(8*(j%4)) ));
                rt.reference(new Key(k), nodes[i]);
            }
        }
    }
    

    /**
     * Print version information
     */
    public static void version() {
        System.out.println(
            Version.nodeName + " version " + Version.nodeVersion
            +", protocol version "         + Version.protocolVersion
            +" (build "                    + Version.buildNumber
            +", last good build "          + Version.lastGoodBuild
            +")"
        );
    }

    /**
     * Print usage information
     */
    public static void usage() {
        version();
        System.out.println("Usage: java "+Main.class.getName()+" [options]");
        System.out.println("");
        System.out.println("Configurable options");
        System.out.println("--------------------");
        Node.config.printUsage(System.out);
        System.out.println("");
        System.out.println("Command-line switches");
        System.out.println("---------------------");
        switches.printUsage(System.out);
        System.out.println("");
        System.out.println("Send support requests to support@freenetproject.org.");
        System.out.println("Bug reports go to devl@freenetproject.org.");
    }

    /**
     * Print HTML manual
     */
    public static void manual() {
        manual(new PrintWriter(System.out));
    }

    public static void manual(PrintWriter out) {
        out.println("<html><body>");
        out.println("<br><br>");
        out.println("<h2>Freenet Reference Daemon Documentation</h2>");
        out.println("<h3>" + Config.htmlEnc(Version.getVersionString()) +
                    "</h3>");
        out.println("<br>");
        java.text.DateFormat df = java.text.DateFormat.getDateTimeInstance();
        out.println("<i>(This manual was automatically generated on " + 
                    Config.htmlEnc(df.format(new Date()))
                    + ". If you have updated Freenet since then, you " +
                    "may wish regenerate it.)</i>");
        out.println("<br><br>");
        out.println("FRED (Freenet REference Daemon) is the standard " +
                    "implementation of Freenet. This is the node, which " +
                    "serves as a router, data cache, and personal gateway " +
                    "all rolled into one. For FRED to run, it requires a " +
                    "configuration file to be present - this can be created " +
                    "either during the installation, by starting the node " +
                    "with the --config switch (see below), or running the "+
                    "freenet.config.Setup class manually.");
        out.println("<br><br>");
        out.println("See the <a href=\"http://www.freenetproject.org/"
                    + "index.php?page=documentation\"> project documentation" +
                    " pages</a> for more information, or ask pointed & " +
                    " specific questions on the <a href=\"" +
                    "http://www.freenetproject.org/index.php?page=lists\">" +
                    "mailing lists</a>.");
        out.println("<br><br>");
        out.println("<br>");
        out.println("<h3>Command line switches: </h3>");
        out.println("<hr>");
        switches.printManual(out);
        out.println("<h3>Configuration options: </h3>");
        out.println("These can reside either in the configuration file " +
                    "or be given as command line arguments.");
        out.println("<hr>");
        Core.config.printManual(out);
        out.println("</body></html>");
    }
}


