package freenet.interfaces;

import freenet.*;
import freenet.thread.ThreadFactory;
import freenet.transport.TCP;
import freenet.support.*;
import java.net.InetAddress;

/**
 * A LocalInterface is connected to by clients
 * and can do allowed hosts checking.
 */
public class LocalInterface extends Interface {

    protected ThreadFactory tf;
    protected ConnectionRunner runner;
    protected Address[] allowedHosts;

    /**
     * Builds a LocalInterface from a FieldSet using the entries:
     *      port, allowedHosts, bindAddress
     * 
     * @throws InterfaceException  if the FieldSet was unusable
     * @throws ListenException     if the new Interface couldn't bind
     *                             to its listening address
     * 
     * TODO - transport independence
     *      - allowed hosts wildcards
     */
    public static LocalInterface make(FieldSet fs, ThreadFactory tf,
                                      ConnectionRunner runner)
                        throws InterfaceException, ListenException {
        
        String port, bindAddress;
        String allowedHosts = null;
        Address[] allowedHostsAddr = null;
        
        port = fs.get("port");
        if (port == null || port.trim().equals("")) 
            throw new InterfaceException("No port given for interface");
    
        allowedHosts = fs.get("allowedHosts");
        if (allowedHosts != null && allowedHosts.trim().equals(""))
            allowedHosts = null;

        bindAddress = fs.get("bindAddress");
        if (bindAddress == null || bindAddress.trim().equals("")) {
            // if no bindAddress was specified, we bind to loopback
            // only unless allowedHosts was specified, in which case
            // we bind to all addresses
            bindAddress = (allowedHosts == null ? "127.0.0.1" : null);
        }

        // treat * as a blank, _after_ making the bindAddress decision
        if (allowedHosts != null && allowedHosts.trim().equals("*"))
            allowedHosts = null;

        Transport tcp;
        try {
            // "*" allows all connections.  e.g. for HTTP gateways.
            if (bindAddress == null || bindAddress.trim().equals("*"))
                tcp = new TCP(1);
            else
                tcp = new TCP(InetAddress.getByName(bindAddress.trim()), 1);
        }
        catch (Exception e) {
            throw new InterfaceException(""+e);
        }

        try {
            if (allowedHosts != null) {
                String[] hosts = Fields.commaList(allowedHosts);
                allowedHostsAddr = new Address[hosts.length];
                for (int i=0; i<hosts.length; ++i)
                    allowedHostsAddr[i] = tcp.getAddress(hosts[i]+":0");
            }
            return new LocalInterface(tcp.getListeningAddress(port.trim()),
                                      tf, runner, allowedHostsAddr);
        }
        catch (BadAddressException e) {
            throw new InterfaceException(e.getMessage());
        }
    }
    
    /**
     * @param tm      the ThreadManager for connection threads
     * @param runner  handles the connection thread
     */
    public LocalInterface(ListeningAddress listenAddr, ThreadFactory tf,
                          ConnectionRunner runner) throws ListenException {
        this(listenAddr, tf, runner, null);
    }
    
    /**
     * @param allowedHosts  set of Addresses to do an equalsHost() check with;
     *                      null means to allow all hosts
     */
    public LocalInterface(ListeningAddress listenAddr, ThreadFactory tf,
                          ConnectionRunner runner, Address[] allowedHosts)
                                                throws ListenException {
        super(listenAddr);
        this.tf = tf;
        this.runner = runner;
        this.allowedHosts = allowedHosts;
    }
    
    protected void dispatch(Connection conn) throws RejectedConnectionException {
        if (allowedHosts != null) {
            Address ha = conn.getPeerAddress();
            int i = 0;
            while (i<allowedHosts.length) {
                if (allowedHosts[i].equalsHost(ha)) break;
                ++i;
            }
            if (i == allowedHosts.length)
                throw new RejectedConnectionException("host not allowed: "+ha);
        }
        ConnectionShell shell = new ConnectionShell(conn);
        tf.getThread(shell, true).start();
    }
    
    protected class ConnectionShell implements Runnable {
        protected final Connection conn;
        protected ConnectionShell(Connection conn) {
            this.conn = conn;
        }
        public void run() {
            try {
                runner.handle(conn);
            }
            catch (Throwable t) {
                Core.logger.log(LocalInterface.this,
                    "Unhandled throwable while handling connection",
                    t, Logger.NORMAL);
                conn.close();
            }
        }
    }
}






