package freenet.fs.dir;

import freenet.fs.acct.*;
import freenet.fs.acct.Fragment.ComparatorBySize;
import freenet.fs.acct.Fragment.ComparatorByPos;
import freenet.fs.acct.sys.*;
import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.BinaryTree.Node;
import java.io.*;

/**
 * Maintains a set of size-ordered fragments for finding best fits
 * when allocating unused fragments.
 *
 * The entries in this map are free fragments.  It is like a photo
 * negative of the FragmentPositionMap (with different ordering).
 * 
 * @see FragmentPositionMap
 * @author tavin
 */
final class FragmentSizeMap extends FragmentMap {
    
    FragmentSizeMap(AccountingProcess proc, Cache cache) {
        super(proc, new FragmentSize.Marshal(), cache);
    }

    
    /**
     * @return  the size of the largest fragment in the map,
     *          or zero if the map is empty
     */
    final long largest() {
        Node n = treeMax();
        return n == null ? 0 : ((Fragment) n.getObject()).size();
    }

    
    /**
     * @return  true, if the entry was inserted (no collision)
     */
    final boolean put(Fragment f) {
        return null == treeInsert(
            new AccountingTreeNode(new FragmentSize(f)), false);
    }

    /**
     * @return  true, if an entry was removed
     */
    final boolean remove(Fragment f) {
        return null != treeRemove(new FragmentSize(f));
    }

    /**
     * @return  true, if there is a matching entry
     */
    final boolean contains(Fragment f) {
        return null != treeSearch(new FragmentSize(f));
    }
    

    /**
     * Removes the smallest fragment at least as large as the requested size,
     * or failing that the largest fragment still at least as large as the
     * minimum size, or failing that, does nothing.
     * @param size     the size to request as lower bound
     * @param minSize  the absolute lower bound
     */
    final Fragment remove(long size, long minSize) {
        Node n = treeMatch(new FragmentSize(size), 1);
        if (n != null) {
            Fragment f = (Fragment) n.getObject();
            if (f.size() >= minSize) {
                treeRemove(n);
                return f;
            }
        }
        return null;
    }


    private static final class FragmentSize extends FragmentRecord {

        private static final class Marshal implements AccountingTreeMarshal {
            public final Comparable readEntry(DataInput din, int len)
                                                throws IOException {
                return new FragmentSize(din.readLong(), din.readLong());
            }
            public final int getEntryLength(Comparable entry) {
                return 16;
            }
            public final void writeEntry(Comparable entry, DataOutput out)
                                                        throws IOException {
                FragmentSize f = (FragmentSize) entry;
                out.writeLong(f.getLowerBound());
                out.writeLong(f.getUpperBound());
            }
        }

        FragmentSize(Fragment f) {
            super(f);
        }
        
        FragmentSize(long lo, long hi) {
            super(lo, hi);
        }

        FragmentSize(long size) {
            super(size);
        }
        
        public final int compareTo(Object o) {
            return compareTo((FragmentSize) o);
        }

        public final int compareTo(FragmentSize f) {
            int cmp = ComparatorBySize.compare(this, f);
            return cmp == 0 ? ComparatorByPos.compare(this, f) : cmp;
        }
    }
}


