package freenet.fs.dir;

import freenet.fs.acct.Fragment;
import freenet.support.BinaryTree.Node;
import java.util.Vector;

/**
 * Maintains a map of free and allocated Fragments.
 * Allocates and reclaims space.
 * @author tavin
 */
class FragmentManager {

    private final FragmentSizeMap fsizeMap;
    private final FragmentPositionMap fposMap;

    
    /**
     * @param fsizeMap  the map of free fragments by size
     * @param fposMap   the map of allocated fragments by position
     */
    FragmentManager(FragmentSizeMap fsizeMap, FragmentPositionMap fposMap) {
        this.fsizeMap = fsizeMap;
        this.fposMap = fposMap;
    }

    
    /**
     * @return  the sum of the sizes of the free fragments
     */
    final long available() {
        return fsizeMap.total();
    }

    /**
     * @return  the size of the largest free fragment
     */
    final long largest() {
        return fsizeMap.largest();
    }


    /**
     * Removes a specific Fragment from free space.  There must be
     * a free Fragment that contains the requested Fragment.
     * @param range  the requested Fragment
     * @param ticketID  the ticket ID that will own the allocation
     * @return false, if the fragment is not free
     */
    boolean allocate(Fragment range, long ticketID) {

        Fragment f = fposMap.allocate(range, ticketID);
        if (f == null)
            return false;
        
        fsizeMap.remove(f);

        if (range.getLowerBound() > f.getLowerBound()) {
            Fragment rem = new Fragment(f.getLowerBound(), range.getLowerBound() - 1);
            fsizeMap.put(rem);
        }
        if (range.getUpperBound() < f.getUpperBound()) {
            Fragment rem = new Fragment(range.getUpperBound() + 1, f.getUpperBound());
            fsizeMap.put(rem);
        }

        return true;
    }


    /**
     * Attempt to allocate a single Fragment with size in [minSize, maxSize].
     * @param maxSize   largest Fragment to return
     * @param minSize   smallest Fragment to return
     * @param ticketID  the ticket ID that will own the allocation
     * @return  the allocated Fragment,
     *          or null if there was no suitable Fragment found
     */
    Fragment allocate(long maxSize, long minSize, long ticketID) {

        Fragment f = fsizeMap.remove(maxSize, minSize);
        if (f == null)
            return null;

        Fragment alloc1 = new Fragment(f.getLowerBound(),
                                       Math.min(f.getUpperBound(),
                                                f.getLowerBound() + maxSize - 1));

        Fragment alloc2 = fposMap.allocate(alloc1, ticketID);
        if (alloc2 == null || !alloc2.equals(f)) {
            throw new DirectoryException("fragment size/position map inconsistency: "
                                         + " (size) " + f + " (pos) " + alloc2);
        }

        if (f.size() > maxSize) {
            // restore remainder to the size-ordered free map
            Fragment rem = new Fragment(f.getLowerBound() + maxSize,
                                        f.getUpperBound());
            fsizeMap.put(rem);
        }

        return alloc1;
    }

    /**
     * Attempts to allocate a set of Fragments totalling the requested size.
     * Stops when there is no more free space.
     * @return  the allocations as an array of Fragments
     *          (may be an empty array, but not null)
     */
    Fragment[] allocate(long size, long ticketID) {
        synchronized (allocVec) {
            try {
                Fragment f;
                // NOTE- should there be a larger than 1-byte minimum fragment size?
                while (size > 0 && null != (f = allocate(size, 1, ticketID))) {
                    size -= f.size();
                    allocVec.addElement(f);
                }
                Fragment[] ret = new Fragment[allocVec.size()];
                allocVec.copyInto(ret);
                return ret;
            }
            finally {
                allocVec.removeAllElements();
            }
        }
    }

    private final Vector allocVec = new Vector();

    
    /**
     * Returns a set of Fragments to free space.
     */
    final void free(Fragment[] f) {
        for (int i=0; i<f.length; ++i)
            free(f[i]);
    }

    /**
     * Returns a given Fragment to free space.
     * @param f  the Fragment to free
     * @return false, if the fragment was not allocated
     */
    boolean free(Fragment f) {

        Fragment fcont = fposMap.free(f);
        if (fcont == null)
            return false;
        
        // coalesce adjacent free fragments
        if (fcont.getLowerBound() < f.getLowerBound())
            fsizeMap.remove(new Fragment(fcont.getLowerBound(), f.getLowerBound() - 1));
        if (fcont.getUpperBound() > f.getUpperBound())
            fsizeMap.remove(new Fragment(f.getUpperBound() + 1, fcont.getUpperBound()));

        fsizeMap.put(fcont);
        
        return true;
    }
}



