package freenet.fs.acct.sys;

import freenet.fs.acct.*;
import freenet.fs.acct.fsck.*;
import freenet.support.Comparable;
import freenet.support.Fields.ByteArrayComparator;
import freenet.support.*;
import freenet.support.sort.*;
import java.io.*;
import java.util.Vector;


/**
 * Performs a syntactic check of the data that has been recorded
 * for a given accounting tree.
 * @author tavin
 */
public class AccountingTreeCheck implements FaultAnalysis, AccountingStruct {

    private final String desc;
    
    private final AccountingProcess proc;
    
    private final Vector faults = new Vector();


    public AccountingTreeCheck(AccountingProcess proc) {
        this(proc, null);
    }

    public AccountingTreeCheck(AccountingProcess proc, String desc) {
        this.proc = proc;
        this.desc = desc;
    }


    /**
     * @return  a descriptive name for the checked structure,
     *          or null if one wasn't given
     */
    public final String getDescription() {
        return desc;
    }


    public final boolean hasFaults() {
        return !faults.isEmpty();
    }

    
    private int faultIndex = 0;

    public final Fault getNextFault() {
        return faultIndex < faults.size()
               ? (Fault) faults.elementAt(faultIndex++)
               : null; 
    }


    /**
     * Throws an IllegalStateException because
     * all inconsistencies at this level are FATAL.
     */
    public final void commitFixes() {
        throw new IllegalStateException();
    }
    
    
    
    public final void found(int bnum, DataInput din) throws IOException {
        check(bnum, din, null);
    }

    public final void found(int bnum, DataInput din, BlockTransaction btx)
                                                    throws IOException {
        check(bnum, din, btx);
    }

    public void found(BlockTransaction btx) throws IOException {
        check(-1, null, btx);
    }


    private final Vector records = new Vector();

    private void check(int bnum, DataInput din, BlockTransaction btx)
                                                throws IOException {
        synchronized (faults) {
        synchronized (records) {
            try {
                if (din != null)
                    check(bnum, -1, din);

                if (btx != null)
                    check(bnum, btx.getTransactionID(),
                          proc.resume(btx, VoidBlockEntry.instance));

                if (records.size() > 1) {
                    for (int i=0; i<records.size(); ++i) {
                        byte[] data = (byte[]) records.elementAt(i);
                        records.setElementAt(new STData(data), i);
                    }
                    QuickSorter.quickSort(new VectorSorter(records));
                    STData stdata = (STData) records.firstElement();
                    for (int i=1; i<records.size(); ++i) {
                        STData stdata2 = (STData) records.elementAt(i);
                        if (stdata.data != null && stdata.compareTo(stdata2) == 0)
                            error(bnum, "duplicated record: "+stdata);
                        stdata = stdata2;
                    }
                }
            }
            finally {
                records.removeAllElements();
            }
        }}
    }
    
    private void check(int bnum, int txid, DataInput din) throws IOException {
        int cmd;
        int cmdno = -1;
        while (true) {
            try {
                cmd = din.readUnsignedShort();
                if (cmd == SerialTree.STOP)
                    return;
            }
            catch (EOFException e) {
                return;
            }

            ++cmdno;
            
            try {
                if (cmd == SerialTree.STORE) {
                    int num = din.readUnsignedShort();
                    if (records.size() < num + 1) {
                        records.setSize(num + 1);
                    }
                    int len = din.readUnsignedShort();
                    if (len == 0) {
                        error(bnum, txid, cmdno, "empty STORE #"+num);
                    }
                    if (records.elementAt(num) == null) {
                        byte[] data = new byte[len];
                        din.readFully(data);
                        records.setElementAt(data, num);
                    }
                    else {
                        error(bnum, txid, cmdno, "duplicate STORE #"+num);
                    }
                }
                else if (cmd == SerialTree.DELETE) {
                    int num = din.readUnsignedShort();
                    if (records.size() <= num || records.elementAt(num) == null) {
                        error(bnum, txid, cmdno, "gratuitous DELETE #"+num);
                    }
                    else {
                        records.setElementAt(null, num);
                    }
                }
                else {
                    error(bnum, txid, cmdno,
                          "bad command code: 0x"+Integer.toHexString(cmd));
                    return;
                }
            }
            catch (EOFException e) {
                error(bnum, txid, cmdno, "premature EOF");
            }
        }
    }


    private void error(int bnum, String msg) {
        faults.addElement(new STFault(bnum + ": " + msg));
    }

    private void error(int bnum, int txid, int cmdno, String msg) {
        synchronized (errbuf) {
            try {
                errbuf.append(bnum);
                if (txid != -1) {
                    errbuf.append(" tx ");
                    errbuf.append(txid);
                }
                errbuf.append(" cmd ");
                errbuf.append(cmdno);
                errbuf.append(": ");
                errbuf.append(msg);
                faults.addElement(new STFault(errbuf.toString()));
            }
            finally {
                errbuf.setLength(0);
            }
        }
    }

    private final StringBuffer errbuf = new StringBuffer();


    private static final class STFault implements Fault {
    
        private final String desc;
        
        STFault(String desc) {
            this.desc = desc;
        }
        
        public final String getDescription() {
            return desc;
        }
    
        public final int getSeverity() {
            return FATAL;
        }
    
        public final void fix() {
            throw new IllegalStateException();
        }
    }


    private static final class STData implements Comparable {
        
        private final byte[] data;
        
        STData(byte[] data) {
            this.data = data;
        }

        public final int compareTo(Object o) {
            return compareTo((STData) o);
        }

        public final int compareTo(STData o) {
            if (data == null)
                return o.data == null ? 0 : -1;
            else
                return o.data == null ? 1 : ByteArrayComparator.compare(data, o.data);
        }

        public final String toString() {
            return data == null ? "(empty)" : Fields.bytesToHex(data);
        }
    }
}


