package freenet.fs.acct;

import freenet.support.*;
import java.util.*;
import java.io.*;

/**
 * An AccountingProcess for use with a single data structure.
 * Do NOT use this in anything but a single threaded context!
 * @author tavin
 */
public class SingleAccountingProcess implements AccountingProcess {

    private AccountingTable acct;
    private int useCount;

    private final BlockList freeBlocks;
    private final BlockList controlBlocks;

    private final BlockList retireQueue = new BlockList();

    private final Hashtable txTable = new Hashtable();
    private int lastID = 0;

    private final OrderedVector btxFlushQueue =
        new OrderedVector(new BlockTransaction.ComparatorByAnnotationLength());
    
    private boolean mustFreeze = false;

    
    /**
     * @param init  the initialization scan
     */
    public SingleAccountingProcess(AccountingInitializer init) {
        acct = init.getAccountingTable();
        freeBlocks = init.getFreeBlocks();
        controlBlocks = init.getControlBlocks();
        useCount = acct.getBlockCount() - freeBlocks.count() - controlBlocks.count();
    }

    /**
     * @return  a summary of the internal state for debugging use
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("total block count: ").append(getBlockCount()).append('\n');
        sb.append("free count: ").append(getFreeCount()).append('\n');
        sb.append("use count: ").append(getUseCount()).append('\n');
        sb.append("free blocks: ").append(freeBlocks).append('\n');
        sb.append("control blocks: ").append(controlBlocks).append('\n');
        sb.append("transactions: ").append(txTable.size()).append('\n');
        sb.append("queued to retire: ").append(retireQueue).append('\n');
        return sb.toString();
    }
    

    public final SharedAccountingProcess share(int tag) {
        return new SharedAccountingProcess(tag, this);
    }

    public final void setAccountingTable(AccountingTable acct) {
        int blockCount = getBlockCount();
        this.acct = acct;
        while (blockCount < acct.getBlockCount())
            freeBlocks.push(blockCount++);
    }

    public final AccountingTable getAccountingTable() {
        return acct;
    }

    private final int getControlDataWidth() {
        return getDataWidth() - ControlBlock.CONTROL_HEADER_WIDTH;
    }

    public final int getDataWidth() {
        return acct.getDataWidth() - ControlBlock.CONTROL_TYPE_WIDTH;
    }

    public final int getBlockWidth() {
        return acct.getBlockWidth();
    }

    public final int getBlockCount() {
        return acct.getBlockCount();
    }

    public final int getUseCount() {
        return useCount;
    }

    public final int getFreeCount() {
        //while (blockCount < acct.getBlockCount())
        //    freeBlocks.push(blockCount++);
        return freeBlocks.count() - txTable.size() - 1;
    }

    public final BlockTransaction create(BlockEntry be) {
        //if (txTable.size() >= getFreeCount())
        //    throw new AccountingException("no free slots left");
        BlockTransaction btx = new BlockTransaction(-1, ++lastID);
        txTable.put(btx, be);
        ++useCount;
        return btx;
    }
    
    public final BlockTransaction modify(int bnum, BlockEntry be) {
        //if (txTable.size() >= getFreeCount())
        //    throw new AccountingException("no free slots left");
        BlockTransaction btx = new BlockTransaction(bnum, ++lastID);
        txTable.put(btx, be);
        return btx;
    }

    public DataInput resume(BlockTransaction btx, BlockEntry be) {
        //if (txTable.size() >= getFreeCount())
        //    throw new AccountingException("no free slots left");
        if (lastID < btx.getTransactionID())
            lastID = btx.getTransactionID();
        txTable.put(btx, be);
        if (btx.getBlockNumber() == -1)
            ++useCount;
        
        try {
            return btx.readAnnotation();
        }
        finally {
            btx.resetAnnotation();
        }
    }

    public final void abandon(BlockTransaction btx) {
        if (controlBlocks.count() > 0)
            mustFreeze = true;
        txTable.remove(btx);
        if (btx.getBlockNumber() == -1)
            --useCount;
    }
    
    public final DataInput load(int bnum) throws IOException {
        DataInput bin = acct.readBlock(bnum);
        return bin == null || bin.readUnsignedShort() != 0 ? null : bin;
    }
    
    public final void retire(int bnum) {
        mustFreeze = true;
        retireQueue.push(bnum);
        --useCount;
    }

    /**
     * This may end up doing a freeze() anyway, but otherwise it will
     * just collect the new annotations and write just one block
     * (or however many are needed to contain those annotations).
     */
    public void flush() throws IOException {
        if (getFreeCount() < 0) {
            throw new AccountingException("not enough space left in table");
        }
        if (mustFreeze) {
            freeze();
            return;
        }
        if (txTable.isEmpty()) {
            return;
        }
        synchronized (btxFlushQueue) {  // just do it once.. silly java
            try {
                Enumeration bte = txTable.keys();
                do {
                    btxFlushQueue.insert(bte.nextElement());
                }
                while (bte.hasMoreElements());
                
                BlockTransaction btx = (BlockTransaction) btxFlushQueue.lastElement();
                if (btx.getRecordLength() > getControlDataWidth()-2) {
                    // an annotation so fat it can't fit inside one block, yeesh
                    freeze();
                    return;
                }
                int i = btxFlushQueue.size();
                btxFlushQueue.removeElementAt(--i);
                
                int abCount = 1;
                AnnotationBlock ab = new AnnotationBlock(freeBlocks.pop(),
                                                         controlBlocks.peek());
                controlBlocks.push(ab.getBlockNumber());
                ab.add(btx);

                while (i-- > 0) {
                    // find an annotation record that can fit
                    // in the space left in the annotation block
                    int j = i;
                    do {
                        btx = (BlockTransaction) btxFlushQueue.elementAt(j);
                        if (ab.getBodySize() + btx.getRecordLength() <= getControlDataWidth()) {
                            btxFlushQueue.removeElementAt(j);
                            break;
                        }
                    }
                    while (--j >= 0);

                    // if there wasn't one, start a new annotation block
                    // and grab the biggest annotation record
                    if (j == -1) {
                        if (getFreeCount() < 1) {
                            // oops, ran out of empty slots, must freeze
                            while (abCount-- > 0)
                                freeBlocks.push(controlBlocks.pop());
                            freeze();
                            return;
                        }
                        ++abCount;
                        ab = new AnnotationBlock(freeBlocks.pop(), ab);
                        controlBlocks.push(ab.getBlockNumber());
                        btx = (BlockTransaction) btxFlushQueue.lastElement();
                        btxFlushQueue.removeElementAt(i);
                    }
                    ab.add(btx);
                }

                // now we have the annotation blocks ready to go,
                // and we know how many there are

                // rule of thumb: if doing a full freeze would
                // free up blocks, then do it.
                if (abCount + controlBlocks.count() > txTable.size()) {
                    while (abCount-- > 0)
                        freeBlocks.push(controlBlocks.pop());
                    freeze();
                    return;
                }

                ControlBlock cb = ab;
                do {
                    DataOutputStream out = acct.writeBlock(cb.getBlockNumber());
                    cb.writeTo(out);
                    out.close();
                    cb = cb.next;
                }
                while (cb != null);
            }
            finally {
                btxFlushQueue.removeAllElements();
            }
        }
    }

    /**
     * Freezes all the active BlockEntry objects, deletes all retired
     * blocks and previously written annotation blocks.  Slower than
     * flush() but periodically necessary.
     */
    public void freeze() throws IOException {

        if (txTable.isEmpty() && !mustFreeze)
            return;

        if (getFreeCount() < 0)
            throw new AccountingException("not enough space left in table");

        BlockList oldQueue = new BlockList(retireQueue);

        Enumeration bte = txTable.keys();
        while (bte.hasMoreElements()) {
            BlockTransaction btx = (BlockTransaction) bte.nextElement();
            if (btx.getBlockNumber() != -1)
                oldQueue.push(btx.getBlockNumber());
        }

        int[] oldBlocks = oldQueue.blocks();
        int[] newBlocks = new int[txTable.size()];
        for (int i=0; i<newBlocks.length; ++i)
            newBlocks[i] = freeBlocks.pop();

        SynchronizationBlock bsync =
            new SynchronizationBlock(freeBlocks.pop(), controlBlocks.peek(),
                                     newBlocks, oldBlocks);
        
        DataOutputStream out = acct.writeBlock(bsync.getBlockNumber());
        bsync.writeTo(out);
        out.close();
        
        Enumeration bes = txTable.elements();
        for (int i=0; i<newBlocks.length; ++i) {
            BlockEntry be = (BlockEntry) bes.nextElement();
            out = acct.writeBlock(newBlocks[i]);
            out.writeShort(0);
            be.freeze(newBlocks[i], out);
            out.close();
        }
        for (int i=0; i<oldBlocks.length; ++i) {
            acct.destroyBlock(oldBlocks[i]);
            freeBlocks.push(oldBlocks[i]);
        }

        int[] cblo = controlBlocks.blocks();
        for (int i=0; i<cblo.length; ++i) {
            acct.destroyBlock(cblo[i]);
            freeBlocks.push(cblo[i]);
        }
        
        acct.destroyBlock(bsync.getBlockNumber());
        freeBlocks.push(bsync.getBlockNumber());

        controlBlocks.clear();
        retireQueue.clear();
        txTable.clear();
        mustFreeze = false;
        lastID = 0;
    }
}


