package freenet.fs;

import java.io.*;

/**
 * The Storage is the layer that provides input and output streams
 * over the randomly addressable bytes of whatever underlying
 * storage system is being used (file, disk partition, etc.)
 * Implementations are not responsible for handling any thread
 * synchronization or locking issues.
 * @author tavin
 */
public interface Storage {

    /**
     * @return  the total number of bytes available
     *          (generally should be a fixed constant)
     */
    long size();

    /**
     * Typically a Storage needs to be initialized before it can be used
     * (i.e., filled with random data).  Also, if the size of the Storage
     * is changed this may effectively truncate the valid portion of the
     * Storage, requiring re-initialization beginning at a certain offset.
     * If the return value of this method is less than the size() value,
     * everything beyond the truncated region needs to be initialized,
     * and any data that was previously stored beyond the truncated
     * region must be discarded.
     * 
     * @return  length of the truncated region, in [0, size()]
     */
    long truncation();

    /**
     * @return an input stream that reads the raw bytes in
     *         the underlying storage
     */
    InputStream getInputStream(long start, long end) throws IOException;

    /**
     * @return an output stream that writes directly to the
     *         underlying storage
     *  
     * The output stream returned by this method MUST perform
     * a disk-syncing operation when it is flushed or closed,
     * so that all bytes written are guaranteed to be physically
     * recorded.
     */
    OutputStream getOutputStream(long start, long end) throws IOException;
}



