package freenet.fs;

import freenet.fs.acct.Fragment;
import java.io.*;

/**
 * A ReadLock provides protection against writing
 * to a range that is currently being read from.
 *
 * @author Scott
 * @author tavin
 */
public class ReadLock extends Lock {

    public ReadLock(LockGrantor lg, long lo, long hi) {
        super(lg, lo, hi);
    }

    public final int getTypeMask() {
        return READ;
    }

    public final int getProtectionMask() {
        return WRITE;
    }

    public final String getTypeName() {
        return "READ";
    }

    public final InputStream getStream() throws IOException {
        return new ReadLockInputStream(ticket.getInputStream());
    }
    
    /**
     * Reads from the locked range, staying behind conflicting locks.
     */
    private class ReadLockInputStream extends FilterInputStream {

        ReadLockInputStream(InputStream in) {
            super(in);
        }

        public int read() throws IOException {
            try {
                waitAvailable(1);
                if (slide.slipped())
                    throw new IOException("bad data");
                int rv = in.read();
                if (rv != -1)
                    move(1);
                return rv;
            }
            catch (LockException e) {
                return -1;
            }
        }

        public int read(byte[] buf, int off, int len) throws IOException {
            try {
                long a = waitAvailable(1);
                if (slide.slipped())
                    throw new IOException("bad data");
                int rv = in.read(buf, off, (int) Math.min(len, a));
                if (rv != -1)
                    move(rv);
                return rv;
            }
            catch (LockException e) {
                return -1;
            }
        }
        
        public final int available() {
            return (int) getAvailable();
        }

        public final boolean markSupported() {
            return false;
        }

        public final void mark(int foo) {}

        public final void reset() {}

        public void close() throws IOException {
            try {
                in.close();
            }
            finally {
                unlock();
            }
        }

        protected void finalize() throws Throwable {
            close();
        }
    }

    /**
     * Creates a stream that invisibly handles the locking and unlocking.
     */
    public static InputStream getInputStream(LockGrantor lg, long lo, long hi)
                                                        throws IOException {
        ReadLock rl = new ReadLock(lg, lo, hi);
        boolean worked = false;
        try {
            InputStream ret = rl.getStream();
            worked = true;
            return ret;
        }
        finally {
            if (!worked)
                rl.unlock();
        }
    }


    public static InputStream getInputStream(LockGrantor lg, Fragment[] ranges)
                                                        throws IOException {
        ReadLock[] locks = new ReadLock[ranges.length];
        synchronized (lg.semaphore()) {
            for (int i=0; i<ranges.length; ++i) {
                locks[i] = new ReadLock(lg,
                                        ranges[i].getLowerBound(),
                                        ranges[i].getUpperBound());
            }
        }
        boolean worked = false;
        try {
            InputStream ret = new LockedInputStream(locks);
            worked = true;
            return ret;
        }
        finally {
            if (!worked) {
                for (int i=0; i<locks.length; ++i)
                    locks[i].unlock();
            }
        }
    }
}

    
