package freenet.fs;

import java.io.*; 

/**
 * A LockedInputStream chains together a group of read locks
 * for reading as a single input stream.
 * @author tavin
 */
public class LockedInputStream extends FilterInputStream {

    protected ReadLock[] locks;
    protected int lx = 0;

    protected boolean closed = false;

    
    /**
     * Create a locked input stream that reads up to the end
     * of an array of read locks.
     * @param locks  the array of read locks
     */
    public LockedInputStream(ReadLock[] locks) throws IOException {
        super(locks[0].getStream());
        this.locks = locks;
    }
    

    public int read() throws IOException {
        return fixate() ? in.read() : -1;
    }

    public int read(byte[] buf, int off, int len) throws IOException {
        return fixate()
               ? in.read(buf, off, (int) Math.min(len, locks[lx].getBytesLeft()))
               : -1;
    }

    public void close() throws IOException {
        if (!closed) {
            closed = true;
            try {
                in.close();
            }
            finally {
                while (lx < locks.length)
                    locks[lx++].unlock();
            }
        }
    }

    private boolean fixate() throws IOException {

        if (closed)
            throw new IOException("stream closed");
        else if (lx >= locks.length)
            return false;
        else if (locks[lx].getBytesLeft() > 0)
            return true;

        in.close();
        locks[lx].unlock();
        if (++lx >= locks.length)
            return false;
        in = locks[lx].getStream();
            
        return true;
    }

    protected void finalize() throws Throwable {
        close();
    }
}


