package freenet.fs;

import freenet.support.Fields;
import java.io.*;

/**
 * The LockTicket is an accounting device for Locks.  It is used
 * cooperatively by Locks and the LockGrantor to orchestrate the
 * "sliding locks" behavior.  LockSignals are registered with the
 * LockTicket, so that when the stream associated with the lock
 * advances or unlocks, the registered LockSignals can be notified
 * (thus informing the blocking locks that they can move forward).
 * The LockGrantor also registers a LockSignal so that the lock
 * is removed from the active locks list when it unlocks.
 * @author tavin
 */
public final class LockTicket {

    /** singly linked list element */
    private static final class SignalTuple {
        private SignalTuple next = null;
        private final LockSignal signal;
        private SignalTuple(LockSignal signal) {
            this.signal = signal;
        }
    }

    /** collection of callback signals */
    private final SignalTuple stHead = new SignalTuple(null);

    

    /** type of the lock owning this ticket */
    final int typeMask;
    
    /** the lower bound of the locked range */
    final long lo;
    
    /** the upper bound of the locked range */
    final long hi;

    
    /** value of last position notification, -1 means unlocked */
    volatile long pos;

    // only the thread using the stream will cause the pos value
    // to be changed.  we want it volatile so we can atomically
    // read the value without synchronizing
    
    
    /** true means it was unlocked prematurely */
    volatile boolean failed = false;

    // volatile so the value we read from this is consistent
    // with the pos value we read immediately prior
    
    
    /** storage the ticket is tied to */
    Storage storage;

    // the storage field is filled in by the LockGrantor
    // when the lock is granted
    

    /**
     * LockTickets are created by Locks.
     * @param typeMask  type flag of the owning lock
     * @param lo        lower bound of lock
     * @param hi        upper bound of lock
     */
    LockTicket(int typeMask, long lo, long hi) {
        this.typeMask = typeMask;
        this.lo = lo;
        this.hi = hi;
        pos = lo;
    }

    /**
     * @return [TYPE: LO->HI @ POS[, failed]
     */
    public final String toString() {
        long p = pos;
        return "[0x" + Integer.toHexString(typeMask) + ": "
               + Fields.longToHex(lo) + "->" + Fields.longToHex(hi)
               + " @ " + (p == -1 ? "-1" : Fields.longToHex(p))
               + (failed ? ", failed]" : "]");
    }

    /**
     * For a read lock.
     */
    final InputStream getInputStream() throws IOException {
        return storage.getInputStream(lo, hi);
    }

    /**
     * For a write lock.
     */
    final OutputStream getOutputStream() throws IOException {
        return storage.getOutputStream(lo, hi);
    }

    /**
     * Called to add a LockSignal to the notification list.
     * @return  the current position value, or -1 if unlocked
     */
    synchronized long register(LockSignal sig) {
        if (pos != -1) {
            SignalTuple st = new SignalTuple(sig);
            st.next = stHead.next;
            stHead.next = st;
        }
        return pos;
    }
    
    /**
     * Called by the Lock when unlocking.
     * All registered LockSignals are notified.
     */
    void unlock() {
        // no synchronization worries here because unlock() and move()
        // are always called in a single threaded context and only they
        // change the pos value
        if (pos != -1) {
            SignalTuple st;
            synchronized (this) {
                if (pos <= hi) {
                    pos = -1;       // update the volatiles in order
                    failed = true;
                }
                else pos = -1;
                st = stHead.next;
            }
            // we need to release the monitor because signalling the unlock
            // on the LockGrantor's LockSignal grabs LockGrantor.semaphore(),
            // which leads to a deadlock since a thread can be holding that
            // semaphore while trying to grab the monitor on this LockTicket
            while (st != null) {
                st.signal.signalUnlock(this, failed);
                st = st.next;
            }
        }
    }

    /**
     * Called by the Lock when advancing its byte-pointer.
     * All registered LockSignals are notified.
     * @param n  how far to advance the byte pointer
     */
    void move(long n) throws LockException {
        // no synchronization worries here because unlock() and move()
        // are always called in a single threaded context and only they
        // change the pos value
        if (pos == -1) {
            throw new LockException("LockTicket.move(): n = " + n +
                                    ", ticket = " + this);
        }
        SignalTuple st;
        synchronized (this) {
            pos += n;
            st = stHead.next;
        }
        // although signalling the move on the LockGrantor's LockSignal
        // shouldn't grab LockGrantor.semaphore(), i'm going to release
        // the monitor anyway.  it's efficient if nothing else.
        while (st != null) {
            st.signal.signalMove(this, pos);
            st = st.next;
        }
    }
}



