package freenet.client.metadata;
import freenet.FieldSet;
import freenet.client.Request;
import freenet.client.RequestProcess;
import freenet.client.GetRequestProcess;
import freenet.client.PutRequestProcess;
import freenet.client.FreenetURI;
import freenet.Core;
import freenet.support.Logger;
import freenet.support.Bucket;
import freenet.support.BucketSink;
import freenet.support.BucketFactory;

/*
  This code is distributed under the GNU Public Licence (GPL)
  version 2.  See http://www.gnu.org/ for further details of the GPL.
*/

/**
 * Parse SplitFiles.
 * <p>
 * @author giannij
 */
public class SplitFile extends MetadataPart {
    public static final String name = "SplitFile";
    
    protected int size;
    protected int blockSize = -1;
    protected String[] blockURIs = new String[0];
    protected String[] checkBlockURIs = new String[0];
    protected FieldSet decoderParams = null;

    // REDFLAG: graph?

    public SplitFile(int size, String[] blockURIs) {
        this.size = size;
	this.blockURIs = blockURIs;
    }

    public SplitFile(int size, int blockSize,
                     String[] blockURIs, String[] checkBlockURIs, FieldSet decoderParams) {
        this.size = size;
        this.blockSize = blockSize;
	this.blockURIs = blockURIs;
	this.checkBlockURIs = checkBlockURIs;
        this.decoderParams = decoderParams;
    }

    public SplitFile(FieldSet fs) throws InvalidPartException {
	if (fs.get("Size") == null) {
	    throw new InvalidPartException(name() + ": Requires Size.");
	}
	try {
	    size = Integer.parseInt(fs.get("Size"), 16);
	}
	catch (NumberFormatException nfe) {
	    throw new InvalidPartException(name() + ": Couldn't read Size.");
	}

	if (fs.get("BlockSize") != null) {
            try {
                blockSize = Integer.parseInt(fs.get("BlockSize"), 16);
            }
            catch (NumberFormatException nfe) {
                Core.logger.log(this, "Couldn't parse BlockSize", Logger.MINOR); 
            }
	}
        else {
            Core.logger.log(this, "BlockSize not specified.", Logger.MINOR); 
        }
        

	int blockCount = 0;
	if (fs.get("BlockCount") == null) {
		throw new InvalidPartException(name() + ": Requires BlockCount.");
	}
	try {
	    blockCount = Integer.parseInt(fs.get("BlockCount"), 16);
	}
	catch (NumberFormatException nfe) {
	    throw new InvalidPartException(name() + ": Couldn't read BlockCount.");
	}
	
	// REDFLAG: graph

	// Start at 1 not 0.

	if (fs.isSet("Block")) {
	    FieldSet blocks = fs.getSet("Block");

	    blockURIs = new String[blockCount];
	    int i;
	    for (i = 0; i < blockURIs.length; i++) {
		blockURIs[i] = blocks.get(Integer.toString(i + 1, 16));
		if (blockURIs[i] == null) {
		    throw new InvalidPartException(name() + ": Couldn't read Block." + 
						   Integer.toString(i + 1, 16) + ".");
		}
	    }
	}
	else {
	    throw new InvalidPartException(name() + ": Couldn't read Blocks.");
	}

        // Handle optional check block data.
	int checkBlockCount = 0;
	if (fs.get("CheckBlockCount") != null) {
            try {
                checkBlockCount = Integer.parseInt(fs.get("CheckBlockCount"), 16);
            }
            catch (NumberFormatException nfe) {
                throw new InvalidPartException(name() + ": Couldn't read CheckBlockCount.");
            }
            
            if (fs.isSet("CheckBlock")) {
                FieldSet blocks = fs.getSet("CheckBlock");
                
                checkBlockURIs = new String[checkBlockCount];
                int i;
                for (i = 0; i < checkBlockURIs.length; i++) {
                    checkBlockURIs[i] = blocks.get(Integer.toString(i + 1, 16));
                    if (checkBlockURIs[i] == null) {
                        throw new InvalidPartException(name() + ": Couldn't read CheckBlock." + 
                                                       Integer.toString(i + 1, 16) + ".");
                    }
                }
            }
	}

        if (fs.isSet("decoder")) {
            decoderParams = fs.getSet("decoder");
        }
    }

    public void addTo(FieldSet fs) {
        FieldSet me = new FieldSet();
        me.put("Size", Integer.toString(size,16));
        if (blockSize != -1) {
            me.put("BlockSize", Integer.toString(blockSize,16));
        }
        me.put("BlockCount", Integer.toString(blockURIs.length,16));
	// REDFLAG: graph
	for (int i = 0; i < blockURIs.length; i++) {
	    // start at 1
	    me.put("Block." + Integer.toString(i + 1, 16), blockURIs[i]);
	}
        // handle optional check block data.
        if (checkBlockURIs.length > 0) {
            me.put("CheckBlockCount", Integer.toString(checkBlockURIs.length,16));
            // REDFLAG: graph
            for (int i = 0; i < checkBlockURIs.length; i++) {
                // start at 1
                me.put("CheckBlock." + Integer.toString(i + 1, 16), checkBlockURIs[i]);
            }
        }

        if (decoderParams != null) {
            me.put("decoder", decoderParams);
        }

        fs.add(name(), me);
    }

    public int getSize() { return size; }
    public int getBlockCount() { return blockURIs.length; }
    public int getCheckBlockCount() { return checkBlockURIs.length; }

    /**
     * Returns the block size in bytes or -1 if this SplitFile doesn't
     * have a fixed BlockSize.
     **/
    public int getBlockSize() { 
        if (blockSize != -1) {
            return blockSize;
        }

        // REDFLAG: HACK, remove when SplitFile spec fixed.
        Core.logger.log(this, "WARNING: Crappy SplitFile insert client didn't set BlockSize.",
                        Logger.MINOR); 
        int[] commonBlockSizes = {3145728,1048576, 524288, 262144};
        for (int i = 0; i < commonBlockSizes.length; i++) {
            int testBlocks = size / commonBlockSizes[i];
            if ((size % commonBlockSizes[i]) != 0) {
                testBlocks++;
            }
            if (testBlocks == blockURIs.length) {
                Core.logger.log(this, "Guessed BlockSize=" + commonBlockSizes[i], Logger.MINOR); 
                return commonBlockSizes[i];
            }
        }
        Core.logger.log(this, "Couldn't guess BlockSize.", Logger.MINOR); 
        return -1;
    }

    public String getDecoderName() {
        if (decoderParams == null) {
            return null;
        }
        return decoderParams.get("name");
    }

    public FieldSet getDecoderParams() {
        return decoderParams;
    }

    // non-const! copy?
    public String[] getBlockURIs() { return blockURIs; }
    public String[] getCheckBlockURIs() { return checkBlockURIs; }

    public String name() {
        return name;
    }

    public boolean isControlPart() {
        return true;
    }

    /**
     * Not implemented.
     **/
    public RequestProcess getGetProcess(FreenetURI furi, int htl, Bucket data, 
                                        BucketFactory ptBuckets, 
                                        int recursionLevel) {

	
	return null;
    }

    /**
     * Not implemented.
     **/
    public RequestProcess getPutProcess(FreenetURI furi, int htl, 
                                        String cipher, Metadata next, 
                                        Bucket data, BucketFactory ptBuckets,
                                        int recursionLevel, boolean descend) {
	return null;
    }

    public String toString() {
	StringBuffer value = new StringBuffer("SplitFile [length=" + Integer.toString(size, 16) + 
					      "] [ blocks=" + 
					      Integer.toString(blockURIs.length, 16) + 
                                              "] [ blockSize=" + 
					      Integer.toString(blockSize, 16) + 
                                              "] [ checkBlocks = " +
                                              Integer.toString(checkBlockURIs.length, 16) + "]\n");
 
	for (int i = 0; i < blockURIs.length; i++) {
	    value.append("   [" + Integer.toString(i+1, 16) + "] " + blockURIs[i] + "\n");
	}

        if (checkBlockURIs.length > 0) {
            value.append("---Check Blocks---\n");
        }
	for (int i = 0; i < checkBlockURIs.length; i++) {
	    value.append("   [" + Integer.toString(i+1, 16) + "] " + checkBlockURIs[i] + "\n");
	}

        if (decoderParams != null) {
            value.append("---Decoder Params---\n");
            value.append(decoderParams.toString() + "\n");
        }
        

        return value.toString();
    }

}






